/**
 * @file
 * @brief 設定ファイル読み込み関数定義ファイル。
 * @author tsntsumi
 * @version 1.2
 * @since v1.0 2013/12
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdarg.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <math.h>
#include "readconf.h"

size_t RCNumRisedErrors = 0;
RCError *RCRisedErrors = NULL;
size_t RCMaxNameLength = 64;
size_t RCMaxStringLength = 128;

/**
 * @brief 設定ファイルを読み込んでいる時の行番号。
 */
static size_t RCReadConfLineNo = 0;
/**
 * @brief 設定ファイル読み込み中に発生したエラーを格納する配列のサイズ。
 */
static size_t RCRisedErrorArrayCapacity = 0;
/**
 * @brief エラーの原因の文字列の最大の長さ。
 */
static const size_t RCMaxReasonLength = 256;
/**
 * @brief デリミタ文字の集合。
 */
static const char *const RCDelimiters = " \t\r";
/**
 * @brief コメント開始文字。
 */
static const char RCCommentChar = '#';

/**
 * @brief 指定された文字がデリミタかどうかを真偽値で返します。
 */
#define IS_DELIMITER(c) (strchr(RCDelimiters, c) != NULL)
/**
 * @brief 指定された文字がコメント開始文字かどうかを真偽値で返します。
 */
#define IS_COMMENT(c) (c == RCCommentChar)

static void readConf(RCConfItem *items, size_t numItems, FILE *conffp);
static void initialize(RCConfItem *items, size_t numItems);
static void readExpression(RCConfItem *items, size_t numItems, FILE *conffp);
static RCConfItem *findItem(RCConfItem *items, int numItems, char *name);
static char *readName(FILE *conffp);
static void readValue(RCConfItem *item, FILE *conffp);
static void readNL(FILE *conffp);
static void readString(RCConfItem *item, FILE *conffp);
static void readInteger(RCConfItem *item, FILE *conffp);
static void readReal(RCConfItem *item, FILE *conffp);
static char *readQuotedString(FILE *conffp);
static char *readBareString(FILE *conffp);
static void skipToEndOfLine(FILE *conffp);
static void skipWhitespaces(FILE *conffp);
static void addError(RCErrorNo errorNo, char *format, ...);


bool readconf(RCConfItem *items, size_t numItems, const char *confPath)
{
  if (numItems <= 0)
    {
      addError(RCErrorNoItem,
               "no item was specified. numItems = %d", numItems);
      return false;
    }
  if (items == NULL)
    {
      addError(RCErrorNullItem, "null items were specified.");
      return false;
    }

  initialize(items, numItems);

  FILE *conffp = fopen(confPath, "r");
  if (conffp == NULL)
    {
      addError(RCErrorOpen,
               "cannot open conf file '%s'. %s", 
               confPath, strerror(errno));
      return false;
    }

  readConf(items, numItems, conffp);

  fclose(conffp);
  return RCNumRisedErrors == 0;
}

void RCFreeStringValueInItems(RCConfItem *items, size_t numItems)
{
  for (int i = 0; i < numItems; i ++)
    {
      if (items[i].type == RCValueTypeString)
	{
	  free(items[i].value.string);
	  items[i].value.string = NULL;
          free(items[i].defaultValue.string);
          items[i].defaultValue.string = NULL;
	}
    }
}

void RCFreeRisedErrors(void)
{
  for (int i = 0; i < RCNumRisedErrors; i ++)
    {
      free(RCRisedErrors[i].reason);
    }
  free(RCRisedErrors);
  RCRisedErrors = NULL;
  RCNumRisedErrors = 0;
  RCRisedErrorArrayCapacity = 0;
}

char *RCDuplicateString(const char *str)
{
  if (str == NULL)
    {
      return NULL;
    }
  const size_t length = strlen(str);
  char *dup = calloc(1, length + 1);
  if (dup != NULL)
    {
      strcpy(dup, str);
    }
  return dup;
}

/**
 * @brief 設定ファイルのストリームから設定を読み込み、指定の構造体に格納します。
 * @param[inout] items 読み込むべき設定項目の名前とデフォルト値を設定した
 *                     構造体配列へのポインタ。
 *                     読み込んだ設定項目の値を格納するのにも使用されます。
 * @param[in] numItems 構造体配列 items の項目数。
 * @param[in] conffp 読み込む設定ファイルのストリーム。
 */
static void readConf(RCConfItem *items, size_t numItems, FILE *conffp)
{
  RCReadConfLineNo = 1;
  while (!feof(conffp))
    {
      readExpression(items, numItems, conffp);
      readNL(conffp);
    }
}

/**
 * @brief 設定ファイルを読み込む前の初期化を行います。
 * @details
 * - 行番号を 0 に設定。
 * - デフォルト値を初期値として設定。
 * - 定義行番号を 0 に設定。
 *
 * @param[inout] items 初期化する設定項目の構造体配列。
 * @param[in]    numItems 設定項目の数。
 */
static void initialize(RCConfItem *items, size_t numItems)
{
  RCConfItem *item = items;

  RCReadConfLineNo = 0;
  for (size_t i = 0; i < numItems; i ++, item ++)
    {
      if (item->type == RCValueTypeString)
        {
          if (item->defaultAsString == NULL)
            {
              item->value.string = NULL;
            }
          else
            {
              item->defaultValue.string = RCDuplicateString(item->defaultAsString);
              if (item->defaultValue.string == NULL)
                {
                  addError(RCErrorNoMemory,
                           "cannot copy default value of %s. %s",
                           item->name,
                           strerror(errno));
                }
              item->value.string = RCDuplicateString(item->defaultValue.string);
              if (item->value.string == NULL)
                {
                  addError(RCErrorNoMemory,
                           "cannot copy default value of %s. %s",
                           item->name,
                           strerror(errno));
                }
            }
        }
      else if (item->type == RCValueTypeInteger)
        {
          char *endptr = NULL;
          item->defaultValue.integer = strtol(item->defaultAsString, &endptr, 0);
          if (*endptr != '\0')
            {
              addError(RCErrorNumberFormat,
                       "default value of %s is not a number. '%c' in '%s'",
                       item->name, *endptr, item->defaultAsString);
            }
          item->value.integer = item->defaultValue.integer;
        }
      else if (item->type == RCValueTypeReal)
        {
          char *endptr = NULL;
          item->defaultValue.real = strtod(item->defaultAsString, &endptr);
          if (*endptr != '\0')
            {
              addError(RCErrorNumberFormat,
                       "default value of %s is not a number. '%c' in '%s'",
                       item->name, *endptr, item->defaultAsString);
            }
          item->value.real = item->defaultValue.real;
        }
      item->lineNo = 0;
    }
}

/**
 * @brief 設定項目の式を読み込みます。
 * @param[inout] items 読み込んだ式の値を格納する設定項目配列。
 * @param[in] numItems 設定項目配列の要素数。
 * @param[in] conffp 設定式を読み込むストリーム。
 */
static void readExpression(RCConfItem *items, size_t numItems, FILE *conffp)
{
  skipWhitespaces(conffp);
  int c = fgetc(conffp);
  ungetc(c, conffp);
  if (c == '\n')
    {
      return;
    }
  char *name = readName(conffp);
  if (name == NULL)
    {
      return;
    }
  skipWhitespaces(conffp);
  c = fgetc(conffp);
  if (c != '=')
    {
      addError(RCErrorSyntax, "syntax error. missing '='.");
      free(name);
      skipToEndOfLine(conffp);
      return;
    }
  RCConfItem *dest = findItem(items, numItems, name);
  if (dest == NULL)
    {
      addError(RCErrorUnexpectedName, "unexpected name '%s' appeared.", name);
      free(name);
      skipToEndOfLine(conffp);
      return;
    }
  if (dest->lineNo > 0)
    {
      addError(RCErrorRedefinition,
               "redefinition of '%s'. previous definition is at line %lu.",
               name, dest->lineNo);
    }
  readValue(dest, conffp);
  free(name);
}

/**
 * @brief 指定された名前の設定項目を検索します。
 * @param[in] items 検索対象の設定項目配列。
 * @param[in] numItems 検索対象の設定項目の数。
 * @param[in] name 検索する名前。
 * @return 見つかった設定項目。見つからなかったら NULL。
 */
static RCConfItem *findItem(RCConfItem *items, int numItems, char *name)
{
  int i;
  RCConfItem *p = items;

  for (i = 0; i < numItems; i ++, p ++)
    {
      if (strcasecmp(p->name, name) == 0)
        return p;
    }
  return NULL;
}

/**
 * @brief 文字列が格納されているメモリ領域を文字列の長さで切り詰めます。
 * @param str 文字列が格納されているメモリ領域。
 * @return 切り詰めた文字列。
 */
static inline char *shrinkString(char *str)
{
  char *shrinked = RCDuplicateString(str);
  free(str);
  return shrinked;
}

/**
 * @brief 設定ファイルから項目名を読み込みます。
 * @param conffp 読み込む設定ファイルストリーム。
 * @return 読み込んだ項目名。
 */
static char *readName(FILE *conffp)
{
  char *name = calloc(RCMaxNameLength, 1);
  int i = 0;
  int c;

  if (name == NULL)
    {
      addError(RCErrorNoMemory, "cannot allocate name. %s", strerror(errno));
      return NULL;
    }

  while ((c = fgetc(conffp)) != EOF)
    {
      if (c == '\n' || c == '=')
        {
          ungetc(c, conffp);
          break;
        }
      if (IS_DELIMITER(c) || IS_COMMENT(c))
        {
          break;
        }
      name[i++] = c;
      if (i > RCMaxNameLength)
        {
          name[i-1] = '\0';
          addError(RCErrorNameTooLong, 
                   "name length is longer than %lu.", RCMaxNameLength);
          skipToEndOfLine(conffp);
          free(name);
          return NULL;
        }
    }

  return shrinkString(name);
}

/**
 * @brief 指定された項目に値を設定します。
 * @param item 値を設定する項目構造体。
 * @param value 設定する値。
 */
static inline void setValue(RCConfItem *item, RCConfValue value)
{
  if (item->type == RCValueTypeString)
    {
      free(item->value.string);
      item->value.string = RCDuplicateString(item->defaultValue.string);
    }
  else
    {
      item->value = item->defaultValue;
    }
}

/**
 * @brief 設定ファイルの値を読み込みます。
 * @param[inout] item 読み込んだ値を格納する設定項目構造体。
 * @param[in] conffp 設定ファイルストリーム。
 * @return 読み込んだ設定ファイルを格納した設定項目構造体へのポインタ。
 */
static void readValue(RCConfItem *item, FILE *conffp)
{
  switch (item->type)
    {
    case RCValueTypeString:
      readString(item, conffp);
      break;
    case RCValueTypeInteger:
      readInteger(item, conffp);
      break;
    case RCValueTypeReal:
      readReal(item, conffp);
      break;
    default:
      return;
    }
  if (item->validator == NULL)
    {
      return;
    }
  if (!item->validator(item->name, item->type, item->value))
    {
      addError(RCErrorValidationFailed, "item value validation failed.");
      setValue(item, item->defaultValue);
    }
}

/**
 * @brief 改行文字を読み込みます。
 * @details 空白文字を読み飛ばした後に改行文字が現れなければ、
 *          エラーになります。
 * @param conffp 設定ファイルストリーム。
 */
static void readNL(FILE *conffp)
{
  skipWhitespaces(conffp);
  int c = fgetc(conffp);
  if (c == EOF)
    {
      return;
    }
  if (c != '\n')
    {
      ungetc(c, conffp);
      char *nextString = readBareString(conffp);
      addError(RCErrorSyntax, "unexpected string '%s' appeared.", nextString);
      free(nextString);
      skipToEndOfLine(conffp);
      return;
    }
  RCReadConfLineNo ++;
}

/**
 * @brief 文字列を読み込み、設定項目構造体に登録します。
 * @details 読み込む文字列は、引用符で囲まれている場合、
 * 両端の引用符を取り除きます。引用符で囲まれていない場合は、
 * コメント開始文字か改行文字の直前までを読み込み、
 * 末尾の空白文字を取り除きます。
 * @param[out] item 読み込んだ文字列を登録する構造体。
 * @param[in]  conffp 設定ファイルストリーム。
 */
static void readString(RCConfItem *item, FILE *conffp)
{
  int c;
  char *string = NULL;

  skipWhitespaces(conffp);
  c = fgetc(conffp);
  if (c == '"' || c == '\'')
    {
      ungetc(c, conffp);
      string = readQuotedString(conffp);
    }
  else
    {
      ungetc(c, conffp);
      string = readBareString(conffp);
    }
  if (string != NULL)
    {
      free(item->value.string);
      item->value.string = string;
      item->lineNo = RCReadConfLineNo;
    }
}

/**
 * @brief 整数値を読み込み、設定項目構造体に登録します。
 * @details 整数を構成する文字以外を読み込んだ場合は、
 * エラーにして登録は行いません。例えば「.」が含まれる場合は、
 * 整数ではなく実数を指定したということでエラーになります。
 *
 * 整数値として「0」や「0x」を接頭辞とする数値を指定できます。
 * 「0」で始まる場合は 8 進数です。そのため「8, 9」が続いた場合は、
 * エラーになります。また、「0x」で始まる場合は 16 進数です。
 * @param[out] item 読み込んだ整数値を登録する構造体。
 * @param[in]  conffp 設定ファイルストリーム。
 */
static void readInteger(RCConfItem *item, FILE *conffp)
{
  skipWhitespaces(conffp);
  char *string = readBareString(conffp);
  if (string == NULL)
    {
      return;
    }
  char *endptr = NULL;
  long n = strtol(string, &endptr, 0);
  if (*endptr != '\0')
    {
      addError(RCErrorNumberFormat,
               "not an integer char '%c' appeared in '%s'.", *endptr, string);
    }
  else
    {
      item->value.integer = n;
      item->lineNo = RCReadConfLineNo;
    }
  free(string);
}

/**
 * @brief 実数値を読み込み、設定項目構造体に登録します。
 * @details 実数を構成する文字以外を読み込んだ場合は、
 * エラーにして登録は行いません。
 *
 * 実数は次の正規表現で表すことが出来ます。
 *   - <code>[-+]?[0-9]*(\.[0-9]*([eE][-+]?[0-9]+)?)?</code>
 *
 * 例えば次のような文字列は実数値です。
 *   - <code>3.14</code>
 *   - <code>+0.</code>
 *   - <code>-.05</code>
 *   - <code>141421356e-8</code>
 *
 * @param[out] item 読み込んだ実数値を登録する構造体。
 * @param[in]  conffp 設定ファイルストリーム。
 */
static void readReal(RCConfItem *item, FILE *conffp)
{
  skipWhitespaces(conffp);
  char *string = readBareString(conffp);
  if (string == NULL)
    {
      return;
    }
  char *endptr = NULL;
  double r = strtod(string, &endptr);
  if (*endptr != '\0')
    {
      addError(RCErrorNumberFormat,
               "not a number char '%c' appeared in '%s'.", *endptr, string);
    }
  else
    {
      item->value.real = r;
      item->lineNo = RCReadConfLineNo;
    }
  free(string);
}

/**
 * @brief 引用符で囲まれた文字列を読み込みます。
 * @details 両端の引用符は取り除きます。
 *
 * 読み込む文字列の最大長は、グローバル変数
 * @link RCMaxStringLength @endlink に格納された値です。
 * この値は終端文字 <code>\\0</code> を含む長さになります。
 * 読み込む際に、この変数の長さのバッファを割り当てて読み込みに使用します。
 * 読み込んだ後に、読み込んだ長さ分の文字列領域を割り当てて、
 * バッファからコピーすることで文字列の使用メモリ容量を節約します。
 *
 * @param conffp 設定ファイルストリーム。
 * @return 読み込んだ文字列。エラーが発生した場合は NULL。
 */
static char *readQuotedString(FILE *conffp)
{
  char *string = calloc(RCMaxStringLength, 1);

  if (string == NULL)
    {
      addError(RCErrorNoMemory, "cannot allocate string. %s", strerror(errno));
      return NULL;
    }

  int i = 0;
  int quote = fgetc(conffp);
  int c;
  while ((c = fgetc(conffp)) != quote)
    {
      if (c == EOF)
        {
          addError(RCErrorSyntax, "unterminated string without %c.", quote);
          free(string);
          return NULL;
        }
      if (c == '\n')
        {
          ungetc(c, conffp);
          addError(RCErrorSyntax, "unterminated string without %c.", quote);
          free(string);
          return NULL;
        }
      string[i++] = c;
      if (i > RCMaxStringLength)
        {
          string[i-1] = '\0';
          addError(RCErrorStringTooLong, 
                   "stirng length longer than %lu.", RCMaxStringLength);
          skipToEndOfLine(conffp);
          free(string);
          return NULL;
        }
    }

  return shrinkString(string);
}

/**
 * @brief コメント開始文字か改行文字が現れるまでを文字列として読み込みます。
 * @details 読み込んだ文字列の末尾の空白文字は取り除きます。
 *
 * 読み込む文字列の最大長は、グローバル変数
 * @link RCMaxStringLength @endlink に格納された値です。
 * この値は終端文字 <code>\\0</code> を含む長さになります。
 * 読み込む際に、この変数の長さのバッファを割り当てて読み込みに使用します。
 * 読み込んだ後に、読み込んだ長さ分の文字列領域を割り当てて、
 * バッファからコピーすることで文字列の使用メモリ容量を節約します。
 *
 * @param conffp 設定ファイルストリーム。
 * @return 読み込んだ文字列。エラーが発生した場合は NULL。
 */
static char *readBareString(FILE *conffp)
{
  char *string = calloc(RCMaxStringLength, 1);
  int i = 0;
  int c;

  if (string == NULL)
    {
      addError(RCErrorNoMemory, "cannot allocate string. %s", strerror(errno));
      return NULL;
    }

  while ((c = fgetc(conffp)) != EOF)
    {
      if (c == '\n' || IS_COMMENT(c))
        {
          ungetc(c, conffp);
          break;
        }
      string[i++] = c;
      if (i > RCMaxNameLength)
        {
          string[i-1] = '\0';
          addError(RCErrorStringTooLong, 
                   "string length is longer than %lu.", RCMaxStringLength);
          skipToEndOfLine(conffp);
          free(string);
          return NULL;
        }
    }

  for (i --; i > 0; i --)
    {
      if (!IS_DELIMITER(string[i]))
        {
          break;
        }
    }

  if (i == 0)
    {
      free(string);
      return NULL;
    }

  i ++;
  string[i] = '\0';

  return shrinkString(string);
}

/**
 * @brief 改行文字の直前までを読み捨てます。
 * @param conffp 設定ファイルストリーム。
 */
static void skipToEndOfLine(FILE* conffp)
{
  int c;

  while ((c = fgetc(conffp)) != EOF)
    {
      if (c == '\n')
        {
          ungetc(c, conffp);
          break;
        }
    }
  return;
}

/**
 * @brief 連続する空白文字を読み捨てます。
 * @details コメント開始文字が現れた場合は、
 * 続くコメントを空白と見なして改行文字の前までを読み捨てます。
 * @param conffp 設定ファイルストリーム。
 */
static void skipWhitespaces(FILE *conffp)
{
  int c;

  while ((c = fgetc(conffp)) != EOF)
    {
      if (IS_COMMENT(c))
        {
          skipToEndOfLine(conffp);
          break;
        }
      if (IS_DELIMITER(c) == false)
        {
          ungetc(c, conffp);
          break;
        }
    }
}

/**
 * @brief 指定されたエラー番号と文字列を、エラー情報構造体配列に追加します。
 * @param errorNo エラー番号。
 * @param format エラー理由書式文字列。
 * @param ... format 引数
 */
static void addError(RCErrorNo errorNo, char *format, ...)
{
  if (RCNumRisedErrors >= RCRisedErrorArrayCapacity)
    {
      if (RCRisedErrorArrayCapacity == 0)
        {
          RCRisedErrorArrayCapacity = 8;
        }
      else
        {
          RCRisedErrorArrayCapacity *= 2;
        }
      RCError *errors = realloc(RCRisedErrors,
                                RCRisedErrorArrayCapacity * sizeof(RCError));
      if (errors == NULL)
        {
          return;
        }
      RCRisedErrors = errors;
    }
  RCRisedErrors[RCNumRisedErrors].errorNo = errorNo;
  RCRisedErrors[RCNumRisedErrors].lineNo = RCReadConfLineNo;
  RCRisedErrors[RCNumRisedErrors].reason = NULL;
  char *reason = calloc(RCMaxReasonLength, 1);
  if (reason != NULL)
    {
      va_list ap;
      va_start(ap, format);
      vsnprintf(reason, RCMaxReasonLength, format, ap);
      va_end(ap);
      RCRisedErrors[RCNumRisedErrors].reason = shrinkString(reason);
    }
  RCNumRisedErrors ++;
}

/*
 * Local variables: ***
 * coding: utf-8-unix ***
 * mode: C ***
 * c-file-style: "gnu" ***
 * tab-width: 8 ***
 * indent-tabs-mode: nil ***
 * End: ***
 */
