
/* --------------------------------------------- */
/*  H8-3069F File access (FAT16/32) function     */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "fat.h"

extern unsigned char eram_start;
extern unsigned char eram_end;

// -------------------------------------------
//  Macros
// -------------------------------------------
#define CLUSTER2SECTOR(X)	( ((X - 2) * BPB_SecPerClus) + FirstDataSector )

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void fat_20ms_handler(void);
int  fat_initialize(void);
int  fat_process(void);
int ui_function_fat(UI_COMMAND uicmd);

// Locals
static int ui_function_fat_debug (UI_COMMAND uicmd, int param);
static int ui_function_fat_mbr_bpb(UI_COMMAND uicmd, int param);
static int ui_function_fat_nop(UI_COMMAND uicmd, int param);

// -------------------------------------------
//  Variables
// -------------------------------------------
// Global
// Locals
static int fat_disable_timer;
static unsigned char fat_proc;
static unsigned char fat_error;
static unsigned char ui_fat_proc;

static unsigned char fat_reset_request;
static unsigned char *p_buffer_addr;

static MBR_INFORMARION MBR_info;
static BPB_INFORMARION BPB_info;

static unsigned long RootDirSectors;
static unsigned long FATSectors;
static unsigned long FirstDataSector;
static unsigned long TotalSect;
static unsigned long DataSectors;
static unsigned long CountofClusters;

enum fat_error_code {
	FAT_ERR_NONE,
	FAT_PROCESS_ERROR = 100,
};

enum fat_process_mode {
	FAT_READY_WAIT,				// 00
	FAT_READ_MBR_01,
	FAT_READ_MBR_02,
	FAT_READ_BPB_01,
	FAT_READ_BPB_02,
	FAT_IDLE = 0x80,
	FAT_RESET,
	FAT_ERROR_STOP = 0xFF,
};


static const MODULE_MENU_TABLE fat_app_table[] = {					// interval time will be ignored.
	{	 0,		0,	ui_function_fat_debug,			"  Debug         ",	},
	{	 1,		0,	ui_function_fat_mbr_bpb,		"  MBR/BPB       ",	},
	{	-1,		0,	ui_function_fat_nop,			"0123456789ABCDEF",	},
};


static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&fat_proc,			},
	{	 1,		 1,	"Error Code:     ",	&fat_error,			},
	{	 1,		 3,	"RootDirSectors: ",	&RootDirSectors,	},
	{	 1,		 3,	"FATSectors:     ",	&FATSectors,		},
	{	 1,		 3,	"FirstDataSector:",	&FirstDataSector,	},
	{	 1,		 3,	"TotalSect:      ",	&TotalSect,			},
	{	 1,		 3,	"DataSectors:    ",	&DataSectors,		},
	{	 1,		 3,	"CountofClusters:",	&CountofClusters,	},
	{	-1,		 1,	"0123456789ABCDEF", 0,					},
};

static const DEBUG_VAR_TABLE mbr_bpb_var_table[] = {
	{	 0,		 1,	"MBR.FAT type:   ",	&MBR_info.fat_type,			},
	{	 1,		 3,	"MBR.BPB sector: ",	&MBR_info.fat_BPB_sector,	},
	{	 2,		 2,	"BPB.ResvdSecCnt:",	&BPB_info.ResvdSecCnt,		},
	{	 3,		 2,	"BPB.BytsPerSec: ",	&BPB_info.BytsPerSec,		},
	{	 4,		 1,	"BPB.NumFATs:    ",	&BPB_info.NumFATs,			},
	{	 5,		 2,	"BPB.RootEntCnt: ",	&BPB_info.RootEntCnt,		},
	{	 6,		 2,	"BPB.FATSz16:    ",	&BPB_info.FATSz16,			},
	{	 7,		 3,	"BPB.FATSz32:    ",	&BPB_info.FATSz32,			},
	{	 8,		 1,	"BPB.SecPerClus: ",	&BPB_info.SecPerClus,		},
	{	-1,		 1,	"0123456789ABCDEF", 0,							},
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void fat_20ms_handler(void) {
	if(fat_disable_timer) fat_disable_timer--;
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int  fat_initialize(void) {
	fat_proc = FAT_READY_WAIT;
	fat_error = FAT_ERR_NONE;
	fat_reset_request = 1;
	fat_disable_timer = 0;
	p_buffer_addr = &eram_start;
	
	// UI related
	ui_fat_proc = 0x00;
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int fat_process(void) {
	READ10_REQ read10_req;
	
	switch( fat_proc ) {
		case FAT_READY_WAIT:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				fat_proc = FAT_READ_MBR_01;
			} else {
				fat_disable_timer = 1000/20; 		// 1000ms Wait
			}
			break;
			
		case FAT_READ_MBR_01:
			read10_req.logical_blocl_address = 0;
			read10_req.transfer_length = 1;
			if( usbms_read10( &read10_req ) == CLASS_REQ_ACCEPTED ) {
				fat_proc = FAT_READ_MBR_02;
			} else {
				fat_disable_timer = 200/20; 		// 200ms Wait
			}
			break;
			
		case FAT_READ_MBR_02:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				// **** Take in MBR information from buffer
				MBR_info.fat_type = p_buffer_addr[450];
				MBR_info.fat_BPB_sector =	p_buffer_addr[454]
										+   p_buffer_addr[455] * 0x100
										+   p_buffer_addr[456] * 0x10000
										+   p_buffer_addr[457] * 0x1000000;
										
				fat_proc = FAT_READ_BPB_01;
			}
			break;
		
		case FAT_READ_BPB_01:
			read10_req.logical_blocl_address = MBR_info.fat_BPB_sector;
			read10_req.transfer_length = 1;
			if( usbms_read10( &read10_req ) == CLASS_REQ_ACCEPTED ) {
				fat_proc = FAT_READ_BPB_02;
				
			} else {
				fat_disable_timer = 200/20; 		// 200ms Wait
			}
			break;
			
		case FAT_READ_BPB_02:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				// **** Take in BPB information from buffer
				BPB_info.BytsPerSec   = p_buffer_addr[ 11] +   p_buffer_addr[ 12] * 0x100;
				BPB_info.ResvdSecCnt  = p_buffer_addr[ 14] +   p_buffer_addr[ 15] * 0x100;
				BPB_info.RootEntCnt   = p_buffer_addr[ 17] +   p_buffer_addr[ 18] * 0x100;
				BPB_info.FATSz16      = p_buffer_addr[ 22] +   p_buffer_addr[ 23] * 0x100;
				BPB_info.NumFATs      = p_buffer_addr[ 16];
				BPB_info.SecPerClus   = p_buffer_addr[ 13];
				BPB_info.TotSec16     = p_buffer_addr[ 19] +   p_buffer_addr[ 20] * 0x100;
				BPB_info.TotSec32     = p_buffer_addr[ 32]
									  + p_buffer_addr[ 33] * 0x100
									  + p_buffer_addr[ 34] * 0x10000
									  + p_buffer_addr[ 35] * 0x1000000;
									  
				if( BPB_info.FATSz16 == 0 ) {				// AREA Only for FAT32
					BPB_info.FATSz32 = p_buffer_addr[ 36]
									+  p_buffer_addr[ 37] * 0x100
									+  p_buffer_addr[ 38] * 0x10000
									+  p_buffer_addr[ 39] * 0x1000000;
				}
				
				
				// **** Calculate file access parameters
				RootDirSectors		= ((BPB_info.RootEntCnt * 32) + (BPB_info.BytsPerSec - 1)) / BPB_info.BytsPerSec;
				FATSectors			= BPB_info.FATSz16 ? BPB_info.FATSz16 : BPB_info.FATSz32;
				FirstDataSector		= BPB_info.ResvdSecCnt + (BPB_info.NumFATs * FATSectors) + RootDirSectors;
				TotalSect			= BPB_info.TotSec16 ? BPB_info.TotSec16 : BPB_info.TotSec32;
				DataSectors			= TotalSect - FirstDataSector;
				CountofClusters		= DataSectors / BPB_info.SecPerClus;
				
				fat_proc = FAT_IDLE;
			}
			break;
		
		case FAT_IDLE:
		default:
			break;
	}
}


// -------------------------------------------
//  UI Function - FAT function
// -------------------------------------------
int ui_function_fat(UI_COMMAND uicmd) {
	static unsigned int current_index;
	static unsigned int cursol_position;
	int ret_val;
	UI_COMMAND ui_cmd_sub;

	ret_val = UI_RET_READY;
	
	switch( ui_fat_proc ) {
		case 0x00:
			// Event check
			switch( uicmd.cmd ) {
				case UI_CMD_NOP:
				case UI_CMD_INTEVAL:
					break;
					
				case UI_CMD_STARTED:
					current_index = 0;
					cursol_position = 0;
					break;
					
				case UI_CMD_KEY_PRESS_BACK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ret_val = UI_RET_QUIT;
					break;
					
				case UI_CMD_KEY_PRESS_UP:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
				case UI_CMD_KEY_PRESS_DOWN:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 0 ) {
						cursol_position = 1;
						current_index++;
					} else {
						if( fat_app_table[current_index+1].num != -1 ) current_index++;
					}
					break;
				case UI_CMD_KEY_PRESS_OK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ui_cmd_sub.cmd = UI_CMD_STARTED;
					fat_app_table[current_index].pExecFunc(ui_cmd_sub, fat_app_table[current_index].param);	// Initialize
					ui_fat_proc = 0x01;
					break;
			}
			
			// Menu view
			if( cursol_position == 0) { // cursol in first line
				sc1602_set_buffer_cursol( 0, fat_app_table[current_index  ].display );
				sc1602_set_buffer       ( 1, fat_app_table[current_index+1].display );
			} else {
				sc1602_set_buffer       ( 0, fat_app_table[current_index-1].display );
				sc1602_set_buffer_cursol( 1, fat_app_table[current_index  ].display );
			}
			break;
		
		case 0x01:
			switch( fat_app_table[current_index].pExecFunc(uicmd, fat_app_table[current_index].param) ) {
				case UI_RET_QUIT:
					ui_fat_proc = 0x00;
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Function - FAT debug
// -------------------------------------------
int ui_function_fat_debug(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_variable3 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 4:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Function - FAT debug
// -------------------------------------------
int ui_function_fat_mbr_bpb(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, mbr_bpb_var_table[current_index].display );
		switch(mbr_bpb_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, mbr_bpb_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, mbr_bpb_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_variable3 ( 1, mbr_bpb_var_table[current_index].p_variable );
				break;
			case 4:
				sc1602_set_buffer_dump (1, mbr_bpb_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( mbr_bpb_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Sub Function - USB nop
// -------------------------------------------
int ui_function_fat_nop(UI_COMMAND uicmd, int param)
{
	return UI_RET_QUIT;
}

