/*
 * CAN/LIN/UART/PORT Checker for RL78/F14
 *
 * Target: QB-R5F10PPJ-TB (RL78/F14, 100pin, 256KB ROM, 20KB RAM)
 * Author: Yasushi Tanaka
 *
 * [ CANhCo ]
 */

#include <string.h>
#include "common.h"
#include "iodefine.h"
#include "cpu.h"
#include "log.h"
#include "timer.h"
#include "can.h"

/*
 * CAN
 * Oo͒`
 */
#define CAN_LOG					(0)
									/* Oo͂ */

/*
 * CAN
 * obt@TCY`
 */
#define CAN_EVENT_BUFFER_SIZE	((u1)(0x20))
									/* Cxgobt@̌ */
#define CAN_RX_BUFFER_SIZE		((u1)(0x40))
									/* Mobt@̌ */
#define CAN_TX_BUFFER_SIZE		((u1)(0x20))
									/* Mobt@̌ */

/*
 * CAN
 * p[^ftHgl
 */
#define CAN_DEFAULT_KBPS		(500)
									/* 500kbps */
#define CAN_DEFAULT_TSEG1		(11)
									/* 16Tq, TvE|Cg75% */
#define CAN_DEFAULT_TSEG2		(4)
									/* 16Tq, TvE|Cg75% */
#define CAN_DEFAULT_SJW			(1)
									/* 16Tq, TvE|Cg75% */

/*
 * CAN
 * n[hEFA萔
 */
#define RL78F_TX_SLOT_MAX		(4)
									/* MXbg */

/*
 * SFRݒl(PER2WX^)
 */
#define RL78F_PER2_CAN0EN		((u1)(0x01))
									/* CANL */

/*
 * SFRݒl(MKmH, MKmLWX^)
 */
#define RL78F_MK2H_CANGRFRMK	((u1)(0x02))
									/* INTCANGRFR}XN */
#define RL78F_MK2H_CANGERRMK	((u1)(0x04))
									/* INTCANGERR}XN */
#define RL78F_MK2L_CAN0ERRMK	((u1)(0x20))
									/* INTCAN0ERR}XN */
#define RL78F_MK2L_CAN0WUPMK	((u1)(0x40))
									/* INTCAN0WUP}XN */
#define RL78F_MK2L_CAN0CFRMK	((u1)(0x80))
									/* INTCAN0CFR}XN */
#define RL78F_MK2H_CAN0TRMMK	((u1)(0x01))
									/* INTCAN0TRM}XN */

/*
 * SFRݒl(PRmH, PRmLWX^)
 */
#define RL78F_PR02H_CANGRFRPR0	((u1)(0x02))
									/* INTCANGRFR荞ݗD揇(bit0) */
#define RL78F_PR12H_CANGRFRPR1	((u1)(0x02))
									/* INTCANGRFR荞ݗD揇(bit1) */
#define RL78F_PR02H_CANGERRPR0	((u1)(0x04))
									/* INTCANGERR荞ݗD揇(bit0) */
#define RL78F_PR12H_CANGERRPR1	((u1)(0x04))
									/* INTCANGERR荞ݗD揇(bit1) */
#define RL78F_PR02L_CAN0ERRPR0	((u1)(0x20))
									/* INTCAN0ERR荞ݗD揇(bit0) */
#define RL78F_PR12L_CAN0ERRPR1	((u1)(0x20))
									/* INTCAN0ERR荞ݗD揇(bit1) */
#define RL78F_PR02L_CAN0WUPPR0	((u1)(0x40))
									/* INTCAN0WUP荞ݗD揇(bit0) */
#define RL78F_PR12L_CAN0WUPPR1	((u1)(0x40))
									/* INTCAN0WUP荞ݗD揇(bit1) */
#define RL78F_PR02L_CAN0CFRPR0	((u1)(0x80))
									/* INTCAN0CFR荞ݗD揇(bit0) */
#define RL78F_PR12L_CAN0CFRPR1	((u1)(0x80))
									/* INTCAN0CFR荞ݗD揇(bit1) */
#define RL78F_PR02H_CAN0TRMPR0	((u1)(0x01))
									/* INTCAN0TRM荞ݗD揇(bit0) */
#define RL78F_PR12H_CAN0TRMPR1	((u1)(0x01))
									/* INTCAN0TRM荞ݗD揇(bit1) */

/*
 * SFRݒl(CANCKSELWX^)
 */
#define RL78F_CANCKSEL_CAN0MCKE	((u1)(0x01))
									/* CANX1NbN */

/*
 * SFRݒl(GSTSLWX^)
 */
#define RL78F_GSTSL_GRSTSTS		((u1)(0x01))
									/* O[oEZbgE[h */
#define RL78F_GSTSL_GHLTSTS		((u1)(0x02))
									/* O[oEeXgE[h */
#define RL78F_GSTSL_GSLPSTS		((u1)(0x04))
									/* O[oEXgbvE[h */
#define RL78F_GSTSL_GRAMINIT	((u1)(0x08))
									/* CANpRAMNA */

/*
 * SFRݒl(GCTRLHWX^)
 */
#define RL78F_GCTRLH_MEIE		((u1)(0x02))
									/* FIFObZ[WEXg荞݋ */

/*
 * SFRݒl(GERFLLWX^)
 */
#define RL78F_GERFLL_DEF		((u1)(0x01))
									/* DLCG[ */
#define RL78F_GERFLL_MES		((u1)(0x02))
									/* FIFObZ[WEXg */
#define RL78F_GERFLL_THLES		((u1)(0x04))
									/* Mobt@EI[o[t[ */

/*
 * SFRݒl(GCTRLLWX^)
 */
#define RL78F_GCTRLL_GMDC		((u1)(0x03))
									/* O[oE[hIrbg */
#define RL78F_GCTRLL_GRST		((u1)(0x01))
									/* O[oEZbgE[h */
#define RL78F_GCTRLL_GTEST		((u1)(0x02))
									/* O[oEeXgE[h */
#define RL78F_GCTRLL_GSLPR		((u1)(0x04))
									/* O[oEXgbvE[h */

/*
 * SFRݒl(CiCTRHWX^)
 */
#define RL78F_CCTRH_CTMS		((u2)(0x0600))
									/* ʐMeXgE[hI */
#define RL78F_CCTRH_CTME		((u2)(0x0100))
									/* ʐMeXgE[h */

/*
 * SFRݒl(CiCTRHLWX^)
 */
#define RL78F_CCTRHL_TAIE		((u1)(0x01))
									/* MA{[g荞݋ */

/*
 * SFRݒl(CiCTRLHWX^)
 */
#define RL78F_CCTRLH_EWIE		((u1)(0x02))
									/* G[E[jO荞݋ */
#define RL78F_CCTRLH_EPIE		((u1)(0x04))
									/* G[EpbVu荞݋ */
#define RL78F_CCTRLH_BOEIE		((u1)(0x08))
									/* oXItJn荞݋ */
#define RL78F_CCTRLH_BORIE		((u1)(0x10))
									/* oXItA荞݋ */
#define RL78F_CCTRLH_OLIE		((u1)(0x20))
									/* I[o[hEt[M荞݋ */
#define RL78F_CCTRLH_BEIE		((u1)(0x40))
									/* oXEbN荞݋ */
#define RL78F_CCTRLH_ALIE		((u1)(0x80))
									/* A[rg[VEXg荞݋ */

/*
 * SFRݒl(CiCTRLLWX^)
 */
#define RL78F_CCTRLL_CHMDC		((u1)(0x03))
									/* `lE[hIrbg */
#define RL78F_CCTRLL_CRST		((u1)(0x01))
									/* `lEZbgE[h */
#define RL78F_CCTRLL_CWAIT		((u1)(0x02))
									/* `lҋ@[h */
#define RL78F_CCTRLL_CSLPR		((u1)(0x04))
									/* `lEXgbvE[h */

/*
 * SFRݒl(CiERFLLLWX^)
 */
#define RL78F_CERFLLL_EWF		((u1)(0x02))
									/* G[E[jOo */
#define RL78F_CERFLLL_EPF		((u1)(0x04))
									/* G[EpbVuo */
#define RL78F_CERFLLL_BOEF		((u1)(0x08))
									/* oXItJno */
#define RL78F_CERFLLL_BORF		((u1)(0x10))
									/* oXItAo */
#define RL78F_CERFLLL_OVLF		((u1)(0x20))
									/* I[o[ho */
#define RL78F_CERFLLL_BLF		((u1)(0x40))
									/* `lEoXEbNo */
#define RL78F_CERFLLL_ALF		((u1)(0x80))
									/* A[rg[VEXgo */

/*
 * SFRݒl(GCFGHWX^)
 */
#define RL78F_GCFGH_FCLK2		((u2)(0x0001))
									/* FIFOpC^[oE^C}fCLK/2œ */

/*
 * SFRݒl(GCFGLWX^)
 */
#define RL78F_GCFGL_16DIV		((u2)(0x0400))
									/* ^CEX^vfCLK/216AȂ킿1us=1JEg */

/*
 * SFRݒl(GAFLCFGLWX^)
 */
#define RL78F_GAFLCFGL_RNC0		((u1)(0x01))
									/* M[(1[) */

/*
 * SFRݒl(GRWCRLWX^)
 */
#define RL78F_GRWCRL_RULE		((u1)(0x00))
									/* EBhE0([)I */
#define RL78F_GRWCRL_RPAGE		((u1)(0x01))
									/* EBhE1(obt@)I */

/*
 * SFRݒl(GAFLIDLjWX^)
 */
#define RL78F_GAFLIDL_ZERO		((u2)(0x0000))
									/* M[ID(Low) */

/*
 * SFRݒl(GAFLIDHjWX^)
 */
#define RL78F_GAFLIDH_ZERO		((u2)(0x0000))
									/* M[ID(High) */

/*
 * SFRݒl(GAFLMLjWX^)
 */
#define RL78F_GAFLML_NOCOMP		((u2)(0x0000))
									/* IDrbgrȂ */

/*
 * SFRݒl(GAFLMHjWX^)
 */
#define RL78F_GAFLMH_NOCOMP		((u2)(0x0000))
									/* IDrbgARTRrbgrȂ */

/*
 * SFRݒl(GAFLPLjWX^)
 */
#define RL78F_GAFLPL_GAFLFDP0	((u2)(0x0001))
									/* MFIFO0Ɋi[ */

/*
 * SFRݒl(GAFLPHjWX^)
 */
#define RL78F_GAFLPH_GAFLPTR	((u2)(0x0000))
									/* Mx */

/*
 * SFRݒl(RMNBLWX^)
 */
#define RL78F_RMNBL_DISABLE		((u1)(0x00))
									/* Mobt@(0) */

/*
 * SFRݒl(RFCCmWX^)
 */
#define RL78F_RFCC_DISABLE		((u2)(0x0000))
									/* MFIFOgpȂ */
#define RL78F_RFCC_RFE			((u2)(0x0001))
									/* MFIFO */
#define RL78F_RFCC_RFIE			((u2)(0x0002))
									/* MFIFO荞݋ */
#define RL78F_RFCC_RFDC			((u2)(0x0300))
									/* MFIFO16bZ[Wi[ */
#define RL78F_RFCC_RFIGCV		((u2)(0x6000))
									/* MFIFO50%i[ꂽ犄荞ݔ */

/*
 * SFRݒl(RFSTSmLWX^)
 */
#define RL78F_RFSTSL_RFEMP		((u1)(0x01))
									/* MFIFOobt@ɖǃbZ[WȂ */
#define RL78F_RFSTSL_RFMLT		((u1)(0x04))
									/* MFIFObZ[WEXg */
#define RL78F_RFSTSL_RFIF		((u1)(0x08))
									/* MFIFO荞ݗv */

/*
 * SFRݒl(RFPCTRmLWX^)
 */
#define RL78F_RFPCTRL_RFPC		((u1)(0xff))
									/* MFIFO̖̎ǃbZ[WɈړ */

/*
 * SFRݒl(CFCCHkWX^)
 */
#define RL78F_CFCCH_DISABLE		((u2)(0x0000))
									/* MFIFO͎M[hŎgp */

/*
 * SFRݒl(CFCCLkWX^)
 */
#define RL78F_CFCCL_DISABLE		((u2)(0x0000))
									/* MFIFO͎gpA荞݂֎~ */

/*
 * SFRݒl(CFSTSkLWX^)
 */
#define RL78F_CFSTSL_CFMLT		((u1)(0x04))
									/* MFIFObZ[WEXg */

/*
 * SFRݒl(THLCCiWX^)
 */
#define RL78F_THLCC_DISABLE		((u2)(0x0000))
									/* Mobt@gpȂ */

/*
 * SFRݒl(THLSTSiLWX^)
 */
#define RL78F_THLSTSL_THLELT	((u1)(0x04))
									/* Mobt@EI[o[t[ */

/*
 * SFRݒl(CiSTSLLWX^)
 */
#define RL78F_CSTSLL_CRSTSTS	((u1)(0x01))
									/* `lEZbgE[h */
#define RL78F_CSTSLL_CHLTSTS	((u1)(0x02))
									/* `lҋ@[h */
#define RL78F_CSTSLL_CSLPSTS	((u1)(0x04))
									/* `lEXgbvE[h */
#define RL78F_CSTSLL_COMSTS		((u1)(0x80))
									/* ʐMXe[^XEtO */

/*
 * SFRݒl(TMCpWX^)
 */
#define RL78F_TMC_TMTR			((u1)(0x01))
									/* Mv */

/*
 * SFRݒl(TMSTSpWX^)
 */
#define RL78F_TMSTS_TMTRF		((u1)(0x06))
									/* Mobt@M */
#define RL78F_TMSTS_NONE		((u1)(0x00))
									/* Ȃ */
#define RL78F_TMSTS_ABORTED		((u1)(0x02))
									/* MA{[g */
#define RL78F_TMSTS_SENT		((u1)(0x04))
									/* M(MA{[gvȂ) */
#define RL78F_TMSTS_SENTABORT	((u1)(0x06))
									/* M(MA{[gv) */
#define RL78F_TMSTS_READY		((u1)(0x00))
									/* Mf[^Zbg\ */
#define RL78F_TMSTS_TMTSTS		((u1)(0x01))
									/* M(can_frame.sts̊rbgƂĎgp) */

/*
 * SFRݒl(TMIECLWX^)
 */
#define RL78F_TMIECL_ENABLE		((u1)(0x0f))
									/* Mobt@0-3荞݋ */

/*
 * SFRݒl(RFPCTRmLWX^)
 */
#define RL78F_RFPCTRL_RFPC		((u1)(0xff))
									/* MFIFO̖̎ǃbZ[WɈړ */

/*
 * CAN
 * ʐMp[^ǗGA
 */
static u2 can_kbps;
									/* CAN{[[g(ݒ\l:125/250/500/1000) */
static u1 can_tseg1;
									/* TSEG1p[^(ݒ\l:4-16) */
static u1 can_tseg2;
									/* TSEG2p[^(ݒ\l:2-8) */
static u1 can_sjw;
									/* SJWp[^(ݒ\l:1-4) */

/*
 * CAN
 * vf[^
 */
static can_stat can_count;
									/* JEgf[^ */

/*
 * CAN
 * ԊǗGA
 */
static timer_info can_time_start;
									/* ^CEX^vJn_̐Ύ */
static SADDR can_time can_time_stamp;
									/* ݂̃^CEX^v */

/*
 * CAN
 * CxgǗGA
 */
static can_event can_event_buffer[CAN_EVENT_BUFFER_SIZE];
									/* Cxgobt@ */
static u1 can_event_readp;
									/* Cxgǂݍ݃|Cg */
static u1 can_event_writep;
									/* Cxg݃|Cg */
static u1 can_event_num;
									/* CxgL */

/*
 * CAN
 * MǗGA
 */
static can_frame can_rx_buffer[CAN_RX_BUFFER_SIZE];
									/* Mobt@ */
static SADDR u1 can_rx_readp;
									/* Mobt@ǂݍ݃|Cg */
static SADDR u1 can_rx_writep;
									/* Mobt@݃|Cg */
static SADDR u1 can_rx_num;
									/* Mobt@L */

/*
 * CAN
 * MǗGA
 */
static can_frame* can_tx_buffer[CAN_TX_BUFFER_SIZE];
									/* Mobt@ */
static SADDR u1 can_tx_readp;
									/* Mobt@ǂݍ݃|Cg */
static SADDR u1 can_tx_writep;
									/* Mobt@݃|Cg */
static SADDR u1 can_tx_num;
									/* Mobt@L */
static can_frame* can_tx_ptr[RL78F_TX_SLOT_MAX];
									/* MXbg̃N */

/*
 * CAN
 * ^CEX^vXV
 *
 * 1ms^C}荞݂܂͊荞݋֎~ԂŌĂ΂
 */
void can_time_update(void)
{
	u2 tsc;

	/* ^CEX^vǂݏo */
	tsc = GTSC;

	/* ŐV̒l菬`FbN */
	if (tsc < can_time_stamp.us)
	{
		/* [vĂ邽߁Acan_time_stamp.us_64kAbv */
		can_time_stamp.us_64k++;

		/* 0܂Ŗ߂Acan_time_stamp.us_4mAbv */
		if (0 == can_time_stamp.us_64k)
		{
			can_time_stamp.us_4m++;
		}
	}

	/* ŐV̒lɍXV */
	can_time_stamp.us = tsc;
}

/*
 * CAN
 * ŐṼ^CEX^v擾
 */
static void can_time_latest(can_time* stamp)
{
	u1 psw;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* ^CEX^vXV */
	can_time_update();

	/* tB[hRs[ */
	stamp->us = can_time_stamp.us;
	stamp->us_64k = can_time_stamp.us_64k;
	stamp->us_4m = can_time_stamp.us_4m;

	/* 荞݋ */
	cpu_ei(psw);
}

/*
 * CAN
 * ߋ̃^CEX^vi[
 */
static void can_time_store(u2 tsc, can_time* stamp)
{
	u1 psw;
	u2 diff;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* ^CEX^vXV */
	can_time_update();

	/* \̂̈ꊇ */
	*stamp = can_time_stamp;

	/* 荞݋ */
	cpu_ei(psw);

	/* ŐVltscƂ̍𓾂 */
	diff = (u2)(stamp->us - tsc);

	/* stamp->us傫ꍇ́A64KJE^߂Kv */
	if (stamp->us < diff)
	{
		/* 64KJE^߂A傤0̏ꍇ4MJE^߂ */
		if (stamp->us_64k == 0)
		{
			stamp->us_4m--;
		}
		stamp->us_64k--;
	}

	/* ŐVl獷̂^CEX^vuslƂ */
	stamp->us -= diff;
}

/*
 * CAN
 * Cxg擾
 */
u1 can_event_get(void)
{
	return can_event_num;
}

/*
 * CAN
 * Cxgo
 */
void can_event_deque(can_event* ptr)
{
	u1 psw;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* Cxg0łȂꍇ */
	if (0 != can_event_num)
	{
		/* \̂̈ꊇ */
		*ptr = can_event_buffer[can_event_readp];

		/* ǂݍ݃|C^XV */
		can_event_readp++;
		if (CAN_EVENT_BUFFER_SIZE == can_event_readp)
		{
			can_event_readp = 0;
		}

		/* CxgXV */
		can_event_num--;
	}

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * CAN
 * Cxg}
 *
 * 荞݋֎~ԂŌĂ΂
 */
static void can_event_enque(u1 event)
{
	can_event *ptr;

	/* CxgőlłȂꍇ */
	if (CAN_EVENT_BUFFER_SIZE != can_event_num)
	{
		/* |C^𓾂 */
		ptr = &can_event_buffer[can_event_writep];

		/* CxgER[hݒ */
		ptr->event = event;

		/* ^CEX^vݒ */
		can_time_latest(&ptr->stamp);

		/* ݃|C^XV */
		can_event_writep++;
		if (CAN_EVENT_BUFFER_SIZE == can_event_writep)
		{
			can_event_writep = 0;
		}

		/* CxgXV */
		can_event_num++;

		/* ߋ̍ő吔𒴂Ă΍XV */
		if (can_event_num > can_count.ev_max)
		{
#if CAN_LOG != 0
			log_u1("CAN:can_count.ev_max=", can_count.ev_max);
#endif /* CAN_LOG != 0 */

			can_count.ev_max = can_event_num;
		}
	}
}

/*
 * CAN
 * `lEG[荞
 */
void can_cerr_isr(void)
{
	u1 cerfll;

	/* C0ERFLLLWX^ǂݏo */
	cerfll = C0ERFLLL;
#if CAN_LOG != 0
	log_u1("CAN:channel error int", cerfll);
#endif /* CAN_LOG != 0 */

	/* G[E[jO荞 */
	if (0 != (cerfll & RL78F_CERFLLL_EWF))
	{
#if CAN_LOG != 0
		log_msg("error warning");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.ew_cnt++;

		/* CAN_EVENT_EWFCxg} */
		can_event_enque(CAN_EVENT_EWF);
	}

	/* G[EpbVu荞 */
	if (0 != (cerfll & RL78F_CERFLLL_EPF))
	{
#if CAN_LOG != 0
		log_msg("error passive");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.ep_cnt++;

		/* CAN_EVENT_EPFCxg} */
		can_event_enque(CAN_EVENT_EPF);
	}

	/* oXItJn荞 */
	if (0 != (cerfll & RL78F_CERFLLL_BOEF))
	{
#if CAN_LOG != 0
		log_msg("enter bus off");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.boe_cnt++;

		/* CAN_EVENT_BOEFCxg} */
		can_event_enque(CAN_EVENT_BOEF);
	}

	/* oXItA荞 */
	if (0 != (cerfll & RL78F_CERFLLL_BORF))
	{
#if CAN_LOG != 0
		log_msg("restore bus off");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.bor_cnt++;

		/* CAN_EVENT_BORFCxg} */
		can_event_enque(CAN_EVENT_BORF);
	}

	/* I[o[[hMo荞 */
	if (0 != (cerfll & RL78F_CERFLLL_OVLF))
	{
#if CAN_LOG != 0
		log_msg("overload frame");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.ovl_cnt++;

		/* CAN_EVENT_OVLFCxg} */
		can_event_enque(CAN_EVENT_OVLF);
	}

	/* `lEoXEbNo荞 */
	if (0 != (cerfll & RL78F_CERFLLL_BLF))
	{
#if CAN_LOG != 0
		log_msg("channel bus lock");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.bl_cnt++;

		/* CAN_EVENT_BLFCxg} */
		can_event_enque(CAN_EVENT_BLF);
	}

	/* A[rg[VEXgo荞 */
	if (0 != (cerfll & RL78F_CERFLLL_ALF))
	{
#if CAN_LOG != 0
		log_msg("arbitration lost");
#endif /* CAN_LOG != 0 */

		/* JEgAbv */
		can_count.al_cnt++;

		/* CAN_EVENT_ALFCxg} */
		can_event_enque(CAN_EVENT_ALF);
	}

	/* ȊÕrbg܂߁A1ɂȂrbgNA */
	C0ERFLLL &= (u1)(~cerfll);
}

/*
 * CAN
 * O[oEG[荞
 */
void can_gerr_isr(void)
{
	u1 gerfl;
	u1 cfsts;
	u1 rfsts;

	/* GERFLLWX^ǂݏo */
	gerfl = GERFLL;
#if CAN_LOG != 0
	log_u1("CAN:global error int", gerfl);
#endif /* CAN_LOG != 0 */

	/* DLCG[(N肦Ȃ) */
	if (0 != (gerfl & RL78F_GERFLL_DEF))
	{
		/* DLCG[NA */
		GERFLL &= (u1)(~RL78F_GERFLL_DEF);
	}

	/* Mobt@EI[o[t[(N肦Ȃ) */
	if (0 != (gerfl & RL78F_GERFLL_THLES))
	{
		/* Mobt@EI[o[t[NA */
		THLSTS0L &= (u1)(~RL78F_THLSTSL_THLELT);
	}

	/* FIFObZ[WEXg(N肦) */
	if (0 != (gerfl & RL78F_GERFLL_MES))
	{
		/* MFIFOobt@(N肦Ȃ) */
		cfsts = CFSTS0L;
		if (0 != (cfsts & RL78F_CFSTSL_CFMLT))
		{
			CFSTS0L &= (u1)(~RL78F_CFSTSL_CFMLT);
		}

		/* MFIFOobt@1(N肦Ȃ) */
		rfsts = RFSTS1L;
		if (0 != (rfsts & RL78F_RFSTSL_RFMLT))
		{
			RFSTS1L &= (u1)(~RL78F_RFSTSL_RFMLT);
		}

		/* MFIFOobt@0(N肦) */
		rfsts = RFSTS0L;
		if (0 != (rfsts & RL78F_RFSTSL_RFMLT))
		{
			RFSTS1L &= (u1)(~RL78F_RFSTSL_RFMLT);

			/* JEgAbv */
			can_count.fifo_cnt++;

#if CAN_LOG != 0
			log_u1("rx fifo overflow=", can_count.fifo_cnt);
#endif /* CAN_LOG != 0 */

			/* CAN_EVENT_RFMLTCxg} */
			can_event_enque(CAN_EVENT_RFMLT);
		}
	}
}

/*
 * CAN
 * M擾
 */
u1 can_rx_get(void)
{
	return can_rx_num;
}

/*
 * CAN
 * Mt[擾
 */
void can_rx_deque(can_frame* ptr)
{
	u1 psw;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* M0łȂꍇ */
	if (0 != can_rx_num)
	{
		/* \̂̈ꊇ */
		*ptr = can_rx_buffer[can_rx_readp];

		/* ǂݍ݃|C^XV */
		can_rx_readp++;
		if (CAN_RX_BUFFER_SIZE == can_rx_readp)
		{
			can_rx_readp = 0;
		}

		/* MXV */
		can_rx_num--;
	}

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * CAN
 * Mt[}
 *
 * O[oM荞݃}XNԂŌĂ΂
 */
static void can_rx_enque(void)
{
	can_frame* ptr;
	volatile u1* src;
	u1 *dst;
	u1 loop;
	u2 rfts;

	/* Mobt@Lől*/
	if (can_rx_num != CAN_RX_BUFFER_SIZE)
	{
		/* Mt[CNg */
		can_count.rx_cnt[0]++;
		if (0 == can_count.rx_cnt[0])
		{
			can_count.rx_cnt[1]++;
		}

		/* |C^𓾂 */
		ptr = &can_rx_buffer[can_rx_writep];

		/* ID(Low)擾 */
		ptr->id_l = RFIDL0;

		/* ID(High)擾BgIDtOƃ[gEt[tO܂ */
		ptr->id_h = RFIDH0;

		/* DLC擾 */
		ptr->dlc = (u1)(RFPTR0H >> 4);

		/* DLCɂ炸A8oCg] */
		src = &(RFDF00L);
		dst = &(ptr->buf[0]);
		for (loop=0; loop < 8; loop++)
		{
			*dst++ = *src++;
		}

		/* M̃^CEX^v擾 */
		rfts = RFTS0;

		/* ߋ̃^CEX^vi[ */
		can_time_store(rfts, &(ptr->stamp));
#if CAN_LOG != 0
		log_u2("CAN:rx frame is enqued. time stamp=", rfts);
#endif /* CAN_LOG != 0 */

		/* ݃|C^ړ */
		can_rx_writep++;
		if (CAN_RX_BUFFER_SIZE == can_rx_writep)
		{
			can_rx_writep = 0;
		}

		/* L+1B荞݃}XNĂ邽ߔr͕KvȂ */
		can_rx_num++;

		/* ߋ̍ő吔𒴂Ă΍XV */
		if (can_rx_num > can_count.rx_max)
		{
#if CAN_LOG != 0
			log_u1("CAN:can_count.rx_max=", can_count.rx_max);
#endif /* CAN_LOG != 0 */

			can_count.rx_max = can_rx_num;
		}
	}
	else
	{
		/* Mobt@@I[o[t[ */
		can_count.ovf_cnt++;
#if CAN_LOG != 0
		log_u1("CAN:rx buffer overflow=", can_count.ovf_cnt);
#endif /* CAN_LOG */
	}
}

/*
 * CAN
 * MFIFOǂݏo
 */
static void can_rx_fifo(void)
{
	u1 psw;
	u1 rfstsl;

	/* CANO[oMFIFO荞݂}XN */
	psw = cpu_di();
	MK2H |= RL78F_MK2H_CANGRFRMK;
	cpu_ei(psw);

	/* RFSTS0Lbit01ɂȂ܂œǂݏo */
	rfstsl = RFSTS0L;

	/* MFIFOɂȂ܂Ń[v */
	while (0 == (rfstsl & RL78F_RFSTSL_RFEMP))
	{
		/* MFIFO1t[M */
		can_rx_enque();

		/* ̖ǃbZ[WɃ[hE|C^i߂ */
		RFPCTR0L = RL78F_RFPCTRL_RFPC;

		/* ēxǂݏo */
		rfstsl = RFSTS0L;
	}

	/* CANO[oMFIFO荞݂ */
	MK2H &= (u1)(~RL78F_MK2H_CANGRFRMK);
}

/*
 * CAN
 * ACh
 */
void can_idle(void)
{
	/* O[oMFIFOf[^ǂݏo */
	can_rx_fifo();
}

/*
 * CAN
 * O[oMFIFO荞
 */
void can_rx_isr(void)
{
#if CAN_LOG != 0
	log_msg("CAN:global rx fifo int");
#endif /* CAN_LOG != 0 */

	/* O[oMFIFOf[^ǂݏo */
	can_rx_fifo();

	/* ̊荞݂󂯕tł悤ARFIFNA */
	RFSTS0L &= (u1)(~RL78F_RFSTSL_RFIF);
}

/*
 * CAN
 * Mobt@瑗Mv
 *
 * `lM荞݃}XNԂŌĂ΂
 */
static void can_tx_req(u1 slot)
{
	u1 loop;
	can_frame *ptr;
	u1 *dat;
	volatile u2* tmidl;
	volatile u1* tmdf;
	volatile u1* tmc;

	/* Mobt@t[f[^ւ̃|C^𓾂 */
	ptr = can_tx_buffer[can_tx_readp];

	/* MXbg̃Nݒ */
	can_tx_ptr[slot] = ptr;

	/* TMIDLɃ|Cg */
	tmidl = &TMIDL0;
	tmidl += (slot * 8);

	/* IDf[^̉ʂZbg */
	tmidl[0] = ptr->id_l;

	/* IDf[^̏ʂZbg */
	tmidl[1] = ptr->id_h;

	/* DLCZbg(tmidl[3]Ȃ̂Œ) */
	tmidl[3] = (u2)(ptr->dlc << 12);

	/* TMDFɃ|Cg */
	tmdf = (volatile u1*)&tmidl[4];

	/* Mf[^ݒ(DLCɂ炸A8oCgݒ肷) */
	dat = &(ptr->buf[0]);
	for (loop=0; loop < 8; loop++)
	{
		*tmdf++ = *dat++;
	}

	/* TMCɃ|Cg */
	tmc = &TMC0;
	tmc += slot;

	/* TMTRrbgZbg */
	*tmc |= RL78F_TMC_TMTR;
}

/*
 * CAN
 * Mobt@ֈړ
 *
 * `lM荞݃}XNԂŌĂ΂
 */
static void can_tx_next(void)
{
	/* ǂݍ݃|Cg */
	can_tx_readp++;
	if (CAN_TX_BUFFER_SIZE == can_tx_readp)
	{
		can_tx_readp = 0;
	}

	/* Mobt@̌ */
	can_tx_num--;
}

/*
 * CAN
 * Mt[擾ёM
 */
static void can_tx_deque(void)
{
	u1 psw;
	u1 slot;
	u1 full;

	/* CAN0M荞݂}XN */
	psw = cpu_di();
	MK2H |= RL78F_MK2H_CAN0TRMMK;
	cpu_ei(psw);

	/* fullJE^ */
	full = 0;

	/* Mobt@Lȏꍇ */
	while ((0 != can_tx_num) && (RL78F_TX_SLOT_MAX != full))
	{
		/* fullJE^ */
		full = 0;

		/* MXbgԂɒׂ */
		for (slot=0; slot < RL78F_TX_SLOT_MAX; slot++)
		{
			/* NULLłΑMXbg󂢂Ă */
			if (NULL == can_tx_ptr[slot])
			{
				/* Mv */
				can_tx_req(slot);

				/* Mobt@ֈړ */
				can_tx_next();

				/* ̃t[͑Mł̂ŁAfor[vI */
				slot = RL78F_TX_SLOT_MAX;
			}
			else
			{
				/* gpȂ̂ŁAfullJE^CNg */
				full++;
			}
		}
	}

	/* CAN0M荞݂ */
	MK2H &= (u1)(~RL78F_MK2H_CAN0TRMMK);
}

/*
 * CAN
 * `lM荞
 */
void can_tx_isr(void)
{
	u1 slot;
	u1 tmtrf;
	volatile u1* tmsts;

	/* TMSTS0WX^Ƀ|Cg */
	tmsts = &TMSTS0;

	/* Mobt@ */
	for (slot=0; slot < RL78F_TX_SLOT_MAX; slot++)
	{
		/* TMTRFrbg擾 */
		tmtrf = (u1)(*tmsts & RL78F_TMSTS_TMTRF);

		/* 񂪂邩*/
		if (RL78F_TMSTS_NONE != tmtrf)
		{
			/* 񂪂̂ŁArbg𗎂Ƃ */
			*tmsts &= (u1)(~RL78F_TMSTS_TMTRF);

			/* MXbgɌѕtꂽ|C^stsZbg */
			can_tx_ptr[slot]->sts = (u1)(tmtrf | RL78F_TMSTS_TMTSTS);

			/* MXbgɌѕtꂽ|C^ */
			can_tx_ptr[slot] = NULL;

			if (RL78F_TMSTS_ABORTED == tmtrf)
			{
				/* JEgAbv */
				can_count.abt_cnt++;

#if CAN_LOG != 0
				log_u1("CAN:tx abort complete. slot=", slot);
#endif /* CAN_LOG != 0 */
			}
			if (RL78F_TMSTS_SENT == tmtrf)
			{
				/* JEgAbv */
				can_count.tx_cnt[0]++;
				if (0 == can_count.tx_cnt[0])
				{
					can_count.tx_cnt[1]++;
				}

#if CAN_LOG != 0
				log_u1("CAN:tx send complete. slot=", slot);
#endif /* CAN_LOG != 0 */
			}
			if (RL78F_TMSTS_SENTABORT == tmtrf)
			{
				/* JEgAbv */
				can_count.txabt_cnt++;
#if CAN_LOG != 0
				log_u1("CAN:tx send+abort complete. slot=", slot);
#endif /* CAN_LOG != 0 */
			}
		}

		/* TMSTSWX^ɐi߂ */
		tmsts++;
	}

	/* Mobt@玟̑Ms */
	can_tx_deque();
}

/*
 * CAN
 * Mt[}
 */
void can_tx_enque(can_frame* ptr)
{
	u1 psw;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* Mobt@tłȂꍇ */
	if (CAN_TX_BUFFER_SIZE != can_tx_num)
	{
		/* Xe[^XNA */
		ptr->sts = 0x00;

		/* |C^L */
		can_tx_buffer[can_tx_writep] = ptr;

		/* ݃|Cgi߂ */
		can_tx_writep++;
		if (CAN_TX_BUFFER_SIZE == can_tx_writep)
		{
			can_tx_writep = 0;
		}

		/* 𑝂₷ */
		can_tx_num++;

		/* ߋ̍ő吔𒴂Ă΍XV */
		if (can_tx_num > can_count.tx_max)
		{
#if CAN_LOG != 0
			log_u1("CAN:can_count.tx_max=", can_count.tx_max);
#endif /* CAN_LOG != 0 */

			can_count.tx_max = can_tx_num;
		}

		/* CANobt@瑗M */
		can_tx_deque();
	}

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * CAN
 * RtBM[V
 */
void can_config(void)
{
	/* CANʐMp[^ǗGA */
	can_kbps = CAN_DEFAULT_KBPS;
	can_tseg1 = CAN_DEFAULT_TSEG1;
	can_tseg2 = CAN_DEFAULT_TSEG2;
	can_sjw = CAN_DEFAULT_SJW;
}

/*
 * CAN
 * M[ݒ
 */
static void can_rule_init(void)
{
	/* M[(1[) */
	GAFLCFGL = RL78F_GAFLCFGL_RNC0;

	/* CANO[oRAMEBhE0([)ɐ؂ւ */
	GRWCRL = RL78F_GRWCRL_RULE;

	/* rf[^w */
	GAFLIDL0 = RL78F_GAFLIDL_ZERO;
	GAFLIDH0 = RL78F_GAFLIDH_ZERO;

	/* IDrbgrȂ */
	GAFLML0 = RL78F_GAFLML_NOCOMP;
	GAFLMH0 = RL78F_GAFLMH_NOCOMP;

	/* MFIFO0Ɋi[ */
	GAFLPL0 = RL78F_GAFLPL_GAFLFDP0;

	/* DLC`FbNȂA12bit̃xt^ */
	GAFLPH0 = RL78F_GAFLPH_GAFLPTR;

	/* CANO[oRAMEBhE1(obt@)ɐ؂ւ */
	GRWCRL = RL78F_GRWCRL_RPAGE;
}

/*
 * CAN
 * obt@ݒ
 */
static void can_buf_init(void)
{
	/* Mobt@0AgpȂ */
	RMNBL = RL78F_RMNBL_DISABLE;

	/* MFIFO10bZ[WAgpȂ */
	RFCC1 = RL78F_RFCC_DISABLE;

	/* MFIFO0bZ[WAgpȂ */
	CFCCH0 = RL78F_CFCCH_DISABLE;
	CFCCL0 = RL78F_CFCCL_DISABLE;

	/* Mobt@͎gpȂ */
	THLCC0 = RL78F_THLCC_DISABLE;

	/* MFIFO016bZ[W(RFErbg̓O[ołȂƃZbgłȂ) */
	RFCC0 = (RL78F_RFCC_RFIE | RL78F_RFCC_RFDC | RL78F_RFCC_RFIGCV);
}

/*
 * CAN
 * 
 */
void can_init(void)
{
	u1 loop;
	u1 global;
	u1 channel;
	u2 cicfgh;
	u2 cicfgl;
	u2 tq;

	/* ӃCl[uEWX^2CANLɂ */
	PER2 |= RL78F_PER2_CAN0EN;

	/* CANO[oMFIFO荞݂CANO[oEG[荞݂}XN */
	MK2H |= (RL78F_MK2H_CANGRFRMK | RL78F_MK2H_CANGERRMK);

	/* 荞ݗD揇ʂx2ɐݒ */
	PR02H |= (RL78F_PR02H_CANGRFRPR0 | RL78F_PR02H_CANGERRPR0);
	PR12H &= (u1)(~(RL78F_PR12H_CANGRFRPR1 | RL78F_PR12H_CANGERRPR1));

	/* CAN0G[荞݁ECAN0EFCNAbv荞݁ECAN0MFIFOM荞݂}XN */
	MK2L |= (RL78F_MK2L_CAN0ERRMK | RL78F_MK2L_CAN0WUPMK | RL78F_MK2L_CAN0CFRMK);

	/* 荞ݗD揇ʂx2ɐݒ */
	PR02L |= (RL78F_PR02L_CAN0ERRPR0 | RL78F_PR02L_CAN0WUPPR0 | RL78F_PR02L_CAN0CFRPR0);
	PR12L &= (u1)(~(RL78F_PR12L_CAN0ERRPR1 | RL78F_PR12L_CAN0WUPPR1 | RL78F_PR12L_CAN0CFRPR1));

	/* CAN0M荞݂}XN */
	MK2H |= RL78F_MK2H_CAN0TRMMK;

	/* 荞ݗD揇ʂx2ɐݒ */
	PR02H |= RL78F_PR02H_CAN0TRMPR0;
	PR12H &= (u1)(~RL78F_PR12H_CAN0TRMPR1);

	/* CANւX1NbNł͂ȂAfCLK/2 */
	CANCKSEL &= (u1)(~RL78F_CANCKSEL_CAN0MCKE);

	/* vf[^ */
	memset(&can_count, 0, sizeof(can_count));

	/* CxgǗGA */
	can_event_readp = 0;
	can_event_writep = 0;
	can_event_num = 0;

	/* CANMǗGA */
	can_rx_readp = 0;
	can_rx_writep = 0;
	can_rx_num = 0;

	/* CANMǗGA */
	can_tx_readp = 0;
	can_tx_writep = 0;
	can_tx_num = 0;
	for (loop=0; loop < RL78F_TX_SLOT_MAX; loop++)
	{
		can_tx_ptr[loop] = NULL;
	}

	/* CANRAMNA܂ő҂ */
	global = GSTSL;
	while (0 != (GSTSL & RL78F_GSTSL_GRAMINIT))
	{
		global = GSTSL;
	}

	/* O[oEXgbvE[hO[oEZbgE[h֑J */
	global = GCTRLL;
	global &= (u1)(~RL78F_GCTRLL_GSLPR);
	global &= (u1)(~RL78F_GCTRLL_GMDC);
	global |= RL78F_GCTRLL_GRST;
	GCTRLL = global;

	/* O[oEZbgE[hɑJڂ܂ő҂ */
	global = GSTSL;
	while (0 == (global & RL78F_GSTSL_GRSTSTS))
	{
		global = GSTSL;
	}

	/* `lEXgbvE[h`lEZbgE[h֑J */
	channel = C0CTRLL;
	channel &= (u1)(~RL78F_CCTRLL_CSLPR);
	channel &= (u1)(~RL78F_CCTRLL_CHMDC);
	channel |= RL78F_CCTRLL_CRST;
	C0CTRLL = channel;

	/* `lEZbgE[hɑJڂ܂ő҂ */
	channel = C0STSLL;
	while (0 == (channel & RL78F_CSTSLL_CRSTSTS))
	{
		channel = C0STSLL;
	}

	/* O[oE[hݒ */
	GCFGH = RL78F_GCFGH_FCLK2;
	GCFGL = RL78F_GCFGL_16DIV;

	/* TSEG1, TSEG2, SJWݒ */
	cicfgh = (u2)(can_sjw - 1);
	cicfgh <<= 8;
	cicfgh |= (u2)((can_tseg2 - 1) << 4);
	cicfgh |= (u2)(can_tseg1 - 1);
	C0CFGH = cicfgh;

	/* {[[gݒ(fCLK/2=16MHzOƂ) */
	tq = (u2)(can_tseg1 + can_tseg2 + 1);
	tq *= can_kbps;
	cicfgl = (u2)(16000 / tq);
	cicfgl--;
	C0CFGL = cicfgl;

	/* M[ݒ */
	can_rule_init();

	/* obt@ݒ */
	can_buf_init();

	/* O[oEG[荞݂́AFIFObZ[WEXĝ݋ */
	GCTRLH = RL78F_GCTRLH_MEIE;

	/* `lEG[荞݂A`lEoXEG[ׂċ */
	C0CTRLH = ( RL78F_CCTRLH_EWIE |
				RL78F_CCTRLH_EPIE |
				RL78F_CCTRLH_BOEIE |
				RL78F_CCTRLH_BORIE |
				RL78F_CCTRLH_OLIE |
				RL78F_CCTRLH_BEIE |
				RL78F_CCTRLH_ALIE);

	/* Mobt@荞݂Sċ */
	TMIECL = RL78F_TMIECL_ENABLE;

	/* M荞ݐݒ(A{[g) */
	C0CTRHL |= RL78F_CCTRHL_TAIE;
}

/*
 * CAN
 * eXgE[hݒ
 */
void can_test(u1 mode)
{
	u1 global;
	u1 channel;
	u2 cctrh;

	/* O[oEZbgE[hO[oEeXgE[h֑J */
	global = GCTRLL;
	global &= (u1)(~RL78F_GCTRLL_GMDC);
	global |= RL78F_GCTRLL_GTEST;
	GCTRLL = global;

	/* O[oEeXgE[hɑJڂ܂ő҂ */
	global = GSTSL;
	while (0 == (global & RL78F_GSTSL_GHLTSTS))
	{
		global = GSTSL;
	}

	/* `lEZbgE[h`lҋ@[h֑J */
	channel = C0CTRLL;
	channel &= (u1)(~RL78F_CCTRLL_CHMDC);
	channel |= RL78F_CCTRLL_CWAIT;
	C0CTRLL = channel;

	/* `lҋ@[hɑJڂ܂ő҂ */
	channel = C0STSLL;
	while (0 == (channel & RL78F_CSTSLL_CHLTSTS))
	{
		channel = C0STSLL;
	}

	/* ʐMeXgE[hݒ */
	cctrh = C0CTRH;
	cctrh &= (u2)(~RL78F_CCTRH_CTMS);
	cctrh |= (u2)(mode << 9);
	cctrh |= RL78F_CCTRH_CTME;
	C0CTRH = cctrh;
}

/*
 * CAN
 * Jn
 */
void can_start(void)
{
	u1 global;
	u1 channel;
	u1 other;

	/* O[oEZbgE[h or O[oEeXgE[h */
	/* O[o샂[h֑J */
	GCTRLL &= (u1)(~RL78F_GCTRLL_GMDC);

	/* O[o샂[hɑJڂ܂ő҂ */
	global = GSTS;
	other = ( RL78F_GSTSL_GRSTSTS | RL78F_GSTSL_GHLTSTS | RL78F_GSTSL_GSLPSTS );
	while (0 != (global & other))
	{
		global = GSTS;
	}

	/* Ń^CEX^vX^[ĝŁA^C}̌ݎԂL */
	timer_getinfo(&can_time_start);

	/* `lEZbgE[h or `lҋ@[h */
	/* `lʐM[h֑J */
	C0CTRLL &= (u1)(~RL78F_CCTRLL_CHMDC);

	/* `lʐM[hɑJڂ܂ő҂ */
	channel = C0STSLL;
	other = (RL78F_CSTSLL_CRSTSTS | RL78F_CSTSLL_CHLTSTS | RL78F_CSTSLL_CSLPSTS);
	while (0 != (channel & other))
	{
		channel = C0STSLL;
	}

	/* CAN0M荞݂ */
	MK2H &= (u1)(~RL78F_MK2H_CAN0TRMMK);

	/* CAN0G[荞݂ */
	MK2L &= (u1)(~RL78F_MK2L_CAN0ERRMK);

	/* CANO[oMFIFO荞݂CANO[oEG[荞݂ */
	MK2H &= (u1)(~(RL78F_MK2H_CANGRFRMK | RL78F_MK2H_CANGERRMK));

	/* MFIFOobt@L */
	RFCC0 |= RL78F_RFCC_RFE;
}
