/***************************************************************************/
/** @file       font.cpp
    @brief      
    @author     shom
    @internal
****************************************************************************/

#include "pch_core.h"

#include "font.h"

#include "core/graphic/2d/image_board.h"


using namespace font_def;


/***************************************************************************
	cfont_character_base
****************************************************************************/

cfont_character_base::cfont_character_base()
:
m_p_iboard_font( CREATE_SPTR( cimage_board ) )//,
{
}

void	cfont_character_base::Initialize( 
	crender* in_p_render,
	const TCHAR* sz_path_texture
	)
{
	m_p_iboard_font->Initialize( in_p_render, sz_path_texture );
}

void	cfont_character_base::Finalize()
{
	m_p_iboard_font->Finalize();
}

void	cfont_character_base::SetContext( const font_context_t& in_context )
{
	m_p_iboard_font->SetSize( in_context.v2_size );
	m_p_iboard_font->SetColor( in_context.color );
}

void	cfont_character_base::Draw( crender* in_p_render, u32 in_character, const vec2f& in_v2_pos )
{
	///--
	vec2f v2_uv_start, v2_uv_end;
	calc_texture_uv( in_character, v2_uv_start, v2_uv_end );
	m_p_iboard_font->SetUv( v2_uv_start, v2_uv_end );

	m_p_iboard_font->SetPos( in_v2_pos );

	m_p_iboard_font->Update();
	///--

	///--
	m_p_iboard_font->Draw( in_p_render );
	///--
}


/***************************************************************************
	cfont_character_ascii
****************************************************************************/

cfont_character_ascii::cfont_character_ascii()
{
}

void	cfont_character_ascii::Initialize( crender* in_p_render )
{
	cfont_character_base::Initialize( in_p_render, _T( "texture/font/font_ascii.png" ) );
}

/*------------------------------------------------------------------------*/
namespace
{
	const u32 sc_nof_chara_on_column_in_texture_ascii = 16;
	const u32 sc_nof_chara_on_row_in_texture_ascii = 16;
} 
/*------------------------------------------------------------------------*/
void	cfont_character_ascii::calc_texture_uv(
	u32 in_character,
	vec2f& out_v2_start, vec2f& out_v2_end
	)
{
	const u32 idx_u = in_character % sc_nof_chara_on_column_in_texture_ascii;
	const u32 idx_v = in_character / sc_nof_chara_on_column_in_texture_ascii;

	const f32 u_start = SCAST<f32>( idx_u     ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_ascii );
	const f32 v_start = SCAST<f32>( idx_v     ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_ascii );
	const f32 u_end   = SCAST<f32>( idx_u + 1 ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_ascii );
	const f32 v_end   = SCAST<f32>( idx_v + 1 ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_ascii );

	out_v2_start.x = u_start;
	out_v2_start.y = v_start;
	out_v2_end.x = u_end;
	out_v2_end.y = v_end;
}


/***************************************************************************
	cfont_character_sjis
****************************************************************************/

cfont_character_sjis::cfont_character_sjis()
{
}

void	cfont_character_sjis::Initialize( crender* in_p_render )
{
	cfont_character_base::Initialize( in_p_render, _T( "texture/font/hiragana_katakana.png" ) );
}

/*------------------------------------------------------------------------*/
namespace
{
	const u32 sc_character_a_hira = 0x829f/**/;
	const u32 sc_character_a_kata = 0x8340/*@*/;	

	const u32 sc_nof_chara_on_column_in_texture_sjis = 16;
	const u32 sc_nof_chara_on_row_in_texture_sjis = 16;

	const u32 sc_character_onbiki = 0x815b/*[*/;
	const u32 idx_onbiki =
		sc_nof_chara_on_column_in_texture_sjis * sc_nof_chara_on_row_in_texture_sjis - 1;
} 
/*------------------------------------------------------------------------*/
void	cfont_character_sjis::calc_texture_uv(
	u32 in_character,
	vec2f& out_v2_start, vec2f& out_v2_end
	)
{
	const b32 b_onbiki = ( in_character == sc_character_onbiki );
	const b32 b_hiragana = in_character < sc_character_a_kata;
	const u32 idx_offset =
		( b_onbiki ) ? idx_onbiki :
		  ( in_character -
		    ( b_hiragana ? sc_character_a_hira : sc_character_a_kata )
			);

	const u32 idx_u = idx_offset % sc_nof_chara_on_column_in_texture_sjis;
	const u32 idx_v = idx_offset / sc_nof_chara_on_column_in_texture_sjis + ( b_hiragana ? 0 : 6 );

	const f32 u_start = SCAST<f32>( idx_u     ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_sjis );
	const f32 v_start = SCAST<f32>( idx_v     ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_sjis );
	const f32 u_end   = SCAST<f32>( idx_u + 1 ) / SCAST<f32>( sc_nof_chara_on_column_in_texture_sjis );
	const f32 v_end   = SCAST<f32>( idx_v + 1 ) / SCAST<f32>( sc_nof_chara_on_row_in_texture_sjis );

	out_v2_start.x = u_start;
	out_v2_start.y = v_start;
	out_v2_end.x = u_end;
	out_v2_end.y = v_end;
}


/***************************************************************************
	cfont
****************************************************************************/

cfont::cfont()
:
m_p_character_ascii( CREATE_SPTR( cfont_character_ascii ) ),
m_p_character_sjis( CREATE_SPTR( cfont_character_sjis ) )//,
{
}

void	cfont::Initialize( crender* in_p_render )
{
	m_p_character_ascii->Initialize( in_p_render );
	m_p_character_sjis->Initialize( in_p_render );
}

void	cfont::Finalize()
{
	m_p_character_ascii->Finalize();
	m_p_character_sjis->Finalize();
}

void	cfont::Draw( crender* in_p_render, const char* const in_sz_msg, const font_context_t& in_context )
{
	m_p_character_ascii->SetContext( in_context );
	m_p_character_sjis->SetContext( in_context );

	u32 idx = 0;
	u32 nof_char = 0;
	while( in_sz_msg[ idx ] != '\0' )
	{
		//--
		const f32 pos_x_offset = in_context.v2_size.x * SCAST<f32>( nof_char );
		const f32 pos_x = in_context.v2_pos.x + pos_x_offset;

		const vec2f	v2_pos( pos_x, in_context.v2_pos.y );
		//--

		if( ( in_sz_msg[ idx ] & 0xff ) < 0x80 )
			//ascii
		{
			m_p_character_ascii->Draw( in_p_render, ( in_sz_msg[ idx ] & 0xff ), v2_pos );

			idx += 1;
		}
		else
			//sjis
		{			
			const u32 character =
				( ( in_sz_msg[ idx ] & 0xff ) << 8 ) + ( in_sz_msg[ idx + 1 ] & 0xff );
	
			m_p_character_sjis->Draw( in_p_render, character, v2_pos );
	
			idx += 2;
		}

		++nof_char;
	}
}
