package kisscelltopng.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ResourceBundle;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;

import kisscelltopng.logic.DataConverter;
import kisscelltopng.logic.DataConverterFactory;
import kisscelltopng.ui.CellsTablePane.MenuItemCallback;
import kisscelltopng.ui.model.CellsTableItem;
import kisscelltopng.ui.model.CellsTablePaneModel;
import kisscelltopng.ui.model.DirChoosePanelModel;
import kisscelltopng.ui.model.DirChoosePanelModel.SourceType;


public class MainFrame extends JFrame {
	private static final long serialVersionUID = 1L;
	
	private DataConverter dataConverter;
	
	private DirChoosePanel dirChoosePanel = new DirChoosePanel();

	private CellsTablePane cellsTablePane = new CellsTablePane();
	
	private CellPreviewPane cellPreviewPane = new CellPreviewPane();


	public MainFrame() {
		setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				onClose();
			}
		});
		initComponent();
		setSize(750, 550);
		setLocationByPlatform(true);
	}
	
	private AbstractAction actOpenArchive = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onOpenArchive();
		}
	};

	private AbstractAction actSaveCells = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onSaveCells();
		}
	};
	
	private AbstractAction actSelectAll = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onSelectAll();
		}
	};
	
	private void initComponent() {
		ResourceBundle res = ResourceBundle.getBundle(getClass().getName());

		setTitle(res.getString("window.caption"));
		
		JMenuBar menubar = new JMenuBar();
		
		JMenu mnuFile = new JMenu(res.getString("menu.file.caption"));
		mnuFile.setMnemonic(res.getString("menu.file.mnemonic").charAt(0));
		menubar.add(mnuFile);
		
		JMenuItem mnuLoadArchive = new JMenuItem(actOpenArchive);
		actOpenArchive.putValue(Action.NAME, res.getString("menu.file.loadArchive.caption"));
		mnuLoadArchive.setMnemonic(res.getString("menu.file.loadArchive.mnemonic").charAt(0));
		mnuFile.add(mnuLoadArchive);

		actSaveCells.putValue(Action.NAME, res.getString("menu.file.saveCells.caption"));
		JMenuItem mnuSaveCells = new JMenuItem(actSaveCells);
		mnuSaveCells.setMnemonic(res.getString("menu.file.saveCells.mnemonic").charAt(0));
		mnuFile.add(mnuSaveCells);

		JMenuItem mnuExit = new JMenuItem(res.getString("menu.file.exit.caption"));
		mnuExit.setMnemonic(res.getString("menu.file.exit.mnemonic").charAt(0));
		mnuExit.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				onClose();
			}
		});
		mnuFile.add(mnuExit);
		
		JMenu mnuEdit = new JMenu(res.getString("menu.edit.caption"));
		mnuEdit.setMnemonic(res.getString("menu.edit.mnemonic").charAt(0));
		menubar.add(mnuEdit);
		
		final JCheckBoxMenuItem mnuSelectAll = new JCheckBoxMenuItem();
		actSelectAll.putValue(Action.NAME, res.getString("menu.edit.selectAll.caption"));
		mnuSelectAll.setAction(actSelectAll);
		mnuSelectAll.setMnemonic(res.getString("menu.edit.selectAll.mnemonic").charAt(0));
		mnuEdit.add(mnuSelectAll);
		
		mnuEdit.addMenuListener(new MenuListener() {
			public void menuSelected(MenuEvent e) {
				CellsTablePaneModel model = cellsTablePane.getModel();
				mnuSelectAll.setSelected(model.isAllItemSelected());
			}
			public void menuDeselected(MenuEvent e) {
				// 何もしない.
			}
			public void menuCanceled(MenuEvent e) {
				// 何もしない.
			}
		});
		
		JMenu mnuView = new JMenu(res.getString("menu.view.caption"));
		mnuView.setMnemonic(res.getString("menu.view.mnemonic").charAt(0));
		menubar.add(mnuView);

		final JMenu mnuSort = new JMenu(res.getString("menu.view.sort.caption"));
		mnuSort.setMnemonic(res.getString("menu.view.sort.mnemonic").charAt(0));
		mnuView.add(mnuSort);
		
		mnuView.addMenuListener(new MenuListener() {
			public void menuSelected(MenuEvent e) {
				Component[] comps = mnuSort.getMenuComponents();
				for (Component comp : comps) {
					mnuSort.remove(comp);
				}
				cellsTablePane.addSortMenuItem(new MenuItemCallback() {
					public void addMenuItem(JMenuItem menuItem) {
						mnuSort.add(menuItem);
					}
				});
			}
			public void menuDeselected(MenuEvent e) {
				// 何もしない.
			}
			public void menuCanceled(MenuEvent e) {
				// 何もしない.
			}
		});

		JMenu mnuHelp = new JMenu(res.getString("menu.help.caption"));
		mnuHelp.setMnemonic(res.getString("menu.help.mnemonic").charAt(0));
		menubar.add(mnuHelp);

		JMenuItem mnuBrowseHelp = new JMenuItem(res.getString("menu.help.browseHelp.caption"));
		mnuBrowseHelp.setMnemonic(res.getString("menu.help.browseHelp.mnemonic").charAt(0));
		mnuBrowseHelp.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				onBrowseHelp();
			}
		});
		mnuHelp.add(mnuBrowseHelp);

		JMenuItem mnuAbout = new JMenuItem(res.getString("menu.help.aboutThis.caption"));
		mnuAbout.setMnemonic(res.getString("menu.help.aboutThis.mnemonic").charAt(0));
		mnuAbout.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				onAbout();
			}
		});
		mnuHelp.add(mnuAbout);

		setJMenuBar(menubar);

		Container contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());
		
		contentPane.add(dirChoosePanel, BorderLayout.NORTH);
		
		dirChoosePanel.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				String cmd = e.getActionCommand();
				if (DirChoosePanel.ACTION_LOAD.equals(cmd)) {
					onLoad();

				} else if (DirChoosePanel.ACTION_RESET.equals(cmd)) {
					onReset();
				}
			}
		});

		JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, cellsTablePane, cellPreviewPane);
		splitPane.resetToPreferredSizes();
		
		contentPane.add(splitPane, BorderLayout.CENTER);
		
		cellsTablePane.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				String cmd = e.getActionCommand();
				if (CellsTablePane.ACTION_SELCHANGE.equals(cmd)) {
					onSelChange(cellsTablePane.getSelectedItem());
				} else if (CellsTablePane.ACTION_SAVE.equals(cmd)) {
					onSaveCells();
				}
			}
		});
		
		dirChoosePanel.getModel().addPropertyChangeListener(
				DirChoosePanelModel.PROP_DEST_DIR,
				new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				onChangeDestDir();
			}
		});

		updateSaveButton();
	}
	
	public void onClose() {
		try {
			if (dataConverter != null) {
				dataConverter.release(null);
				dataConverter = null;
			}

		} catch (IOException ex) {
			// クローズ中に失敗はログに書くのみで継続する.
			ex.printStackTrace();
		}
		dispose();
	}
	
	protected void onBrowseHelp() {
		// TODO:
		System.out.println("help");
	}
	
	public void onAbout() {
		// TODO:
		System.out.println("about");
	}
	
	protected void onLoad() {
		try {
			DirChoosePanelModel dirModel = dirChoosePanel.getModel();
			SourceType sourceType = dirModel.getSourceType();
			File src = dirModel.getSourceFile();
			if (src == null || !src.exists()) {
				Toolkit tk = Toolkit.getDefaultToolkit();
				tk.beep();
				return;
			}

			int numOfCells;
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try {
				if (dataConverter != null) {
					dataConverter.release(null);
				}

				// アーカイブを解析する.
				DataConverterFactory factory = new DataConverterFactory();
				dataConverter = factory.createDataConverter(sourceType);
				dataConverter.load(src);
				
				// テーブルモデルに設定する.
				CellsTablePaneModel tableModel = cellsTablePane.getModel();
				dataConverter.updateCellsTablePaneModel(tableModel);
				numOfCells = tableModel.getRowCount();
				
				// 列を再設定する.(パレットの有無により表示列数が異なるため)
				cellsTablePane.setupColumn();

			} finally {
				setCursor(Cursor.getDefaultCursor());
			}

			// 出力先フォルダ候補のリフレッシュ
			onChangeDestDir();

			// ボタンのリフレッシュ
			updateSaveButton();
			
			if (numOfCells <= 0) {
				// セル数がゼロの場合はメッセージを表示する.
				ResourceBundle res = ResourceBundle.getBundle(getClass().getName());
				JOptionPane.showMessageDialog(this, res.getString("message.no_data"));

			} else {
				// 成功し、セルが1つ以上ある場合のみ
				dirModel.setLock(true);
			}
			
		} catch (Exception ex) {
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
		}
	}
	
	protected void reset() throws IOException {
		DirChoosePanelModel model = dirChoosePanel.getModel();
		if (dataConverter != null) {
			dataConverter.release(cellsTablePane.getModel());
			dataConverter = null;
		}
		model.setLock(false);
	}
	
	protected void onReset() {
		try {
			reset();
			updateSaveButton();

		} catch (Exception ex) {
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
		}
	}
	
	protected void onSelChange(CellsTableItem item) {
		BufferedImage img = null;
		try {
			if (dataConverter != null) {
				img = dataConverter.getImage(item);
			}

		} catch (Exception ex) {
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
		}
		cellPreviewPane.setImage(img);

		updateSaveButton();
	}
	
	protected void onChangeDestDir() {
		File destDir = dirChoosePanel.getModel().getDestDir();
		CellsTablePaneModel tableModel = cellsTablePane.getModel();
		tableModel.setDestDir(destDir);
		
		updateSaveButton();
	}
	
	protected void updateSaveButton() {
		boolean enabled = canSave();
		actSaveCells.setEnabled(enabled);
		cellsTablePane.setEnabledSave(enabled);
	}
	
	protected void onSelectAll() {
		CellsTablePaneModel model = cellsTablePane.getModel();
		model.selectAll(!model.isAllItemSelected());
	}
	
	protected void onOpenArchive() {
		try {
			if (dirChoosePanel.browseSourceFile()) {
				reset();
				onLoad();
			}

		} catch (Exception ex) {
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
		}
	}
	
	protected boolean canSave() {
		DirChoosePanelModel dirModel = dirChoosePanel.getModel();
		File destDir = dirModel.getDestDir();
		if (destDir == null || destDir.isFile()) {
			return false;
		}

		int count = 0;
		CellsTablePaneModel tableModel = cellsTablePane.getModel();
		final int mx = tableModel.getRowCount();
		for (int idx = 0; idx < mx; idx++) {
			CellsTableItem item = tableModel.get(idx);
			if (item.isSelected()) {
				count++;
				break; // 1つ以上あれば十分なので検査は打ち切る.
			}
		}
		return count > 0;
	}

	/**
	 * セル名の重複をチェックする.<br>
	 * 同一セル名があるファイル名の一覧が返される.<br>
	 * 重複がなければ戻り値は空のリストとなる.<br>
	 * @return 同一セル名のリスト、なければ空
	 */
	protected List<File> checkCellNames() {
		CellsTablePaneModel tableModel = cellsTablePane.getModel();
		final int mx = tableModel.getRowCount();
		HashMap<File, Integer> cnts = new HashMap<File, Integer>();
		ArrayList<File> overs = new ArrayList<File>();
		for (int idx = 0; idx < mx; idx++) {
			CellsTableItem item = tableModel.get(idx);
			if ( !item.isSelected()) {
				continue;
			}
			File cellName = new File(item.getCellName());
			Integer cnt = cnts.get(cellName);
			if (cnt == null) {
				cnt = Integer.valueOf(1);
			} else {
				cnt = Integer.valueOf(cnt.intValue() + 1);
			}
			if (cnt.intValue() > 1) {
				overs.add(cellName);
			}
			cnts.put(cellName, cnt);
		}
		return overs;
	}
	
	protected void onSaveCells() {
		// 出力先、出力対象のチェックがあるか判定する.
		if ( !canSave()) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			tk.beep();
			return;
		}
		
		try {
			ResourceBundle res = ResourceBundle.getBundle(getClass().getName());

			// 同一のセル名が指定されていないか検査する.
			List<File> dupCellNames = checkCellNames();
			if ( !dupCellNames.isEmpty()) {
				StringBuilder buf = new StringBuilder();
				buf.append(res.getString("message.error.duplicate_cell_names")); 
				if (dupCellNames.size() <= 1) {
					buf.append(System.getProperty("line.separator"));
					buf.append(dupCellNames.get(0).toString());
				}
				JOptionPane.showMessageDialog(this, buf.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
				return;
			}

			// モデルの取得
			DirChoosePanelModel dirModel = dirChoosePanel.getModel();
			File destDir = dirModel.getDestDir();
			CellsTablePaneModel tableModel = cellsTablePane.getModel();
			
			// 出力先が存在しなければ作成するか問い合わせる.
			boolean checkDir = false;
			if ( !destDir.exists()) {
				int ret = JOptionPane.showConfirmDialog(this,
						res.getString("message.confirm.create_directory"),
						"CONFIRM", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
				if (ret != JOptionPane.YES_OPTION) {
					return;
				}
				checkDir = destDir.mkdirs();
			
			} else {
				checkDir = destDir.isDirectory();
			}

			// 出力先がディレクトリでなければエラー
			if ( !checkDir) {
				JOptionPane.showMessageDialog(this,
						res.getString("message.error.not_directory"),
						"ERROR", JOptionPane.ERROR_MESSAGE);
				return;
			}
			
			// 出力処理.
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try {
				dataConverter.save(destDir, tableModel);

			} finally {
				setCursor(Cursor.getDefaultCursor());
			}

			// 完了メッセージ
			JOptionPane.showMessageDialog(this, res.getString("message.save_done"));

		} catch (Exception ex) {
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.toString(), "ERROR", JOptionPane.ERROR_MESSAGE);
		}
	}
}
