package kisscelltopng.logic;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import charactermanaj.model.io.ArchivedEntry;
import charactermanaj.model.io.ArchivedFile;
import charactermanaj.model.io.ArchivedFileFactory;

import kisscelltopng.kiss.io.KiSSCellWriter;
import kisscelltopng.ui.model.CellsTableItem;
import kisscelltopng.ui.model.CellsTablePaneModel;

public class PNGDataConverter extends AbstractDataConverter {

	private ArchivedFile<? extends ArchivedEntry> archivedFile;

	private List<String> pictureFiles = Collections.emptyList();
	
	private Caches<BufferedImage> caches = new Caches<BufferedImage>();
	
	private ImageReader imageReader;
	
	private KiSSCellWriter cellWriter = new KiSSCellWriter();
	
	public void load(File file) throws IOException {
		if (this.archivedFile != null) {
			throw new IllegalStateException("already opened.");
		}
		if (file == null) {
			throw new IllegalArgumentException();
		}
		this.archivedFile = ArchivedFileFactory.getArchivedFile(file);

		ArrayList<String> pictureFiles = new ArrayList<String>();
		for (ArchivedEntry entry : archivedFile) {
			if ( !entry.isDirectory()) {
				String fileName = entry.getName();
				String lcName = fileName.toLowerCase();
				if (lcName.endsWith(".png")) {
					pictureFiles.add(fileName);
				}
			}
		}
		this.pictureFiles = pictureFiles;
		caches.clear();
	}

	public void release(CellsTablePaneModel model) throws IOException {
		if (imageReader != null) {
			imageReader.dispose();
			imageReader = null;
		}
		if (archivedFile != null) {
			archivedFile.close();
			archivedFile = null;
		}
		if (model != null) {
			model.clear();
		}
		pictureFiles.clear();
		caches.clear();
	}

	public void updateCellsTablePaneModel(CellsTablePaneModel model) throws IOException {
		if (model == null) {
			throw new IllegalArgumentException();
		}
		if (archivedFile == null) {
			throw new IllegalStateException("archive is not opened.");
		}
		
		ImageReader imageReader = getImageReader();

		ArrayList<CellsTableItem> items = new ArrayList<CellsTableItem>();
	
		for (String fileName : pictureFiles) {
			ArchivedEntry entry = archivedFile.getEntry(fileName);
			if (entry == null) {
				continue;
			}

			CellsTableItem item = new CellsTableItem();
			item.setFileName(fileName);
	
			String cellName;
			int st = fileName.lastIndexOf('/');
			if (st < 0) {
				cellName = fileName;
			} else {
				cellName = fileName.substring(st + 1);
			}
			int en = cellName.lastIndexOf('.');
			if (en > 0) {
				cellName = cellName.substring(0, en);
			}
			
			item.setCellName(cellName);
			item.setSelected(true);
			item.setPaletteGroup(0);

			item.setIndexedColor(false);
			
			int w, h;
			InputStream is = new BufferedInputStream(archivedFile.getInputStream(entry));
			try {
				ImageInputStream iis = ImageIO.createImageInputStream(is);
				try {
					imageReader.setInput(iis);
					w = imageReader.getWidth(0);
					h = imageReader.getHeight(0);
					imageReader.abort();
					
				} finally {
					iis.close();
				}
			} finally {
				is.close();
			}
			
			item.setSize(new Dimension(w, h));
			
			items.add(item);
		}
		
		Collections.sort(items);
		
		model.clear();
		model.add(items);

		imageReader.reset();
	}

	public BufferedImage getImage(CellsTableItem item) throws IOException {
		if (item == null) {
			return null;
		}
		String fileName = item.getFileName();
		BufferedImage image = caches.get(fileName);
		if (image == null) {
			image = loadPicture(fileName);
			caches.put(fileName, image);
		}
		return image;
	}
	
	protected BufferedImage loadPicture(String fileName) throws IOException {
		ArchivedEntry entry = archivedFile.getEntry(fileName);
		if (entry == null) {
			return null;
		}
		
		ImageReader imageReader = getImageReader();
		BufferedImage image;
		
		InputStream is = new BufferedInputStream(archivedFile.getInputStream(entry));
		try {
			ImageInputStream iis = ImageIO.createImageInputStream(is);
			try {
				imageReader.setInput(iis);
				image = imageReader.read(0);
				
			} finally {
				iis.close();
			}
		} finally {
			is.close();
		}
		imageReader.reset();
		
		return convertIntARGB(image);
	}

	/**
	 * 32ビット透過ARGB形式に補正して返す.<br>
	 * @param img 元画像
	 * @return 32ビット透過ARGB形式
	 */
	protected BufferedImage convertIntARGB(BufferedImage img) {
		if (img == null) {
			return null;
		}

		int width = img.getWidth();
		int height = img.getHeight();
		BufferedImage cnv = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		
		Graphics2D g = cnv.createGraphics();
		try {
			g.drawImage(img, 0, 0, width, height, null);
			
		} finally {
			g.dispose();
		}
		
		return cnv;
	}
	
	protected ImageReader getImageReader() {
		ImageReader imageReader = this.imageReader;
		if (imageReader == null) {
			Iterator<ImageReader> ite = ImageIO.getImageReadersBySuffix("png");
			imageReader = ite.next();
			this.imageReader = imageReader;
		}
		return imageReader;
	}
	
	@Override
	protected void writeImage(File outDir, String cellName, BufferedImage img)
			throws IOException {
		File outFile = new File(outDir, cellName + ".cel");
		cellWriter.write(outFile, img);
	}
}
