package littlemaid.entity;

import java.util.Random;
import java.util.UUID;

import littlemaid.LittleMaidCore;
import littlemaid.entity.ai.EntityAIAttackOnCollide;
import littlemaid.entity.ai.EntityAIFollowOwner;
import littlemaid.entity.ai.EntityAILanding;
import littlemaid.entity.ai.EntityAIPutTorch;
import littlemaid.entity.ai.EntityAIRecover;
import littlemaid.entity.ai.EntityAIWait;
import littlemaid.gui.GuiHandlerLittleMaid;
import littlemaid.inventory.InventoryLittleMaid;
import littlemaid.network.MessageSyncEquip;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityCreature;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.IEntityOwnable;
import net.minecraft.entity.SharedMonsterAttributes;
import net.minecraft.entity.ai.EntityAISwimming;
import net.minecraft.entity.ai.EntityAIWatchClosest;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.EnumDyeColor;
import net.minecraft.item.ItemArmor;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.item.ItemSword;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.network.play.server.S0BPacketAnimation;
import net.minecraft.pathfinding.PathNavigateGround;
import net.minecraft.potion.Potion;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.MathHelper;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraftforge.fml.common.network.NetworkRegistry.TargetPoint;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class EntityLittleMaid extends EntityCreature implements IEntityOwnable {

	/* インベントリ */
	public InventoryLittleMaid inventory = new InventoryLittleMaid(this);

	/** Float used to smooth the rotation of the head */
	private float headRotationCourse;
	private float headRotationCourseOld;

    // On/Offの切り替え値を持つAIはメンバにもたせてあとで操作しやすくする
    /** 待機用AI */
    protected EntityAIWait aiWait = new EntityAIWait(this);

    // いろいろ状態保持
    /** メイドさんの役割 */
    private EnumMaidRole maidRole;

	public EntityLittleMaid(World worldIn)
	{
		super(worldIn);
		this.setSize(0.7F, 1.5F);

		((PathNavigateGround) this.getNavigator()).func_179690_a(true);

		// AI設定 priorityは小さい方が優先度が高い コメント後ろの()はMutexBits値
		// 頭動作…2（2bit目） 移動…4（3bit目）
		/* 生存本能的AI */
		this.tasks.addTask(1, new EntityAISwimming(this)); // 泳ぐ (*4)
		this.tasks.addTask(1, new EntityAILanding(this)); // 着陸する (0)
		this.tasks.addTask(1, new EntityAIRecover(this)); // 手持ちの砂糖を食べる (0)
		/* 基礎行動AI */
		this.tasks.addTask(2, this.aiWait); // 待機 (0)
		/* 役割行動AI */
		this.tasks.addTask(3, new EntityAIAttackOnCollide(this, 0.7D, true)); // 攻撃する (4)
		this.tasks.addTask(3, new EntityAIPutTorch(this, 0.6D)); // 松明設置 (4)
		/* 普段行動AI */
		this.tasks.addTask(4, new EntityAIFollowOwner(this, 0.7D, 3.0F, 8.0F)); // ご主人に着いて行く (6)
		this.tasks.addTask(4, new littlemaid.entity.ai.EntityAIBeg(this, 8.0F)); // おねだりする (4)
		this.tasks.addTask(5, new EntityAIWatchClosest(this, EntityLivingBase.class, 10F)); // 近くのエンティティ(EntityLivingBase)を見る (*2)

		// ターゲットAI
		this.targetTasks.addTask(1, new littlemaid.entity.ai.EntityAIOwnerHurtTarget(this));

		// 初期体力値設定
		setHealth(getMaxHealth());

		// 未契約状態に設定
		setContract(false);
		setLanding(false);
	}

// ==============================================================================
//  初期化
// ------------------------------------------------------------------------------

	/**
	 * エンティティの初期化処理（Entityクラスのコンストラクタで呼ばれる）
	 */
	@Override
	protected void entityInit() {
		super.entityInit();
		// ID:16 MaidState(bit1:待機 bit2:着地 bit3:契約)
		this.dataWatcher.addObject(16, new Byte((byte) 0));
		// ID:17 OwnerID
		this.dataWatcher.addObject(17, "");
//		// ID:18 Health
//		this.dataWatcher.addObject(18, this.getHealth());
		// ID:19 Beg(おねだり)
		this.dataWatcher.addObject(19, new Byte((byte) 0));
		// ID:20 Color
		this.dataWatcher.addObject(20, Byte.valueOf((byte) EnumDyeColor.BLACK.getMetadata()));
		// ID:xx Model
	}

	/**
	 * エンティティ属性の適用（EntityLivingBaseクラスのコンストラクタで呼ばれる）
	 */
	@Override
	protected void applyEntityAttributes() {
		super.applyEntityAttributes();

		// 歩く早さ
        this.getEntityAttribute(SharedMonsterAttributes.movementSpeed).setBaseValue(0.7D);
        // 最大ライフ（野生用）
		this.getEntityAttribute(SharedMonsterAttributes.maxHealth).setBaseValue(8.0D);
		// 素手でのダメージ値
		this.getAttributeMap().registerAttribute(SharedMonsterAttributes.attackDamage);
		this.getEntityAttribute(SharedMonsterAttributes.attackDamage).setBaseValue(2.0D);
	}

// ==============================================================================
//  イベント
// ------------------------------------------------------------------------------

	/**
	 * 相互作用（右クリックアクション）
	 */
	@Override
	protected boolean interact(EntityPlayer player) {
		ItemStack itemstack = player.inventory.getCurrentItem();
		LittleMaidCore.getLogger().info("EntityLittleMaid_interact");

		// ケーキだ！
		if (itemstack != null && itemstack.getItem() == Items.cake) {

			// もらったケーキは食べとく（ハート5個回復）
			this.heal(10);

			if (!player.capabilities.isCreativeMode && --itemstack.stackSize <= 0)
			{
				player.inventory.setInventorySlotContents(player.inventory.currentItem, (ItemStack) null);
			}

			// まだ契約してない（契約手続き）
			if (!this.isContract()) {

				if (!this.worldObj.isRemote)
				{
					this.setContract(true);
					this.navigator.clearPathEntity();
					this.setAttackTarget((EntityLivingBase) null);
					this.aiWait.setWaiting(true);
					this.setHealth(20.0F);
					this.setOwnerId(player.getUniqueID().toString());
					//				this.playTameEffect(true);
					this.worldObj.setEntityState(this, (byte) 7);
				}

				// インベントリ初期化
				this.inventory.clear();

				// 契約時挨拶
				this.playSound("littlemaid:basic.contracted", 1.0F, 1.0F);

				// 実績解除
				player.triggerAchievement(LittleMaidCore.employMaid);

				// 統計加算
				player.addStat(LittleMaidCore.employmentNumber, 1);
			}

			return true;
		}

		// お砂糖だ！
		if (itemstack != null && itemstack.getItem() == Items.sugar)
		{
			// もらったお砂糖は食べとく（ハート半個回復）
			this.heal(1);

			if (!player.capabilities.isCreativeMode && --itemstack.stackSize <= 0)
			{
				player.inventory.setInventorySlotContents(player.inventory.currentItem, (ItemStack) null);
			}

			// くれた人はご主人さまだった
			if (this.isContract() && this.getOwner() == player)
			{
				// まず落ち着く
				this.isJumping = false;
				this.navigator.clearPathEntity();
				this.setAttackTarget((EntityLivingBase) null);

				// 自分は何をすべきか考える（モード切替）
				if (this.getHeldItem() == null) {
					this.setRole(EnumMaidRole.NONE);
				} else {
					if (this.getHeldItem().getItem() instanceof ItemSword) {
						this.setRole(EnumMaidRole.ATTACKER);
					}
					if (this.getHeldItem().getItem() instanceof ItemBlock
						&& ((ItemBlock)this.getHeldItem().getItem()).getBlock().equals(Blocks.torch)) {
						this.setRole(EnumMaidRole.TORCHER);
					}
				}

				// 待機状態の切り替え
				this.aiWait.setWaiting(!this.isWaiting());

				// 統計加算
				player.addStat(LittleMaidCore.giveSugarNumber, 1);
			}

			return true;
		}

		// ここからはご主人さまの時のみの反応
		if (this.isContract() && this.getOwner() == player)
		{
			// お化粧を渡された（色変更）
			if (itemstack != null && itemstack.getItem() == Items.dye)
			{
				if (!player.capabilities.isCreativeMode && --itemstack.stackSize <= 0)
				{
					player.inventory.setInventorySlotContents(player.inventory.currentItem, (ItemStack) null);
				}

				this.setFeature(EnumDyeColor.byDyeDamage(Items.dye.getMetadata(itemstack)));

				return true;
			}

			// インベントリを開く（どれにも該当しなかった）
			if (!this.worldObj.isRemote)
			{
				player.openGui(LittleMaidCore.instance, GuiHandlerLittleMaid.GUI_ID_INVVENTORY, player.worldObj, this.getEntityId(), 0, 0);
			}

			return true;
		}

		return super.interact(player);
	}

	/**
	 * this.worldObj.setEntityState の第二引数の値を受け取る
	 * サーバーから対象Entityを認知しているクライアント全てに通知されるみたい。
	 */
	@SideOnly(Side.CLIENT)
	public void handleHealthUpdate(byte state)
	{
		if (state == 7)
		{
			this.playContractEffect(true);
		}
		else if (state == 6)
		{
			this.playContractEffect(false);
		}
		else
		{
			super.handleHealthUpdate(state);
		}
	}

// ==============================================================================
//  機能
// ------------------------------------------------------------------------------

	/**
	 * 契約時エフェクトを出す
	 */
	@SideOnly(Side.CLIENT)
	protected void playContractEffect(boolean success)
	{
		EnumParticleTypes enumparticletypes = EnumParticleTypes.HEART;

		if (!success)
		{
			enumparticletypes = EnumParticleTypes.SMOKE_NORMAL;
		}

		double d0 = this.rand.nextGaussian() * 0.02D;
		double d1 = this.rand.nextGaussian() * 0.02D;
		double d2 = this.rand.nextGaussian() * 0.02D;
		this.worldObj.spawnParticle(enumparticletypes, this.posX, this.posY + (double) this.height, this.posZ, d0, d1, d2, new int[0]);
	}

	/**
	 * デスポーン可能かどうか
	 * @return
	 */
	@Override
	protected boolean canDespawn() {
		// 契約していたらデスポーンしない
		return !this.isContract();
	}

	/**
	 * おねだり状態を設定
	 * @param bbeg
	 */
	public void setBeg(boolean bbeg)
	{
		if (bbeg)
		{
			this.dataWatcher.updateObject(19, Byte.valueOf((byte) 1));
		}
		else
		{
			this.dataWatcher.updateObject(19, Byte.valueOf((byte) 0));
		}
	}

	/**
	 * おねだり状態を取得
	 * @return
	 */
	public boolean isBeg()
	{
		return this.dataWatcher.getWatchableObjectByte(19) == 1;
	}

	/**
	 * 契約状態を設定する
	 * @param contract
	 */
	public void setContract(boolean contract) {
		byte b0 = this.dataWatcher.getWatchableObjectByte(16);

		if (contract)
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 | 4)));
			this.getEntityAttribute(SharedMonsterAttributes.maxHealth).setBaseValue(20.0D);
		}
		else
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 & -5)));
			this.getEntityAttribute(SharedMonsterAttributes.maxHealth).setBaseValue(8.0D);
		}

		// 役割をクリア
		this.maidRole = EnumMaidRole.NONE;
		this.setupContractAI();
	}

	/**
	 * 契約状態に応じてAIの再セットアップを行う場合はここで
	 */
	protected void setupContractAI()
	{
	}

	/**
	 * 契約メイドかどうかを取得
	 * @return boolean 契約済みならtrue
	 */
	public boolean isContract() {
		return (this.dataWatcher.getWatchableObjectByte(16) & 4) != 0;
	}

	/**
	 * ご主人様のIDを設定
	 * @param ownerUuid ご主人様のUUID文字列
	 */
	public void setOwnerId(String ownerUuid)
	{
		this.dataWatcher.updateObject(17, ownerUuid);
	}

	/**
	 * ご主人様のUUID文字列を取得
	 */
	@Override
	public String getOwnerId() {
		return this.dataWatcher.getWatchableObjectString(17);
	}

	/**
	 * ご主人様のEntityを取得
	 * @return ご主人様のEntityPlayerインスタンス
	 */
	public EntityLivingBase getOwnerEntity()
	{
		try
		{
			UUID uuid = UUID.fromString(this.getOwnerId());
			return uuid == null ? null : this.worldObj.getPlayerEntityByUUID(uuid);
		}
		catch (IllegalArgumentException illegalargumentexception)
		{
			return null;
		}
	}

	@Override
	public Entity getOwner() {
		return getOwnerEntity();
	}

	/**
	 * このメイドさんのご主人様かどうかを取得
	 * @param entityIn ご主人と思しきEntity
	 * @return ご主人様ならtrue
	 */
	public boolean isOwner(EntityLivingBase entityIn)
	{
		return entityIn == this.getOwnerEntity();
	}

	/**
	 * 待機状態を設定
	 * @param waiting
	 */
	public void setWaiting(boolean waiting)
	{
		byte b0 = this.dataWatcher.getWatchableObjectByte(16);

		if (waiting)
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 | 1)));
		}
		else
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 & -2)));
		}
	}

	/**
	 * 待機状態を取得
	 * @return 待機状態ならtrue
	 */
	public boolean isWaiting()
	{
		return (this.dataWatcher.getWatchableObjectByte(16) & 1) != 0;
	}

	/**
	 * 着地状態を設定
	 * @param landing
	 */
	public void setLanding(boolean landing)
	{
		byte b0 = this.dataWatcher.getWatchableObjectByte(16);

		if (landing)
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 | 2)));
		}
		else
		{
			this.dataWatcher.updateObject(16, Byte.valueOf((byte) (b0 & -3)));
		}
	}

	/**
	 * 着地状態を取得
	 * @return 着地状態ならtrue
	 */
	public boolean isLanding()
	{
		return (this.dataWatcher.getWatchableObjectByte(16) & 2) != 0;
	}

// ==============================================================================
//  テクスチャ変更
// ------------------------------------------------------------------------------

	/**
	 * メイドさんの特色を設定
	 * @param featurecolor
	 */
	public void setFeature(EnumDyeColor featurecolor)
	{
		this.dataWatcher.updateObject(20, Byte.valueOf((byte) (featurecolor.getMetadata() & 15)));
	}

	/**
	 * メイドさんの特色を取得
	 * @return
	 */
	public EnumDyeColor getFeature()
	{
		return EnumDyeColor.byMetadata(this.dataWatcher.getWatchableObjectByte(20) & 15);
	}

// ==============================================================================
//  持ち物と武器防具
// ------------------------------------------------------------------------------

	@Override
	public ItemStack getHeldItem() {
		// 先頭のアイテムを持ってることにする
		return this.inventory.getStackInSlot(0);
	}

	/**
	 * 装備中の装備品を取得
	 * slotIn…3:Head,2:Chest,1:Leggings,0:Boots
	 */
	@Override
	public ItemStack getCurrentArmor(int slotIn) {
		return this.inventory.getStackInSlot(this.inventory.getSizeInventory() - (slotIn + 1));
	}

	/**
	 * エンティティがモブを攻撃
	 */
	@Override
	public boolean attackEntityAsMob(Entity target) {
		// 基礎ダメージ値取得
		float f = (float) this.getEntityAttribute(SharedMonsterAttributes.attackDamage).getAttributeValue();
		int i = 0;
		boolean flag_c = false;

		if (target instanceof EntityLivingBase)
		{
			// クリティカル条件確認
			flag_c = this.fallDistance > 0.0F && !this.onGround && !this.isOnLadder() && !this.isInWater() && !this.isPotionActive(Potion.blindness) && this.ridingEntity == null && target instanceof EntityLivingBase;
			if (flag_c && f > 0.0F) f *= 1.5F;

			// 手持ちアイテムによる上乗せダメージ値算出
			f += EnchantmentHelper.func_152377_a(this.getHeldItem(), ((EntityLivingBase) target).getCreatureAttribute());
			// ノックバック値取得
			i += EnchantmentHelper.getKnockbackModifier(this);
			// ダッシュ状態ならノックバック値上乗せ
			if (this.isSprinting()) ++i;
		}

		// ターゲットにダメージ付与
		boolean flag = target.attackEntityFrom(DamageSource.causeMobDamage(this), f);

		if (flag)
		{
			this.setLastAttacker(target);

			// ノックバックの適用
			if (i > 0)
			{
				target.addVelocity((double) (-MathHelper.sin(this.rotationYaw * (float) Math.PI / 180.0F) * (float) i * 0.5F), 0.1D, (double) (MathHelper.cos(this.rotationYaw * (float) Math.PI / 180.0F) * (float) i * 0.5F));
				this.motionX *= 0.6D;
				this.motionZ *= 0.6D;
			}

			// 手持ちアイテムに炎上エンチャントがついてたら炎上させる？
			int j = EnchantmentHelper.getFireAspectModifier(this);

			if (j > 0)
			{
				target.setFire(j * 4);
			}

			// クリティカルアニメーションの表示
			if (flag_c)
			{
				((WorldServer) this.worldObj).getEntityTracker().func_151248_b(this, new S0BPacketAnimation(target, 4));
			}

			// 与ダメ、被ダメ関係のエンチャントを適用する？
			this.func_174815_a(this, target);
		}

		return flag;
	}

	/**
	 * 装備している防具のダメージ軽減量を取得
	 */
	@Override
	public int getTotalArmorValue() {
        int i = 1;

        for (int k = 1; k <= 4; ++k)
        {
            ItemStack itemstack = this.inventory.getStackInSlot(this.inventory.getSizeInventory() - k);

            if (itemstack != null && itemstack.getItem() instanceof ItemArmor)
            {
                int l = ((ItemArmor)itemstack.getItem()).damageReduceAmount;
                i += l;
            }
        }

        return i;
	}

	/**
	 * 装備品の耐久値を減らす
	 */
	@Override
	protected void damageArmor(float amount) {
		this.inventory.damageArmor(amount);

		// 装備品をクライアントへ同期する
		this.syncEquip();
	}

	/**
	 * 装備品情報を半径64ブロック内にいるプレイヤーのクライアントへ通知
	 */
	public void syncEquip() {
		LittleMaidCore.MESSAGE.sendToAllAround(
			new MessageSyncEquip(this),
			new TargetPoint(this.dimension, this.posX, this.posY, this.posZ, 64.0D));
	}


// ==============================================================================
//  モデル動作
// ------------------------------------------------------------------------------

	@Override
	public void onUpdate() {
		super.onUpdate();

		this.headRotationCourseOld = this.headRotationCourse;

		if (this.isBeg())
		{
			this.headRotationCourse += (1.0F - this.headRotationCourse) * 0.4F;
		}
		else
		{
			this.headRotationCourse += (0.0F - this.headRotationCourse) * 0.4F;
		}
	}

	@SideOnly(Side.CLIENT)
	public float getInterestedAngle(float partialTicks)
	{
		// 27度くらい首を傾ける
		return (this.headRotationCourseOld + (this.headRotationCourse - this.headRotationCourseOld) * partialTicks) * -0.1F * (float) Math.PI;
	}

	/**
	 * メイドさんがやられた
	 */
	@Override
	public void onDeath(DamageSource cause) {
		// チャットに死亡メッセージを出力
		if (!this.worldObj.isRemote && this.worldObj.getGameRules().getGameRuleBooleanValue("showDeathMessages")
			&& this.getOwnerEntity() instanceof EntityPlayerMP)
		{
			((EntityPlayerMP) this.getOwnerEntity()).addChatMessage(this.getCombatTracker().getDeathMessage());
		}

		// メイドさんを失った統計を加算
		if (this.getOwnerEntity() instanceof EntityPlayerMP) {
			EntityPlayer player = (EntityPlayer) this.getOwnerEntity();
			player.addStat(LittleMaidCore.lostNumber, 1);
		}

		super.onDeath(cause);
	}

// ==============================================================================
//  音声
// ------------------------------------------------------------------------------

	/**
	 * 泳ぐ
	 */
	protected String getSwimSound()
	{
		return "game.player.swim";
	}

	/**
	 * 飛沫
	 */
	protected String getSplashSound()
	{
		return "game.player.swim.splash";
	}

	/**
	 * 被ダメ
	 */
	protected String getHurtSound()
	{
		return "game.player.hurt";
	}

	/**
	 * 無念…
	 */
	public String getDeathSound() {
		return "basic.death";
	}

// ==============================================================================
//  状態保持
// ------------------------------------------------------------------------------

	/**
	 * メイドさんの役割（お仕事）を取得する
	 * @return
	 */
	public EnumMaidRole getRole() {
		return this.maidRole;
	}

	/**
	 * メイドさんの役割（お仕事）を設定する
	 * @param role
	 */
	public void setRole(EnumMaidRole role) {
		this.maidRole = role;
	}

	/**
	 * セーブデータからメイドさんの状態を読み込む
	 */
	@Override
	public void readEntityFromNBT(NBTTagCompound tagCompound) {
		super.readEntityFromNBT(tagCompound);
		LittleMaidCore.getLogger().info("EntityLittleMaid_readEntityFromNBT");

		// ご主人ID
		String s = tagCompound.getString("OwnerUUID");
		if (s.length() > 0) {
			this.setOwnerId(s);
			this.setContract(true);
		}

		// 待機
		this.aiWait.setWaiting(tagCompound.getBoolean("Waiting"));

		// 特色
		this.setFeature(EnumDyeColor.byMetadata(tagCompound.getInteger("Feature")));

		// インベントリ
		NBTTagList nbtList = (NBTTagList) tagCompound.getTag("Inventory");
		for (int i = 0; i < nbtList.tagCount(); ++i)
		{
			NBTTagCompound nbtStack = nbtList.getCompoundTagAt(i);
			int j = nbtStack.getByte("Slot") & 255;
			ItemStack itemstack = ItemStack.loadItemStackFromNBT(nbtStack);

			if (itemstack != null)
			{
				this.inventory.setInventorySlotContents(j, itemstack);
			}
		}

		// 役割
		if (EnumMaidRole.getEnum(tagCompound.getString("Role")) != null) {
			this.maidRole = EnumMaidRole.getEnum(tagCompound.getString("Role"));
		}
	}

	/**
	 * メイドさんの状態を保存する
	 */
	@Override
	public void writeEntityToNBT(NBTTagCompound tagCompound) {
		super.writeEntityToNBT(tagCompound);
		LittleMaidCore.getLogger().info("EntityLittleMaid_writeEntityToNBT");

		// ご主人ID
		if (this.getOwnerId() == null)
		{
			tagCompound.setString("OwnerUUID", "");
		}
		else
		{
			tagCompound.setString("OwnerUUID", this.getOwnerId());
		}

		// 待機
		tagCompound.setBoolean("Waiting", this.isWaiting());

		// 特色
		tagCompound.setByte("Feature", (byte)this.getFeature().getMetadata());

		// インベントリ
		NBTTagList nbtList = new NBTTagList();
		for (int i = 0; i < this.inventory.getSizeInventory(); ++i)
		{
			if (this.inventory.getStackInSlot(i) != null)
			{
				NBTTagCompound nbtStack = new NBTTagCompound();
				nbtStack.setByte("Slot", (byte) i);
				this.inventory.getStackInSlot(i).writeToNBT(nbtStack);
				nbtList.appendTag(nbtStack);
			}
		}
		tagCompound.setTag("Inventory", nbtList);

		// 役割
		tagCompound.setString("Role", this.maidRole.getName());

	}

// ==============================================================================
//  便利関数
// ------------------------------------------------------------------------------
	/**
	 * Randomインスタンスを取得
	 * @return
	 */
	public Random getRandom()
	{
		return this.rand;
	}


}
