/* ----------------------------------------------------------------- */
/*           The HMM-Based Speech Synthesis System (HTS)             */
/*           Open JTalk developed by HTS Working Group               */
/*           http://open-jtalk.sourceforge.net/                      */
/* ----------------------------------------------------------------- */
/*                                                                   */
/*  Copyright (c) 2008-2010  Nagoya Institute of Technology          */
/*                           Department of Computer Science          */
/*                                                                   */
/* All rights reserved.                                              */
/*                                                                   */
/* Redistribution and use in source and binary forms, with or        */
/* without modification, are permitted provided that the following   */
/* conditions are met:                                               */
/*                                                                   */
/* - Redistributions of source code must retain the above copyright  */
/*   notice, this list of conditions and the following disclaimer.   */
/* - Redistributions in binary form must reproduce the above         */
/*   copyright notice, this list of conditions and the following     */
/*   disclaimer in the documentation and/or other materials provided */
/*   with the distribution.                                          */
/* - Neither the name of the HTS working group nor the names of its  */
/*   contributors may be used to endorse or promote products derived */
/*   from this software without specific prior written permission.   */
/*                                                                   */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND            */
/* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,       */
/* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF          */
/* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE          */
/* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS */
/* BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,          */
/* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED   */
/* TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,     */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON */
/* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,   */
/* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY    */
/* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/* POSSIBILITY OF SUCH DAMAGE.                                       */
/* ----------------------------------------------------------------- */


#include <string>
#include <vector>
#include "open_jtalk_lib.h"

#define MAXBUFLEN 1024
#pragma comment(lib,"..\\Debug\\open_jtalk_lib.lib")
#pragma comment(lib,"winmm.lib")
#pragma comment(lib,"Advapi32.lib")
/* Getfp: wrapper for fopen */
FILE *Getfp(const char *name, const char *opt)
{
   FILE *fp = fopen(name, opt);

   if (fp == NULL) {
      fprintf(stderr, "ERROR: Getfp() in open_jtalk.c: Cannot open %s.\n", name);
      exit(1);
   }

   return (fp);
}

int main(int argc, char **argv)
{
   FILE *txtfp = stdin;
   char *txtfn = NULL;
   FILE *wavfp = NULL;
   FILE *logfp = NULL;

   /* text */
   char buff[MAXBUFLEN];

   /* engine */
   OpenJTalk open_jtalk;

   /* directory name of dictionary */
   char *dn_mecab = NULL;

   /* file names of models */
   char *fn_ms_lf0 = NULL;
   char *fn_ms_mcp = NULL;
   char *fn_ms_dur = NULL;

   /* file names of trees */
   char *fn_ts_lf0 = NULL;
   char *fn_ts_mcp = NULL;
   char *fn_ts_dur = NULL;

   /* file names of windows */
   char **fn_ws_lf0;
   char **fn_ws_mcp;
   int num_ws_lf0 = 0, num_ws_mcp = 0;

   /* file names of global variance */
   char *fn_ms_gvl = NULL;
   char *fn_ms_gvm = NULL;

   /* file names of global variance trees */
   char *fn_ts_gvl = NULL;
   char *fn_ts_gvm = NULL;

   /* file names of global variance switch */
   char *fn_gv_switch = NULL;

   /* global parameter */
   int sampling_rate = 16000;
   int fperiod = 80;
   double alpha = 0.42;
   int stage = 0;               /* gamma = -1.0/stage */
   double beta = 0.0;
   int audio_buff_size = 1600;
   double uv_threshold = 0.5;
   double gv_weight_lf0 = 0.7;
   double gv_weight_mcp = 1.0;
   HTS_Boolean use_log_gain = FALSE;


   /* delta window handler for log f0 */
   fn_ws_lf0 = (char **) calloc(argc, sizeof(char *));
   /* delta window handler for mel-cepstrum */
   fn_ws_mcp = (char **) calloc(argc, sizeof(char *));

   /* read command */
   while (--argc) {
      if (**++argv == '-') {
         switch (*(*argv + 1)) {
         case 'x':
            dn_mecab = *(++argv);
            --argc;
            break;
         case 't':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               fn_ts_lf0 = *(++argv);
               break;
            case 'm':
               fn_ts_mcp = *(++argv);
               break;
            case 'd':
               fn_ts_dur = *(++argv);
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-t%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'm':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               fn_ms_lf0 = *(++argv);
               break;
            case 'm':
               fn_ms_mcp = *(++argv);
               break;
            case 'd':
               fn_ms_dur = *(++argv);
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-m%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'd':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               fn_ws_lf0[num_ws_lf0] = *(++argv);
               num_ws_lf0++;
               break;
            case 'm':
               fn_ws_mcp[num_ws_mcp] = *(++argv);
               num_ws_mcp++;
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-d%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'o':
            switch (*(*argv + 2)) {
            case 'w':
               wavfp = Getfp(*(++argv), "wb");
               break;
            case 't':
               logfp = Getfp(*(++argv), "w");
               break;
            default:
               fprintf(stderr, "ERROR: main() in open_jtalk.c: Invalid option '-o%c'.\n",
                       *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 's':
            sampling_rate = atoi(*++argv);
            --argc;
            break;
         case 'p':
            fperiod = atoi(*++argv);
            --argc;
            break;
         case 'a':
            alpha = atof(*++argv);
            --argc;
            break;
         case 'g':
            stage = atoi(*++argv);
            --argc;
            break;
         case 'l':
            use_log_gain = TRUE;
            break;
         case 'b':
            beta = atof(*++argv);
            --argc;
            break;
         case 'u':
            uv_threshold = atof(*++argv);
            --argc;
            break;
         case 'e':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               fn_ts_gvl = *(++argv);
               break;
            case 'm':
               fn_ts_gvm = *(++argv);
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-e%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'c':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               fn_ms_gvl = *(++argv);
               break;
            case 'm':
               fn_ms_gvm = *(++argv);
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-c%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'j':
            switch (*(*argv + 2)) {
            case 'f':
            case 'p':
               gv_weight_lf0 = atof(*(++argv));
               break;
            case 'm':
               gv_weight_mcp = atof(*(++argv));
               break;
            default:
               fprintf(stderr,
                       "ERROR: main() in open_jtalk.c: Invalid option '-j%c'.\n", *(*argv + 2));
               exit(1);
            }
            --argc;
            break;
         case 'k':
            fn_gv_switch = *++argv;
            --argc;
            break;
         case 'z':
            audio_buff_size = atoi(*++argv);
            --argc;
            break;
         default:
            fprintf(stderr, "ERROR: main() in open_jtalk.c: Invalid option '-%c'.\n", *(*argv + 1));
            exit(1);
         }
      } else {
         txtfn = *argv;
         txtfp = Getfp(txtfn, "rt");
      }
   }
   /* dictionary directory check */
   if (dn_mecab == NULL) {
      fprintf(stderr, "ERROR: main() in open_jtalk.c: No dictionary directory.\n");
      exit(1);
   }
   /* number of models,trees check */
   if (fn_ms_dur == NULL || fn_ms_mcp == NULL || fn_ms_lf0 == NULL ||
       fn_ts_dur == NULL || fn_ts_mcp == NULL || fn_ts_lf0 == NULL ||
       fn_ws_mcp == NULL || fn_ws_lf0 == NULL) {
      fprintf(stderr,
              "ERROR: main() in open_jtalk.c: Specify models (trees) for each parameter.\n");
      exit(1);
   }

   /* initialize and load */
   OpenJTalk_initialize(&open_jtalk, sampling_rate, fperiod, alpha, stage, beta, audio_buff_size,
                        uv_threshold, use_log_gain, gv_weight_mcp, gv_weight_lf0);
   OpenJTalk_load(&open_jtalk, dn_mecab, fn_ms_dur, fn_ts_dur, fn_ms_mcp, fn_ts_mcp, fn_ws_mcp,
                  num_ws_mcp, fn_ms_lf0, fn_ts_lf0, fn_ws_lf0, num_ws_lf0, fn_ms_gvm, fn_ts_gvm,
                  fn_ms_gvl, fn_ts_gvl, fn_gv_switch);

	
   /* synthesis */
   fgets(buff, MAXBUFLEN - 1, txtfp);

   char *sample = "eXgRg";

  
   
	OpenJTalk_synthesis(&open_jtalk, buff, wavfp, logfp);
	OpenJTalk_synthesis(&open_jtalk, sample, wavfp, logfp);
	   
	Sleep(10000);
   ::OpenJTalk_clear(&open_jtalk);
   Sleep(10000);
   /* free */
//   OpenJTalk_clear(&open_jtalk);
//   free(fn_ws_mcp);
//   free(fn_ws_lf0);
   if (txtfn != NULL)
      fclose(txtfp);
//   if (wavfp != NULL)
//      fclose(wavfp);
//   if (logfp != NULL)
//      fclose(logfp);


   return 0;
}
