/*                                                                   */
/*  Copyright (c) 2008-2010  Nagoya Institute of Technology          */
/*                           Department of Computer Science          */
/*                                                                   */
/* All rights reserved.                                              */
/*                                                                   */
/* Redistribution and use in source and binary forms, with or        */
/* without modification, are permitted provided that the following   */
/* conditions are met:                                               */
/*                                                                   */
/* - Redistributions of source code must retain the above copyright  */
/*   notice, this list of conditions and the following disclaimer.   */
/* - Redistributions in binary form must reproduce the above         */
/*   copyright notice, this list of conditions and the following     */
/*   disclaimer in the documentation and/or other materials provided */
/*   with the distribution.                                          */
/* - Neither the name of the HTS working group nor the names of its  */
/*   contributors may be used to endorse or promote products derived */
/*   from this software without specific prior written permission.   */
/*                                                                   */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND            */
/* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,       */
/* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF          */
/* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE          */
/* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS */
/* BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,          */
/* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED   */
/* TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,     */
/* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON */
/* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,   */
/* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY    */
/* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/* POSSIBILITY OF SUCH DAMAGE.                                       */
/* ----------------------------------------------------------------- */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>

#include "mecab.h"
#include "njd.h"
#include "jpcommon.h"
#include "HTS_engine.h"

#define OPENJTALK_EXPORTS
#include "open_jtalk_lib.h"

void OpenJTalk_initialize(OpenJTalk * open_jtalk, int sampling_rate, int fperiod, double alpha,
                          int stage, double beta, int audio_buff_size, double uv_threshold,
                          HTS_Boolean use_log_gain, double gv_weight_mcp, double gv_weight_lf0)
{
   Mecab_initialize(&open_jtalk->mecab);
   NJD_initialize(&open_jtalk->njd);
   JPCommon_initialize(&open_jtalk->jpcommon);
   HTS_Engine_initialize(&open_jtalk->engine, 2);
   HTS_Engine_set_sampling_rate(&open_jtalk->engine, sampling_rate);
   HTS_Engine_set_fperiod(&open_jtalk->engine, fperiod);
   HTS_Engine_set_alpha(&open_jtalk->engine, alpha);
   HTS_Engine_set_gamma(&open_jtalk->engine, stage);
   HTS_Engine_set_log_gain(&open_jtalk->engine, use_log_gain);
   HTS_Engine_set_beta(&open_jtalk->engine, beta);
   HTS_Engine_set_audio_buff_size(&open_jtalk->engine, audio_buff_size);
   HTS_Engine_set_msd_threshold(&open_jtalk->engine, 1, uv_threshold);
   HTS_Engine_set_gv_weight(&open_jtalk->engine, 0, gv_weight_mcp);
   HTS_Engine_set_gv_weight(&open_jtalk->engine, 1, gv_weight_lf0);
}

void OpenJTalk_load(OpenJTalk * open_jtalk, char *dn_mecab, char *fn_ms_dur, char *fn_ts_dur,
                    char *fn_ms_mcp, char *fn_ts_mcp, char **fn_ws_mcp, int num_ws_mcp,
                    char *fn_ms_lf0, char *fn_ts_lf0, char **fn_ws_lf0, int num_ws_lf0,
                    char *fn_ms_gvm, char *fn_ts_gvm, char *fn_ms_gvl, char *fn_ts_gvl,
                    char *fn_gv_switch)
{
   Mecab_load(&open_jtalk->mecab, dn_mecab);
   HTS_Engine_load_duration_from_fn(&open_jtalk->engine, &fn_ms_dur, &fn_ts_dur, 1);
   HTS_Engine_load_parameter_from_fn(&open_jtalk->engine, &fn_ms_mcp, &fn_ts_mcp,
                                     fn_ws_mcp, 0, FALSE, num_ws_mcp, 1);
   HTS_Engine_load_parameter_from_fn(&open_jtalk->engine, &fn_ms_lf0, &fn_ts_lf0,
                                     fn_ws_lf0, 1, TRUE, num_ws_lf0, 1);
   if (fn_ms_gvm != NULL) {
      if (fn_ts_gvm != NULL)
         HTS_Engine_load_gv_from_fn(&open_jtalk->engine, &fn_ms_gvm, &fn_ts_gvm, 0, 1);
      else
         HTS_Engine_load_gv_from_fn(&open_jtalk->engine, &fn_ms_gvm, NULL, 0, 1);
   }
   if (fn_ms_gvl != NULL) {
      if (fn_ts_gvl != NULL)
         HTS_Engine_load_gv_from_fn(&open_jtalk->engine, &fn_ms_gvl, &fn_ts_gvl, 1, 1);
      else
         HTS_Engine_load_gv_from_fn(&open_jtalk->engine, &fn_ms_gvl, NULL, 1, 1);
   }
   if (fn_gv_switch != NULL)
      HTS_Engine_load_gv_switch_from_fn(&open_jtalk->engine, fn_gv_switch);
}

void OpenJTalk_synthesis(OpenJTalk * open_jtalk, char *txt, FILE * wavfp, FILE * logfp)
{
   char *buff = (char *) calloc(2 * strlen(txt) + 1, sizeof(char));

   text2mecab(buff, txt);
   Mecab_analysis(&open_jtalk->mecab, buff);
   free(buff);
   mecab2njd(&open_jtalk->njd, Mecab_get_feature(&open_jtalk->mecab),
             Mecab_get_size(&open_jtalk->mecab));
   njd_set_pronunciation(&open_jtalk->njd);
   njd_set_digit(&open_jtalk->njd);
   njd_set_accent_phrase(&open_jtalk->njd);
   njd_set_accent_type(&open_jtalk->njd);
   njd_set_unvoiced_vowel(&open_jtalk->njd);
   njd_set_long_vowel(&open_jtalk->njd);
   njd2jpcommon(&open_jtalk->jpcommon, &open_jtalk->njd);

	
	JPCommon_make_label(&open_jtalk->jpcommon);
   if (JPCommon_get_label_size(&open_jtalk->jpcommon) > 2) {
      HTS_Engine_load_label_from_string_list(&open_jtalk->engine,
                                             JPCommon_get_label_feature(&open_jtalk->jpcommon),
                                             JPCommon_get_label_size(&open_jtalk->jpcommon));

   	
   	HTS_Engine_create_sstream(&open_jtalk->engine);
      HTS_Engine_create_pstream(&open_jtalk->engine);

   	HTS_Engine_create_gstream(&open_jtalk->engine);

   	
   	if (wavfp != NULL)
         HTS_Engine_save_riff(&open_jtalk->engine, wavfp);

   	
   	if (logfp != NULL) {
         NJD_fprint(&open_jtalk->njd, logfp);
         HTS_Engine_save_label(&open_jtalk->engine, logfp);
         HTS_Engine_save_information(&open_jtalk->engine, logfp);
      }
      HTS_Engine_refresh(&open_jtalk->engine);
   }

	
	JPCommon_refresh(&open_jtalk->jpcommon);
   NJD_refresh(&open_jtalk->njd);
   Mecab_refresh(&open_jtalk->mecab);


}

void OpenJTalk_clear(OpenJTalk * open_jtalk)
{
   Mecab_clear(&open_jtalk->mecab);
   NJD_clear(&open_jtalk->njd);
   JPCommon_clear(&open_jtalk->jpcommon);
   HTS_Engine_clear(&open_jtalk->engine);
}
