!
! Copyright (C) 2017 Mitsuaki Kawamura
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
MODULE sctk_tetra
  !
  IMPLICIT NONE
  !
CONTAINS
!
! Bi-linear interpolation
!
SUBROUTINE interpol_indx(ng,kvec,kintp,wintp)
  !
  USE kinds, ONLY : DP
  USE ktetra, ONLY : tetra_type
  !
  IMPLICIT NONE
  !
  INTEGER,INTENT(in) :: ng(3)
  REAL(DP),INTENT(in) :: kvec(3)
  INTEGER,INTENT(out) :: kintp(20)
  REAL(DP),INTENT(out) :: wintp(20)
  !
  INTEGER :: ikv(3,20), dikv(3,3), ii
  REAL(DP) :: x, y, z, xv(3)
  !
  ! Search nearest neighbor grid points.
  !
  xv(1:3) = kvec(1:3) * REAL(ng(1:3), dp)
  ikv(1:3,1) = NINT(xv(1:3))
  dikv(1:3,1:3) = 0
  DO ii = 1, 3
     dikv(ii,ii) = ikv(ii,1) - FLOOR(xv(ii))
     dikv(ii,ii) = 1 - 2 * dikv(ii,ii)
  END DO
  xv(1:3) = ABS(xv(1:3) - REAL(ikv(1:3,1), dp))
  x = xv(1)
  y = xv(2)
  z = xv(3)
  !
  ikv(1:3, 2) = ikv(1:3,1) + dikv(1:3,1)
  ikv(1:3, 3) = ikv(1:3,1) + dikv(1:3,2)
  ikv(1:3, 4) = ikv(1:3,1) + dikv(1:3,3)
  !
  IF(tetra_type == 1) THEN
     !
     wintp(1) = 0.2_dp * (1.0_dp - x - y - z)
     wintp(2) = 0.2_dp * x
     wintp(3) = 0.2_dp * y
     wintp(4) = 0.2_dp * z
     DO ii = 1, 4
        ikv(1:3, 4*ii+1:4*ii+4) = ikv(1:3, 1:4)
        wintp(   4*ii+1:4*ii+4) = wintp(   1:4)
     END DO
     !
  ELSE
     !
     ikv(1:3, 5) = ikv(1:3,1) + SUM(dikv(1:3,1:3), 2)
     !
     ikv(1:3, 6) = ikv(1:3,1) - dikv(1:3,1)
     ikv(1:3, 7) = ikv(1:3,1) - dikv(1:3,2)
     ikv(1:3, 8) = ikv(1:3,1) - dikv(1:3,3)
     !
     ikv(1:3, 9) = ikv(1:3,1) + 2*dikv(1:3,1)
     ikv(1:3,10) = ikv(1:3,1) + 2*dikv(1:3,2)
     ikv(1:3,11) = ikv(1:3,1) + 2*dikv(1:3,3)
     !
     ikv(1:3,12) = ikv(1:3,1) + dikv(1:3,2) + dikv(1:3,3)
     ikv(1:3,13) = ikv(1:3,1) + dikv(1:3,3) + dikv(1:3,1)
     ikv(1:3,14) = ikv(1:3,1) + dikv(1:3,1) + dikv(1:3,2)
     !
     ikv(1:3,15) = ikv(1:3,1) - dikv(1:3,1) + dikv(1:3,3)
     ikv(1:3,16) = ikv(1:3,1) - dikv(1:3,2) + dikv(1:3,1)
     ikv(1:3,17) = ikv(1:3,1) - dikv(1:3,3) + dikv(1:3,2)
     !
     ikv(1:3,18) = ikv(1:3,1) + dikv(1:3,1) - dikv(1:3,3)
     ikv(1:3,19) = ikv(1:3,1) + dikv(1:3,2) - dikv(1:3,1)
     ikv(1:3,20) = ikv(1:3,1) + dikv(1:3,3) - dikv(1:3,2)
     !
     wintp( 1) = ( (x - 2.0_dp)*(x - 1.0_dp)*(1.0_dp + x) &
     &           + (y - 2.0_dp)*(y - 1.0_dp)*(1.0_dp + y) &
     &           + (z - 2.0_dp)*(z - 1.0_dp)*(1.0_dp + z) &
     &           + 2.0_dp*(x*y + y*z + z*x)*(x + y + z - 1.0_dp) &
     &           - 8.0_dp*x*y*z - 4.0_dp) * 0.5_dp
     wintp( 2) = x * ( 2.0_dp + x*(1.0_dp - x - y - z) &
     &               + y*(1.0_dp - 2.0_dp*y + z) &
     &               + z*(1.0_dp - 2.0_dp*z + y)) * 0.5_dp
     wintp( 3) = y * ( 2.0_dp + y*(1.0_dp - x - y - z) &
     &               + x*(1.0_dp - 2.0_dp*x + z) &
     &               + z*(1.0_dp - 2.0_dp*z + x)) * 0.5_dp
     wintp( 4) = z * ( 2.0_dp + z*(1.0_dp - x - y - z) &
     &               + y*(1.0_dp - 2.0_dp*y + x) &
     &               + x*(1.0_dp - 2.0_dp*x + y)) * 0.5_dp
     wintp( 5) = x * y * z
     wintp( 6) = x * (1.0_dp - x) * (    x + 3.0_dp*y + 3.0_dp*z - 2.0_dp) / 6.0_dp
     wintp( 7) = y * (1.0_dp - y) * (3.0_dp*x +     y + 3.0_dp*z - 2.0_dp) / 6.0_dp
     wintp( 8) = z * (1.0_dp - z) * (3.0_dp*x + 3.0_dp*y +     z - 2.0_dp) / 6.0_dp
     wintp( 9) = x * (x - 1.0_dp) * (x + 1.0_dp) / 6.0_dp
     wintp(10) = y * (y - 1.0_dp) * (y + 1.0_dp) / 6.0_dp
     wintp(11) = z * (z - 1.0_dp) * (z + 1.0_dp) / 6.0_dp
     wintp(12) = y * z * (y + z - 2.0_dp * x) * 0.5_dp
     wintp(13) = z * x * (z + x - 2.0_dp * y) * 0.5_dp
     wintp(14) = x * y * (x + y - 2.0_dp * z) * 0.5_dp
     wintp(15) = x * z * (x - 1.0_dp) * 0.5_dp
     wintp(16) = x * y * (y - 1.0_dp) * 0.5_dp
     wintp(17) = y * z * (z - 1.0_dp) * 0.5_dp
     wintp(18) = x * z * (z - 1.0_dp) * 0.5_dp
     wintp(19) = x * y * (x - 1.0_dp) * 0.5_dp
     wintp(20) = y * z * (y - 1.0_dp) * 0.5_dp
     !
  END IF
  !
  DO ii = 1, 20
     ikv(1:3,ii) = MODULO(ikv(1:3,ii), ng(1:3))
     kintp(ii) = 1 + ikv(3,ii) + ikv(2,ii)*ng(3) + ikv(1,ii)*ng(2)*ng(3)
  END DO
  !
END SUBROUTINE interpol_indx
!
! Compute DOS at each k flagment
!
SUBROUTINE calc_dosk(dosd)
  !
  USE kinds, ONLY : DP
  USE mp_world, ONLY : world_comm
  USE klist, ONLY : nks
  USE wvfct, ONLY : et
  USE ktetra, ONLY : wlsm, tetra, ntetra
  USE el_phon, ONLY : elph_nbnd_min, elph_nbnd_max
  USE ener, ONLY : ef
  !
  USE sctk_val, ONLY : nx, xi0
  !
  IMPLICIT NONE
  !
  REAL(dp),INTENT(OUT) :: dosd(nx,elph_nbnd_min:elph_nbnd_max,nks)
  !
  INTEGER :: ibnd, it, ii, ix, itetra(4), ntetra0, ntetra1
  REAL(dp) :: ei(4,elph_nbnd_min:elph_nbnd_max), e(4), a(4,4), w1(nx,4), V
  !
  dosd(1:nx, elph_nbnd_min:elph_nbnd_max, 1:nks) = 0.0_dp
  !
  CALL divide(world_comm, ntetra, ntetra0, ntetra1)
  !
  !$OMP PARALLEL DEFAULT(NONE) &
  !$OMP & SHARED(ntetra0,ntetra1,elph_nbnd_min,elph_nbnd_max,nx,wlsm,et,ef,xi0,dosd,tetra) &
  !$OMP & PRIVATE(it,ii,ibnd,ix,ei,w1,a,e,V,itetra)
  !
  DO it = ntetra0, ntetra1
     !
     DO ibnd = elph_nbnd_min, elph_nbnd_max
        ei(1:4, ibnd) = MATMUL(wlsm(1:4,1:20), et(ibnd,tetra(1:20, it))) - ef
     END DO
     !
     !$OMP DO
     DO ibnd = elph_nbnd_min, elph_nbnd_max
        !
        itetra(1) = 0
        e(1:4) = ei(1:4,ibnd)
        CALL hpsort (4, e, itetra)
        !
        DO ix = 1, nx
           !
           DO ii = 1, 4
              a(ii,1:4) = (xi0(ix) - e(1:4)) / (e(ii) - e(1:4))
           END DO
           !
           IF(e(1) < xi0(ix) .AND. xi0(ix) <= e(2)) THEN
              !
              V = a(2,1) * a(3,1) * a(4,1) / (xi0(ix) - e(1))
              !
              w1(ix,itetra(1)) = a(1,2) + a(1,3) + a(1,4)
              w1(ix,itetra(2:4)) = a(2:4,1)
              w1(ix,1:4) = w1(ix,1:4) * V
              !
           ELSE IF(e(2) < xi0(ix) .AND. xi0(ix) <= e(3)) THEN
              !
              V = a(2,3) * a(3,1) + a(3,2) * a(2,4)
              !
              w1(ix,itetra(1)) = a(1,4) * V + a(1,3) * a(3,1) * a(2,3)
              w1(ix,itetra(2)) = a(2,3) * V + a(2,4) * a(2,4) * a(3,2)
              w1(ix,itetra(3)) = a(3,2) * V + a(3,1) * a(3,1) * a(2,3)
              w1(ix,itetra(4)) = a(4,1) * V + a(4,2) * a(2,4) * a(3,2)
              !
              V = 1.0_dp / (e(4) - e(1))
              w1(ix,1:4) = w1(ix,1:4) * V
              !
           ELSE IF(e(3) < xi0(ix) .AND. xi0(ix) < e(4)) THEN
              !
              V = a(1,4) * a(2,4) * a(3,4) / (e(4) - 0.0_dp)
              w1(ix,itetra(1:3)) = a(1:3,4)
              w1(ix,itetra(4)) = a(4,1) + a(4,2) + a(4,3)
              w1(ix,1:4) = w1(ix,1:4) * V
              !
           END IF
           !
        END DO ! ix
        !
        DO ix = 1, nx
           dosd(1:nx, ibnd, tetra(1:20, it)) = dosd(1:nx,ibnd,    tetra(1:20, it)) &
           &                            + MATMUL(w1(1:nx,1:4), wlsm(1:4,1:20))
        END DO ! ix
        !
     END DO ! ibnd
     !$OMP END DO NOWAIT
     !
  END DO ! it
  !$OMP END PARALLEL
  !
  dosd(  1:nx, elph_nbnd_min:elph_nbnd_max, 1:nks) = &
  & dosd(1:nx, elph_nbnd_min:elph_nbnd_max, 1:nks) / REAL(ntetra, dp)
  !
END SUBROUTINE calc_dosk
!
! Integration weight with tetrahedron method
!
SUBROUTINE tetraweight(nkd0,indx1,indx2,wghtd)
  !
  USE wvfct, ONLY : nbnd
  USE mp_world, ONLY : world_comm
  USE kinds, ONLY : DP
  USE cell_base, ONLY : omega
  USE wvfct, ONLY : et, nbnd
  USE ktetra, ONLY : wlsm, tetra, ntetra
  USE klist, ONLY : nks
  USE ener, ONLY : ef
  !
  USE sctk_val, ONLY : nmf
  !
  !
  IMPLICIT NONE
  !
  INTEGER,INTENT(IN) :: nkd0, indx1(20,6 * nks), indx2(20,6 * nks)
  COMPLEX(dp),INTENT(OUT) :: wghtd(nbnd*(nmf+1),nbnd,nkd0)
  !
  INTEGER :: it, ibnd, ii, ntetra0, ntetra1, itetra(4)
  REAL(dp) :: thr = 1e-8_dp, V
  REAL(dp) :: e(4), a(4,4), ei0(4,nbnd), ej0(4,nbnd), ei1(4), ej1(4,nbnd), tsmall(4,4)
  COMPLEX(dp) :: w1(nbnd*(nmf+1),4), w2(nbnd*(nmf+1),4)
  !
  CALL divide(world_comm, ntetra, ntetra0, ntetra1)
  !
  wghtd(1:nbnd*(nmf+1),1:nbnd,1:nkd0) = 0.0_dp
  !
  !$OMP PARALLEL DEFAULT(NONE) &
  !$OMP & SHARED(ntetra0,ntetra1,nbnd,wlsm,nmf,et,wghtd,thr,ef,tetra,indx1,indx2) &
  !$OMP & PRIVATE(it,ii,ibnd,itetra,ei0,ej0,ei1,ej1,w1,w2,a,e,V,tsmall)
  !
  DO it = ntetra0, ntetra1
     !
     DO ibnd = 1, nbnd
        ei0(1:4,ibnd) = MATMUL(wlsm(1:4,1:20), et(ibnd, tetra(1:20,it))) - ef
        ej0(1:4,ibnd) = MATMUL(wlsm(1:4,1:20), et(ibnd, indx1(1:20,it))) - ef
     END DO
     !
     !$OMP DO
     DO ibnd = 1, nbnd
        !
        w1(1:(nmf+1)*nbnd,1:4) = 0.0_dp
        !
        itetra(1) = 0
        e(1:4) = ei0(1:4, ibnd)
        CALL hpsort (4, e, itetra)
        !
        DO ii = 1, 4
           a(ii,1:4) = (0.0_dp - e(1:4)) / (e(ii) - e(1:4))
        END DO
        !
        IF(e(1) <= 0.0_dp .AND. 0.0_dp < e(2) ) THEN
           !
           ! A - 1
           !
           V = a(2,1) * a(3,1) * a(4,1)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/a(1,2), a(2,1), 0.0_dp, 0.0_dp/)
              tsmall(3, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
              tsmall(4, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
        ELSE IF(e(2) <= 0.0_dp .AND. 0.0_dp < e(3)) THEN
           !
           ! B - 1
           !
           V = a(3,1) * a(4,1) * a(2,4)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
              tsmall(3, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
              tsmall(4, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
           ! B - 2
           !
           V = a(3,2) * a(4,2)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/0.0_dp, 1.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(3, 1:4) = (/0.0_dp, a(2,3), a(3,2), 0.0_dp/)
              tsmall(4, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
           ! B - 3
           !
           V = a(2,3) * a(3,1) * a(4,2)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
              tsmall(3, 1:4) = (/0.0_dp, a(2,3), a(3,2), 0.0_dp/)
              tsmall(4, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
        ELSE IF(e(3) <= 0.0_dp .AND. 0.0_dp < e(4)) THEN
           !
           ! C - 1
           !
           V = a(4,3)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/0.0_dp, 1.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(3, 1:4) = (/0.0_dp, 0.0_dp, 1.0_dp, 0.0_dp/)
              tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, a(3,4), a(4,3)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
           ! C - 2
           !
           V = a(3,4) * a(4,2)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/0.0_dp, 1.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(3, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
              tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, a(3,4), a(4,3)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
           ! C - 3
           !
           V = a(3,4) * a(2,4) * a(4,1)
           !
           IF(V > thr) THEN
              !
              tsmall(1, 1:4) = (/1.0_dp, 0.0_dp, 0.0_dp, 0.0_dp/)
              tsmall(2, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
              tsmall(3, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
              tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, a(3,4), a(4,3)/)
              !
              ei1(1:4       ) = MATMUL(tsmall(1:4,1:4), ei0(itetra(1:4),   ibnd))
              ej1(1:4,1:nbnd) = MATMUL(tsmall(1:4,1:4), ej0(itetra(1:4), 1:nbnd))
              !
              CALL tetra2(ei1,ej1,w2)
              !
              w1(1:(nmf+1)*nbnd,itetra(1:4)) = w1(1:(nmf+1)*nbnd,          itetra(1:4)) &
              &                   + V * MATMUL(w2(1:(nmf+1)*nbnd,1:4), tsmall(1:4,1:4))
              !
           END IF
           !
        ELSE IF(e(4) <= 0.0_dp ) THEN
           !
           ! D - 1
           !
           ei1(1:4       ) = ei0(itetra(1:4),   ibnd)
           ej1(1:4,1:nbnd) = ej0(itetra(1:4), 1:nbnd)
           !
           CALL tetra2(ei1,ej1,w2)
           !
           w1(1:(nmf+1)*nbnd,1:4) = w1(1:(nmf+1)*nbnd,1:4) &
           &                      + w2(1:(nmf+1)*nbnd,1:4)
           !
        ELSE
           !
           CYCLE
           !
        END IF
        !
        wghtd(1:(nmf+1)*nbnd,ibnd,indx2(1:20,it)) = wghtd(1:(nmf+1)*nbnd,ibnd,      indx2(1:20,it)) &
        &                                   + MATMUL(w1(1:(nmf+1)*nbnd,1:4), wlsm(1:4,1:20))
        !
     END DO ! ibnd
     !$OMP END DO NOWAIT
     !
  END DO ! it
  !
  !$OMP END PARALLEL
  !
  wghtd(1:(nmf+1)*nbnd,1:nbnd,1:nkd0) = wghtd(1:(nmf+1)*nbnd,1:nbnd,1:nkd0) &
  &                            * 4.0_dp / (REAL(ntetra, dp) * omega)
  !
END SUBROUTINE tetraweight
!
!-----------------------------------------------------------------------
SUBROUTINE tetra2(ei0,ej0,w0)
  !---------------------------------------------------------------------
  !
  ! This routine take the unoccupied region.
  !
  USE kinds, ONLY : dp
  USE wvfct, ONLY : nbnd
  !
  USE sctk_val, ONLY : nmf
  !
  IMPLICIT NONE
  !
  REAL(dp),INTENT(IN) :: ei0(4), ej0(4,nbnd)
  COMPLEX(dp),INTENT(OUT) :: w0(0:nmf,nbnd,4)
  !
  INTEGER :: ii, ibnd, itetra(4)
  REAL(dp) :: V, de(4), thr = 1.0e-8_dp
  REAL(dp) :: e(4), a(4,4), tsmall(4,4)
  COMPLEX(dp) :: w1(0:nmf,4)
  !
  w0(0:nmf,1:nbnd,1:4) = 0.0_dp
  !
  DO ibnd = 1, nbnd
     !
     e(1:4) = ej0(1:4,ibnd)
     itetra(1) = 0
     call hpsort (4, e, itetra)
     !
     DO ii = 1, 4
        a(ii,1:4) = ( 0.0_dp - e(1:4) ) / (e(ii) - e(1:4))
     END DO
     !
     IF(0_dp <= e(1)) THEN
        !
        ! A - 1
        !
        de(1:4) = e(1:4) - ei0(itetra(1:4))
        CALL lindhard(de,w1)
        w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) + w1(0:nmf,1:4)
        !
     ELSE IF((e(1) < 0.0_dp .AND. 0.0_dp <= e(2)) .OR. (e(1) <= 0.0_dp .AND. 0.0_dp < e(2))) THEN
        !
        ! B - 1
        !
        V = a(1,2)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,2), a(2,1), 0.0_dp, 0.0_dp/)
           tsmall(2, 1:4) = (/0.0_dp, 1.0_dp, 0.0_dp, 0.0_dp/)
           tsmall(3, 1:4) = (/0.0_dp, 0.0_dp, 1.0_dp, 0.0_dp/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !       
        END IF
        !
        ! B - 2
        !
        V = a(1,3) * a(2,1)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,2), a(2,1), 0.0_dp, 0.0_dp/)
           tsmall(2, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
           tsmall(3, 1:4) = (/0.0_dp, 0.0_dp, 1.0_dp, 0.0_dp/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !
        END IF
        !
        ! B - 3
        !
        V = a(1,4) * a(2,1) * a(3,1)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,2), a(2,1), 0.0_dp, 0.0_dp/)
           tsmall(2, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
           tsmall(3, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !       
        END IF
        !          
     ELSE IF((e(2) < 0.0_dp .AND. 0.0_dp <= e(3)) .OR. (e(2) <= 0.0_dp .AND. 0.0_dp < e(3))) THEN
        !          
        ! C - 1
        !
        V = a(2,4) * a(1,4) * a(3,1)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
           tsmall(2, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
           tsmall(3, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !      
        END IF
        !
        ! C - 2
        !
        V = a(1,3) * a(2,3)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
           tsmall(2, 1:4) = (/0.0_dp, a(2,3), a(3,2), 0.0_dp/)
           tsmall(3, 1:4) = (/0.0_dp, 0.0_dp, 1.0_dp, 0.0_dp/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !
        END IF
        !
        ! C - 3
        ! 
        V = a(1,3) * a(2,4) * a(3,2)
        !
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,3), 0.0_dp, a(3,1), 0.0_dp/)
           tsmall(2, 1:4) = (/0.0_dp, a(2,3), a(3,2), 0.0_dp/)
           tsmall(3, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !
        END IF
        !          
     ELSE IF((e(3) < 0.0_dp .AND. 0.0_dp <= e(4)) .OR. (e(3) <= 0.0_dp .AND. 0.0_dp < e(4))) THEN
        !
        ! D - 1
        !
        V = a(3,4) * a(2,4) * a(1,4) 
        !          
        IF(V > thr) THEN
           !
           tsmall(1, 1:4) = (/a(1,4), 0.0_dp, 0.0_dp, a(4,1)/)
           tsmall(2, 1:4) = (/0.0_dp, a(2,4), 0.0_dp, a(4,2)/)
           tsmall(3, 1:4) = (/0.0_dp, 0.0_dp, a(3,4), a(4,3)/)
           tsmall(4, 1:4) = (/0.0_dp, 0.0_dp, 0.0_dp, 1.0_dp/)
           !
           de(1:4) = MATMUL(tsmall(1:4,1:4), (e(1:4) - ei0(itetra(1:4))))
           CALL lindhard(de,w1)
           w0(0:nmf,ibnd,itetra(1:4)) = w0(0:nmf,ibnd,itetra(1:4)) &
           &               + V * MATMUL(w1(0:nmf,1:4), tsmall(1:4,1:4))
           !        
        END IF
        !
     END IF
     !
  END DO
  !
END SUBROUTINE tetra2
!
! Tetarahedra method for lindhard function
!
SUBROUTINE lindhard(de,w)
  !
  USE kinds, ONLY : DP
  USE dfpt_tetra_mod, ONLY : dfpt_tetra_lindhard_1234, dfpt_tetra_lindhard_1231, &
  &                          dfpt_tetra_lindhard_1233, dfpt_tetra_lindhard_1221, &
  &                          dfpt_tetra_lindhard_1222, dfpt_tetra_lindhard_1211
  !
  USE sctk_val, ONLY : nmf, mf
  !
  IMPLICIT NONE
  !
  REAL(dp),INTENT(IN) :: de(4)
  COMPLEX(dp),INTENT(inout) :: w(0:nmf,4)
  !
  INTEGER :: ii, imf, itetra(4)
  REAL(dp) :: e(4), le(4), thr, thr2
  !
  ! Static part
  !
  itetra(1) = 0
  e(1:4) = de(1:4)
  call hpsort (4, e, itetra)
  !
  thr = MAXVAL(e(1:4)) * 1e-3_dp
  thr2 = 1e-8_dp
  !
  DO ii = 1, 4
     IF(e(ii) < thr2) THEN
        IF(ii == 3) THEN
           CALL errore("lindhard", "Nesting occurs.", 0)
        END IF
        le(ii) = 0.0_dp
        e(ii) = 0.0_dp
     ELSE
        le(ii) = LOG(e(ii))
     END IF
  END DO
  !
  IF(ABS(e(4) - e(3)) < thr ) THEN
     IF(ABS(e(4) - e(2)) < thr ) THEN
        IF(ABS(e(4) - e(1)) < thr ) THEN
           !
           ! e(4) = e(3) = e(2) = e(1)
           !
           w(0,itetra(4)) = 0.25_dp / e(4)
           w(0,itetra(3)) = w(0,itetra(4))
           w(0,itetra(2)) = w(0,itetra(4))
           w(0,itetra(1)) = w(0,itetra(4))
           !
        ELSE
           !
           ! e(4) = e(3) = e(2)
           !
           w(0,itetra(4)) = dfpt_tetra_lindhard_1211(e(4),e(1),le(4),le(1))
           w(0,itetra(3)) = w(0,itetra(4))
           w(0,itetra(2)) = w(0,itetra(4))
           w(0,itetra(1)) = dfpt_tetra_lindhard_1222(e(1),e(4),le(1),le(4))
           !
           IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
              WRITE(*,'(100e15.5)') e(1:4)
              WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
              CALL errore("lindhard", "4=3=2", 0)
           END IF
           !
        END IF
     ELSE IF(ABS(e(2) - e(1)) < thr ) THEN
        !
        ! e(4) = e(3), e(2) = e(1)
        !
        w(0,itetra(4)) = dfpt_tetra_lindhard_1221(e(4),e(2), le(4),le(2))
        w(0,itetra(3)) = w(0,itetra(4))
        w(0,itetra(2)) = dfpt_tetra_lindhard_1221(e(2),e(4), le(2),le(4))
        w(0,itetra(1)) = w(0,itetra(2))
        !
        IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
           WRITE(*,'(100e15.5)') e(1:4)
           WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
           CALL errore("lindhard", "4=3 2=1", 0)
        END IF
        !
     ELSE
        !
        ! e(4) = e(3)
        !
        w(0,itetra(4)) = dfpt_tetra_lindhard_1231(e(4),e(1),e(2),le(4),le(1),le(2))
        w(0,itetra(3)) = w(0,itetra(4))
        w(0,itetra(2)) = dfpt_tetra_lindhard_1233(e(2),e(1),e(4),le(2),le(1),le(4))
        w(0,itetra(1)) = dfpt_tetra_lindhard_1233(e(1),e(2),e(4),le(1),le(2),le(4))
        !
        IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
           WRITE(*,'(100e15.5)') e(1:4)
           WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
           CALL errore("lindhard", "4=3", 0)
        END IF
        !
     END IF
  ELSE IF(ABS(e(3) - e(2)) < thr) THEN
     IF(ABS(e(3) - e(1)) < thr) THEN
        !
        ! e(3) = e(2) = e(1)
        !
        w(0,itetra(4)) = dfpt_tetra_lindhard_1222(e(4),e(3), le(4),le(3))
        w(0,itetra(3)) = dfpt_tetra_lindhard_1211(e(3),e(4), le(3),le(4))
        w(0,itetra(2)) = w(0,itetra(3))
        w(0,itetra(1)) = w(0,itetra(3))
        !
        IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
           WRITE(*,'(100e15.5)') e(1:4)
           WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
           CALL errore("lindhard", "3=2=1", 0)
        END IF
        !
     ELSE
        !
        ! e(3) = e(2)
        !
        w(0,itetra(4)) = dfpt_tetra_lindhard_1233(e(4),e(1),e(3),le(4),le(1),le(3))
        w(0,itetra(3)) = dfpt_tetra_lindhard_1231(e(3),e(1),e(4),le(3),le(1),le(4))
        w(0,itetra(2)) = w(0,itetra(3))
        w(0,itetra(1)) = dfpt_tetra_lindhard_1233(e(1),e(4),e(3),le(1),le(4),le(3))
        !
        IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
           WRITE(*,'(100e15.5)') e(1:4)
           WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
           CALL errore("lindhard", "3=2", 0)
        END IF
        !
     END IF
  ELSE IF(ABS(e(2) - e(1)) < thr) THEN
     !
     ! e(2) = e(1)
     !
     w(0,itetra(4)) = dfpt_tetra_lindhard_1233(e(4),e(3),e(2),le(4),le(3),le(2))
     w(0,itetra(3)) = dfpt_tetra_lindhard_1233(e(3),e(4),e(2),le(3),le(4),le(2))
     w(0,itetra(2)) = dfpt_tetra_lindhard_1231(e(2),e(3),e(4),le(2),le(3),le(4))
     w(0,itetra(1)) = w(0,itetra(2))
     !
     IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
        WRITE(*,'(100e15.5)') e(1:4)
        WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
        CALL errore("lindhard", "2=1", 0)
     END IF
     !
  ELSE
     !
     ! DIFferent each other.
     !
     w(0,itetra(4)) = dfpt_tetra_lindhard_1234(e(4),e(1),e(2),e(3),le(4),le(1),le(2),le(3))
     w(0,itetra(3)) = dfpt_tetra_lindhard_1234(e(3),e(1),e(2),e(4),le(3),le(1),le(2),le(4))
     w(0,itetra(2)) = dfpt_tetra_lindhard_1234(e(2),e(1),e(3),e(4),le(2),le(1),le(3),le(4))
     w(0,itetra(1)) = dfpt_tetra_lindhard_1234(e(1),e(2),e(3),e(4),le(1),le(2),le(3),le(4))
     !
     IF(ANY(REAL(w(0,itetra(1:4)), dp) < 0.0_dp)) THEN
        WRITE(*,'(100e15.5)') e(1:4)
        WRITE(*,'(100e15.5)') REAL(w(0,itetra(1:4)), dp)
        CALL errore("lindhard", "Something wrong.", 0)
     END IF
     !
  END IF
  !
  ! Dynamical part
  !
  DO imf = 1, nmf
     w(imf,1:4) = 0.25_dp / CMPLX(de(1:4), mf(imf), DP)
  END DO
  !
END SUBROUTINE lindhard
!
END MODULE sctk_tetra
