!
! Copyright (C) 2017 Mitsuaki Kawamura
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
MODULE sctk_read_files
  !
  IMPLICIT NONE
  !
CONTAINS
!
! Read elph*.dat
!
SUBROUTINE read_elph()
  !
  USE kinds, ONLY : DP
  USE mp_world, ONLY : world_comm
  USE modes, ONLY : nmodes
  USE io_global, ONLY : ionode, ionode_id, stdout
  USE mp, ONLY : mp_bcast
  USE disp,  ONLY : nq1, nq2, nq3, x_q, nqs
  USE el_phon, ONLY : elph_nbnd_min, elph_nbnd_max
  !
  USE sctk_val, ONLY : gg0, nqbz, omg0
  !
  USE sctk_cnt_dsp, ONLY : cnt_and_dsp
  !
  IMPLICIT NONE
  !
  INTEGER :: nb0, iq, fi, iqv(3), cnt, dsp
  REAL(dp) :: qvec(3)
  !
  CHARACTER(LEN=6) :: int_to_char
  INTEGER, EXTERNAL :: find_free_unit
  !
  ! Read # of k, bands, modes
  !
  IF(ionode) THEN
     fi = find_free_unit()
     OPEN(fi, file = "elph1.dat" )
     READ(fi,*) nq1, nq2, nq3
     READ(fi,*) elph_nbnd_min, elph_nbnd_max
     READ(fi,*) qvec(1:3)
     READ(fi,*) nmodes
     CLOSE(fi)
  END IF
  !
  CALL mp_bcast(nq1, ionode_id, world_comm)
  CALL mp_bcast(nq2, ionode_id, world_comm)
  CALL mp_bcast(nq3, ionode_id, world_comm)
  CALL mp_bcast(nmodes, ionode_id, world_comm)
  CALL mp_bcast(elph_nbnd_min, ionode_id, world_comm)
  CALL mp_bcast(elph_nbnd_max, ionode_id, world_comm)
  WRITE(stdout,*) "  k grid for MEs ", nq1, nq2, nq3
  WRITE(stdout,*) "  # of modes : ", nmodes
  !
  ! Search irreducible k points
  !
  CALL q_points()
  nqbz = nq1 * nq2 * nq3
  !
  CALL cnt_and_dsp(nqs,cnt,dsp)
  !
  ALLOCATE(gg0(nmodes,elph_nbnd_min:elph_nbnd_max,elph_nbnd_min:elph_nbnd_max,nqbz,cnt), omg0(nmodes,cnt))
  !
  ! Read omega & gg
  !
  !$OMP PARALLEL DEFAULT(NONE) &
  !$OMP & SHARED(cnt,dsp,nmodes,elph_nbnd_min,elph_nbnd_max,nqbz,nq1,nq2,nq3,x_q,omg0,gg0) &
  !$OMP & PRIVATE(fi,iq,nb0,qvec,iqv)
  !
  !$OMP DO
  DO iq = 1, cnt
     !
     !$OMP CRITICAL
     fi = find_free_unit()
     OPEN(fi, file = "elph"//TRIM(int_to_char(iq))//".dat", form = 'unformatted')
     !$OMP END CRITICAL
     !
     READ(fi,*) iqv(1:3)
     IF(.NOT. ALL(iqv(1:3) == (/nq1, nq2, nq3/))) &
     & CALL errore("read_elph", "k grid. iq = ", dsp + iq)
     !
     READ(fi,*) iqv(1:2)
     IF(iqv(1) /= elph_nbnd_min) &
     & CALL errore("read_elph", "First band. iq = ", dsp + iq)
     !
     IF(iqv(2) /= elph_nbnd_max) &
     & CALL errore("read_elph", "Last band. iq = ", dsp + iq)
     !
     READ(fi,*) qvec(1:3)
     !
     IF(ANY(ABS(qvec(1:3) - x_q(1:3,iq + dsp)) > 1e-5_dp)) &
     & CALL errore("read_elph", "qvec. iq = ", dsp + iq)
     !
     READ(fi,*) nb0
     IF(nb0 /= nmodes) &
     & CALL errore("read_elph", "# of modes. iq = ", dsp + iq)
     !     
     READ(fi,*) omg0(1:nmodes,iq)
     READ(fi,*) gg0(1:nmodes,elph_nbnd_min:elph_nbnd_max,elph_nbnd_min:elph_nbnd_max,1:nqbz,iq)
     !
     CLOSE(fi)
     !
  END DO
  !$OMP END DO
  !
  !$OMP END PARALLEL
  !
END SUBROUTINE read_elph
!
! Read Screened Coulomb matrix elements
!
SUBROUTINE read_Coulomb()
  !
  USE wvfct, ONLY : nbnd
  USE kinds, ONLY : DP
  USE mp_world, ONLY : world_comm
  USE io_global, ONLY : ionode, ionode_id, stdout
  USE mp, ONLY : mp_bcast
  USE disp,  ONLY : nq1, nq2, nq3, x_q, nqs
  !
  USE sctk_val, ONLY : ncf, nqbz, nmf, Vc0
  !
  USE sctk_cnt_dsp, ONLY : cnt_and_dsp
  !
  IMPLICIT NONE
  !
  INTEGER :: fi, iq, iqv(3), nb0, cnt, dsp
  REAL(dp) :: qvec(3)
  !
  CHARACTER(LEN=6) :: int_to_char
  INTEGER, EXTERNAL :: find_free_unit
  !
  ! READ # of points for Chebyshev interpolation
  !
  IF(ionode) THEN
     fi = find_free_unit()
     OPEN(fi, file = "vel1.dat", form = 'unformatted')
     READ(fi) iqv(1:3)
     READ(fi) nb0
     READ(fi) qvec(1:3)
     READ(fi) ncf
     CLOSE(fi)
  END IF
  !
  CALL mp_bcast(ncf, ionode_id, world_comm)
  WRITE(stdout,*) "  Dimension of Chebyshev interpolation : ", ncf
  !
  CALL cnt_and_dsp(nqs,cnt,dsp)
  ALLOCATE(vc0(ncf,nbnd,nbnd,nqbz,cnt))
  !
  !$OMP PARALLEL DEFAULT(NONE) &
  !$OMP & SHARED(cnt,dsp,nqs,nqbz,nbnd,vc0,nq1,nq2,nq3,x_q,ncf) &
  !$OMP & PRIVATE(fi,iq,nb0,iqv,qvec)
  !
  !$OMP DO
  DO iq = 1, cnt
     !
     !$OMP CRITICAL
     fi = find_free_unit()
     OPEN(fi, file = "vel"//TRIM(int_to_char(iq))//".dat", form = 'unformatted')
     !$OMP END CRITICAL
     !
     READ(fi) iqv(1:3)
     IF(.NOT. ALL(iqv(1:3) == (/nq1,nq2,nq3/))) &
     &  CALL errore("rea_coulomb", "kgrid. iq = ", dsp + iq)
     !
     READ(fi) nb0
     IF(nbnd /= nb0) &
     &  CALL errore("rea_coulomb", "# of bands. iq = ", dsp + iq)
     !
     READ(fi) qvec(1:3)
     IF(ANY(ABS(qvec(1:3) - x_q(1:3,iq + dsp)) > 1e-5_dp)) &
     &  CALL errore("rea_coulomb", "q point. iq = ", dsp + iq)
     !
     READ(fi) nb0
     IF(nb0 /= ncf) &
     &  CALL errore("rea_coulomb", "# of Chevyshev interpolation. iq = ", dsp + iq)
     !
     READ(fi) vc0(1:ncf,1:nbnd,1:nbnd,1:nqbz,iq)
     !
     CLOSE(fi)
     !
  END DO
  !$OMP END DO
  !
  !$OMP END PARALLEL
  !
  IF(nmf < 0) THEN
     vc0(1:ncf,1:nbnd,1:nbnd,1:nqbz,1:cnt) = 0.0_dp
     nmf = 0
  END IF
  !
END SUBROUTINE read_Coulomb
!
! Read from data-file.xml
!
SUBROUTINE read_a2fsave()
  !
  USE kinds, ONLY : DP
  USE parameters, ONLY : npk
  USE mp_world, ONLY : world_comm
  USE io_files, ONLY : prefix
  USE io_global, ONLY : ionode, ionode_id, stdout
  USE mp, ONLY : mp_bcast
  USE start_k, ONLY : nk1, nk2, nk3
  USE fermisurfer_common, ONLY : rotate_k_fs
  USE wvfct, ONLY : nbnd, wg, et
  USE klist, ONLY : nks, xk, wk, nelec
  USE ener, ONLY : ef
  USE cell_base, ONLY : at, bg
  USE ktetra, ONLY : tetra, ntetra
  USE lsda_mod, ONLY : nspin, isk
  USE ktetra, ONLY : opt_tetra_init, opt_tetra_weights
  !
  INTEGER :: fi, nbnd0, ik, i1, i2, i3, s_dummy(3,3,48), t_rev_dummy(48)
  INTEGER,ALLOCATABLE :: equiv(:,:,:)
  REAL(8),ALLOCATABLE :: et0(:,:)
  INTEGER, EXTERNAL :: find_free_unit
  !
  IF(ALLOCATED(et)) DEALLOCATE(et)
  IF(ALLOCATED(wg)) DEALLOCATE(wg)
  !
  IF ( ionode ) THEN
     fi = find_free_unit()
     OPEN(fi, file = TRIM(prefix) // ".a2Fsave")
     READ(fi,*) nbnd0, nks
     IF (nbnd0 /= nbnd ) CALL errore('read_a2fsave','# of band is incorrect', nbnd0)
     ALLOCATE(et0(nbnd0,nks))
     READ(fi,*) et0(1:nbnd,1:nks)
     READ(fi,*) xk(   1:3,1:nks)
     READ(fi,*) wk(       1:nks)
     READ(fi,*) nk1, nk2, nk3
     CLOSE(fi)
  END IF
  !
  CALL mp_bcast (nk1, ionode_id, world_comm)
  CALL mp_bcast (nk2, ionode_id, world_comm)
  CALL mp_bcast (nk3, ionode_id, world_comm)
  CALL mp_bcast (nks, ionode_id, world_comm)
  IF(.NOT. ionode) ALLOCATE(et0(nbnd,nks))
  CALL mp_bcast (et0, ionode_id, world_comm)
  CALL mp_bcast (xk, ionode_id, world_comm)
  CALL mp_bcast (wk, ionode_id, world_comm)
  !
  ! ... Find equivalent k point in irr-BZ for whole BZ
  !
  WRITE(stdout,*) "  Dense k-grid : ", nk1, nk2, nk3
  WRITE(stdout,*) "  # of k(dense) : ", nks
  ALLOCATE(equiv(nk1, nk2, nk3))
  CALL rotate_k_fs(equiv)
  !
  ! ... Map e_k into whole BZ (Measured from E_F)
  !
  nks = nk1 * nk2 * nk3
  ALLOCATE(et(nbnd,nks), wg(nbnd,nks))
  ik = 0
  DO i1 = 1, nk1
     DO i2 = 1, nk2
        DO i3 = 1, nk3
           ik = ik + 1
           et(1:nbnd,ik) = et0(1:nbnd, equiv(i1,i2,i3))
           xk(1:3,ik) = REAL((/i1, i2, i3/), DP) / REAL((/nk1, nk2, nk3/), DP)
           xk(1:3,ik) = MATMUL(bg(1:3,1:3), xk(1:3,ik))
           wk(ik) = 1.0_dp / REAL(nks, DP)
        END DO
     END DO
  END DO
  !
  ! ... Find Fermi energy
  !
  IF(ALLOCATED(tetra)) DEALLOCATE(tetra)
  s_dummy(1:3,1:3,1:48) = 0
  t_rev_dummy(1:48) = 0
  DO i1 = 1, 3
     s_dummy(i1,i1,1:48) = 1
  END DO
  CALL opt_tetra_init(1, s_dummy, .False., t_rev_dummy, at, bg, npk, &
  &                   0, 0, 0, nk1, nk2, nk3, nks, xk, 1)
  !
  CALL opt_tetra_weights (nks, nspin, nbnd, nelec, ntetra,&
  &                       tetra, et, ef, wg, 0, isk)
  !
  DEALLOCATE(et0,equiv)
  !
END SUBROUTINE read_a2fsave
!
END MODULE sctk_read_files
