!
! Copyright (C) 2017 Mitsuaki Kawamura
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
MODULE sctk_broyden
  !
  IMPLICIT NONE
  !
CONTAINS
!>
!> Solve gap equation with the Broyden method
!>
SUBROUTINE broyden_gapeq()
  !
  USE kinds, ONLY : DP
  USE mp_world, ONLY : mpime
  USE io_global, ONLY : stdout
  USE control_flags, ONLY : niter, tr2
  USE sctk_val, ONLY : delta, dk, emin, ngap, ngap1, ngap2, xi
  !
  USE sctk_gapeq_rhs, ONLY : gapeq_rhs
  USE sctk_io_delta, ONLY : out_delta
  !
  INTEGER :: itr, jtr
  REAL(dp) :: res, alpha = 0.2_dp, dd(ngap,2), rhs(ngap,2), rhs0(ngap,2), drhs(ngap,2), &
  &          jacob1(ngap,2,niter), jacob2(ngap,2,niter), ave(2), dave(2)
  LOGICAL :: lfermi(ngap)
  REAL(dp),EXTERNAL :: ddot
  !
  itr = 0
  WRITE(stdout,*) "  Iteration ", itr
  !
  CALL gapeq_rhs(rhs)
  res = ddot(ngap * 2, rhs, 1, rhs, 1)
  res = SQRT(res) / REAL(ngap * 2, dp)
  !
  ! $$$$$  Information of conversience  $$$$$
  !
  ave(1:2) = 0.0_dp
  dave(1:2) = 0.0_dp
  !
  lfermi(1:ngap1) = ABS(xi(1:ngap1,1)) < emin * 1e-2_dp
  dave(1) = SUM(delta(1:ngap1,1) * dk(1:ngap1,1), lfermi(1:ngap1))
  ave(1)  = SUM(                 dk(1:ngap1,1), lfermi(1:ngap1))
  !
  lfermi(1:ngap2) = ABS(xi(1:ngap2,2)) < emin * 1e-2_dp
  dave(2) = SUM(delta(1:ngap2,2) * dk(1:ngap2,2), lfermi(1:ngap2))
  ave(2)  = SUM(                 dk(1:ngap2,2), lfermi(1:ngap2))
  !
  dave(1:2) = dave(1:2) / ave(1:2) * 13.6057d3
  !
  WRITE(stdout,*) "      Residual[Ry] : ", res
  WRITE(stdout,*) "       Delta [meV] : ", dave(1:2)
  WRITE(stdout,*) ""
  !
  ! $$$$$  End information of conversience  $$$$$
  !
  IF(res < tr2) GOTO 5
  !
  dd(1:ngap,1:2) = - alpha * rhs(1:ngap,1:2)
  !
  DO itr = 1, niter
     !
     WRITE(stdout,*) "  Iteration ", itr
     !
     delta(1:ngap,1:2) = delta(1:ngap,1:2) + dd(1:ngap,1:2)
     !
     rhs0(1:ngap,1:2) = rhs(1:ngap,1:2)
     CALL gapeq_rhs(rhs)
     res = ddot(ngap, rhs, 1, rhs, 1)
     res = SQRT(res) / REAL(ngap, dp)
     !
     ! $$$$$  Information of conversience  $$$$$
     !
     ave(1:2) = 0.0_dp
     dave(1:2) = 0.0_dp
     !
     lfermi(1:ngap1) = ABS(xi(1:ngap1,1)) < emin * 1e-2_dp
     dave(1) = SUM(delta(1:ngap1,1) * dk(1:ngap1,1), lfermi(1:ngap1))
     ave(1)  = SUM(                 dk(1:ngap1,1), lfermi(1:ngap1))
     !
     lfermi(1:ngap2) = ABS(xi(1:ngap2,2)) < emin * 1e-2_dp
     dave(2) = SUM(delta(1:ngap2,2) * dk(1:ngap2,2), lfermi(1:ngap2))
     ave(2)  = SUM(                 dk(1:ngap2,2), lfermi(1:ngap2))
     !
     dave(1:2) = dave(1:2) / ave(1:2) * 13.6057d3
     !
     WRITE(stdout,*) "      Residual[Ry] : ", res
     WRITE(stdout,*) "       Delta [meV] : ", dave(1:2)
     WRITE(stdout,*) ""
     !
     ! $$$$$  End information of conversience  $$$$$
     !
     IF(res < tr2) THEN
        !       
        delta(1:ngap,1) = delta(1:ngap,1) * SIGN(1.0_dp, dave(1))
        delta(1:ngap,2) = delta(1:ngap,2) * SIGN(1.0_dp, dave(2))
        !
        GOTO 5
        !
     END IF
     !
     IF(mpime == 0) THEN
        IF(MODULO(itr, 2) == 0) THEN
           CALL out_delta("delta.evn")
           CALL system("rm -f delta.odd")
        ELSE
           CALL out_delta("delta.odd")
           CALL system("rm -f delta.evn")
        END IF
     END IF
     !
     ! Update Jacobian with drhs
     !
     drhs(1:ngap,1:2) = rhs(1:ngap,1:2) - rhs0(1:ngap,1:2)
     !
     jacob1(1:ngap,1:2,itr) = - alpha * drhs(1:ngap,1:2)
     DO jtr = 1, itr - 1
        jacob1(1:ngap,1:2,itr) = jacob1(1:ngap,1:2,itr) - jacob1(1:ngap,1:2,jtr) &
        &          * ddot(ngap, jacob2(1:ngap,1:2,jtr), 1, drhs(1:ngap,1:2), 1)
     END DO
     jacob1(1:ngap,1:2,itr) = dd(1:ngap,1:2) + jacob1(1:ngap,1:2,itr)
     jacob2(1:ngap,1:2,itr) = drhs(1:ngap,1:2) / ddot(ngap, drhs(1:ngap,1:2), 1, drhs(1:ngap,1:2), 1)
     !
     ! Compute dd with new Jacobian & rhs
     !
     dd(1:ngap,1:2) = - alpha * rhs(1:ngap,1:2)
     DO jtr = 1, itr
        dd(1:ngap,1:2) = dd(1:ngap,1:2) - jacob1(1:ngap,1:2,jtr) &
        &        * ddot(ngap, jacob2(1:ngap,1:2,jtr), 1, rhs(1:ngap,1:2), 1)
     END DO
     !
  END DO ! itr
  !
  IF(mpime == 0) THEN
     CALL out_delta("delta.dat")
     CALL system("rm -f delta.evn delta.odd")
  END IF
  !
  WRITE(stdout,*) ""
  WRITE(stdout,*) '  Not converged! res = ',res
  RETURN
  !
5 continue
  !
  IF(mpime == 0) THEN
     CALL out_delta("delta.dat")
     CALL system("rm -f delta.evn delta.odd")
  END IF
  !
  WRITE(stdout,*) ""
  WRITE(stdout,*) '  Converged! iter = ',itr
  !
END SUBROUTINE broyden_gapeq
  !
END MODULE sctk_broyden
