﻿using System;
using System.Linq;
using System.Text;
using System.Dynamic;
using System.Runtime;

using Vintagestory.API.Common;
using Vintagestory.API.MathTools;
using Vintagestory.API.Config;
using Vintagestory.API.Server;
using Vintagestory.API.Client;

using Vintagestory.GameContent;


//using Vintagestory.ServerMods;

namespace AdjustorMod
{
	public class ItemAdjustor : Item
	{
		private ICoreAPI Api { get; set; }
		private ICoreServerAPI ServerApi { get; set; }
		private ILogger Logger { get; set; }
		private ICoreClientAPI ClientApi { get; set; }

		private const string _meshAngleKey = @"MeshAngle";
		private const string _toolMode = @"toolMode";
		private const string _sideKey = @"side";
		private const string _typeKey = @"type";

		private const string _rotKey = @"rot";//Slabs use this key instead....
		private const string _axialRotationKey = @"rotation";//Trees and Pillars; ns, we, ud - axial rotation
		private const string _verticalCode = @"vertical";


		private const string _chuteTypeStraight = @"straight";
		private const string _chuteTypeCross = @"cross";
		private const string _chuteTypeTjunction = @"t";


		private static string _simpleCoatingClass = @"BlockSimpleCoating";//Similar to slabs...


		private static string[] _cardinalCodes = new string[]
		{
			@"north",
			@"south",
			@"east",
			@"west",
		};

		private static string[] _verticalCodes = new string[]
		{
			@"up",
			@"down",
		};

		private static string[] _axialRotationCodes = new string[]
		{			
			//tree logs, and tree log-like things; pillars, axles, some chutes....;
			@"ud",
			@"ns",
			@"we",
			@"ud-n", 
			@"ud-e", 
			@"ud-s", 
			@"ud-w"
		};

		private static string[] _blacklistedClasses = new string[]
		{
			@"BlockBed",
			@"BlockDoor",
			@"BlockTroughDoubleBlock",
			@"BlockFence", //FIXME: Blacklist of Multi-block entites until futher notice.
			@"BlockAxle",
			@"BlockWindmillRotor",
			@"BlockAngledGears",
		};

		private static string[] _blacklistedEntityClasses = new string[]
		{
			//@"GenericTypedContainer",
		};


		private static string[] _verticalVariantKeys = new string[]
		{
			"verticalorientation",
			"rot",
			"side"//linen, plates, ect... used inconsistently
		};

		private static string[] _horizontalVariantKeys = new string[]
		{
			"horizontalorientation",
			"rot",
			"side"
		};

		private static string[] _slabesqueCodes = new string[]
		{
			"slab",//many things with 'slab' in name, some at begin some at end...

		};



		public override void OnLoaded(ICoreAPI api)
		{
			base.OnLoaded(api);
			this.Api = base.api;

			if (api.Side.IsServer( )) {
				this.ServerApi = ( ICoreServerAPI )api;
				this.Logger = this.ServerApi.World.Logger;
			}

			if (api.Side.IsClient( )) {
				this.ClientApi = ( ICoreClientAPI )api;
				this.Logger = this.ClientApi.World.Logger;
			}
			#if DEBUG
			Logger.VerboseDebug("{0} >> ItemAdjustor ~ OnLoaded", this.Code);
			#endif
		}

		public override bool OnHeldInteractCancel(float secondsUsed, ItemSlot slot, EntityAgent byEntity, BlockSelection blockSel, EntitySelection entitySel, EnumItemUseCancelReason cancelReason)
		{
			return false;
		}

		/// <summary>
		/// Check if the block should even be considered for Adjustment...(On CLIENT)
		/// </summary>
		/// <param name="itemslot">Itemslot.</param>
		/// <param name="byEntity">By entity.</param>
		/// <param name="blockSel">Block sel.</param>
		/// <param name="entitySel">Entity sel.</param>
		/// <param name="handHandling">Hand handling.</param>
		public override void OnHeldInteractStart(ItemSlot itemslot, EntityAgent byEntity, BlockSelection blockSel, EntitySelection entitySel, bool firstEvent, ref EnumHandHandling handHandling)
		{			
			if (blockSel == null) {
				return;
			}

			handHandling = EnumHandHandling.Handled;

			if (api.Side.IsClient( )) {

				if (blockSel != null) {
					BlockPos position = blockSel.Position;
					Block thatBlock = byEntity.World.BlockAccessor.GetBlock(position);
					var mode = CurrentRotationMode(itemslot);

					if (RotatableByVariant(thatBlock) || RotatableByBehavior(thatBlock) || RotatableByAngledMeshEntity(thatBlock, position) || thatBlock.IsChute( )) {
						#if DEBUG
						Logger.VerboseDebug("[{0}] Looks Rotatable, FromFace: {1}", thatBlock.Code.Path, blockSel.Face.Code);
						ClientApi.ShowChatMessage(string.Format("Appears Rotatable: {0}", thatBlock.Code));
						#endif

						handHandling = EnumHandHandling.PreventDefault;
					} else {
						#if DEBUG
						Logger.VerboseDebug(string.Format("Non-Rotatable: {0} !", (thatBlock != null ? thatBlock.Code.ToString( ) : "Entity?")));
						#endif
						handHandling = EnumHandHandling.NotHandled;
					}
				}
			}
		}

		/// <summary>
		/// Perform actual rotate action (On SERVER)
		/// </summary>
		/// <param name="secondsUsed">Seconds used.</param>
		/// <param name="slot">Slot.</param>
		/// <param name="byEntity">By entity.</param>
		/// <param name="blockSel">Block sel.</param>
		/// <param name="entitySel">Entity sel.</param>
		public override bool OnHeldInteractStep(float secondsUsed, ItemSlot slot, EntityAgent byEntity, BlockSelection blockSel, EntitySelection entitySel)
		{
			if (blockSel == null) {
				return false;
			}

			if (api.Side.IsServer( )) {
				bool madeAdjustment = false;

				BlockPos position = blockSel.Position;
				Block thatBlock = byEntity.World.BlockAccessor.GetBlock(position);
				IPlayer thePlayer = null;

				if (byEntity is EntityPlayer) {
					thePlayer = ServerApi.World.PlayerByUid((( EntityPlayer )byEntity).PlayerUID);
				}

				if (IsBlacklisted(thatBlock)) {
					ServerApi.SendMessage(thePlayer, GlobalConstants.CurrentChatGroup, "That isn't adjustable...", EnumChatType.Notification);
					return false;
				}

				EnumWorldAccessResponse why = ServerApi.World.Claims.TestAccess(thePlayer, position, EnumBlockAccessFlags.BuildOrBreak);
				if (why != EnumWorldAccessResponse.Granted) {
					ServerApi.SendMessage(thePlayer, GlobalConstants.CurrentChatGroup, string.Format("Not permitted, {0}", why), EnumChatType.Notification);
					return false;
				}
				var meshSpin = RotatableByAngledMeshEntity(thatBlock, position);

				if (meshSpin) {
					madeAdjustment = true;
					MeshRotation(this.CurrentRotationMode(slot), thePlayer, thatBlock, position);
				}
				else if (this.CurrentRotationMode(slot) == RotationModes.Free) {
				madeAdjustment = FreeRotation(blockSel, byEntity, thePlayer, thatBlock, position);
				}
				else {
				madeAdjustment = FixedRotation(this.CurrentRotationMode(slot), blockSel, byEntity, thePlayer, thatBlock, position);
				}

				if (madeAdjustment) DamageItem(ServerApi.World, byEntity, slot);

				return secondsUsed < 0.7;
			}

			return false;
		}

		/// <summary>
		/// For tooltip. (rotate state)
		/// </summary>
		/// <returns>The held item info.</returns>
		/// <param name="stack">Stack.</param>
		/// <param name="dsc">Dsc.</param>
		/// <param name="world">World.</param>
		/// <param name="withDebugInfo">With debug info.</param>
		public override void GetHeldItemInfo(ItemSlot inSlot, StringBuilder dsc, IWorldAccessor world, bool withDebugInfo)
		{
			base.GetHeldItemInfo(inSlot, dsc, world, withDebugInfo);
			//Toolmode
			dsc.AppendFormat("Rotation Mode: {0}\n",this.CurrentRotationMode(inSlot.Itemstack));
		}

		#region Tool Modes

		/// <summary>
		/// Show Rotation mode changer?
		/// </summary>
		/// <returns>The quantity tool modes.</returns>
		/// <param name="slot">Slot.</param>
		/// <param name="byPlayer">By player.</param>
		/// <param name="blockSelection">Block selection.</param>
		/// <remarks>How clunky; </remarks>
		/* 

		RegisterToolModes<T: Enum>(); --Mabey even a GUI callback
		
		*/

		public override SkillItem[ ] GetToolModes(ItemSlot slot, IClientPlayer forPlayer, BlockSelection blockSel)
		{
		if (blockSel == null) return null;

		Block block = forPlayer.Entity.World.BlockAccessor.GetBlock(blockSel.Position);

		if (block.MatterState == EnumMatterState.Solid)
			return AdjustorSkills.DefaultSkills( api as ICoreClientAPI);
		else return null;
		}



		public override int GetToolMode(ItemSlot slot, IPlayer byPlayer, BlockSelection blockSelection)
		{
			return slot.Itemstack.Attributes.GetInt(_toolMode);
		}

		public override void SetToolMode(ItemSlot slot, IPlayer byPlayer, BlockSelection blockSelection, int toolMode)
		{
			slot.Itemstack.Attributes.SetInt(_toolMode, toolMode);
		}

		/// <summary>
		/// Internal decoded 'toolmode'
		/// </summary>
		/// <returns>The rotation mode.</returns>
		/// <param name="slot">Slot.</param>
		public RotationModes CurrentRotationMode(ItemSlot slot)
		{
			if (slot.Itemstack.Attributes.HasAttribute(_toolMode)) {
				return (RotationModes)slot.Itemstack.Attributes.GetInt(_toolMode);
			}
			return RotationModes.Free;
		}

		/// <summary>
		/// Internal decoded 'toolmode'
		/// </summary>
		/// <returns>The rotation mode.</returns>
		/// <param name="stack">ItemStack.</param>
		public RotationModes CurrentRotationMode(ItemStack stack)
		{
			if (stack.Attributes.HasAttribute(_toolMode)) {
				return ( RotationModes )stack.Attributes.GetInt(_toolMode);
			}
			return RotationModes.Free;
		}

		#endregion

		#region Private Methods

		/// <summary>
		/// Determines if its a Blacklisted Entity or BlockEntity
		/// </summary>
		/// <returns><c>true</c> if is blacklisted the specified thatBlock; otherwise, <c>false</c>.</returns>
		/// <param name="thatBlock">That block.</param>
		private static bool IsBlacklisted(Block thatBlock)
		{
			if (_blacklistedEntityClasses.Contains(thatBlock.EntityClass)
				|| _blacklistedClasses.Contains(thatBlock.Class))
				return true;

			return false;
		}

		/// <summary>
		/// Is it Rotatable, possibly with Variant key that is known to be a direction?
		/// </summary>
		/// <returns><c>true</c>, if rotatable, <c>false</c> otherwise.</returns>
		/// <param name="thatBlock">A block.</param>
		private static bool RotatableByVariant(Block thatBlock)
		{
			if (thatBlock != null) {

			if (thatBlock.IsMissing == false && thatBlock.MatterState == EnumMatterState.Solid
				&& (_verticalVariantKeys.Any(vvk => thatBlock.Variant.Keys.Contains(vvk)) ||
				    _horizontalVariantKeys.Any(hvk => thatBlock.Variant.Keys.Contains(hvk)) ||
				    thatBlock.Variant.Keys.Contains(_axialRotationKey)
				   )
			)
				{
					return true;
				}
			}

			return false;
		}

		/// <summary>
		/// Is it Rotatable, possibly via Behavior?
		/// </summary>
		/// <returns><c>true</c>, if rotatable, <c>false</c> otherwise.</returns>
		/// <param name="thatBlock">A block.</param>
		private bool RotatableByBehavior(Block thatBlock)
		{
			if (thatBlock != null) {

				if (thatBlock.BlockBehaviors.Length > 0) {
					#if DEBUG
					Logger.VerboseDebug("Block {0} has {1} #behaviors", thatBlock.Code.ToShortString( ), thatBlock.BlockBehaviors.Length);
					#endif
				}

				if (thatBlock.HasBehavior("OmniRotatable", ClientApi.ClassRegistry)
					|| thatBlock.HasBehavior("HorizontalOrientable", ClientApi.ClassRegistry)
					|| thatBlock.HasBehavior("HorUDOrientable", ClientApi.ClassRegistry)
					|| thatBlock.HasBehavior("NWOrientable", ClientApi.ClassRegistry)
					) {
					return true;
				}
			}

			return false;
		}

		private bool RotatableByAngledMeshEntity(Block thatBlock, BlockPos pos)
		{
		//Coulda: (IAngledMeshBlockEnt >>> public virtual float MeshAngle ) ---but Nooooo....
		//Annnd the 1st thing is to try for the BlockEntity...
		var mabeyBE = Api.World.BlockAccessor.GetBlockEntity(pos.Copy( ));
		if (mabeyBE != null) 
		{
		var someType = mabeyBE.GetType( );
		if (someType.GetProperty(_meshAngleKey) != null ||
			someType.GetField(_meshAngleKey) != null
			) {
			#if DEBUG
			Logger.VerboseDebug("[{0}] apparently supports 'MeshAngle'", thatBlock.Code);
			#endif
			return true;
			}
		
		}


		return false;
		}

		/// <summary>
		/// Determines if slab, sheet, plate type block in a N/E/W/S configuration
		/// </summary>
		/// <returns><c>true</c> if is slab NEW the specified thatBlock; otherwise, <c>false</c>.</returns>
		/// <param name="thatBlock">That block.</param>
		private static bool IsSpecialCaseNEWS(Block thatBlock)
		{
			if (_simpleCoatingClass.Equals(thatBlock.Class, StringComparison.OrdinalIgnoreCase)) {

				if (thatBlock.Variant.ContainsKey(_sideKey)) {

					if (_cardinalCodes.Contains(thatBlock.Variant[_sideKey])) return true;
				}
			}

			if (_slabesqueCodes.Any(thatBlock.Code.Path.Contains)) {

				string rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_horizontalVariantKeys.Contains);

				if (_cardinalCodes.Contains(thatBlock.Variant[rotationKey])) return true;
			}
			return false;
		}

		/// <summary>
		/// Determines if slab, sheet, plate type block in a U/D configuration
		/// </summary>
		/// <returns><c>true</c> if is slab NEW the specified thatBlock; otherwise, <c>false</c>.</returns>
		/// <param name="thatBlock">That block.</param>
		private static bool IsSpecialCaseUD(Block thatBlock)
		{
			if (_simpleCoatingClass.Equals(thatBlock.Class, StringComparison.OrdinalIgnoreCase)) {

				if (thatBlock.Variant.ContainsKey(_sideKey)) {

					if (_verticalCodes.Contains(thatBlock.Variant[_sideKey])) return true;
				}
			}

			if (_slabesqueCodes.Any(thatBlock.Code.Path.Contains)) {

				string rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_verticalVariantKeys.Contains);

				if (_verticalCodes.Contains(thatBlock.Variant[rotationKey])) return true;
			}
			return false;
		}

		/// <summary>
		/// Changes AssetLocation from a N/E/W/S Slab into a UP slab.
		/// </summary>
		/// <returns>The slab vertical flip case.</returns>
		/// <param name="thatBlock">That block.</param>
		private static AssetLocation SpecialVerticalFlipCase(Block thatBlock)
		{			
			string rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_horizontalVariantKeys.Contains);//There better be only 1 dimension here!
			string redirectionPath = null;

			if (!string.IsNullOrEmpty(rotationKey)) {
				string horizontalValue = thatBlock.Variant[rotationKey];
				//Slab in North / East / West / South - but needs to be UP 

				redirectionPath = thatBlock.Code.Path.Replace(horizontalValue, @"up");
			}

			return new AssetLocation(thatBlock.Code.Domain, redirectionPath);
		}

		/// <summary>
		/// Changes AssetLocation from a UP/DOWN Slab into a NORTH slab.
		/// </summary>
		/// <returns>The slab vertical flip case.</returns>
		/// <param name="thatBlock">That block.</param>
		private static AssetLocation SpecialHorizontalFlipCase(Block thatBlock)
		{
			string rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_verticalVariantKeys.Contains);//There better be only 1 dimension here!
			string redirectionPath = null;

			if (!string.IsNullOrEmpty(rotationKey)) {
				string verticalValue = thatBlock.Variant[rotationKey];
				//Slab in Up / Down - but needs to be NORTH

				redirectionPath = thatBlock.Code.Path.Replace(verticalValue, @"north");
			}

			return new AssetLocation(thatBlock.Code.Domain, redirectionPath);
		}

		/// <summary>
		/// For flipping Axial oriented things
		/// </summary>
		/// <returns>The omni flip.</returns>
		/// <param name="thatBlock">That block.</param>
		/// <param name="axis">Axis.</param>
		/// <param name="axisNormal">Axis normal.</param>
		private static AssetLocation AxialOmniFlip(Block thatBlock, EnumAxis axis, bool axisNormal)
		{
			string oppositeDirection = string.Empty;

			//Axial block: ud / ns / we - change to opposite
			string directionName = thatBlock.Variant[_axialRotationKey] ?? thatBlock.Variant[_sideKey];
			if (axisNormal) {
				switch (directionName) {
				case @"ud":
					oppositeDirection = (axis == EnumAxis.X || axis == EnumAxis.Z) ? @"ns" : @"we";
					break;
				case @"ns":
					oppositeDirection = (axis == EnumAxis.Y) ? @"ud" : @"we";
					break;
				case @"we":
					oppositeDirection = (axis == EnumAxis.Y) ? @"ud" : @"ns";
					break;
				}
			} else {
				switch (directionName) {
				case @"ud":
					oppositeDirection = (axis == EnumAxis.X) ? @"ns" : @"we";
					break;
				case @"ns":
					oppositeDirection = @"ud";
					break;
				case @"we":
					oppositeDirection = @"ud";
					break;
				}
			}

			var redirectionPath = thatBlock.Code.Path.Replace(directionName, oppositeDirection);

			return new AssetLocation(thatBlock.Code.Domain, redirectionPath);
		}

		/// <summary>
		/// Custom Block variant Determinator - as Chutes are...."SPECIAL"
		/// </summary>
		/// <returns>The specific flip.</returns>
		/// <param name="someChute">That block.</param>
		/// <param name="axis">Axis.</param>
		/// <param name="rotateHorizontal">Rotate horizontal.</param>
		private AssetLocation ChuteSpecificFlip(Block someChute, EnumAxis axis, bool rotateHorizontal)
		{
		/*Is: ?
		{ code: "vertical", states: ["up", "down" ] },
		{ code: "side", states: ["north", "east", "south", "west"] }
		>> NEWS-UD
		*/
		AssetLocation translatedCode = null;
		string side = someChute.Variant[_sideKey];
		string type = someChute.Variant[_typeKey];

		if (someChute.Variant.ContainsKey(_verticalCode)) 
		{
			//'Standard' Chute >> ["elbow", "3way"] 
			string vert = someChute.Variant[_verticalCode];

			switch (axis) 
			{					
				case EnumAxis.X:
					if (side == "east") { translatedCode = someChute.CodeWithVariant(_sideKey, "west"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "east"); }
				break;

				case EnumAxis.Y:
					if (vert == "up") { translatedCode = someChute.CodeWithVariant(_verticalCode, "down"); }
					else { translatedCode = someChute.CodeWithVariant(_verticalCode, "up"); }
				break;

				case EnumAxis.Z:
					if (side == "north") { translatedCode = someChute.CodeWithVariant(_sideKey, "south"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "north"); }
				break;
			}
		}
		/*
		//////////// Special Case CITY.
		{ code: "side", states: ["ns", "we", "ud-n", "ud-e", "ud-s", "ud-w"] } << t
		{ code: "side", states: ["ns", "we", "ground"] } << cross
		{ code: "side", states: ["ns", "we", "ud"] } << straight
		*/
		else 
		{
		switch (type) {
			case _chuteTypeStraight:
			switch (axis) 
			{
				case EnumAxis.X:
					if (side == "we") { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
				break;

				case EnumAxis.Y:
					if (side == "ud") { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
					else { translatedCode = someChute.CodeWithVariant(_verticalCode, "ud"); }
				break;

				case EnumAxis.Z:
					if (side == "ns") { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
				break;
			}
			break;

			case _chuteTypeCross:
			switch (axis) 
			{
				case EnumAxis.X:
					if (side == "we") { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
				break;

				case EnumAxis.Y:
					if (side == "ground") { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
					else { translatedCode = someChute.CodeWithVariant(_verticalCode, "ground"); }
				break;

				case EnumAxis.Z:
					if (side == "ns") { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
					else { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
				break;
			}
			break;

			case _chuteTypeTjunction:
				bool uprightMode = side.StartsWith("ud-", StringComparison.Ordinal);
				if (uprightMode == false) 
				{
					switch (axis) 
					{
						case EnumAxis.X:
							if (side == "we") { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
							else { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
						break;

						case EnumAxis.Y:
							translatedCode = someChute.CodeWithVariant(_sideKey, "ud-n");
						break;

						case EnumAxis.Z:
							if (side == "ns") { translatedCode = someChute.CodeWithVariant(_sideKey, "we"); }
							else { translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); }
						break;
					}
				}
				else 
				{
					switch (axis) 
					{
						case EnumAxis.X:
							if (side.EndsWith("n", StringComparison.Ordinal)) { translatedCode = someChute.CodeWithVariant(_sideKey, "ud-s"); }
							else { translatedCode = someChute.CodeWithVariant(_sideKey, "ud-n"); }
						break;

						case EnumAxis.Y:
							translatedCode = someChute.CodeWithVariant(_sideKey, "ns");
						break;

						case EnumAxis.Z:
							if (side.EndsWith("e", StringComparison.Ordinal)) { translatedCode = someChute.CodeWithVariant(_sideKey, "ud-w"); }
							else { translatedCode = someChute.CodeWithVariant(_sideKey, "ud-e"); }
						break;
					}
				}
			break;
		}	
		}

		return translatedCode;
		}


		private AssetLocation ChuteSpecificForce(Block someChute, RotationModes direction)
		{
		/*Is: ?
		{ code: "vertical", states: ["up", "down" ] },
		{ code: "side", states: ["north", "east", "south", "west"] }
		>> NEWS-UD
		*/
		AssetLocation translatedCode = null;
		string side = someChute.Variant[_sideKey];
		string type = someChute.Variant[_typeKey];

		if (someChute.Variant.ContainsKey(_verticalCode)) {
		//'Standard' Chute >> ["elbow", "3way"] 
		string vert = someChute.Variant[_verticalCode];

		switch (direction) {
		case RotationModes.East:
		case RotationModes.West:
		case RotationModes.North:
		case RotationModes.South:
			translatedCode = someChute.CodeWithVariant(_sideKey, direction.ToString().ToLower()); 			
		break;

		case RotationModes.Up:
		case RotationModes.Down:
			translatedCode = someChute.CodeWithVariant(_verticalCode, direction.ToString( ).ToLower( ));		
		break;

		
		default:
			translatedCode = someChute.Code.Clone();
		break;

		}
		}
		else {
		/*
		//////////// Special Case CITY.
		{ code: "side", states: ["ns", "we", "ud-n", "ud-e", "ud-s", "ud-w"] } << t
		{ code: "side", states: ["ns", "we", "ground"] } << cross
		{ code: "side", states: ["ns", "we", "ud"] } << straight
		*/
		switch (type) {
			case _chuteTypeStraight:
				switch (direction) {
				case RotationModes.West:
				case RotationModes.East:					
					translatedCode = someChute.CodeWithVariant(_sideKey, "we"); 
					break;

				case RotationModes.Up:
				case RotationModes.Down:					
					translatedCode = someChute.CodeWithVariant(_sideKey, "ud"); 
					break;

				case RotationModes.North:
				case RotationModes.South:					
					translatedCode = someChute.CodeWithVariant(_sideKey, "ns"); 
					break;
				}
			break;

			case _chuteTypeCross:
				switch (direction) {
				case RotationModes.West:
				case RotationModes.East:					
					 translatedCode = someChute.CodeWithVariant(_sideKey, "we"); 
				break;
										
				case RotationModes.Up:
				case RotationModes.Down:				
					translatedCode = someChute.CodeWithVariant(_sideKey, "ground"); 
				break;

				case RotationModes.North:
				case RotationModes.South:					
					translatedCode = someChute.CodeWithVariant(_sideKey, "ns");
				break;
				}
			break;

			case _chuteTypeTjunction:
			bool uprightMode = side.StartsWith("ud-", StringComparison.Ordinal);

			if (uprightMode == false) 
				{
					switch (direction) 
					{
					case RotationModes.West:
					case RotationModes.East:
						translatedCode = someChute.CodeWithVariant(_sideKey, "we");
					break;

					case RotationModes.Up:
					case RotationModes.Down:
						translatedCode = someChute.CodeWithVariant(_sideKey, "ud-n");
					break;

					case RotationModes.North:
					case RotationModes.South:
						translatedCode = someChute.CodeWithVariant(_sideKey, "ns");
					break;
					}
				}
				else 
				{

				switch (direction) {
				case RotationModes.North:
					translatedCode = someChute.CodeWithVariant(_sideKey, "ud-n");
				break;
								
				case RotationModes.East:
					translatedCode = someChute.CodeWithVariant(_sideKey, "ud-e");
				break;

				case RotationModes.West:
					translatedCode = someChute.CodeWithVariant(_sideKey, "ud-w");
				break;

				case RotationModes.South:
					translatedCode = someChute.CodeWithVariant(_sideKey, "ud-s");
				break;

				case RotationModes.Up:
				case RotationModes.Down:
					translatedCode = someChute.CodeWithVariant(_sideKey, "ns");
				break;
				}

			}
			break;
			}
		}

		return translatedCode;
		}

		/// <summary>
		/// Is this a axial type thing / wood log / chute?
		/// </summary>
		/// <returns>If its axial.</returns>
		/// <param name="thatBlock">That block.</param>
		private static bool IsAxialOriented(Block thatBlock)
		{
		if (thatBlock.Variant.ContainsKey(_axialRotationKey)) return true;

		//Axial variants - vary...
		if (_axialRotationCodes.Any(arc => arc == thatBlock.Variant[_sideKey])) return true;

		return false;
		}





		/// <summary>
		/// Rotation by any 'free' axis available to block
		/// </summary>
		/// <param name="blockSel">Block sel.</param>
		/// <param name="byEntity">By entity.</param>
		/// <param name="thePlayer">The player.</param>
		private bool FreeRotation(BlockSelection blockSel, EntityAgent byEntity, IPlayer thePlayer, Block thatBlock, BlockPos position)
		{
			bool rotateHorizontal = false;
			AssetLocation renamedAsset = null;

			switch (blockSel.Face.Axis) {
			case EnumAxis.X:
			case EnumAxis.Z:
				rotateHorizontal = !byEntity.Controls.Sneak;
				break;

			case EnumAxis.Y:
				rotateHorizontal = byEntity.Controls.Sneak;
				break;
			}

			if (rotateHorizontal) {
				#if DEBUG
				ServerApi.SendMessage(thePlayer, GlobalConstants.CurrentChatGroup, "Angle N/E/W/S.", EnumChatType.Notification);
				#endif

				//So many HORIZONTAL Special Cases...
				renamedAsset = thatBlock.GetRotatedBlockCode(90);//Except Fences 

				if (IsSpecialCaseUD(thatBlock)) {					
					renamedAsset = SpecialHorizontalFlipCase(thatBlock);
					#if DEBUG
					Logger.VerboseDebug("{0} Special UP/DN Case {1}", thatBlock.Code, renamedAsset);
					#endif
				}

				if (IsAxialOriented(thatBlock)) {					
					renamedAsset = AxialOmniFlip(thatBlock, blockSel.Face.Axis, rotateHorizontal); //Wood logs also special

					#if DEBUG
					Logger.VerboseDebug("{0} Special Axial Case {1}", thatBlock.Code, renamedAsset);
					#endif
				}

				if (thatBlock.IsChute()) {
					renamedAsset = ChuteSpecificFlip(thatBlock, blockSel.Face.Axis, rotateHorizontal);
				}

				if (renamedAsset != thatBlock.Code) {
					#if DEBUG
					Logger.VerboseDebug("{0} Horizontalize: {1}", thatBlock.Code, renamedAsset);
					#endif

					var rotatedBlock = ServerApi.World.GetBlock(renamedAsset);
					if (rotatedBlock != null) {
						ServerApi.World.BlockAccessor.ExchangeBlock(rotatedBlock.BlockId, position);
						ServerApi.TryUnborkChute(rotatedBlock, position);
						return true;
					} else {
						Logger.VerboseDebug("{0} CAN'T EXCHANGE {1}", thatBlock.Code, renamedAsset);
					}
				}
			} else {
				#if DEBUG
				ServerApi.SendMessage(thePlayer, GlobalConstants.CurrentChatGroup, "Angle U/D.", EnumChatType.Notification);
				#endif

				renamedAsset = thatBlock.GetVerticallyFlippedBlockCode( );

				//Special Case for Slabs, Sheets in N/E/S/W mode when trying to rotate them back 'up'
				if (IsSpecialCaseNEWS(thatBlock)) {
					renamedAsset = SpecialVerticalFlipCase(thatBlock);
					#if DEBUG
					Logger.VerboseDebug("{0} Special N/E/W/S Case {1}", thatBlock.Code, renamedAsset);
					#endif
				}

				if (IsAxialOriented(thatBlock)) 
				{
					renamedAsset = AxialOmniFlip(thatBlock, blockSel.Face.Axis, rotateHorizontal); //Wood logs also special
					#if DEBUG
					Logger.VerboseDebug("{0} Special Axial Case {1}", thatBlock.Code, renamedAsset);
					#endif
				}

				if (thatBlock.IsChute( )) {
				renamedAsset = ChuteSpecificFlip(thatBlock, blockSel.Face.Axis, rotateHorizontal);
				}

				if (renamedAsset != thatBlock.Code) {

					#if DEBUG
					Logger.VerboseDebug("{0} Verticallize: {1}", thatBlock.Code, renamedAsset);
					#endif

					var rotatedBlock = ServerApi.World.GetBlock(renamedAsset);
					if (rotatedBlock != null) {
						ServerApi.World.BlockAccessor.ExchangeBlock(rotatedBlock.BlockId, position);
						ServerApi.TryUnborkChute(rotatedBlock, position);
						return true;
					} else {
						Logger.VerboseDebug("{0} CAN'T EXCHANGE {1}", thatBlock.Code, renamedAsset);
					}
				}
			}

			return false;
		}

		/// <summary>
		/// Fixed by type - force to that rotation; 'state'
		/// </summary>
		/// <param name="rotationMode">Rotation modes.</param>
		/// <param name="blockSel">Block sel.</param>
		/// <param name="byEntity">By entity.</param>
		/// <param name="thePlayer">The player.</param>
		/// <param name="thatBlock">That block.</param>
		/// <param name="position">Position.</param>
		private bool FixedRotation(RotationModes rotationMode, BlockSelection blockSel, EntityAgent byEntity, IPlayer thePlayer, Block thatBlock, BlockPos position)
		{
		AssetLocation renamedAsset;
		if (thatBlock.IsChute( )) 
				{	renamedAsset = ChuteSpecificForce(thatBlock, rotationMode);	}
			else { renamedAsset = RenameBlockFace(thatBlock, rotationMode); }

			if (renamedAsset != null) {

				if (renamedAsset.Equals(thatBlock.Code) == false) {
					#if DEBUG
					Logger.VerboseDebug("{0} Renamed: {1}", thatBlock.Code, renamedAsset);
					#endif

					var rotatedBlock = ServerApi.World.GetBlock(renamedAsset);
					if (rotatedBlock != null) {
						ServerApi.World.BlockAccessor.ExchangeBlock(rotatedBlock.BlockId, position);
						ServerApi.TryUnborkChute(rotatedBlock, position);
						return true;
					} else {
						Logger.VerboseDebug("{0} CAN'T EXCHANGE {1}", thatBlock.Code, renamedAsset);
					}
				} else {
					#if DEBUG
					Logger.VerboseDebug("{0} IDENTICAL!? {1}", thatBlock.Code, renamedAsset);
					#endif
				}
			}

			return false;
		}

		/// <summary>
		/// rotation by/to MESH - not block!
		/// </summary>
		/// <returns>The rotation.</returns>
		/// <param name="rotationModes">Rotation modes.</param>
		/// <param name="thePlayer">The player.</param>
		/// <param name="thatBlock">That block.</param>
		/// <param name="position">Position.</param>
		private void MeshRotation(RotationModes rotationModes, IPlayer thePlayer, Block thatBlock, BlockPos position)
		{//Context: SERVER!
		
		dynamic subjectBE = ServerApi.World.BlockAccessor.GetBlockEntity(position);

		//Fix angle to
		switch (rotationModes) {
			case RotationModes.Free:
				subjectBE.MeshAngle += GameMath.DEG2RAD * 45;
				break;		
			//UNIT IS RADIANS !
			case RotationModes.North:
				subjectBE.MeshAngle = GameMath.DEG2RAD * 0;
				break;
			case RotationModes.East:
				subjectBE.MeshAngle = GameMath.DEG2RAD * 90;
				break;
			case RotationModes.West:
				subjectBE.MeshAngle = GameMath.DEG2RAD * 270;
				break;
			case RotationModes.South:
				subjectBE.MeshAngle = GameMath.DEG2RAD * 180;
				break;
			default:
				subjectBE.MeshAngle -= GameMath.DEG2RAD * 45;
				break;
		}		
		#if DEBUG
		Logger.VerboseDebug("[{0}] Mesh angle: {1}(RADs)", thatBlock.Code ,subjectBE.MeshAngle);
		#endif

		subjectBE.MarkDirty(true);		
					
		}

		private AssetLocation RenameBlockFace(Block thatBlock, RotationModes rotationMode)
		{
			string renamedLocationPath = null;
			string rotationKey = null;

			switch (rotationMode) {

			case RotationModes.Up:
			case RotationModes.Down:

				rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_verticalVariantKeys.Contains);//There better be only 1 dimension here!
				if (!string.IsNullOrEmpty(rotationKey))
				{
					string roatableValue = thatBlock.Variant[rotationKey];

					renamedLocationPath = thatBlock.Code.Path.Replace(roatableValue, rotationMode.ToString().ToLowerInvariant() );

					#if DEBUG
					Logger.VerboseDebug("Variant vertical rotation {0}={1} to: {2} byKey[{3}]", rotationKey, roatableValue, renamedLocationPath, rotationKey);
					#endif

				}

				break;

			case RotationModes.North:
			case RotationModes.East:
			case RotationModes.West:
			case RotationModes.South:

				rotationKey = thatBlock.Variant.Keys.FirstOrDefault(_horizontalVariantKeys.Contains);//There better be only 1 dimension here!
				if (!string.IsNullOrEmpty(rotationKey)) {
					string roatableValue = thatBlock.Variant[rotationKey];

					renamedLocationPath = thatBlock.Code.Path.Replace(roatableValue, rotationMode.ToString( ).ToLowerInvariant( ));

					#if DEBUG
					Logger.VerboseDebug("Variant horizontal rotation {0}={1} to: {2} byKey[{3}]", rotationKey, roatableValue, renamedLocationPath, rotationKey);
					#endif

				}
				break;

			default:
				throw new InvalidOperationException("Rotation Mode not Permissible!" );
			}


			if (!string.IsNullOrEmpty(renamedLocationPath)) {

				return new AssetLocation(thatBlock.Code.Domain, renamedLocationPath);
			}

			return null;
		}

		#endregion


	}		
}



//thatBlock.BlockBehaviors[

/*
								{
									name: "OmniRotatable",
									properties: {
										"rotateSides": true,
										"facing": "block"
									}
								}
								*/

// SuggestedHVOrientation
// GetRotatedBlockCode(int angle)
//GetVerticallyFlippedBlockCode()
//GetHorizontallyFlippedBlockCode(EnumAxis