package com.tryjava.ext.xml;

import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXB;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.xml.sax.SAXException;

import com.tryjava.ext.xml.schema.PersonWithXmlns;

/**
 * JAXB（Java Architecture for XML Binding）を使ってみる練習です。
 * <p>
 * ・JAXB（Java Architecture for XML Binding）<br/>
 * ・Java SE 6 以降で使えます。<br/>
 * </p>
 * <p>
 * 参考ページ<br/>
 * ・JAXBと名前空間と - Starlight<br/>
 * http://d.hatena.ne.jp/Kazuhira/20120716/1342441007<br/>
 * </p>
 */
public class Jaxb02Schema {
	public void run() throws JAXBException, SAXException {
		marshalPackageInfo();
		unmarshalPackageInfo();
		marshalXsd();
		unmarshalXsd();
	}

	/**
	 * XMLスキーマを使うXMLを出力する練習です。（スキーマ検証なし）
	 */
	String marshalPackageInfo() throws JAXBException, SAXException {
		// 入力オブジェクト
		// xmlns設定があるパッケージのクラスを使用する。
		PersonWithXmlns person = new PersonWithXmlns();
		person.name = "山田 太郎";
		person.age = 20;

		// 実行
		// 出力XMLにxmlns属性が付く
		StringWriter sw = new StringWriter();
		JAXB.marshal(person, sw);
		String xml = sw.toString();
		System.out.println(xml);

		return xml;
	}

	/**
	 * XMLスキーマ使用を使うXMLを読み込む練習です。
	 */
	PersonWithXmlns unmarshalPackageInfo() {
		// 入力XML
		String input = "" //
				+ "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" //
				+ "<person xmlns=\"http://www.example.org/person/\">\n" //
				+ "    <name>山田 太郎</name>\n" //
				+ "    <age>20</age>\n" //
				+ "</person>\n" //
				;

		// 実行
		StringReader sr = new StringReader(input);
		PersonWithXmlns person = JAXB.unmarshal(sr, PersonWithXmlns.class);
		System.out.println(ToStringBuilder.reflectionToString(person));

		return person;
	}

	/**
	 * XMLスキーマを使うXMLを出力する練習です。（スキーマ検証あり）
	 *
	 * <p>
	 * 上手く出来なくて、次のようなエラーが発生してしまいます。解決方法不明。<br/>
	 * →解決しました。person.xsdに「elementFormDefault="qualified"」を付加することで解決できました。
	 * </p>
	 *
	 * <pre>
	 * javax.xml.bind.MarshalException
	 *  - with linked exception:
	 * [org.xml.sax.SAXParseException; lineNumber: 0; columnNumber: 0; cvc-complex-type.2.4.a: 要素'name'で始まる無効なコンテンツが見つかりました。'{name}'のいずれかが必要です。]
	 * </pre>
	 */
	String marshalXsd() throws JAXBException, SAXException {
		// 入力オブジェクト
		PersonWithXmlns person = new PersonWithXmlns();
		person.name = "山田 太郎";
		person.age = 20;

		// スキーマインスタンス生成
		SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		Schema schema = schemaFactory.newSchema(getClass().getResource("/com/tryjava/text/xml/person.xsd"));

		// Marshallerインスタンス生成
		JAXBContext context = JAXBContext.newInstance(person.getClass());
		Marshaller marshaller = context.createMarshaller();
		marshaller.setSchema(schema);
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

		// 実行
		StringWriter sw = new StringWriter();
		marshaller.marshal(person, sw);
		String xml = sw.toString();
		System.out.println(xml);

		return xml;
	}

	/**
	 * XMLスキーマ使用を使うXMLを読み込む練習です。
	 *
	 * @throws JAXBException
	 * @throws SAXException
	 */
	PersonWithXmlns unmarshalXsd() throws JAXBException, SAXException {
		// 入力XML
		String input = "" //
				+ "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" //
				+ "<person xmlns=\"http://www.example.org/person/\">\n" //
				+ "    <name>山田 太郎</name>\n" //
				+ "    <age>20</age>\n" //
				+ "</person>\n" //
				;

		// スキーマインスタンス生成
		SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		Schema schema = schemaFactory.newSchema(getClass().getResource("/com/tryjava/text/xml/person.xsd"));

		// Unmarshallerインスタンス生成
		JAXBContext context = JAXBContext.newInstance(PersonWithXmlns.class);
		Unmarshaller unmarshaller = context.createUnmarshaller();
		unmarshaller.setSchema(schema);

		// 実行
		StringReader sr = new StringReader(input);
		PersonWithXmlns person = (PersonWithXmlns) unmarshaller.unmarshal(sr);
		System.out.println(ToStringBuilder.reflectionToString(person));

		return person;
	}

}
