package com.tryjava.etc.junit.s19_1;

import java.util.List;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;

/**
 * Listクラスを比較するカスタムMatcher
 */
public class ListMatcher extends BaseMatcher<List<?>> {
	private final Object[] items;
	private List<?> actual;
	private int idx = 0;

	/**
	 * コンストラクタ<br>
	 * 期待値を保持する。
	 *
	 * @param items
	 *            期待値
	 */
	public ListMatcher(Object[] items) {
		this.items = items;
	}

	/**
	 * 比較検証する。<br>
	 *
	 * @param actual
	 *            実測値
	 * @return 期待値と実測値が一致する場合true, それ以外の場合falseを返却する。
	 */
	@Override
	public boolean matches(Object actual) {
		if (!(actual instanceof List)) {
			return false;
		}
		List<?> list = (List<?>) actual;
		this.actual = list;
		if (list.size() != items.length) {
			return false;
		}
		for (Object obj : list) {
			Object other = items[idx];
			if (obj != null && !obj.equals(other)) {
				return false;
			} else if (obj == null && other != null) {
				return false;
			}
			idx++;
		}
		return true;
	}

	/**
	 * 比較検証で不一致の場合のメッセージを組み立てる。
	 */
	@Override
	public void describeTo(Description desc) {
		if (actual == null) {
			desc.appendValue(items);
		} else {
			desc.appendValue(items[idx]).appendText(", but ")
					.appendValue(actual.get(idx))
					.appendText(" at index of " + idx);
		}
	}

	/**
	 * ファクトリメソッド<br>
	 *
	 * 使用例
	 *
	 * <pre>
	 * import static tryjunit.s19_1.ListMatcher.*;
	 * assertThat(actual, is(list(1, 2, 3)));
	 * </pre>
	 *
	 * @param items
	 *            期待値
	 * @return Matcherオブジェクト
	 */
	public static <T> Matcher<List<?>> list(T... items) {
		return new ListMatcher(items);
	}
}
