package com.tryjava.etc.junit.s19_1;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;

/**
 * 日付を比較するカスタムMatcher
 */
public class DateMatcher extends BaseMatcher<Date> {
	private final int expectedYear;
	private final int expectedMonth;
	private final int expectedDay;
	private Date actualDate;

	/**
	 * コンストラクタ<br>
	 * 期待値を保持する。
	 */
	public DateMatcher(int year, int month, int day) {
		this.expectedYear = year;
		this.expectedMonth = month;
		this.expectedDay = day;
	}

	/**
	 * 比較検証する。<br>
	 *
	 * @param actual
	 *            実測値
	 * @return 期待値と実測値が一致する場合true, それ以外の場合falseを返却する。
	 */
	@Override
	public boolean matches(Object actual) {
		if (!(actual instanceof Date)) {
			return false;
		}
		this.actualDate = (Date) actual;

		Calendar cal = Calendar.getInstance();
		cal.setTime(actualDate);
		if (expectedYear != cal.get(Calendar.YEAR)
				|| expectedMonth != cal.get(Calendar.MONTH)
				|| expectedDay != cal.get(Calendar.DATE)) {
			return false;
		}
		return true;
	}

	/**
	 * 比較検証で不一致の場合のメッセージを組み立てる。
	 */
	@Override
	public void describeTo(Description desc) {
		desc.appendValue(String.format("%d/%02d/%02d", expectedYear,
				expectedMonth, expectedDay));
		if (actualDate != null) {
			desc.appendText(" but actual is ");
			desc.appendValue(new SimpleDateFormat("yyyy/MM/dd")
					.format(actualDate));
		}
	}

	/**
	 * ファクトリメソッド<br>
	 *
	 * 使用例
	 *
	 * <pre>
	 * import static tryjunit.s19_1.DateMatcher.*;
	 * assertThat(actual, is(dateOf(2012, 1, 12)));
	 * </pre>
	 */
	public static Matcher<Date> list(int yyyy, int mm, int dd) {
		return new DateMatcher(yyyy, mm, dd);
	}
}
