/*
 * $Id: game2c.c,v 1.6 2017/01/09 19:23:29 localhost Exp $
 */
/*
Copyright (c) 2017, k_mia
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

void usage()
{
	fprintf(stderr,"usage: game2c [-g|-b|-t] [file]\n");
	fprintf(stderr,"       -g: game source is binary.\n");
	fprintf(stderr,"       -b: same as -g.\n");
	fprintf(stderr,"       -t: game source is text.(default)\n");
	fprintf(stderr," ex) game2c pi.gt >translated_game.gc\n");
	fprintf(stderr,"     cc -o pi translated_game.c (or cc translated_game_LE.c for LE; GAME80,MZ,...)\n");
}

void outstrs(FILE *fp, char *ss[]);
int getlino(FILE *fp);
int getcE(FILE *fp);
void errout(int lino, int c, char *str);
int skipwhite(FILE *fp, int c);
int skipAtoZ(FILE *fp);
void push(char *s);
void insert(int pos, char* s);
void pull(FILE* fp);
int artexpsub(FILE *ifp, FILE *ofp);
int artexp2(FILE *ifp, FILE *ofp);
int artexp(FILE *ifp, FILE *ofp);
int term(FILE *ifp, FILE *ofp);


int vflag=1;
/* NOW stderr is not constant. you must load after main. */
FILE *vfp=NULL; //stderr;

/* game source is text? */
int text_src=1;

int is_eol(c)
int c;
{
	if (c==0) { return 1; }
	if (text_src) {
		if (c=='\n') { return 1; }
		return 0;
	}
	return 0;
}


#define MAXLINE (1024*32/8)
int linotbl[MAXLINE];
int nlino=0;

int ifend=0;

int lino=1;

int loop_label_no=70000;

int main(ac,av)
int ac;
char **av;
{
	FILE *ofp=stdout;
	FILE *ifp=stdin;
	int linolast=1;
	int c, c2;
	extern char *optarg;
	extern int optind;
	int errflg=0;

	while ((c=getopt(ac,av,"gbth"))!= -1) {
		switch (c) {
		case 'g':
		case 'b':
			text_src=0;
			break;
		case 't':
			text_src=1;
			break;
		case '?':
			errflg++;
			break;
		case 'h':
			usage();
			exit(0);
		}
	}
	if (errflg) {
		usage();
		exit(2);
	}
	if (optind<ac) {
		FILE* ffp=fopen(av[optind],"r");
		if (ffp==NULL) {
			fprintf(stderr,"ERROR: %s cannot open.\n",av[optind]);
			exit (1);
		}
		ifp=ffp;
		fprintf(ofp,"/* original source is %s */\n",av[optind]);
	}

	vfp=stderr;

	for (; (lino=getlino(ifp))>0; ) {
		for ( ; linolast<=lino; linolast++) {
			fprintf(ofp," case %d: ;\n",linolast);
		}
		linotbl[nlino++]=lino;
		if (vflag) fprintf(vfp,"%5d",lino);
		if ((c=getcE(ifp))==EOF) break;
		if (c!=' ') {
			/* remark */
			for (; ; c=getcE(ifp)) {
				if (is_eol(c)||c==EOF) break;
			}
			if (vflag) fprintf(vfp,"\n");
			if (c==EOF) {
				errout(lino,' ',"unexpected EOF");
				break;
			}
			continue;
		}
		if (vflag) fprintf(vfp,"%c",c);
		/* put label */
		for ( ; ; ) {
			if ((c=skipwhite(ifp,c))==EOF) break;
			if (is_eol(c)) {
				/* EOL */
				for (; ifend>0; ifend--) {
					fprintf(ofp,"}\n");
				}
				break;
			}
			/* statement */
			if (isupper(c)) {
				/* variable, for */
				switch (c2=skipAtoZ(ifp)) {
				case '=':
					c2=c;
					fprintf(ofp,"  %c=",c2);
					c=artexp(ifp,ofp);
					fprintf(ofp,";\n");
					if (c==',') {
						/* for */
						//fprintf(ofp,"for(;%c<=(",c2);
						fprintf(ofp,"  push_loop(%d,(",loop_label_no);
						c=artexp(ifp,ofp);
						fprintf(ofp,"));\n case %d: ;\n",loop_label_no);
						loop_label_no++;
					}
					break;
				case '(':
					fprintf(ofp,"  POKEW(%c,",c);
					c=artexp(ifp,ofp);
					if (c!=')'||getcE(ifp)!='=') {
						errout(lino,c,"illegal left value");
					}
					fprintf(ofp,",");
					c=artexp(ifp,ofp);
					fprintf(ofp,");\n");
					break;
				case ':':
					fprintf(ofp,"  POKEB(%c,",c);
					c=artexp(ifp,ofp);
					if (c!=')'||getcE(ifp)!='=') {
						errout(lino,c,"illegal left value");
					}
					fprintf(ofp,",");
					c=artexp(ifp,ofp);
					fprintf(ofp,");\n");
					break;
				default:
					errout(lino,c2,"illegal left value");
				}
			}
			else if (c==';') {
				/* if */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				fprintf(ofp,"  if (");
				c=artexp(ifp,ofp);
				fprintf(ofp,") {\n"); //,lino+1);
				ifend++;
			}
			else if (c=='#') {
				/* goto */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				fprintf(ofp,"  lino=");
				c=artexp(ifp,ofp);
				fprintf(ofp,"; if (lino) { goto gameloop; }\n");
			}
			else if (c=='!') {
				/* function */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				fprintf(ofp,"  { int linos=");
				c=artexp(ifp,ofp);
				fprintf(ofp,"; if (linos) { game(linos); } }\n");
			}
			else if (c=='@') {
				/* do, until, next */
				switch (c=getcE(ifp)) {
				case 0:
				case ' ':
					/* do */
					//fprintf(ofp,"do{\n");
					fprintf(ofp,"  push_loop(%d,(",loop_label_no);
					fprintf(ofp,"0));\n case %d: ;\n",loop_label_no);
					loop_label_no++;
					break;
				case '=':
					/* until, next */
					c=getc(ifp);
					ungetc(c,ifp);
					if (isupper(c)) {
						/* next */
						fprintf(ofp,"  %c=",c);
						c2=artexp(ifp,ofp);
						//fprintf(ofp,"; continue;\n}\n");
						fprintf(ofp,"; if((%c",c);
						//fprintf(ofp,")<=peek_loop_value()){lino=peek_loop_lino();goto gameloop;}else{lino=%d;pull_loop();}\n",lino);
						fprintf(ofp,")<=peek_loop_value()){lino=peek_loop_lino();goto gameloop;}else{pull_loop();}\n");
						c=c2;
					}
					else if (c=='(') {
						/* until */
						//fprintf(ofp,"}while((");
						fprintf(ofp,"if((");
						c=artexp(ifp,ofp);
						//fprintf(ofp,")<=0);\n");
						//fprintf(ofp,")<=peek_loop_value()){lino=peek_loop_lino();goto gameloop;}else{lino=%d;pull_loop();}\n",lino);
						//fprintf(ofp,")<=0){lino=peek_loop_lino();goto gameloop;}else{lino=%d;pull_loop();}\n",lino);
						fprintf(ofp,")<=0){lino=peek_loop_lino();goto gameloop;}else{pull_loop();}\n");
					}
					else {
						/* break */
						c=artexp(ifp,ofp);
						fprintf(ofp,"break;\n");
					}
					break;
				default:
					errout(lino,c,"@ syntax");
				}
			}
			else if (c=='>') {
				/* asm func */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				//errout(lino,' ',"called asm function(>=...).");
				fprintf(ofp,"  asmsub_value=asmsub(");
				c=artexp(ifp,ofp);
				fprintf(ofp,");\n");
			}
			else if (c=='\'') {
				/* set random reg */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				fprintf(ofp,"  randseed=");
				c=artexp(ifp,ofp);
				fprintf(ofp,";\n");
			}
			else if (c=='$') {
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				/* printf("%c", */
				fprintf(ofp,"  printf(\"%%c\",");
				c=artexp(ifp,ofp);
				fprintf(ofp,");\n");
			}
			else if (c=='?') {
				/* printf("%d","%nd","%02X","%04X", */
				switch (c=getcE(ifp)) {
				case '=':	/* ?= */
					fprintf(ofp,"printf(\"%%d\",");
					c=artexp(ifp,ofp);
					fprintf(ofp,");\n");
					break;
				case '?':	/* ??= */
					if (getcE(ifp)!='=') {
						errout(lino,c,"need equal(=)");
					}
					fprintf(ofp,"printf(\"%%04X\",(");
					c=artexp(ifp,ofp);
					fprintf(ofp,")&0xFFFF);\n");
					break;
				case '$':	/* ?$= */
					if (getcE(ifp)!='=') {
						errout(lino,c,"need equal(=)");
					}
					fprintf(ofp,"printf(\"%%02X\",(");
					c=artexp(ifp,ofp);
					fprintf(ofp,")&0xFF);\n");
					break;
				case '(':	/* ?(n)= */
					fprintf(ofp,"pr(");
					if ((c=artexp(ifp,ofp))!=')') {
						errout(lino,c,"need ')'");
					}
					if (getcE(ifp)!='=') {
						errout(lino,c,"need equal(=)");
					}
					fprintf(ofp,",");
					c=artexp(ifp,ofp);
					fprintf(ofp,");\n");
					break;
				default:
					errout(lino,c,"need =,?,$,(");
				}
			}
			else if (c=='.') {
				/* printf(" ") */
				if (getcE(ifp)!='=') {
					errout(lino,c,"need equal(=)");
				}
				fprintf(ofp,"  tab(");
				c=artexp(ifp,ofp);
				fprintf(ofp,");\n");
			}
			else if (c=='/') {
				/* printf("\n") */
				fprintf(ofp,"  printf(\"\\n\");\n");
				c=getcE(ifp);
			}
			else if (c=='"') {
				/* printf */
				fprintf(ofp,"  fputs(\"");
				for (; (c=getcE(ifp))!=EOF; ) {
					if (c=='"') {
						c=getcE(ifp);
						break;
					}
					if (is_eol(c)) {
					    errout(lino,' ',"unexpected EOL");
					}
					fprintf(ofp,"%c",c);
				}
				if (c==EOF) {
					errout(lino,' ',"unexpected EOF");
				}
				fprintf(ofp,"\",stdout);\n");
			}
			else if (c==']') {
				/* return */
				fprintf(ofp,"  return 0;\n");
				c=getcE(ifp);
			}
			else {
				/* dummy */
				c=getcE(ifp);
			}
		}
		if (vflag) fprintf(vfp,"\n");
		if (c==EOF) {
			errout(lino,' ',"unexpected EOF");
			break;
		}
	}

	return 0;
}


#define MAXPBUF	1024

//static moduloflag=0;
int artterm=0;
static char pbuf[MAXPBUF];

int artexp(ifp,ofp)
FILE *ifp, *ofp;
{
	int c;

	c=artexpsub(ifp,ofp);
	for (; artterm>0; artterm--) {
		fprintf(ofp,"(");
	}
	pull(ofp);
	return c;
}

int artexp2(ifp,ofp)
FILE *ifp, *ofp;
{
	char buf2[1024];
	char buf3[1024];
	int artterm2;
	int artterm3;
	int c;

	artterm2=artterm;
	strcpy(buf2,pbuf);
	artterm=0;
	pbuf[0]=0;

	c=artexpsub(ifp,ofp);
	artterm3=artterm;
	strcpy(buf3,pbuf);

	strcpy(pbuf,buf2);
	artterm=artterm2;

	for (; artterm3>0; artterm3--) {
		push("(");
	}
	push(buf3);

	return c;
}


int artexpsub(ifp,ofp)
FILE *ifp, *ofp;
{
	int c;
	char buf[256];
	int add_kokka=0;

	for (;;) {
		//int pbuf_len=strlen(pbuf);
		c=term(ifp,ofp);
		if (add_kokka) {
			while (add_kokka>0) {
				push(")");
				add_kokka--;
			}
		}
		switch (c) {
		case 0:
		case ' ':
		case EOF:
			return c;
		case '\n':
			return 0;
		case '+':
		case '-':
		case '*':
			sprintf(buf,")%c",c);
			push(buf);
			artterm++;
			break;
		case '/':
			//if (moduloflag) c='%';
			//sprintf(buf,")%c",c);
			for (; artterm>0; artterm--) {
				insert(0,"(");
			}
			insert(0,"division(");
			push(",");
			add_kokka++;
			break;
		case '=':
			push(")==");
			artterm++;
			break;
		case '<':
			switch (c=getc(ifp)) {
			case '=':
				if (vflag) fprintf(vfp,"%c",c);
				push(")<=");
				artterm++;
				break;
			case '>':
				if (vflag) fprintf(vfp,"%c",c);
				push(")!=");
				artterm++;
				break;
			default:
				ungetc(c,ifp);
				push(")<");
				artterm++;
				break;
			}
			break;
		case '>':
			switch (c=getc(ifp)) {
			case '=':
				if (vflag) fprintf(vfp,"%c",c);
				push(")>=");
				artterm++;
				break;
			default:
				ungetc(c,ifp);
				push(")>");
				artterm++;
				break;
			}
			break;
		case ',':
			/* for */
			return c;
		case ')':
			return c;
		default:
			errout(lino,c,"illegal operator");
		}
	}
	return 0;
}

int term(ifp,ofp)
FILE *ifp, *ofp;
{
	int c, c2;
	int n;
	char buf[256];

	c=getcE(ifp);
	if (isdigit(c)) {
		/* number */
		for (n=0; isdigit(c); c=getcE(ifp)) {
			n=n*10+c-'0';
		}
		sprintf(buf,"%d",n);
		push(buf);
		return c;
	}
	if (isupper(c)) {
		/* variable */
		c2=c;
		switch (c=skipAtoZ(ifp)) {
		case '(':
			sprintf(buf,"PEEKW(%c,",c2);
			push(buf);
			c=artexp2(ifp,ofp);
			if (c!=')') {
				errout(lino,c,"unmatch ( )");
			}
			push(")");
			c=getcE(ifp);
			break;
		case ':':
			sprintf(buf,"PEEKB(%c,",c2);
			push(buf);
			c=artexp2(ifp,ofp);
			if (c!=')') {
				errout(lino,c,"illegal left value");
			}
			push(")");
			c=getcE(ifp);
			break;
		default:
			sprintf(buf,"%c",c2);
			push(buf);
			break;
		}
		return c;
	}
	switch (c) {
	case EOF:
	case 0:
		return c;
	case '\n':
		return 0;
	case '"':
		c=getcE(ifp);
		sprintf(buf,"%d",c);
		push(buf);
		c=getcE(ifp);
		if (c!='"') {
			errout(lino,c,"unmatch \"x\"");
		}
		return getcE(ifp);

	case '=':
		/* prog start addr -> 0 */
		push("0");
		return getcE(ifp);

	case '&':
		/* prog start addr -> 0 */
		push("0");
		return getcE(ifp);

	case '*':
		/* ram end addr -> 0xFFFF */
		push("0xFFFF");
		return getcE(ifp);

	case '\'':
		/* random func */
		push("myrandom(");
		c=term(ifp,ofp);
		push(")");
		return c;

	case '$':
		/* keyin or HEX */
		c=getcE(ifp);
		if (isxdigit(c)) {
			/* HEX */
			push("0x");
			for (n=0; isxdigit(c); c=getcE(ifp)) {
				buf[0]=c; buf[1]=0;
				push(buf);
			}
			return c;
		}
		else {
			push("getachr()");
			return c;
		}
	
	case '(':
		/* artexp */
		push("(");
		if ((c=artexp2(ifp,ofp))==')') {
			push(")");
			return getcE(ifp);
		}
		errout(lino,c,"illegal term");
	
	case '+':
		/* abs */
		push("ABS(");
		c=term(ifp,ofp);
		push(")");
		return c;
	
	case '-':
		/* negate */
		push("-(");
		c=term(ifp,ofp);
		push(")");
		return c;

	case '%':
		/* modulo */
		//moduloflag=1;
		push("modulo(");
		c=term(ifp,ofp);
		push(")");
		//moduloflag=0;
		return c;

	case '?':
		/* input */
		push("input()");
		return getcE(ifp);

	case '>':
		/* asm func return */
		push("asmsub_value");
		return getcE(ifp);

	default:
		errout(lino,c,"illegal term");
	}
	return 0;
}

void errout(lino,c,str)
int lino;
int c;
char *str;
{
	fprintf(stderr,"\nERROR: line %d %s near %c\n",lino,str,c);
	exit(1);
}

void outstrs(fp,ss)
FILE *fp;
char *ss[];
{
	while (*ss) fprintf(fp,"%s\n",*ss++);
}

int getlino(fp)
FILE *fp;
{
	int n, c;

	if (text_src) {
		/* source is TEXT */
		while ((c=getc(fp))!=EOF) {
			if (isspace(c)) continue;
			if (! isdigit(c)) {
				errout(lino,c,"Line number required");
				//exit(2);
			}
			n=c-'0';
			while ((c=getc(fp))!=EOF) {
				if (! isdigit(c)) break;
				n=n*10+c-'0';
			}
			if (c==EOF||c=='\n') {
				errout(n,' ',"line contents required");
				//exit(2);
			}
			ungetc(c,fp);
			return n;
		}
		return -1;
	}
	else {
		/* source is BINARY */
		if ((n=getc(fp))==EOF) {
			return -1;
		}
		if (n>=0x80) return -1;
		if ((c=getc(fp))==EOF) {
			return -1;
		}
		return ((n<<8)+c);
	}
}

int skipwhite(fp,c)
FILE *fp;
int c;
{
	if (c!=' ') return c;
	for ( ; (c=getcE(fp))!=EOF; ) {
		if (c==' ') continue;
		return c;
	}
	return EOF;
}

int skipAtoZ(fp)
FILE *fp;
{
	int c;

	for (; (c=getcE(fp))!=EOF; ) {
		if (isupper(c)) continue;
		return c;
	}
	return c;
}

int getcE(fp)
FILE *fp;
{
	int c;

	switch (c=getc(fp)) {
	case 0:
	case EOF:
		return c;
	case '\n':
		return 0;
	default:
		if (vflag) fprintf(vfp,"%c",c);
		return c;
	}
}

void push(s)
char *s;
{
	if (strlen(pbuf)+strlen(s)>=MAXPBUF-1) {
		errout(lino,' ',"too complex artexp");
	}
	strcat(pbuf,s);
}

void insert(pos,s)
int pos;
char* s;
{
	int len=strlen(pbuf);
	int addlen=strlen(s);
	if (len+addlen>=MAXPBUF-1) {
		errout(lino,' ',"too complex artexp");
	}
	if (len<pos) {
		errout(lino,' ',"internal error insert");
	}
	memmove(pbuf+pos+addlen,pbuf+pos,len-pos+1);
	memmove(pbuf+pos,s,addlen);
}

void pull(fp)
FILE *fp;
{
	fprintf(fp,"%s",pbuf);
	pbuf[0]=0;
}
