#include "TCPIPConfig.h"

#include "TCPIP Stack/TCPIP.h"
#include "TCPIP Stack/StackTsk.h"

#include <stdlib.h>
#include <ctype.h>
#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "FreeRTOS.h"
#include "queue.h"
#include "lcd.h"
#include "usb_config.h"
#include "USB/usb.h"
#include "USB/usb_host_generic.h"
#include "../include/USB/timer.h"
#include "../USB/usb_host_local.h"
#include "TCPIP Stack/Tick.h"
#include "MDD File System/FSIO.h"
#include "MDD File System/FSDefs.h"
#include <time.h>
#include "node.h"
#include "task.h"

// Application States
typedef enum
{
    USB_INITIALIZE = 0,                // Initialize the app when a device is attached
	USB_STATE_IDLE,                    // Inactive State
    USB_STATE_LOOP                    // Inactive State
} USB_STATE;

// *****************************************************************************
// *****************************************************************************
// Global Variables
// *****************************************************************************
// *****************************************************************************

AD_PARAM	AdParam;

extern	BYTE		header[41], voice[12];
static	BYTE        deviceAddress;  // Address of the device on the USB
static	USB_STATE	USBState;      // Current state of the application

static void	send_header_USB (BYTE pkt[], BYTE PacketID[]);
static void	send_voice_USB (BYTE pkt[], BYTE seq);
static void	from_Inet(void );
static void	from_Rig(void);
static void	send_header_queue(void);
static void	send_voice_queue( void );
static	void	Inet_CallsignDisp(BYTE RFheader[], BYTE RcvPacketID[]);
BOOL	callsign_check (BYTE CallSign[]);
static	BOOL	gps_sumcheck(BYTE string[]);

static	BYTE 	LastFrame0[6] = {0x55,0x55,0x55,0x55,0xc8,0x7a};
static	BYTE 	LastFrame1[6] = {0x55,0x55,0x55,0x55,0x55,0x55};
static	BYTE 	LastFrameCheckRx[6];
static	BYTE 	LastFrameCheckTx[6];
static	packet	pkt;
static	BYTE	ReSyncFrame[3] = {0x55, 0x2d, 0x16};

static	void	RF_CallsignDisp( BYTE header[]);
static	void	RF_BlockedCallsignDisp(BYTE RFheader[]);
static	DWORD	usb_control_msg (BYTE bmRequestType, BYTE request, WORD Value, WORD Index, BYTE *data, WORD Length, WORD timeout);
static	void	LoadAdParam( void );
static	void	SaveAdParam( void );
void	vTaskDelay( portTickType xTicksToDelay );
static	void	USB_CallDisplay(BYTE RFheader[]);
void	ConnectMSG (BYTE CallSign[]);
void	MACWriteArray(BYTE address, BYTE *val, WORD wLen);
void	Ptt_Off( void );
void	BeepSend (BYTE callsign[]);
void	DisConnectedMsgSend(BYTE callsign[]);
void	NotFoundMsgSend (BYTE callsign[]);
void	UnLinkMsgSend (BYTE callsign[]);
WORD 	swaps_usb(WORD v);
static	void	DprsData(BYTE Sdata[]);
static	void	GPGLL (BYTE string[]);
static	void	GPGGA (BYTE string[]);
static	void	GPRMC (BYTE string[]);
static	void	dprs_message(BYTE string[]);
static	BOOL	GPS_A_SumCheck(BYTE string[]);
static	void	GPS_A (BYTE string[]);
static	WORD	BitsCheck(BYTE pattern1[], BYTE pattern2[], WORD cnt);
static	void	DprsMsgSendToRF (void);

BOOL	headerRead;
static	BOOL	SendHeader;

extern	BYTE	PicVer[2];
extern	WORD	DefaultVid;
extern	WORD	DefaultPid;
extern	BOOL	PTT_On;
extern	BOOL	COS_On;
extern	WORD	SSN;

static	BYTE	Text[32];
static	BYTE	SlowData[6];

void	JitterBufferInit(void);
void	JitterBufferSend (void);
void	JitterBufferSave (BYTE VoicePacket[], BYTE tx_seq);
BYTE	SendConnectMsg(BYTE CallSign[], BYTE resync);

static	BYTE	JitterBuffer[21][12];		
static	WORD	VoicePacketCnt;
static	WORD	RemainVoicePacketCnt;
static	WORD	JitterBufferPnt;
static	WORD	JitterBufferSize = 21;
static	BOOL	VoicePacketSW;

static	DWORD	timer;
static	BOOL	header_gen;

static	WORD	Packet_ID;
static	BYTE	Packet_Seq;

extern	BYTE	HeaderID[2];
extern	BYTE	VoiceID[2];
extern	BYTE	LastFrameID[2];
extern	BYTE	LinkID[2];
extern	BYTE	PttOff[4];
extern	BYTE	CurUser[8];
extern	struct	MessageSW	Msg;

static	BYTE	SendPttOff[10] ={0x0a, 0xc0, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
extern	BOOL	FSInitSW;
extern	xQueueHandle xDstarSndQueue;
//extern	xTaskHandle TCPHandle;
extern	xQueueHandle xDstarLogQueue;
extern	xQueueHandle xDstarRefQueue;
extern	xQueueHandle xDstarDprsMsgQueue;

extern	xTaskHandle FSHandle;
extern	xTaskHandle USBHandle;

extern	BYTE	RefName[8];
extern	BYTE	RefNameSave[8];
extern	BYTE	RefNameSet;

extern	BYTE	TimeDate[20];

static	char	text[120];

extern	BYTE	Radio_Callsign_Check_String[9];
extern	BOOL	FS_Skip_Req;
extern	BOOL	FS_Skip_Ack; 

extern	BOOL	CallCheck_Skip;
BOOL	VoiceMsg;

extern	BYTE	RadioMsg[20];
extern	BYTE	RadioLat[8],RadioLong[9];
extern	BYTE	RadioAtitude[8];
extern	BYTE	RadioSpeed[8];
extern	BYTE	RadioDirection[8];
extern	BYTE	RadioCall[8];
extern	BOOL	RadioDprsSend;
extern	struct	DPRSinfo DPRS;

extern	WORD	crc_tabccitt[256];

extern	BYTE	Radio_GPS_A_MSG[120];
extern	BOOL	Radio_GPS_A_Msg_send;
static	BOOL	SendToInet;

static	DWORD	send_packet_cnt;

BYTE	NullVoice0[12] = {0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x55,0x2d,0x16};
BYTE	NullVoice1[12] = {0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x16,0x29,0xf5};
BYTE	NullVoice_ID31[9] = {0xb2,0x4d,0x22,0x48,0xc0,0x16,0x28,0x26,0xc8};

extern	BOOL	DebugSW;

//******************************************************************************
//******************************************************************************
// Local Routines
//******************************************************************************
//******************************************************************************

/*************************************************************************
 * Function:        CheckForNewAttach
 *
 * Preconditions:   None
 *
 * Input:           None
 *
 * Output:          deviceAddress (global)
 *                  Updates the device address when an attach is found.
 *
 * Returns:         TRUE if a new device has been attached.  FALSE,
 *                  otherwise.
 *
 * Side Effects:    Prints attach message
 *
 * Overview:        This routine checks to see if a new device has been
 *                  attached.  If it has, it records the address.
 *************************************************************************/

BOOL CheckForNewAttach ( void )
{
	static	BYTE	*pDesc;

    // Try to get the device address, if we don't have one.
    if (deviceAddress == 0)
    {
        static	GENERIC_DEVICE_ID DevID;

        DevID.vid   = NodeAdapterVID;
        DevID.pid   = NodeAdapterPID;
        #ifdef USB_GENERIC_SUPPORT_SERIAL_NUMBERS
            DevID.serialNumberLength = 0;
            DevID.serialNumber = NULL;
        #endif

        if (USBHostGenericGetDeviceAddress(&DevID))
        {
            deviceAddress = DevID.deviceAddress;

			pDesc = USBHostGetDeviceDescriptor(deviceAddress);
			pDesc += 12;
			PicVer[1] = *pDesc;
			PicVer[0] = *(pDesc + 1);
			if (PicVer[1] < 0x60)
			{
				memcpy (Text, "Node Adapter    " "  Wrong Firmware", 32);
				xMessage.pcMessage = (char *)&Text;
 				xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
				xQueueSend( xLCDQueue, &xMessage, 0 );
				while (1) ;			
			}
            return TRUE;
        }
    }

    return FALSE;

} // CheckForNewAttach

/*************************************************************************
 * Function:        ManageDemoState
 *
 * Preconditions:   The DemoState global variable must be initialized to
 *                  USB_STATE_IDLE (0).  (This occurs on reset.)
 *
 * Input:           DemoState (global)
 *                  Actions selected based value of DemoState on function
 *                  entry.
 *
 *                  deviceAddress (global)
 *                  May use device address to access device, depending on
 *                  state.
 *
 *                  DataPacket (global)
 *                  May read data from packet buffer, depending on state.
 *
 * Output:          DemoState (global)
 *                  Updates demo state as appropriate.
 *
 *                  DataPacket (global)
 *                  May cause data in the packet buffer to be updated,
 *                  depending on state.
 *
 * Returns:         None
 *
 * Side Effects:    Depend on state transition
 *
 * Overview:        This routine maintains the state of the application,
 *                  updateing global data and taking actions as necessary
 *                  to maintain the custom demo operations.
 *************************************************************************/
static void ManageState ( void )
{
	static	BYTE	*pDesc;
	static	BYTE	status;
	static	BOOL	attachMsgSW;
	static	DWORD	timer;

    // Watch for device detaching
    if (USBHostGenericDeviceDetached(deviceAddress) && deviceAddress != 0)
    {
        USBState = USB_INITIALIZE;
        deviceAddress   = 0;
		memcpy (Text, "device detached " "                ", 32);
		xMessage.pcMessage = (char *)&Text;
 		xMessage.xMinDisplayTime = 3000 / portTICK_RATE_MS;
		xQueueSend( xLCDQueue, &xMessage, 0 );
    }

    switch (USBState)
    {
  		case USB_STATE_LOOP:
			from_Rig();
			from_Inet();
			if (AdParam.UpdateFlag == 0x02) SaveAdParam();
			else if (AdParam.UpdateFlag == 0x00) LoadAdParam();

			if (Msg.notfoundSW)
			{
				NotFoundMsgSend (RefName);
				Msg.notfoundSW = FALSE;
			}
			usb_control_msg (0xC0, GET_AD_STATUS, 0, 0, &status, 1, 100);
			if (status & PTT_OnOff) PTT_On = TRUE;
			else					PTT_On = FALSE; 

			if (!PTT_On && !COS_On)
			{
				DprsMsgSendToRF();
			}
			break;

    	case USB_INITIALIZE:
        	USBState = USB_STATE_IDLE;
			headerRead = FALSE;
			SendHeader = FALSE;
			attachMsgSW = TRUE;
			Msg.connectSW = FALSE;
			Msg.notfoundSW = FALSE;
			timer = TickGet();
        	break;

    	case USB_STATE_IDLE:
        	if (CheckForNewAttach())
        	{
				pDesc  = USBHostGetDeviceDescriptor(1);
		    	if (pDesc)
				{
	       			USBState = USB_STATE_LOOP;
					pDesc += 12;
					PicVer[1] = *pDesc++;
					PicVer[0] = *pDesc;
					if (PicVer[1] < 0x60)
					{
						memcpy (Text, "Node Adapter    " "   Wrong Version", 32);
						xMessage.pcMessage = (char *)&Text;
	 					xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
						xQueueSend( xLCDQueue, &xMessage, 0 );
						while (1) ;			
					}
					LoadAdParam();
					timer = 0;
					header_gen = FALSE;
					usb_control_msg (0x40, SET_AD_INIT, 0, 0, &status, 0, 100);
//					vTaskResume (TCPHandle);
        		}
//			} else {
//				if ((attachMsgSW) && ((TickGet() - timer) > 60* TICK_SECOND))
//				{
//					memcpy (Text, "Attach          " "    Node Adapter", 32);
//					xMessage.pcMessage = (char *)&Text;
//					xMessage.xMinDisplayTime = 500 / portTICK_RATE_MS;
//					xQueueSend( xLCDQueue, &xMessage, 0 );
//					attachMsgSW = FALSE;
//				}
//				vTaskSuspend (TCPHandle);
			}
       		break;
    }
} // ManageDemoState


//******************************************************************************
//******************************************************************************
// USB Support Functions
//******************************************************************************
//******************************************************************************

/*************************************************************************
 * Function:        USB_ApplicationEventHandler
 *
 * Preconditions:   The USB must be initialized.
 *
 * Input:           event       Identifies the bus event that occured
 *
 *                  data        Pointer to event-specific data
 *
 *                  size        Size of the event-specific data
 *
 * Output:          deviceAddress (global)
 *                  Updates device address when an attach or detach occurs.
 *
 *                  DemoState (global)
 *                  Updates the demo state as appropriate when events occur.
 *
 * Returns:         TRUE if the event was handled, FALSE if not
 *
 * Side Effects:    Event-specific actions have been taken.
 *
 * Overview:        This routine is called by the Host layer or client
 *                  driver to notify the application of events that occur.
 *                  If the event is recognized, it is handled and the
 *                  routine returns TRUE.  Otherwise, it is ignored (or
 *                  just "sniffed" and the routine returns FALSE.
 *************************************************************************/

BOOL USB_ApplicationEventHandler ( BYTE address, USB_EVENT event, void *data, DWORD size )
{
	static	BYTE	*pDesc;

	// Handle specific events.
    switch (event)
    {
        case EVENT_GENERIC_ATTACH:
            if (size == sizeof(GENERIC_DEVICE_ID))
            {
                deviceAddress   = ((GENERIC_DEVICE_ID *)data)->deviceAddress;
                USBState = USB_STATE_LOOP;
				pDesc = USBHostGetDeviceDescriptor(deviceAddress);
				pDesc += 12;
				PicVer[1] = *pDesc;
				PicVer[0] = *(pDesc + 1);
				if (PicVer[1] < 0x60)
				{
					memcpy (Text, "Node Adapter    " "  Wrong Firmware", 32);
					xMessage.pcMessage = (char *)&Text;
	 				xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
					xQueueSend( xLCDQueue, &xMessage, 0 );
					while (1) ;			
				}
                return TRUE;
            }
            break;

//        case EVENT_GENERIC_DETACH:
//            deviceAddress   = 0;
//            USBState = USB_INITIALIZE;
//            return TRUE;
//
//        case EVENT_GENERIC_TX_DONE:           // The main state machine will poll the driver.
//        case EVENT_GENERIC_RX_DONE:
//            return TRUE;
//
        case EVENT_VBUS_REQUEST_POWER:
        case EVENT_VBUS_RELEASE_POWER:
        case EVENT_HUB_ATTACH:
        case EVENT_UNSUPPORTED_DEVICE:
        case EVENT_CANNOT_ENUMERATE:
        case EVENT_CLIENT_INIT_ERROR:
        case EVENT_OUT_OF_MEMORY:
        case EVENT_UNSPECIFIED_ERROR:   // This should never be generated.
        case EVENT_SUSPEND:
        case EVENT_DETACH:
        case EVENT_RESUME:
        case EVENT_BUS_ERROR:
            return TRUE;
            break;

        default:
            break;
    }

    return FALSE;

} // USB_ApplicationEventHandler



/*************************************************************************
 * Function:        main
 *
 * Preconditions:   None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Returns:         Never exits
 *
 * Side Effects:    Runs the application
 *
 * Overview:        This is the USB Custom Demo Application's main entry
 *                  point.
 *************************************************************************/

int prvTaskUSB ( void )
{
   // Initialize the processor and peripherals.
    USBState = USB_INITIALIZE;
	AdParam.UpdateFlag = 0x00;

	if (!FSInitSW)
	{
		while (!FSInit())vTaskDelay (5000 / portTICK_RATE_MS);
		Reset(); 
	}

    if ( USBHostInit(0) == TRUE )
    {
			memcpy (Text, "USB Interface   " "     Initialized", 32);
			xMessage.pcMessage = (char *)&Text;
	 		xMessage.xMinDisplayTime = 500 / portTICK_RATE_MS;
			xQueueSend( xLCDQueue, &xMessage, 0 );
    }
    else
    {
		memcpy (Text, "USB Interface   " "            Halt", 32);
		xMessage.pcMessage = (char *)&Text;
 		xMessage.xMinDisplayTime = 500 / portTICK_RATE_MS;
		xQueueSend( xLCDQueue, &xMessage, 0 );
        while (1);
    }

	LED2_TRIS = 0;		/* output */

	JitterBufferInit();

    // Main Processing Loop
    while (1)
    {
        // This demo does not check for overcurrent conditions.  See the
        // USB Host - Data Logger for an example of overcurrent detection
        // with the PIC24F and the USB PICtail Plus.

        // Maintain USB Host State
        USBHostTasks();
        // Maintain Application State
        ManageState();
    }

    return 0;

} // main

/**************************************************************/
/*  From INET */
void	from_Inet()
{
	static	packet	q_pkt;
	extern	xQueueHandle xDstarRcvQueue;

	if (xQueueReceive (xDstarRcvQueue, &q_pkt, 0) == pdTRUE)
	{
		if (!memcmp((char *)q_pkt.DPlusID, (char *)HeaderID, 2) || !memcmp((char *)q_pkt.DPlusID, (char *)LinkID, 2))
		{
			send_header_USB(q_pkt.header, q_pkt.DPlusID);
		}
		else if (!memcmp((char *)q_pkt.DPlusID, (char *)VoiceID, 2) 
					|| !memcmp((char *)q_pkt.DPlusID, (char *)LastFrameID, 2))
		{
			send_voice_USB(q_pkt.voice, q_pkt.B_header.seq);
		}
		else if (!memcmp(&q_pkt, PttOff, 4))
		{
			Ptt_Off();
		}
	}
	if (VoicePacketSW && VoicePacketCnt) JitterBufferSend();
}

static void	send_header_USB(BYTE pkt[], BYTE RcvPacketID[])
{
	extern	xQueueHandle xDstarRcvQueue;

	RemainVoicePacketCnt = VoicePacketCnt;
	
    usb_control_msg(0x40, SET_MyCALL2, 0, 0, &pkt[35], 4, 100);
	usb_control_msg(0x40, SET_MyCALL, 0, 0, &pkt[27], 8, 100);
	usb_control_msg(0x40, SET_YourCALL, 0, 0, &pkt[19], 8, 100);
	usb_control_msg(0x40, SET_RPT1CALL, 0, 0, &pkt[11], 8, 100);
	usb_control_msg(0x40, SET_RPT2CALL, 0, 0, &pkt[3], 8, 100);
   	usb_control_msg(0x40, SET_FLAGS, 0, 0, &pkt[0], 3, 100);

	if (!memcmp(RcvPacketID, LinkID, 2)) Inet_CallsignDisp (pkt, RcvPacketID);
	VoicePacketCnt = 0;
	JitterBufferPnt = 0;
	VoicePacketSW = FALSE;
	PTT_On = FALSE;
}

void	send_voice_USB (BYTE pkt[], BYTE seq)
{
	static	BYTE	i;

	JitterBufferSave (pkt, seq); 

	for (i = 0 ; i < 12 ; i++)
	{
		LastFrameCheckTx[0] = LastFrameCheckTx[1];
		LastFrameCheckTx[1] = LastFrameCheckTx[2];
		LastFrameCheckTx[2] = LastFrameCheckTx[3];
		LastFrameCheckTx[3] = LastFrameCheckTx[4];
		LastFrameCheckTx[4] = LastFrameCheckTx[5];
		LastFrameCheckTx[5] = pkt[i];
		if (!memcmp(LastFrameCheckTx, LastFrame0, 6) || !memcmp(LastFrameCheckTx, LastFrame1, 6)) Ptt_Off();
	}
}

void	Ptt_Off( void )
{
	static	BYTE	dummy;

	VoicePacketSW = TRUE;
	while (VoicePacketCnt) JitterBufferSend ();
	VoicePacketCnt = 0;
	JitterBufferPnt = 0;
	/* PTT OFF */
	usb_control_msg(0x40, SET_PTT, OFF, 0, &dummy, 0, 100);
	PTT_On = FALSE;
	VoicePacketSW = FALSE;
	LED2_IO = 0;
	return;
}


/*************************************************/
/*  From RIG */
static	void	from_Rig()
{
	static	DWORD	length;
	static	DWORD	ret;
	static	WORD	i;
	static	WORD 	voice_pnt;
	extern	BYTE	rig_buffer[32];
	static	BYTE	status;
	static	BYTE	l;

	/* rf header send */
	if (!headerRead)
	{
		ret = usb_control_msg (0xC0, GET_HEADER, 0, 0, header, 32, 100);
        if (ret != 32)
		{
			return;  /*RF header read? */
		}

        ret = usb_control_msg (0xC0, GET_HEADER, 0, 0, &header[32], 9, 100);
  		usb_control_msg (0xC0, GET_AD_STATUS, 0, 0, &status, 1, 100);
		if (status & HeaderGen)
		{
			if ((TickGet() - timer) > AppConfig.HeaderGenTime)
			{
				header_gen = FALSE;
				return;
			}
			header_gen = TRUE;
		}
ReCheck:
		memcpy (Radio_Callsign_Check_String,&header[27], 8);
		if (!CallCheck_Skip)
		{
			Radio_Callsign_Check_String[8] = 'S';
			while (Radio_Callsign_Check_String[8] == 'S') vTaskDelay(20 / portTICK_RATE_MS);
			if (CallCheck_Skip) goto ReCheck;
			if (Radio_Callsign_Check_String[8] == 'N')
			{
				RF_BlockedCallsignDisp(header);
				if (DebugSW)
				{
					sprintf (text, "%19.19s From Radio %8.8s Deny\r\n",TimeDate,&header[27]);
					xQueueSend( xDstarLogQueue, &text, 0 );
				}
				return;
			}
		} else {
			l = strlen((char *)AppConfig.DefaultAuthCall);
			if (l > 8) l = 8;
			if (memcmp (&header[27], AppConfig.DefaultAuthCall, l))
			{
				RF_BlockedCallsignDisp(header);
				if (DebugSW)
				{
					sprintf (text, "%19.19s From Radio %8.8s Deny\r\n",TimeDate,&header[27]);
					xQueueSend( xDstarLogQueue, &text, 0 );
				}
				return;
			}
		}
		RF_CallsignDisp(header);
		if (!AppConfig.Flags.DisableRFcmd)
		{
			if ((header[26] == 'L') && !Msg.connectSW && !Msg.notfoundSW)
			{
				memcpy (RefNameSave, &header[19], 6);
				RefNameSave[6] = 0x20;
				RefNameSave[7] = header[25];
				if (RefNameSave[7] == 0x20) RefNameSave[7] = '*';
				if (memcmp (RefName, RefNameSave, 8))
				{
					RefNameSet = 2;
					sprintf (text, "%19.19s From Radio Link Command accept %8.8s\r\n",TimeDate,RefNameSave);
					xQueueSend( xDstarLogQueue, &text, 0 );
				} 
				else
				{	
					sprintf (text, "%19.19s From Radio Already Linked %8.8s\r\n",TimeDate,RefNameSave);
					xQueueSend( xDstarLogQueue, &text, 0 );
				}
			}
			else if ((header[26] == 'U') && !Msg.connectSW && !Msg.notfoundSW)
			{
				sprintf (text, "%19.19s From Radio UnLink Command accept %8.8s\r\n",TimeDate,&header[19]);
				xQueueSend( xDstarLogQueue, &text, 0 );
				if (!AppConfig.Flags.RepeaterSW) DisConnectedMsgSend(&header[27]);	/* unlink message */
				memset (&header[19], 0x20, 7);
				RefNameSet = 1;
				memset (RefName, 0x20, 8);
			}
			else
			{
//				send_header_queue();
				send_packet_cnt = 0;
				headerRead = TRUE;
				SendHeader = TRUE;
				SendToInet = TRUE;
				voice_pnt = 0;
//				COS_On = TRUE;
			}
		} else {
			if ((header[26] != 'L') && (header[26] != 'U'))
			{
//				send_header_queue();
				send_packet_cnt = 0;
				headerRead = TRUE;
				SendHeader = TRUE;
				SendToInet = TRUE;
				voice_pnt = 0;
//				COS_On = TRUE;
			}
		}
	} else {
		/* rf voice read */
		length = usb_control_msg (0xC0, GET_DATA, 0, 0, rig_buffer, 32, 100);
		if ((length <= 32 ) && (length > 0))
		{
			if (header_gen) timer = TickGet();	
			for (i = 0 ; i < length ; i++)
			{
				voice[voice_pnt] = rig_buffer[i];
				voice_pnt++;
				LastFrameCheckRx[0] = LastFrameCheckRx[1]; 
				LastFrameCheckRx[1] = LastFrameCheckRx[2]; 
				LastFrameCheckRx[2] = LastFrameCheckRx[3]; 
				LastFrameCheckRx[3] = LastFrameCheckRx[4]; 
				LastFrameCheckRx[4] = LastFrameCheckRx[5];
				LastFrameCheckRx[5] = rig_buffer[i];
 
				if (!memcmp(LastFrameCheckRx, LastFrame0, 6))
				{
					memcpy ((char *)pkt.DPlusID, (char *)LastFrameID, 2);
					pkt.B_header.seq = Packet_Seq | 0x40;
					if (SendToInet) send_voice_queue();
					Packet_Seq++;
					if (Packet_Seq > 20) Packet_Seq = 0;
					headerRead = FALSE;
					SendHeader = FALSE;
					COS_On = FALSE;
					LED2_IO = 0;
					header_gen = FALSE;
					/* ptt off packet generated */
					SendPttOff[4] = (Packet_ID >> 8) & 0xff;
					SendPttOff[5] = Packet_ID & 0xff;
					SendPttOff[6] = Packet_Seq;
					if (SendToInet)
					{
						xQueueSend (xDstarSndQueue, &SendPttOff, 20 / portTICK_RATE_MS);
						sprintf (text, "%19.19s From Radio Send packets : %ld\r\n", TimeDate, send_packet_cnt);
					} else {
						sprintf (text, "%19.19s From Radio Ignore packets : %ld (%8.8s)\r\n", TimeDate, send_packet_cnt, &header[27]);
					}
					xQueueSend( xDstarLogQueue, &text, 0 );
					if (!AppConfig.Flags.RepeaterSW && (RefName[7] != 'E')) 
					{
						BeepSend (&header[27]); /* skip the beep on echo modeul */
					}
					return;
				}

				if (voice_pnt == 12)
				{
//					memcpy ((char *)pkt.DPlusID, (char *)VoiceID, 2);
					if (!memcmp (&voice[9], ReSyncFrame, 3)) Packet_Seq = 0;
					pkt.B_header.seq = Packet_Seq;
					if (DPRS.RadioSW && DPRS.DprsSW)
					{
						if (Packet_Seq != 0)
						{
							if (Packet_Seq % 2)
							{
								memcpy (SlowData, &voice[9], 3);
								SlowData[0] ^= 0x70;
								SlowData[1] ^= 0x4f;
								SlowData[2] ^= 0x93;
							} else {
								memcpy (&SlowData[3], &voice[9], 3);
								SlowData[3] ^= 0x70;
								SlowData[4] ^= 0x4f;
								SlowData[5] ^= 0x93;
								if ((SlowData[0] & 0xf0) == 0x30) DprsData (SlowData);
							}
						}
					}
					if (((BitsCheck((BYTE *)voice, NullVoice0, 9) < 5) || (BitsCheck((BYTE *)voice, NullVoice_ID31, 9) < 5)) && SendHeader)
					{
						SendToInet = FALSE;
						SendHeader = FALSE;	/* header sending switch */
					}
					if (SendToInet) send_voice_queue();
					Packet_Seq++;
					if (Packet_Seq > 20) Packet_Seq = 0;
					send_packet_cnt++;
					voice_pnt = 0;
					LED2_IO ^= 1;
				}
			}
		} else if (length == 0)
		{
			usb_control_msg (0xC0, GET_AD_STATUS, 0, 0, &status, 1, 100);
			if (!(status & COS_OnOff))
			{
				headerRead = FALSE;
				COS_On = FALSE;
				header_gen = FALSE;
				SendHeader = FALSE;
				return;
			}
		}
		usb_control_msg (0xC0, GET_AD_STATUS, 0, 0, &status, 1, 100);
		if (status & HeaderDecodeDone)
		{
				headerRead = FALSE;
				COS_On = FALSE;
				header_gen = FALSE;
				SendHeader = FALSE;
				return;
		}
	}
}

static	BYTE	dprs_data[150];
static	BYTE	dprs_pnt = 0;

static	void	DprsData(BYTE Sdata[])
{
	static	BYTE	len;
		len = Sdata[0] & 0x0f;
		if ((dprs_pnt + len) >= 120)
		{
			dprs_pnt = 0;
			return;
		} 
		memcpy (&dprs_data[dprs_pnt], &Sdata[1], len);
		dprs_pnt += len;
		if (dprs_data[dprs_pnt - 1] == 0x0d) dprs_data[dprs_pnt++] = 0x0a;
		if (dprs_data[dprs_pnt - 1] == 0x0a)
		{
			if (DebugSW) xQueueSend( xDstarLogQueue, &dprs_data, 0 );
			if (gps_sumcheck(dprs_data))
			{
					dprs_data[dprs_pnt - 1] = 0x00;
					if(!memcmp (dprs_data, "$GPGLL,", 7)) GPGLL (&dprs_data[7]);
					else if(!memcmp (dprs_data, "$GPGGA,", 7)) GPGGA (&dprs_data[7]);
					else if(!memcmp (dprs_data, "$GPRMC,", 7)) GPRMC (&dprs_data[7]);
					else if(!memcmp (dprs_data, "$$CRC", 5)) GPS_A (&dprs_data[10]);
					else if(dprs_data[0] != '$') dprs_message(dprs_data);
			}

			dprs_pnt = 0;
		} 
}


static	void	send_voice_queue() 	/* read from rig (DV packet) */
{
	extern	BYTE voice[12];

	if (SendHeader)
	{
				send_header_queue();
				SendHeader = FALSE;
				COS_On = TRUE;
				pkt.B_header.seq = Packet_Seq;
	}
	memcpy ((char *)pkt.DPlusID, (char *)VoiceID, 2);
	memcpy ((char *)pkt.ID,  "DSVT" ,4);
	pkt.flags[0] = 0x20;
	pkt.flags[1] = 0x00;
	pkt.reserve[0] = 0x81;
	pkt.reserve[1] = 0x00;
	pkt.B_header.B_ID = 0x20;
	pkt.B_header.DestID = 0x00;
	pkt.B_header.SrcID = 0x01;
	pkt.B_header.SrcTermID = 0x02;
//	pkt.B_header.seq = Packet_Seq;
	pkt.B_header.PacketID[0] = (Packet_ID >> 8) & 0xff;
	pkt.B_header.PacketID[1] = Packet_ID & 0xff;
	memcpy ((char *)pkt.voice, (char *)voice, 12);
	xQueueSend (xDstarSndQueue, &pkt, 20 / portTICK_RATE_MS);
}

static	void	send_header_queue() 	/* read from rig (DV packet) */
{
	extern BYTE header[41];

/* RF header send */
		memcpy ((char *)pkt.DPlusID, (char *)HeaderID, 2);
		memcpy ((char *)pkt.ID,  "DSVT" ,4);
		pkt.flags[0] = 0x10;
		pkt.flags[1] = 0x00;
		pkt.reserve[0] = 0x81;
		pkt.reserve[1] = 0x00;
		pkt.B_header.B_ID = 0x20;
		pkt.B_header.DestID = 0x00;
		pkt.B_header.SrcID = 0x01;
		pkt.B_header.SrcTermID = 0x02;
		pkt.B_header.seq = 0x80;
		Packet_ID = LFSRRand();
		pkt.B_header.PacketID[0] = (Packet_ID >> 8) & 0xff;
		pkt.B_header.PacketID[1] = Packet_ID & 0xff;
		Packet_Seq = 0;

		sprintf (text, "%19.19s From Radio MyCall:%8.8s YourCall:%8.8s RPT1:%8.8s RPT2:%8.8s ID:%2.2x%2.2x\r\n",
				TimeDate,&header[27],&header[19],&header[11],&header[3],pkt.B_header.PacketID[0],pkt.B_header.PacketID[1]);
		xQueueSend( xDstarLogQueue, &text, 0 );
		if (!(header[0] & 0x40))			/* check dup ? */
		{
			sprintf (text, "%19.19s From Radio Header replace RPT1: %8.8s->%8.8s  RPT2: %8.8s->%8.8s\r\n",
							TimeDate,&header[11], AppConfig.DefaultNodeName, &header[3], RefName);
			xQueueSend( xDstarLogQueue, &text, 0 );
 			memcpy (&header[3], RefName, 8);
			memcpy (&header[11], AppConfig.DefaultNodeName, 8);
		}
		memcpy ((char *)&pkt.header, (char *)header, 41);
		xQueueSend (xDstarSndQueue, &pkt, 20 / portTICK_RATE_MS);

		sprintf (text, "%19.19s From Radio MyCall:%8.8s YourCall:%8.8s RPT1:%8.8s RPT2:%8.8s ID:%2.2x%2.2x\r\n",
							TimeDate,&header[27],&header[19],&header[11],&header[3],pkt.B_header.PacketID[0],
							pkt.B_header.PacketID[1]);
		xQueueSend( xDstarLogQueue, &text, 0 );
}

static	void	RF_CallsignDisp(BYTE RFheader[])
{
		memset (Text, 0x20, 32);
		memcpy (&Text[27], "Radio", 5);
		USB_CallDisplay(RFheader);
	return;
}

static	void	RF_BlockedCallsignDisp(BYTE RFheader[])
{
		memset (Text, 0x20, 32);
		memcpy (&Text[27], " Deny", 5);
		USB_CallDisplay(RFheader);
	return;
}

static	void	Inet_CallsignDisp(BYTE RFheader[], BYTE RcvPacketID[])
{
	memset (Text, 0x20, 32);
	if (!memcmp(RcvPacketID, HeaderID, 2)) memcpy (&Text[27]," Inet",5);
	else	memcpy (&Text[27]," Link",5);
	USB_CallDisplay(RFheader);
	return;
}

static	void	USB_CallDisplay (BYTE RFheader[])
{
	static	xLCDMessage xMessage;
	extern	xQueueHandle xLCDQueue;
	static	BYTE i;
	static	ROM BYTE hex[16] = {"0123456789ABCDEF"};

	memcpy (&Text[0],&RFheader[27], 8);
	memcpy (CurUser, &RFheader[27], 8);
	Text[14] = hex[RFheader[0]>>4];
	Text[15] = hex[RFheader[0] & 0x0f];
	memcpy (&Text[16],&RFheader[19], 8);
	if (memcmp (&RFheader[35],"    ",4))
	{
		Text[8] = '/';
		memcpy (&Text[9],&RFheader[35], 4);
	}
	for (i = 0; i < 32 ; i++)
	{
		if (Text[i] == 0x00) Text[i] = 0x20; 
	}
	xMessage.pcMessage = (char *)&Text;
	xMessage.xMinDisplayTime = 500 / portTICK_RATE_MS;
	xQueueSend( xLCDQueue, &xMessage, 50 / portTICK_RATE_MS );
	return;
}

static	DWORD  usb_control_msg (BYTE bmRequestType, BYTE bRequest, WORD wValue, WORD wIndex, BYTE *data, WORD wLength, WORD timeout)
{
	static	DWORD byteCount;
	static	BYTE errorCode;
	static	BYTE RetCode;
	static	DWORD dwRetCode;

	if (bmRequestType & 0x80)
	{
		RetCode = USBHostIssueDeviceRequest( 1, bmRequestType, bRequest, wValue, wIndex, wLength, data, USB_DEVICE_REQUEST_GET, 0xff);
		if (RetCode== USB_SUCCESS)
		{
			if (timeout == 0)
			{
				return 0ul;
			}
			while (!USBHostTransferIsComplete( 1, 0, &errorCode, &byteCount)) vTaskDelay (1 / portTICK_RATE_MS);
			return byteCount;
		} else if (RetCode == USB_ENDPOINT_BUSY) return 0ul;
		else
		{
			dwRetCode = RetCode + 1000;
			return dwRetCode;
		}
	} else {
		RetCode = USBHostIssueDeviceRequest( 1, bmRequestType, bRequest, wValue, wIndex, wLength, data, USB_DEVICE_REQUEST_SET, 0xff);
		if (RetCode == USB_SUCCESS)
		{
			if (timeout == 0)
			{
				if (USBHostTransferIsComplete( 1, 0, &errorCode, &byteCount)) return byteCount;
				else return 1021ul;
			}
			while (!USBHostTransferIsComplete( 1, 0, &errorCode, &byteCount )) 	vTaskDelay (1 / portTICK_RATE_MS);
			return byteCount;
		} else if (RetCode == USB_ENDPOINT_BUSY) return 1000ul;
		else return 1020ul;
	}
	return byteCount;
}


static void	LoadAdParam()
{
	static BYTE dummy;

		AdParam.UpdateFlag = 0x01;
  	    usb_control_msg (0xC0, GET_DelayTime, 0, 0, &AdParam.DelayTime, 1, 100);
  	    usb_control_msg (0xC0, GET_TimeOut, 0, 0, &AdParam.TimeOut, 1, 100);
  	    usb_control_msg (0xC0, GET_KeepAlive, 0, 0, &AdParam.KeepAlive, 1, 100);	
  	    usb_control_msg (0xC0, GET_JitterBufferSize, 0, 0, &AdParam.JitterSize, 1, 100);	
  	    usb_control_msg (0xC0, GET_MODE, 0, 0, &AdParam.Mode, 1, 100);	
  	    usb_control_msg (0xC0, GET_MODE2, 0, 0, &AdParam.Mode2, 1, 100);	
  	    usb_control_msg (0xC0, GET_INVERT_STATUS, 0, 0, &AdParam.Invert, 1, 100);	
  	    usb_control_msg (0xC0, GET_SN_SQ, 0, 0, (BYTE *)&AdParam.SN_Squelch, 2, 100);
		AdParam.SN_Squelch = swaps_usb(AdParam.SN_Squelch) >> 6;	

  	    usb_control_msg (0xC0, GET_BeaconMessageInit, 0, 0, &dummy, 0, 100);	
  	    usb_control_msg (0xC0, GET_BeaconMessage, 0, 0, (BYTE *)&AdParam.BeaconMessage, 20, 100);	
  	    usb_control_msg (0xC0, GET_MyRPTCALL, 0, 0, (BYTE *)&AdParam.BeaconCall, 8, 100);	
  	    usb_control_msg (0xC0, GET_BeaconTimer, 0, 0, &AdParam.BeaconInterval, 1, 100);	

  	    usb_control_msg (0xC0, GET_AltMyCall, 0, 0, (BYTE *)&AdParam.AltMyCall, 8, 100);	
  	    usb_control_msg (0xC0, GET_AltMyCall2, 0, 0, (BYTE *)&AdParam.AltMyCall2, 4, 100);	
  	    usb_control_msg (0xC0, GET_AltYourCall, 0, 0, (BYTE *)&AdParam.AltYourCall, 8, 100);	
  	    usb_control_msg (0xC0, GET_AltRPT1Call, 0, 0, (BYTE *)&AdParam.AltRPT1Call, 8, 100);	
  	    usb_control_msg (0xC0, GET_AltRPT2Call, 0, 0, (BYTE *)&AdParam.AltRPT2Call, 8, 100);
  	    usb_control_msg (0xC0, GET_RepeaterMode, 0, 0, (BYTE *)&AdParam.AltFlags, 3, 100);
	
}

static	void	SaveAdParam()
{
	static	BYTE	dummy;
	static	DWORD	ret, k;
	static	WORD	i;

  	    ret = usb_control_msg (0x40, SET_DelayTime, AdParam.DelayTime, 0, &dummy, 0, 100);
  	    ret = usb_control_msg (0x40, SET_TimeOut, AdParam.TimeOut, 0, &dummy, 0, 100);
  	    ret = usb_control_msg (0x40, SET_KeepAlive, AdParam.KeepAlive, 0, &dummy, 0, 100);	
  	    ret = usb_control_msg (0x40, SET_JitterBufferSize, AdParam.JitterSize, 0, &dummy, 0, 100);	
		if (AdParam.Mode & 0x02)
		{
			usb_control_msg(0x40, SET_CRC_CHECK, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg(0x40, SET_CRC_CHECK, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Mode & 0x04)
		{
			usb_control_msg (0x40, SET_COS_CHECK, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_COS_CHECK, OFF, 0, &dummy, 0, 100);
		}	
		if (AdParam.Mode & 0x08)
		{
			usb_control_msg (0x40, SET_LastFrame, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_LastFrame, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Mode & 0x80)
		{
			usb_control_msg (0x40, SET_HalfFull, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_HalfFull, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Mode2 & 0x01)
		{
			usb_control_msg (0x40, SET_HeaderGen, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_HeaderGen, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Mode2 & 0x02)
		{
			usb_control_msg (0x40, SET_HeaderGenType, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_HeaderGenType, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Mode2 & 0x04)
		{
			usb_control_msg (0x40, SET_HeaderRep, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_HeaderRep, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Invert & 0x80)
		{
			usb_control_msg (0x40, SET_AutoRxDetect, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AutoRxDetect, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Invert & 0x01)
		{
			usb_control_msg (0x40, SET_TX_Invert, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_TX_Invert, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.Invert & 0x02)
		{
			usb_control_msg (0x40, SET_RX_Invert, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_RX_Invert, OFF, 0, &dummy, 0, 100);
		}
  	    ret = usb_control_msg (0x40, SET_SN_SQ, AdParam.SN_Squelch, 0, &dummy, 0, 100);	
		vTaskDelay(500 / portTICK_RATE_MS);
  	    usb_control_msg (0x40, SET_SlowDataSetInit, 0, 0, &dummy, 0, 100);	
  	    usb_control_msg (0x40, SET_BeaconMessage, 0, 0, (BYTE *)&AdParam.BeaconMessage, 20, 100);
		vTaskDelay(500 / portTICK_RATE_MS);
  	    usb_control_msg (0x40, SET_MyRPTCALL, 0, 0, (BYTE *)&AdParam.BeaconCall, 8, 100);
		k = AdParam.BeaconInterval;	
  	    usb_control_msg (0x40, SET_BeaconTimer, k, 0, &dummy, 0, 100);	

  	    usb_control_msg (0x40, SET_AltMyCall, 0, 0, (BYTE *)&AdParam.AltMyCall, 8, 100);	
  	    usb_control_msg (0x40, SET_AltMyCall2, 0, 0, (BYTE *)&AdParam.AltMyCall2, 4, 100);	
  	    usb_control_msg (0x40, SET_AltYourCall, 0, 0, (BYTE *)&AdParam.AltYourCall, 8, 100);	
  	    usb_control_msg (0x40, SET_AltRPT1Call, 0, 0, (BYTE *)&AdParam.AltRPT1Call, 8, 100);	
  	    usb_control_msg (0x40, SET_AltRPT2Call, 0, 0, (BYTE *)&AdParam.AltRPT2Call, 8, 100);
		i = AdParam.AltFlags[2];
  	    usb_control_msg (0x40, SET_AltFlagSet, i, 0, &dummy, 0, 100);

		if (AdParam.AltFlags[0] & 0x08)
		{
			usb_control_msg (0x40, SET_AltMyCallSet, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AltMyCallSet, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.AltFlags[0] & 0x10)
		{
			usb_control_msg (0x40, SET_AltMyCall2Set, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AltMyCall2Set, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.AltFlags[0] & 0x20)
		{
			usb_control_msg (0x40, SET_AltYourCallSet, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AltYourCallSet, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.AltFlags[0] & 0x40)
		{
			usb_control_msg (0x40, SET_AltRPT1CallSet, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AltRPT1CallSet, OFF, 0, &dummy, 0, 100);
		}
		if (AdParam.AltFlags[0] & 0x80)
		{
			usb_control_msg (0x40, SET_AltRPT2CallSet, ON, 0, &dummy, 0, 100);
		} else {
			usb_control_msg (0x40, SET_AltRPT2CallSet, OFF, 0, &dummy, 0, 100);
		}

		AdParam.UpdateFlag = 0x00;
			
}


/* Jitter Buffer */
void	JitterBufferInit()
{
	static	int	i;

	for (i = 0 ; i < JitterBufferSize ; i++)
	{
		if (i%21 == 0)
		{
			memcpy (&JitterBuffer[i][0], NullVoice0, 12);
		} else {
			memcpy (&JitterBuffer[i][0], NullVoice1, 12);
		}
	}
	JitterBufferPnt = 0;
	VoicePacketCnt = 0;
	VoicePacketSW = FALSE;
	SendHeader = FALSE;
	PTT_On = FALSE;
}

void	JitterBufferSave (BYTE VoicePacket[], BYTE sequence)
{
	memcpy (&JitterBuffer[sequence%JitterBufferSize][0], VoicePacket, 12);
	VoicePacketCnt++;
	if (VoicePacketCnt > AppConfig.JitterBufferWait) VoicePacketSW = TRUE;
}

void	JitterBufferSend ()
{
	static	BYTE	len;
	static	BYTE	dummy;
	extern	BOOL	headerRead;

	if (!PTT_On)
	{
		if (headerRead)
		{
			JitterBufferPnt = 0;
			VoicePacketCnt = 0;
			VoicePacketSW = FALSE;
			return;
		}	
		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >=95)
			{
				usb_control_msg(0x40, SET_PTT, ON, 0, &dummy, 0, 100);
				PTT_On = TRUE;
				LED2_IO = 1;
				vTaskDelay (150 / portTICK_RATE_MS);
				break;
			} else {
				vTaskDelay (200 / portTICK_RATE_MS);
			}
		}
	}	

	while (1)
	{	
	   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
		if (len >= 12)
		{
   			usb_control_msg(0x40, PUT_DATA, 0, 0, &JitterBuffer[JitterBufferPnt][0],12, 100);
			VoicePacketCnt--;
			LED2_IO ^= 1;

			if (JitterBufferPnt == 0) memcpy (&JitterBuffer[0][0], NullVoice0, 12);
			else 					  memcpy (&JitterBuffer[JitterBufferPnt][0], NullVoice1, 12);

			JitterBufferPnt++;
			JitterBufferPnt = JitterBufferPnt % JitterBufferSize;
	
			if (RemainVoicePacketCnt)
			{
				RemainVoicePacketCnt--;
				if (!RemainVoicePacketCnt)
				{
					while (1)
					{
					   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
						if (len >=95)
						{
							usb_control_msg(0x40, SET_PTT, ON, 0, &dummy, 0, 100);
							PTT_On = TRUE;
							LED2_IO = 1;
							break;
						} else {
							vTaskDelay (200 / portTICK_RATE_MS);
						}
					}
				}
			}
			break;
		} else {
			vTaskDelay (50 / portTICK_RATE_MS);
		}
	}
}

BYTE	ConnectedMsg[13][12] = {
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x55,0x2d,0x16},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x30,0x0c,0xdc},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x3e,0x01,0xd6},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x31,0x0c,0xc7},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x35,0x0b,0xb3},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x32,0x62,0xb3},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x22,0x0a,0xd5},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x33,0x7f,0xa7},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x47,0x6f,0xb3},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x16,0x29,0xf5},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x16,0x29,0xf5},
		{0x9e,0x8d,0x32,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x55,0x55,0x55},
		{0x55,0xc8,0x7a,0x88,0x26,0x1a,0x3f,0x61,0xe8,0x55,0x55,0x55}};


void	BeepSend (BYTE callsign[])
{
	static	WORD	i;
	static	BYTE	buff[12], flags[3];
	static	BYTE	len;

		/* Call Sign set */
		flags[0] = 0x00;
		flags[1] = 0x00;
		flags[2] = 0x00;
        usb_control_msg (0x40, SET_MyCALL, 0, 0, AppConfig.DefaultNodeName, 8, 100);
        usb_control_msg (0x40, SET_MyCALL2, 0, 0, (BYTE *)"    ", 4, 100);
        usb_control_msg (0x40, SET_YourCALL, 0, 0, callsign, 8, 100);
        usb_control_msg (0x40, SET_RPT2CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_RPT1CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_FLAGS, 0, 0, flags, 3, 100);

		/* PTT ON */
		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >=95)
			{
				usb_control_msg(0x40, SET_PTT, ON, 0, buff, 0, 100);
				PTT_On = TRUE;
				break;
			} else {
				vTaskDelay (200 / portTICK_RATE_MS);
			}
		}

		for (i = 0 ; i < 11 ; i++)
		{
			memcpy (buff, &ConnectedMsg[i][0], 12);
			if (memcmp(RefName, "        ", 8))
			{
				switch (i)
				{
					case 1:
						buff[10] = 'L' ^ 0x4f;
						buff[11] = 'I' ^ 0x93;
						break;
					case 2:
						buff[ 9] = 'N' ^ 0x70;
						buff[10] = 'K' ^ 0x4f;
						buff[11] = 'E' ^ 0x93;
						break;
					case 3:
						buff[10] = 'D' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
					case 4:
						buff[ 9] = '-' ^ 0x70;
						buff[10] = ' ' ^ 0x4f;
						buff[11] = RefName[0] ^ 0x93;
						break;
					case 5:
						buff[10] = RefName[1] ^ 0x4f;
						buff[11] = RefName[2] ^ 0x93;
						break;
					case 6:
						buff[ 9] = RefName[3] ^ 0x70;
						buff[10] = RefName[4] ^ 0x4f;
						buff[11] = RefName[5] ^ 0x93;
						break;
					case 7:
						buff[10] = RefName[6] ^ 0x4f;
						buff[11] = RefName[7] ^ 0x93;
						break;
					case 8:
						buff[ 9] = ' ' ^ 0x70;
						buff[10] = ' ' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
				}
			} else {
				switch (i)
				{
					case 1:
						buff[10] = 'L' ^ 0x4f;
						buff[11] = 'I' ^ 0x93;
						break;
					case 2:
						buff[ 9] = 'N' ^ 0x70;
						buff[10] = 'K' ^ 0x4f;
						buff[11] = 'E' ^ 0x93;
						break;
					case 3:
						buff[10] = 'D' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
					case 4:
						buff[ 9] = '-' ^ 0x70;
						buff[10] = ' ' ^ 0x4f;
						buff[11] = 'N' ^ 0x93;
						break;
					case 5:
						buff[10] = 'O' ^ 0x4f;
						buff[11] = 'N' ^ 0x93;
						break;
					case 6:
						buff[ 9] = 'E' ^ 0x70;
						buff[10] = ' ' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
					case 7:
						buff[10] = ' ' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
					case 8:
						buff[ 9] = ' ' ^ 0x70;
						buff[10] = ' ' ^ 0x4f;
						buff[11] = ' ' ^ 0x93;
						break;
				}
			}
			while (1)
			{
			   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
				if (len >= 12)
				{
					usb_control_msg(0x40, PUT_DATA, 0, 0, buff, 12, 100);
					break;
				} else {
					vTaskDelay (30 / portTICK_RATE_MS);
				}
			}
		}

		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >= 15)
			{
				usb_control_msg(0x40, PUT_DATA, 0, 0, &ConnectedMsg[11][0], 15, 100);
				break;
			} else {
				vTaskDelay (50 / portTICK_RATE_MS);
			}
		}

		/* PTT OFF */
		usb_control_msg(0x40, SET_PTT, OFF, 0, buff, 0, 100);
}

void	DisConnectedMsgSend (BYTE callsign[])
{
	static	WORD	i;
	static	BYTE	buff[12], flags[3];
	static	BYTE	len;


		/* Call Sign set */
		flags[0] = 0x00;
		flags[1] = 0x00;
		flags[2] = 0x00;
        usb_control_msg (0x40, SET_MyCALL, 0, 0, AppConfig.DefaultNodeName, 8, 100);
        usb_control_msg (0x40, SET_MyCALL2, 0, 0, (BYTE *)"    ", 4, 100);
        usb_control_msg (0x40, SET_YourCALL, 0, 0, callsign, 8, 100);
        usb_control_msg (0x40, SET_RPT2CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_RPT1CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_FLAGS, 0, 0, flags, 3, 100);

		/* PTT ON */
		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >=95)
			{
				usb_control_msg(0x40, SET_PTT, ON, 0, buff, 0, 100);
				PTT_On = TRUE;
				break;
			} else {
				vTaskDelay (200 / portTICK_RATE_MS);
			}
		}

		for (i = 0 ; i < 11 ; i++)
		{
			memcpy (buff, &ConnectedMsg[i][0], 12);
			switch (i)
			{
				case 1:
					buff[10] = 'U' ^ 0x4f;
					buff[11] = 'N' ^ 0x93;
					break;
				case 2:
					buff[ 9] = 'L' ^ 0x70;
					buff[10] = 'I' ^ 0x4f;
					buff[11] = 'N' ^ 0x93;
					break;
				case 3:
					buff[10] = 'K' ^ 0x4f;
					buff[11] = 'E' ^ 0x93;
					break;
				case 4:
					buff[ 9] = 'D' ^ 0x70;
					buff[10] = ' ' ^ 0x4f;
					buff[11] = '-' ^ 0x93;
					break;
				case 5:
					buff[10] = ' ' ^ 0x4f;
					buff[11] = RefName[0] ^ 0x93;
					break;
				case 6:
					buff[ 9] = RefName[1] ^ 0x70;
					buff[10] = RefName[2] ^ 0x4f;
					buff[11] = RefName[3] ^ 0x93;
					break;
				case 7:
					buff[10] = RefName[4] ^ 0x4f;
					buff[11] = RefName[5] ^ 0x93;
					break;
				case 8:
					buff[ 9] = RefName[6] ^ 0x70;
					buff[10] = RefName[7] ^ 0x4f;
					buff[11] = ' ' ^ 0x93;
					break;
			}
			while (1)
			{
			   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
				if (len >= 12)
				{
					usb_control_msg(0x40, PUT_DATA, 0, 0, buff, 12, 100);
					break;
				} else {
					vTaskDelay (50 / portTICK_RATE_MS);
				}
			}
		}

		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >= 15)
			{
				usb_control_msg(0x40, PUT_DATA, 0, 0, &ConnectedMsg[11][0], 15, 100);
				break;
			} else {
				vTaskDelay (50 / portTICK_RATE_MS);
			}
		}

		/* PTT OFF */
		usb_control_msg(0x40, SET_PTT, OFF, 0, buff, 0, 100);
}

void	NotFoundMsgSend (BYTE callsign[])
{
	static	WORD	i;
	static	BYTE	buff[12], flags[3];
	static	BYTE	len;


		/* Call Sign set */
		flags[0] = 0x00;
		flags[1] = 0x00;
		flags[2] = 0x00;
        usb_control_msg (0x40, SET_MyCALL, 0, 0, AppConfig.DefaultNodeName, 8, 100);
        usb_control_msg (0x40, SET_MyCALL2, 0, 0, (BYTE *)"    ", 4, 100);
        usb_control_msg (0x40, SET_YourCALL, 0, 0, callsign, 8, 100);
        usb_control_msg (0x40, SET_RPT2CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_RPT1CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_FLAGS, 0, 0, flags, 3, 100);

		/* PTT ON */
		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >=95)
			{
				usb_control_msg(0x40, SET_PTT, ON, 0, buff, 0, 100);
				PTT_On = TRUE;
				break;
			} else {
				vTaskDelay (200 / portTICK_RATE_MS);
			}
		}

		for (i = 0 ; i < 11 ; i++)
		{
			memcpy (buff, &ConnectedMsg[i][0], 12);
			switch (i)
			{
				case 1:
					buff[10] = 'N' ^ 0x4f;
					buff[11] = 'O' ^ 0x93;
					break;
				case 2:
					buff[ 9] = 'T' ^ 0x70;
					buff[10] = ' ' ^ 0x4f;
					buff[11] = 'F' ^ 0x93;
					break;
				case 3:
					buff[10] = 'O' ^ 0x4f;
					buff[11] = 'U' ^ 0x93;
					break;
				case 4:
					buff[ 9] = 'N' ^ 0x70;
					buff[10] = 'D' ^ 0x4f;
					buff[11] = ' ' ^ 0x93;
					break;
				case 5:
					buff[10] = '-' ^ 0x4f;
					buff[11] = ' ' ^ 0x93;
					break;
				case 6:
					buff[ 9] = RefName[0] ^ 0x70;
					buff[10] = RefName[1] ^ 0x4f;
					buff[11] = RefName[2] ^ 0x93;
					break;
				case 7:
					buff[10] = RefName[3] ^ 0x4f;
					buff[11] = RefName[4] ^ 0x93;
					break;
				case 8:
					buff[ 9] = RefName[5] ^ 0x70;
					buff[10] = RefName[6] ^ 0x4f;
					buff[11] = RefName[7] ^ 0x93;
					break;
			}
			while (1)
			{
			   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
				if (len >= 12)
				{
					usb_control_msg(0x40, PUT_DATA, 0, 0, buff, 12, 100);
					break;
				} else {
					vTaskDelay (50 / portTICK_RATE_MS);
				}
			}
		}

		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >= 15)
			{
				usb_control_msg(0x40, PUT_DATA, 0, 0, &ConnectedMsg[11][0], 15, 100);
				break;
			} else {
				vTaskDelay (50 / portTICK_RATE_MS);
			}
		}

		/* PTT OFF */
		usb_control_msg(0x40, SET_PTT, OFF, 0, buff, 0, 100);
}

WORD swaps_usb(WORD v)
{
	WORD_VAL t;
	BYTE b;

	t.Val   = v;
	b       = t.v[1];
	t.v[1]  = t.v[0];
	t.v[0]  = b;

	return (WORD)t.Val;
}

static	BOOL	gps_sumcheck(BYTE string[])
{
	BYTE	*pnt;
	static	BYTE	sum, csum, tmp;

	if (!memcmp (string, "$$CRC", 5))
	{
		return GPS_A_SumCheck(string);
	}
	pnt = string;
	sum = 0;
	if (*pnt == '$') pnt++;
	while (*pnt !='*')
	{
		if ((*pnt == 0x0a) || (*pnt == 0x0d)) return FALSE;
		sum ^= *pnt;
		pnt++;
	}
	pnt++;
	tmp = *pnt - '0';
	if (tmp > 16) tmp -= 7;
	csum = tmp << 4;
	pnt++;
	tmp = *pnt - '0';
	if (tmp > 16) tmp -= 7;
	csum += tmp;
	if (csum == sum) return TRUE;
	return FALSE;
}

static	void	GPGLL (BYTE string[])
{
	static	BYTE	*pnt;
	static	BYTE	i;

	pnt = string;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 7)
		{
			RadioLat[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLat[7] = *pnt;
	pnt += 2;

	i = 0;
	while (*pnt != ',')
	{
		if (i < 8)
		{
			RadioLong[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLong[8] = *pnt;
}

static	void	GPGGA (BYTE string[])
{
	static	BYTE	*pnt;
	static	BYTE	i;
	static	DWORD	d;
	static	BYTE	tmp[10];
	static	WORD	k;

	pnt = string;
	while (*pnt != ',') pnt++;
	pnt++;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 7)
		{
			RadioLat[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLat[7] = *pnt;
	pnt += 2;

	i = 0;
	while (*pnt != ',')
	{
		if (i < 8)
		{
			RadioLong[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLong[8] = *pnt;

	pnt += 2;
	while (*pnt != ',') pnt++;
	pnt++;
	while (*pnt != ',') pnt++;
	pnt++;
	while (*pnt != ',') pnt++;
	pnt++;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 8)
		{
			tmp[i] = *pnt;
			i++;
		}
		pnt++;
	}
	tmp[i] = 0x00;
	d = atof ((char *)tmp);
	k = d / 0.3048 + 0.5;
	sprintf ((char *)RadioAtitude, "%06d", k); 
}

static	void	GPRMC (BYTE string[])
{
	static	DWORD	d;
	static	BYTE	tmp[10];
	static	WORD	k;
	static	BYTE	*pnt;
	static	BYTE	i;

	pnt = string;
	while (*pnt != ',') pnt++;
	pnt++;
	while (*pnt != ',') pnt++;
	pnt++;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 7)
		{
			RadioLat[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLat[7] = *pnt;
	pnt += 2;

	i = 0;
	while (*pnt != ',')
	{
		if (i < 8)
		{
			RadioLong[i] = *pnt;
			i++;
		}
		pnt++;
	}
	pnt++;
	RadioLong[8] = *pnt;
	pnt += 2;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 9)
		{
			tmp[i] = *pnt;
			i++;
		}
		pnt++;
	}
	tmp[i] = 0x00;
	d = atof ((char *)tmp);
	k = d + 0,5;
	sprintf ((char *)RadioSpeed, "%03d", k);

	pnt++;
	i = 0;
	while (*pnt != ',')
	{
		if (i < 9)
		{
			tmp[i] = *pnt;
			i++;
		}
		pnt++;
	}
	tmp[i] = 0x00;
	d = atof ((char *)tmp);
	k = d + 0,5;
	sprintf ((char *)RadioDirection, "%03d", k);
}

static	void	dprs_message (BYTE string[])
{
	static	BYTE	*pnt;
	static	BYTE	i, k;
	static	BYTE	len;

//	len = strlen ((char *)&string);
//	if (len <= 9) return;

	pnt = string;
	i = 0;
	memset (RadioCall, 0x20, 8);
	while (*pnt != ',')
	{
		if (*pnt == 0x00) return;
		if (i < 8)
		{
			RadioCall[i] = *pnt;
			i++;
		}
		pnt++;
	}
	k = 0;
	for (i = 0 ; i < 8 ; i++)
	{
		if (RadioCall[i] != 0x20)
		{
			RadioCall[k] = RadioCall[i];
			k++;
		} else {
			if (RadioCall[k-1] != '-')
			{
				RadioCall[k] = '-';
				k++;
			}
		}
	}

	if (RadioCall[k-1] == '-') RadioCall[k-1] = 0x00;

	if (k < 7)
	{
		for (i = k ; i < 8 ; i++)
		{
			RadioCall[k] = 0x00;
		}
	}

	memset (RadioMsg, 0x00, 20);
	len = strlen ((char *)&string[9]);
	if (len == 0) return;
	memcpy (RadioMsg, &string[9], len);
	RadioDprsSend = TRUE;
}

/*
	CRC update routine
 */

WORD	update_crc_dstar( WORD crc, BYTE c ) {

    WORD tmp, short_c;

    short_c  = 0x00ff &  c;
	
    tmp = (crc & 0x00ff) ^ short_c;
    crc = (crc >> 8)  ^ crc_tabccitt[tmp];

    return crc;

}  /* update_crc_dstar */


/*
	CRC reslut routine
 */

WORD		result_crc_dstar(WORD crc) {

    WORD tmp;

      crc = ~crc;
      tmp = crc;
      crc = (crc << 8) | (tmp >> 8 & 0xff);

    return crc;

}  /* result_crc_dstar */

static BOOL	GPS_A_SumCheck(BYTE string[])
{
	static	WORD	crc_dstar_ffff, k, k0, k1, k2, k3;
	static	BYTE	*pnt;

    crc_dstar_ffff = 0xffff;	/* nornal value 0xffff */
	pnt = string + 10;

	while (*pnt != 0x0a)
	{
		crc_dstar_ffff = update_crc_dstar( crc_dstar_ffff, *pnt);
		pnt++;
	}

	crc_dstar_ffff = result_crc_dstar(crc_dstar_ffff);

	k0 = string[5] - '0';
	if (k0 > 16) k0 -= 7;
	k1 = string[6] - '0';
	if (k1 > 16) k1 -= 7;
	k2 = string[7] - '0';
	if (k2 > 16) k2 -= 7;
	k3 = string[8] - '0';
	if (k3 > 16) k3 -= 7;
	k1 += k0 * 16;
	k3 += k2 * 16;
    k = k1 | (k3 << 8);

	if (k == crc_dstar_ffff) return TRUE;
	return	FALSE;
}

void	GPS_A (BYTE string[])
{
	memcpy (Radio_GPS_A_MSG, string, strlen((char *)string));
	Radio_GPS_A_Msg_send = TRUE;
}

static	WORD	BitsCheck(BYTE pattern1[], BYTE pattern2[], WORD cnt)
{
	static	BYTE	temp;
	static	WORD	i, j, k;
	
	k = 0;
	for (i = 0 ; i < cnt ; i++)
	{
		temp = pattern1[i] ^ pattern2[i];
		for (j = 0 ; j < 8 ; j++)
		{
			if (temp & (0x80 >> i)) k++; 
		}
	}
	return k;
}

BYTE	DprsMyCall[9];

static	void	DprsMsgSendToRF ()
{
	static	WORD	i;
	static	BYTE	buff[12], flags[3];
	static	BYTE	tmp[40], MyCall[9], YourCall[8];
	static	BYTE	len;


		if (xQueueReceive (xDstarDprsMsgQueue, tmp, 0) != pdTRUE) return;

		if (DebugSW)
		{
			sprintf ((char *)text, "%19.19s APRS Message on RF from %9.9s to %9.9s %20.20s\r\n",TimeDate, (char *)&tmp, (char *)&tmp[10], (char *)&tmp[20]);
			xQueueSend( xDstarLogQueue, &text, 0 );
		}
	
		/* Call Sign set */
		memset (MyCall, 0x20, 9);
		memset (YourCall, 0x20, 8);
		memset (DprsMyCall, 0x20, 9);
		for (i = 0 ; i < 9 ; i++)
		{
			if (tmp[i] == '>') break;
			if (tmp[i] == '-')
			{
				MyCall[i] = 0x20;
			} else {
				MyCall[i] = tmp[i];
			}
		}
		if (MyCall[8] != 0x20)
		{
			MyCall[6] = MyCall[7];
			MyCall[7] = MyCall[8];
		}				
		for (i = 0 ; i < 8 ; i++)
		{
			if (tmp[i+10] == '-') break;
			YourCall[i] = tmp[i+10];
		}
		for (i = 0 ; i < 9 ; i++)
		{
			if (tmp[i] == '>') break;
			DprsMyCall[i] = tmp[i];
		}

		flags[0] = 0x00;
		flags[1] = 0x00;
		flags[2] = 0x00;
        usb_control_msg (0x40, SET_MyCALL, 0, 0, MyCall, 8, 100);
        usb_control_msg (0x40, SET_MyCALL2, 0, 0, (BYTE *)"APRS", 4, 100);
        usb_control_msg (0x40, SET_YourCALL, 0, 0, YourCall, 8, 100);
        usb_control_msg (0x40, SET_RPT2CALL, 0, 0, (BYTE *)"DIRECT  ", 8, 100);
        usb_control_msg (0x40, SET_RPT1CALL, 0, 0, AppConfig.DefaultNodeName, 8, 100);
        usb_control_msg (0x40, SET_FLAGS, 0, 0, flags, 3, 100);

		/* PTT ON */
		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >=95)
			{
				usb_control_msg(0x40, SET_PTT, ON, 0, buff, 0, 100);
				PTT_On = TRUE;
				break;
			} else {
				return;
			}
		}

		for (i = 0 ; i < 11 ; i++)
		{
			memcpy (buff, &ConnectedMsg[i][0], 12);
			switch (i)
			{
				case 1:
					buff[10] = tmp[20] ^ 0x4f;
					buff[11] = tmp[21] ^ 0x93;
					break;
				case 2:
					buff[ 9] = tmp[22] ^ 0x70;
					buff[10] = tmp[23] ^ 0x4f;
					buff[11] = tmp[24] ^ 0x93;
					break;
				case 3:
					buff[10] = tmp[25] ^ 0x4f;
					buff[11] = tmp[26] ^ 0x93;
					break;
				case 4:
					buff[ 9] = tmp[27] ^ 0x70;
					buff[10] = tmp[28] ^ 0x4f;
					buff[11] = tmp[29] ^ 0x93;
					break;
				case 5:
					buff[10] = tmp[30] ^ 0x4f;
					buff[11] = tmp[31] ^ 0x93;
					break;
				case 6:
					buff[ 9] = tmp[32] ^ 0x70;
					buff[10] = tmp[33] ^ 0x4f;
					buff[11] = tmp[34] ^ 0x93;
					break;
				case 7:
					buff[10] = tmp[35] ^ 0x4f;
					buff[11] = tmp[36] ^ 0x93;
					break;
				case 8:
					buff[ 9] = tmp[37] ^ 0x70;
					buff[10] = tmp[38] ^ 0x4f;
					buff[11] = tmp[39] ^ 0x93;
					break;
			}
			while (1)
			{
			   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
				if (len >= 12)
				{
					usb_control_msg(0x40, PUT_DATA, 0, 0, buff, 12, 100);
					break;
				} else {
					vTaskDelay (30 / portTICK_RATE_MS);
				}
			}
		}

		while (1)
		{
		   	usb_control_msg(0xC0, GET_REMAINSPACE, 0, 0, &len ,1, 100);
			if (len >= 15)
			{
				usb_control_msg(0x40, PUT_DATA, 0, 0, &ConnectedMsg[11][0], 15, 100);
				break;
			} else {
				vTaskDelay (50 / portTICK_RATE_MS);
			}
		}

		/* PTT OFF */
		usb_control_msg(0x40, SET_PTT, OFF, 0, buff, 0, 100);
		PTT_On = FALSE;
}
