/*********************************************************************
 *
 *  Main Application Entry Point and TCP/IP Stack Demo
 *  Module for Microchip TCP/IP Stack
 *   -Demonstrates how to call and use the Microchip TCP/IP stack
 *	 -Reference: Microchip TCP/IP Stack Help (TCPIP Stack Help.chm)
 *
 *********************************************************************
 * FileName:        MainDemo.c
 * Dependencies:    TCPIP.h
 * Processor:       PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 * Compiler:        Microchip C32 v1.11b or higher
 *					Microchip C30 v3.24 or higher
 *					Microchip C18 v3.36 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright (C) 2002-2010 Microchip Technology Inc.  All rights
 * reserved.
 *
 * Microchip licenses to you the right to use, modify, copy, and
 * distribute:
 * (i)  the Software when embedded on a Microchip microcontroller or
 *      digital signal controller product ("Device") which is
 *      integrated into Licensee's product; or
 * (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
 *		ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device
 *		used in conjunction with a Microchip ethernet controller for
 *		the sole purpose of interfacing with the ethernet controller.
 *
 * You should refer to the license agreement accompanying this
 * Software for additional information regarding your rights and
 * obligations.
 *
 * THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * MICROCHIP BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
 * BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE
 * THEREOF), ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER
 * SIMILAR COSTS, WHETHER ASSERTED ON THE BASIS OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR OTHERWISE.
 *
 * File Description:
 * Change History:
 * Rev   Description
 * ----  -----------------------------------------
 * 1.0   Initial release
 * V5.36 ---- STACK_USE_MPFS support has been removed 
 ********************************************************************/
/*
 * This macro uniquely defines this file as the main entry point.
 * There should only be one such definition in the entire project,
 * and this file must define the AppConfig variable as described below.
 */
#define THIS_IS_STACK_APPLICATION

// Include all headers for any enabled TCPIP Stack functions
#include "TCPIP Stack/TCPIP.h"


// Include functions specific to this stack application
#include "prvTaskTCPIP.h"

#include "FreeRTOS.h"
#include "queue.h"
#include "lcd.h"
#include "task.h"
#include "node.h"

#include <string.h>

// Declare AppConfig structure and some other supporting stack variables
static unsigned short wOriginalAppConfigChecksum;	// Checksum of the ROM defaults for AppConfig

// Private helper functions.
// These may or may not be present in all applications.
static 	void InitAppConfig(void);
void	DstarDPlusClientUDP(void);
void	ReadMacFromEEPROM(BYTE *SerializedMACAddress);
static	BYTE SerializedMACAddress[6];
void 	DisplayIPValue(IP_ADDR IPVal, BYTE XLCDText[]);
void	TCPClient(void);
void	KeepAliveSend ( void );
int		FSInit( void );
void	FTPInit(void);
void	FTPServer(void);

#define DefaultHeaderGenTime	(5ul*TICK_SECOND)

static BYTE	LCDText[32];
extern	struct	MessageSW	Msg;
//static	WORD	DPlusPort = 20001;

extern	enum
	{
		DS_HOME = 0,
		DS_HOME_FIRST_TIME,
		DS_HOME_SECOND_TIME,
		DS_SOCKET_OBTAINED,
		DS_PROCESS_CONNECTED,
		DS_PROCESS_READY,
		DS_PROCESS_READ,
		DS_DPRS_OPEN,
		DS_DPRS_OPEN_AUTOLINK,
		DS_DPRS_OPEN_AUTORELINK,
		DS_DPRS_OPEN_DONE_AUTOLINK,
		DS_DPRS_OPEN_DONE_AUTORELINK,
		DS_DPRS_ACCEPT,
		DS_DPRS_ACCEPT_DONE,
		DS_WAIT_LINK_COMMAND,
		DS_WAIT_LINK,
		DS_UDP_OPEN_FIRST,
		DS_UDP_OPEN_SECOND,
		DS_UDP_OPEN,
		DS_UDP_IS_OPENED,
		DS_UDP_USB_DONE,
		DS_UDP_SEND,
		DS_UDP_RECV,
		DS_UDP_SEND_CALL,
		DS_UDP_RECV_OK,
		DS_UDP_SUB_SOCKET_OPEN,
		DS_LOOP
	} NodeDPlusState;

extern	UDP_SOCKET	ServerSocket;
extern	UDP_SOCKET	RcvDPlusSocket;
extern	TCP_SOCKET	DprsSocket;
extern	BYTE	Sync[3];
extern	BOOL	FSInitSW;
extern	BYTE	MACReadWriteSW;
extern	BOOL	FWUpdate;
extern	BOOL	DprsOpen;
extern	BYTE	ReqDisConnect[5];

extern	BOOL	COS_On;
extern	BOOL	PTT_On;
//
// Main application entry point.
//
void prvTaskTCPIP (void *pvParameters)
{
	static	DWORD dwLastIP;
//	static	WORD	len;
//	static	BYTE	RecvTemp[10];

	// Initialize stack-related hardware components that may be 
	// required by the UART configuration routines
    TickInit();

	MPFSInit();

	// Initialize Stack and application related NV variables into AppConfig.
	InitAppConfig();

	dwLastIP = AppConfig.MyIPAddr.Val;

	// Initialize core stack layers (MAC, ARP, TCP, UDP) and
	// application modules (HTTP, SNMP, etc.)
    StackInit();

	BUTTON0_TRIS = 1;			/* SW1 input */
	BUTTON1_TRIS = 1;			/* SW2 input */
	BUTTON2_TRIS = 1;			/* SW3 input */

	LED1_TRIS = 0;				/* LED1 outout for TCP/IP */
	LED1_IO = 0;

	if (!memcmp(AppConfig.DefaultAuthCall, "        ", 8))
	{
		memcpy (LCDText, "Please Set      " "  Auth. Callsign",32);
		xMessage.pcMessage = (char *)LCDText;
		xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
		xQueueSend( xLCDQueue, &xMessage, 0 );
	}

	while(1)
    {
		StackTask();
		if (FSInitSW)
		{
			DHCPServerTask();
			NBNSTask();
			if (!memcmp(AppConfig.DefaultAuthCall, "        ", 8))
			{
				HTTPServer();
			}
			else
			{
				if (FWUpdate)
				{
					if (DprsSocket != INVALID_SOCKET)
					{
						TCPDiscard(DprsSocket);
						TCPClose (DprsSocket);
						DprsSocket = INVALID_SOCKET;
					}
					if (ServerSocket != INVALID_UDP_SOCKET)
					{
						while (UDPIsPutReady(ServerSocket) < 5) StackTaskUDP();
						UDPPutArray((BYTE*)&ReqDisConnect, 5);
						UDPFlush();
						UDPDiscard();
						UDPClose(RcvDPlusSocket);
						RcvDPlusSocket = INVALID_UDP_SOCKET;
						UDPClose(ServerSocket);
						ServerSocket = INVALID_UDP_SOCKET;
						NodeDPlusState = DS_WAIT_LINK;
					}
					HTTPServer();
				}
				else 
				{
					DstarDPlusClientUDP();
					FTPServer();
					SNTPClient();			/* time server access */
					if (SMTPIsBusy()) SMTPTask();
					if ((NodeDPlusState >= DS_WAIT_LINK) && !Msg.connectSW && !Msg.notfoundSW) HTTPServer();

					/* check SW1 */
					if (!BUTTON0_IO)
					{
						memset((void*)&AppConfig, 0x00, sizeof(AppConfig));
						MACReadWriteSW = 2; /* set read */
						while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);
						Reset();
					}
	
					/* check SW2 */	
					if (!BUTTON1_IO) 
					{
						Reset();
					}
		
					/* check SW3 */
					if (!BUTTON2_IO)
					{
						memcpy (LCDText, "Cur. IP Address ",16);
						DisplayIPValue(AppConfig.MyIPAddr, LCDText);
					}
		
   			    	// If the local IP address has changed (ex: due to DHCP lease change)
    			    // write the new IP address to the LCD display, UART, and Announce 
        			// service
					if(dwLastIP != AppConfig.MyIPAddr.Val)
					{
						dwLastIP = AppConfig.MyIPAddr.Val;
						memcpy (LCDText, "New IP Address  ",16);
						DisplayIPValue(AppConfig.MyIPAddr, LCDText);
						NodeDPlusState = DS_HOME;	// DS_HOME
					}	
				}	
			}
		}
	}
}


// Writes an IP address to the LCD display and the UART as available
void DisplayIPValue(IP_ADDR IPVal, BYTE XLCDText[])
{
    static	BYTE IPDigit[4];
	static	BYTE i;
	static	BYTE j;
	static	BYTE LCDPos;

	LCDPos = 16;
	for(i = 0; i < sizeof(IP_ADDR); i++)
	{
	    uitoa((WORD)IPVal.v[i], IPDigit);


			for(j = 0; j < strlen((char*)IPDigit); j++)
			{
				XLCDText[LCDPos++] = IPDigit[j];
			}
			if(i == sizeof(IP_ADDR)-1)
				break;
			XLCDText[LCDPos++] = '.';

	}
		XLCDText[LCDPos] = 0x00;

		if(LCDPos < 32u)
			LCDText[LCDPos] = 0;
			xMessage.pcMessage = (char *)XLCDText;
 			xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
			xQueueSend( xLCDQueue, &xMessage, 0 );

}


/*********************************************************************
 * Function:        void InitAppConfig(void)
 *
 * PreCondition:    MPFSInit() is already called.
 *
 * Input:           None
 *
 * Output:          Write/Read non-volatile config variables.
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            None
 ********************************************************************/

static void InitAppConfig(void)
{
	extern	MAC_ADDR	MyMACAddr;              // Application MAC address

		MACReadWriteSW = 1; /* set read */
		while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);
		wOriginalAppConfigChecksum = CalcIPChecksum((BYTE*)&AppConfig, sizeof(AppConfig)-2);
		if (wOriginalAppConfigChecksum != AppConfig.CheckSum)
		{
			memset((void*)&AppConfig, 0x00, sizeof(AppConfig));
			AppConfig.Flags.bIsDHCPEnabled = TRUE;
			AppConfig.Flags.bInConfigMode = TRUE;
			AppConfig.Flags.AutoReConnect = FALSE;
			memcpypgm2ram((void*)&MyMACAddr, (ROM void*)SerializedMACAddress, sizeof(MyMACAddr));
			AppConfig.MyIPAddr.Val = MY_DEFAULT_IP_ADDR_BYTE1 | MY_DEFAULT_IP_ADDR_BYTE2<<8ul | MY_DEFAULT_IP_ADDR_BYTE3<<16ul | MY_DEFAULT_IP_ADDR_BYTE4<<24ul;
			AppConfig.DefaultIPAddr.Val = AppConfig.MyIPAddr.Val;
			AppConfig.MyMask.Val = MY_DEFAULT_MASK_BYTE1 | MY_DEFAULT_MASK_BYTE2<<8ul | MY_DEFAULT_MASK_BYTE3<<16ul | MY_DEFAULT_MASK_BYTE4<<24ul;
			AppConfig.DefaultMask.Val = AppConfig.MyMask.Val;
			AppConfig.MyGateway.Val = MY_DEFAULT_GATE_BYTE1 | MY_DEFAULT_GATE_BYTE2<<8ul | MY_DEFAULT_GATE_BYTE3<<16ul | MY_DEFAULT_GATE_BYTE4<<24ul;
			AppConfig.PrimaryDNSServer.Val = MY_DEFAULT_PRIMARY_DNS_BYTE1 | MY_DEFAULT_PRIMARY_DNS_BYTE2<<8ul  | MY_DEFAULT_PRIMARY_DNS_BYTE3<<16ul  | MY_DEFAULT_PRIMARY_DNS_BYTE4<<24ul;
			AppConfig.SecondaryDNSServer.Val = MY_DEFAULT_SECONDARY_DNS_BYTE1 | MY_DEFAULT_SECONDARY_DNS_BYTE2<<8ul  | MY_DEFAULT_SECONDARY_DNS_BYTE3<<16ul  | MY_DEFAULT_SECONDARY_DNS_BYTE4<<24ul;
//			AppConfig.DefaultDPlusPort = DPlusPort;
			strncpy ((char *)AppConfig.ServerName, "58.1.243.168", 32);
			strncpy ((char *)AppConfig.DefaultRefName, "REF047 C", 8);
			strncpy ((char *)AppConfig.DefaultAuthCall, "        ", 8);
			memset ((char *)AppConfig.DefaultAuthPin, 0x00, 8);
			strncpy ((char *)AppConfig.DefaultNodeName, "        ", 8);
			strncpy ((char *)AppConfig.NTPserver, "pool.ntp.org", 32);
			strncpy ((char *)AppConfig.UserID, "admin",16);
			strncpy ((char *)AppConfig.PASSWORD, "microchip", 16);
			AppConfig.TmZone = 0x09;
			AppConfig.TmZoneMinutes = 0x00;
			AppConfig.JitterBufferWait = 0x0f;
			AppConfig.HeaderGenTime = DefaultHeaderGenTime;
			AppConfig.CheckSum = CalcIPChecksum((BYTE*)&AppConfig, sizeof(AppConfig)-2);
			MACReadWriteSW = 2; /* set read */
			while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);
		}

		ReadMacFromEEPROM(SerializedMACAddress);
		memcpy (&MyMACAddr, &SerializedMACAddress, 6);
}
