/*********************************************************************
 *
 *  Application to Demo HTTP2 Server
 *  Support for HTTP2 module in Microchip TCP/IP Stack
 *	 -Implements the application 
 *	 -Reference: RFC 1002
 *
 *********************************************************************
 * FileName:        CustomHTTPApp.c
 * Dependencies:    TCP/IP stack
 * Processor:       PIC18, PIC24F, PIC24H, dsPIC30F, dsPIC33F, PIC32
 * Compiler:        Microchip C32 v1.05 or higher
 *					Microchip C30 v3.12 or higher
 *					Microchip C18 v3.30 or higher
 *					HI-TECH PICC-18 PRO 9.63PL2 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright (C) 2002-2010 Microchip Technology Inc.  All rights
 * reserved.
 *
 * Microchip licenses to you the right to use, modify, copy, and
 * distribute:
 * (i)  the Software when embedded on a Microchip microcontroller or
 *      digital signal controller product ("Device") which is
 *      integrated into Licensee's product; or
 * (ii) ONLY the Software driver source files ENC28J60.c, ENC28J60.h,
 *		ENCX24J600.c and ENCX24J600.h ported to a non-Microchip device
 *		used in conjunction with a Microchip ethernet controller for
 *		the sole purpose of interfacing with the ethernet controller.
 *
 * You should refer to the license agreement accompanying this
 * Software for additional information regarding your rights and
 * obligations.
 *
 * THE SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * MICROCHIP BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
 * BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE
 * THEREOF), ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER
 * SIMILAR COSTS, WHETHER ASSERTED ON THE BASIS OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE), BREACH OF WARRANTY, OR OTHERWISE.
 *
 *
 * Author               Date    Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Elliott Wood     	6/18/07	Original
 ********************************************************************/
#define __CUSTOMHTTPAPP_C

#include "TCPIPConfig.h"

#include "FreeRTOS.h"
#include "queue.h"
#include "lcd.h"
#include "TCPIP Stack/StackTsk.h"
#include "MDD File System/FSIO.h"
#include "MDD File System/FSDefs.h"

#if defined(STACK_USE_HTTP2_SERVER)

#include "TCPIP Stack/TCPIP.h"
#include "prvTaskTCPIP.h"		// Needed for SaveAppConfig() prototype
#include <ctype.h>
#include "node.h"
#include "task.h"
#include "NVMem.h"


#define APP_FLASH_BASE_ADDRESS 0x9d00600
#define	USER_APP_RESET_ADDRESS	(0x9d006000 + 0x1000 + 0x970)

AD_PARAM AdParam;
extern	WORD	SSN;
extern	BYTE	PicVer[2];
extern	BOOL	PTT_On;
extern	BOOL	COS_On;
extern	BYTE	RefNameSet;

extern	struct	DPRSinfo DPRS;
extern	BYTE	DPRS_SW;

WORD CalcIPChecksum(BYTE *AppConfig, WORD length);
void	Ref2IP(BYTE refrecter[]);
UINT __attribute__((nomips16)) NVMemOperation(UINT nvmop);
UINT NVMemWriteWord(void* address, UINT data);
void delay_us(UINT us);
void	flushData(void);


BYTE	REF_IP_address[16];
BYTE	CurUser[8];
WORD	AccCtlCnt = 0;
static	BYTE msg[21];

extern	BYTE	RefNameSet;
extern	BYTE	RefName[8];
extern	BYTE	RefNameSave[8];
extern	BYTE	MyCallSign[8];

extern	BYTE	TimeDate[20];
extern	BOOL	FSInitSW;
extern	BYTE	Text[32];

extern	struct	MessageSW 	Msg;

static	BYTE	notice[60];

extern	xQueueHandle xDstarLogQueue;
extern	xQueueHandle xDstarHttpQueue;
extern	xQueueHandle xDstarAccQueue;
extern	xQueueHandle xDstarDprsAccQueue;
extern	xTaskHandle FSHandle;
extern	xTaskHandle USBHandle;
extern	xTaskHandle LCDHandle;
extern	xTaskHandle FSHandle;

extern	TCP_SOCKET	DprsSocket;

extern	BYTE	MACReadWriteSW;
extern	BOOL	FWUpdate;
extern	BOOL	DprsOpen;
extern	BOOL	AccessFileReadSW;
extern	BOOL	CallCheck_Skip;
extern	BOOL	DprsAccessFileReadSW;
extern	BOOL	ReqLoadAprsCallCheck;
extern	BYTE	EmailTemp[130];

APP_CONFIG newAppConfig;

extern	enum
	{
		DS_HOME = 0,
		DS_HOME_FIRST_TIME,
		DS_HOME_SECOND_TIME,
		DS_SOCKET_OBTAINED,
		DS_PROCESS_CONNECTED,
		DS_PROCESS_READY,
		DS_PROCESS_READ,
		DS_DPRS_OPEN,
		DS_DPRS_OPEN_AUTOLINK,
		DS_DPRS_OPEN_AUTORELINK,
		DS_DPRS_OPEN_DONE_AUTOLINK,
		DS_DPRS_OPEN_DONE_AUTORELINK,
		DS_DPRS_ACCEPT,
		DS_DPRS_ACCEPT_DONE,
		DS_WAIT_LINK_COMMAND,
		DS_WAIT_LINK,
		DS_UDP_OPEN_FIRST,
		DS_UDP_OPEN_SECOND,
		DS_UDP_OPEN,
		DS_UDP_IS_OPENED,
		DS_UDP_USB_DONE,
		DS_UDP_SEND,
		DS_UDP_RECV,
		DS_UDP_SEND_CALL,
		DS_UDP_RECV_OK,
		DS_UDP_SUB_SOCKET_OPEN,
		DS_LOOP
	} NodeDPlusState;

/****************************************************************************
  Section:
	Function Prototypes and Memory Globalizers
  ***************************************************************************/
#if defined(HTTP_USE_POST)
	#if defined(STACK_USE_HTTP_MD5_DEMO)
		#if !defined(STACK_USE_MD5)
			#error The HTTP_MD5_DEMO requires STACK_USE_MD5
		#endif
		static HTTP_IO_RESULT HTTPPostMD5(void);
	#endif
	#if defined(STACK_USE_HTTP_APP_RECONFIG)
		static HTTP_IO_RESULT HTTPConfig(void);
		static HTTP_IO_RESULT HTTPAd1Config(void);
		static HTTP_IO_RESULT HTTPAd2Config(void);
		static HTTP_IO_RESULT HTTPAd3Config(void);
		static HTTP_IO_RESULT HTTPUserPassword(void);
		static HTTP_IO_RESULT HTTPAuthCallPin(void);
		static HTTP_IO_RESULT HTTPLinkUnlink(void);
		static HTTP_IO_RESULT HTTPRefSelectLink(void);
		static HTTP_IO_RESULT HTTPAprsMsgSend(void);
		static HTTP_IO_RESULT HTTPUpdate(void);
		static HTTP_IO_RESULT HTTPAccessFile(void);
		static HTTP_IO_RESULT HTTPDprsFile(void);
		static HTTP_IO_RESULT HTTPDprsCtrlFile(void);
		static HTTP_IO_RESULT HTTPDebugSet(void);
	#endif
	#if defined(STACK_USE_HTTP_EMAIL_DEMO) || defined(STACK_USE_SMTP_CLIENT)
		#if !defined(STACK_USE_SMTP_CLIENT)
			#error The HTTP_EMAIL_DEMO requires STACK_USE_SMTP_CLIENT
		#endif
//		static HTTP_IO_RESULT HTTPPostEmail(void);
	#endif
#endif


// Sticky status message variable.
// This is used to indicated whether or not the previous POST operation was 
// successful.  The application uses these to store status messages when a 
// POST operation redirects.  This lets the application provide status messages
// after a redirect, when connection instance data has already been lost.
static BOOL lastSuccess = FALSE;

// Stick status message variable.  See lastSuccess for details.
static BOOL lastFailure = FALSE;

static	BYTE RefBuf[10];

static	BYTE	RefNameTemp[8];
static	char	text[121];

extern	BOOL	VoiceMsg;

BOOL	ReqRefList;

static	char	temp[11];

BYTE	AprsMessage[70];
BYTE	AprsDest[9];
BYTE	AprsSenderCall[10];
DWORD	AprsSeq;
DWORD	RcvAprsSeq = 100000;
BOOL	ReqAprsSeq = FALSE;

extern	BOOL	DebugSW;

static	BYTE	*pnt;


/****************************************************************************
  Section:
	Authorization Handlers
  ***************************************************************************/
  
/*****************************************************************************
  Function:
	BYTE HTTPNeedsAuth(BYTE* cFile)
	
  Internal:
  	See documentation in the TCP/IP Stack API or HTTP2.h for details.
  ***************************************************************************/
#if defined(HTTP_USE_AUTHENTICATION)
BYTE HTTPNeedsAuth(BYTE* cFile)
{
	// If the filename begins with the folder "protect", then require auth
	if(memcmppgm2ram(cFile, (ROM void*)"protect", 7) == 0)
		return 0x00;		// Authentication will be needed later

	// If the filename begins with the folder "snmp", then require auth
	if(memcmppgm2ram(cFile, (ROM void*)"snmp", 4) == 0)
		return 0x00;		// Authentication will be needed later

	#if defined(HTTP_MPFS_UPLOAD_REQUIRES_AUTH)
	if(memcmppgm2ram(cFile, (ROM void*)"mpfsupload", 10) == 0)
		return 0x00;
	#endif

	// You can match additional strings here to password protect other files.
	// You could switch this and exclude files from authentication.
	// You could also always return 0x00 to require auth for all files.
	// You can return different values (0x00 to 0x79) to track "realms" for below.

	return 0x80;			// No authentication required
}
#endif

/*****************************************************************************
  Function:
	BYTE HTTPCheckAuth(BYTE* cUser, BYTE* cPass)
	
  Internal:
  	See documentation in the TCP/IP Stack API or HTTP2.h for details.
  ***************************************************************************/
#if defined(HTTP_USE_AUTHENTICATION)
BYTE HTTPCheckAuth(BYTE* cUser, BYTE* cPass)
{
	if(strcmppgm2ram((char *)cUser,(ROM char *)AppConfig.UserID) == 0
		&& strcmppgm2ram((char *)cPass, (ROM char *)AppConfig.PASSWORD) == 0)
		return 0x80;		// We accept this combination
	
	// You can add additional user/pass combos here.
	// If you return specific "realm" values above, you can base this 
	//   decision on what specific file or folder is being accessed.
	// You could return different values (0x80 to 0xff) to indicate 
	//   various users or groups, and base future processing decisions
	//   in HTTPExecuteGet/Post or HTTPPrint callbacks on this value.
	
	return 0x00;			// Provided user/pass is invalid
}
#endif

/****************************************************************************
  Section:
	GET Form Handlers
  ***************************************************************************/
  
/*****************************************************************************
  Function:
	HTTP_IO_RESULT HTTPExecuteGet(void)
	
  Internal:
  	See documentation in the TCP/IP Stack API or HTTP2.h for details.
  ***************************************************************************/
HTTP_IO_RESULT HTTPExecuteGet(void)
{
	BYTE filename[20];
	
	// Load the file name
	// Make sure BYTE filename[] above is large enough for your longest name
	MPFSGetFilename(curHTTP.file, filename, 20);
	
	
	return HTTP_IO_DONE;
}


/****************************************************************************
  Section:
	POST Form Handlers
  ***************************************************************************/
#if defined(HTTP_USE_POST)

/*****************************************************************************
  Function:
	HTTP_IO_RESULT HTTPExecutePost(void)
	
  Internal:
  	See documentation in the TCP/IP Stack API or HTTP2.h for details.
  ***************************************************************************/
HTTP_IO_RESULT HTTPExecutePost(void)
{
	// Resolve which function to use and pass along
	BYTE filename[32];
	
	// Load the file name
	// Make sure BYTE filename[] above is large enough for your longest name
	MPFSGetFilename(curHTTP.file, filename, sizeof(filename));

	if(!memcmppgm2ram(filename, "protect/update.htm", 18))
		return HTTPUpdate();
	if(!memcmppgm2ram(filename, "protect/config.htm", 18))
		return HTTPConfig();
	if(!memcmppgm2ram(filename, "protect/user.htm", 16))
		return HTTPUserPassword();
	if(!memcmppgm2ram(filename, "protect/gatewayauth.htm", 23))
		return HTTPAuthCallPin();
	if(!memcmppgm2ram(filename, "nodeadapter/config.htm", 22))
		return HTTPAd1Config();
	if(!memcmppgm2ram(filename, "nodeadapter/config2.htm", 23))
		return HTTPAd2Config();
	if(!memcmppgm2ram(filename, "nodeadapter/config3.htm", 23))
		return HTTPAd3Config();
	if(!memcmppgm2ram(filename, "status.htm", 10))
		return HTTPLinkUnlink();
	if(!memcmppgm2ram(filename, "refselect.htm", 13))
		return HTTPRefSelectLink();
	if(!memcmppgm2ram(filename, "aprsmsg.htm", 11))
		return HTTPAprsMsgSend();
	if(!memcmppgm2ram(filename, "protect/edit.htm", 16))
		return HTTPAccessFile();
	if(!memcmppgm2ram(filename, "protect/dprs.htm", 16))
		return HTTPDprsFile();
	if(!memcmppgm2ram(filename, "protect/dprsedit.htm", 20))
		return HTTPDprsCtrlFile();
	if(!memcmppgm2ram(filename, "protect/debug.htm", 17))
		return HTTPDebugSet();
	return HTTP_IO_DONE;
}


/*****************************************************************************
  Function:
	static HTTP_IO_RESULT HTTPPostConfig(void)

  Summary:
	Processes the configuration form on config/index.htm

  Description:
	Accepts configuration parameters from the form, saves them to a
	temporary location in RAM, then eventually saves the data to EEPROM or
	external Flash.
	
	When complete, this function redirects to config/reboot.htm, which will
	display information on reconnecting to the board.

	This function creates a shadow copy of the AppConfig structure in 
	RAM and then overwrites incoming data there as it arrives.  For each 
	name/value pair, the name is first read to curHTTP.data[0:5].  Next, the 
	value is read to newAppConfig.  Once all data has been read, the new
	AppConfig is saved back to EEPROM and the browser is redirected to 
	reboot.htm.  That file includes an AJAX call to reboot.cgi, which 
	performs the actual reboot of the machine.
	
	If an IP address cannot be parsed, too much data is POSTed, or any other 
	parsing error occurs, the browser reloads config.htm and displays an error 
	message at the top.

  Precondition:
	None

  Parameters:
	None

  Return Values:
  	HTTP_IO_DONE - all parameters have been processed
  	HTTP_IO_NEED_DATA - data needed by this function has not yet arrived
  ***************************************************************************/
#if defined(STACK_USE_HTTP_APP_RECONFIG)
static HTTP_IO_RESULT HTTPConfig(void)
{
	BYTE 	*ptr;
	WORD	i, k;

	// Check to see if the browser is attempting to submit more data than we 
	// can parse at once.  This function needs to receive all updated 
	// parameters and validate them all before committing them to memory so that
	// orphaned configuration parameters do not get written (for example, if a 
	// static IP address is given, but the subnet mask fails parsing, we 
	// should not use the static IP address).  Everything needs to be processed 
	// in a single transaction.  If this is impossible, fail and notify the user.
	// As a web devloper, if you add parameters to AppConfig and run into this 
	// problem, you could fix this by to splitting your update web page into two 
	// seperate web pages (causing two transactional writes).  Alternatively, 
	// you could fix it by storing a static shadow copy of AppConfig someplace 
	// in memory and using it instead of newAppConfig.  Lastly, you could 
	// increase the TCP RX FIFO size for the HTTP server.  This will allow more 
	// data to be POSTed by the web browser before hitting this limit.
	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto ConfigFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	
	// Use current config in non-volatile memory as defaults
	
	MACReadWriteSW = 3; /* set read */
	while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);

	// Start out assuming that DHCP is disabled.  This is necessary since the 
	// browser doesn't submit this field if it is unchecked (meaning zero).  
	// However, if it is checked, this will be overridden since it will be 
	// submitted.
	newAppConfig.Flags.bIsDHCPEnabled = 0;
	newAppConfig.Flags.AutoConnect = 0;
	newAppConfig.Flags.AutoReConnect = 0;
	newAppConfig.Flags.RepeaterSW = 0;
	newAppConfig.Flags.DisableRFcmd = 0;
	newAppConfig.Flags.VoiceMsgSW = 0;
	newAppConfig.Flags.ircDDB = 0;


	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto ConfigFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto ConfigFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ip"))
		{// Read new static IP Address
			if(!StringToIPAddress(curHTTP.data+6, &newAppConfig.MyIPAddr))
				goto ConfigFailure;
				
			newAppConfig.DefaultIPAddr.Val = newAppConfig.MyIPAddr.Val;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"gw"))
		{// Read new gateway address
			if(!StringToIPAddress(curHTTP.data+6, &newAppConfig.MyGateway))
				goto ConfigFailure;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"sub"))
		{// Read new static subnet
			if(!StringToIPAddress(curHTTP.data+6, &newAppConfig.MyMask))
				goto ConfigFailure;

			newAppConfig.DefaultMask.Val = newAppConfig.MyMask.Val;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dns1"))
		{// Read new primary DNS server
			if(!StringToIPAddress(curHTTP.data+6, &newAppConfig.PrimaryDNSServer))
				goto ConfigFailure;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dns2"))
		{// Read new secondary DNS server
			if(!StringToIPAddress(curHTTP.data+6, &newAppConfig.SecondaryDNSServer))
				goto ConfigFailure;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"host"))
		{// Read new server name (host name)
			i = strlen ((void*)curHTTP.data+6);
			memcpy (newAppConfig.ServerName,(void*)curHTTP.data+6 ,i);
			newAppConfig.ServerName[i] = 0x00;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dhcp"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.bIsDHCPEnabled = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ReNm"))
		{// Read new Repeater/Reflector Name
			ptr = curHTTP.data+6;
			i = strlen ((void*)curHTTP.data+6);
			if (i < 7)
			{
				ptr += i; 
				for (k = i ; k < 7 ; k++)
				{
					*ptr = ' ';
					ptr++;
				}
			}
			memcpy (newAppConfig.DefaultRefName, (void*)curHTTP.data+6, 7);				
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"band"))
		{// Read new Repeater/Reflector Name Band Info.
			newAppConfig.DefaultRefName[7] = *(curHTTP.data+6);
			for (i = 0 ; i < 8 ; i++)
			{
				if (newAppConfig.DefaultRefName[i] == 0x00) newAppConfig.DefaultRefName[i] = 0x20;
			}
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"zone"))
		{
			newAppConfig.TmZone = atoi((void *)curHTTP.data+6);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"zneM"))
		{
			i = atoi((void *)curHTTP.data+6);
			if (i >= 60) i = 0;
			newAppConfig.TmZoneMinutes = i;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"Refn"))
		{
//			Ref2IP((void*)curHTTP.data+6);
			memcpy (REF_IP_address, (void*)curHTTP.data+6, 7);
			REF_IP_address[15] = 'S';
			while (REF_IP_address[15] == 'S') vTaskDelay (20 / portTICK_RATE_MS); 
			memcpy (newAppConfig.ServerName,REF_IP_address,16);
			memcpy (newAppConfig.DefaultRefName,(void*)curHTTP.data+6,7);
			for (i = 0 ; i < 8 ; i++)
			{
				if (newAppConfig.DefaultRefName[i] == 0x00) newAppConfig.DefaultRefName[i] = 0x20;
			}
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"jitt"))
		{// Read new DHCP Enabled flag
			newAppConfig.JitterBufferWait = atoi((void *)curHTTP.data+6);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ntp"))
		{
			i = strlen ((void*)curHTTP.data+6);
			memcpy (newAppConfig.NTPserver,(void*)curHTTP.data+6 ,i);
			newAppConfig.NTPserver[i] = 0x00;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"reco"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.AutoReConnect = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"autc"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.AutoConnect = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rept"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.RepeaterSW = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rfcm"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.DisableRFcmd = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"vice"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.VoiceMsgSW = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ircd"))
		{
			if(curHTTP.data[6] == '1')
				newAppConfig.Flags.ircDDB = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"irct"))
		{
			if(curHTTP.data[6] == '0') newAppConfig.ircDDBtype = 0;
			else if(curHTTP.data[6] == '1') newAppConfig.ircDDBtype = 1;
			else if(curHTTP.data[6] == '2') newAppConfig.ircDDBtype = 2;
		}
	}

	// All parsing complete!  Save new settings and force a reboot
	//	SaveAppConfig(&newAppConfig);
	newAppConfig.CheckSum = CalcIPChecksum((BYTE*)&newAppConfig, sizeof(newAppConfig)-2);
	MACReadWriteSW = 4; /* set read */
	while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);

	
	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "/protect/reboot.htm?");
//	memcpy((void*)(curHTTP.data+20), " New IP address", 16);
//	curHTTP.data[20+16] = 0x00;	// Force null termination
//	for(i = 20; i < 20u+16u; i++)
//	{
//		if(curHTTP.data[i] == ' ')
//			curHTTP.data[i] = 0x00;
//	}		
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


ConfigFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/protect/config.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPAd1Config(void)
{
	BYTE	sMode, sMode2, sInvert;
	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto Ad1ConfigFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	
	sMode = 0x00;
	sMode2 = 0x00;
	sInvert = 0x00;
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto Ad1ConfigFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto Ad1ConfigFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"crc"))
		{
				sMode |= 0x02;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"cos"))
		{
				sMode |= 0x04;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"last"))
		{
				sMode |= 0x08;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"h_f"))
		{
				sMode |= 0x80;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"hdrg"))
		{
				sMode2 |= 0x01;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"hdrt"))
		{
				sMode2 |= 0x02;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"txhr"))
		{
				sMode2 |= 0x04;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rxpl"))
		{
				sInvert |= 0x80;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rxpo"))
		{
				sInvert |= 0x02;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"txpo"))
		{
				sInvert |= 0x01;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dely"))
		{
			AdParam.DelayTime = atoi((void *)curHTTP.data+6) /10 ;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"tout"))
		{
			AdParam.TimeOut = atoi((void *)curHTTP.data+6) / 10;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"keep"))
		{
			AdParam.KeepAlive = atoi((void *)curHTTP.data+6) / 10;
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"jitt"))
		{
			AdParam.JitterSize = atoi((void *)curHTTP.data+6);
		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"sn"))
		{
			AdParam.SN_Squelch = atoi((void *)curHTTP.data+6);
		}
	}

	AdParam.Mode = sMode;
	AdParam.Mode2 = sMode2;
	AdParam.Invert = sInvert;
	AdParam.UpdateFlag = 0x02;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


Ad1ConfigFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPAd2Config(void)
{
	static	WORD	i, len;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto Ad2ConfigFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	

	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto Ad2ConfigFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto Ad2ConfigFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"beac"))
		{
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			len = strlen((void*)pnt);
			if (len > 20) len = 20;
			memcpy (AdParam.BeaconMessage,(void*)curHTTP.data+6 ,len);
			if (len < 20)
			{
				for (i = len ; i < 20 ; i++)
				{
					AdParam.BeaconMessage[i] = 0x20;
				}
			}

		} else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"bcal"))
		{
			len = strlen((void*)curHTTP.data+6);
			if (len > 8) len = 8;
			memcpy (AdParam.BeaconCall,(void*)curHTTP.data+6 ,len);
			if (len < 8)
			{
				for (i = len ; i < 8 ; i++)
				{
					AdParam.BeaconCall[i] = 0x20;
				}
			}

		} else if (!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"intv"))
		{
				AdParam.BeaconInterval = atoi((void *)curHTTP.data+6) / 10;
		}
	}

	AdParam.UpdateFlag = 0x02;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config2.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;

Ad2ConfigFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config2.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPAd3Config(void)
{
	static	BYTE	sFlag, aFlag, sRepMode2;

	sFlag = 0x00;
	AdParam.Mode2 &= 0xfb;

	aFlag = AdParam.AltFlags[0];
	aFlag &= 0x07;

	sRepMode2 = AdParam.AltFlags[1];
	sRepMode2 = 0xfe;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto Ad3ConfigFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	

	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto Ad3ConfigFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto Ad3ConfigFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"myc"))
		{
			memcpy (AdParam.AltMyCall,(void*)curHTTP.data+6 ,8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"myc2"))
		{
			memcpy (AdParam.AltMyCall2,(void*)curHTTP.data+6 ,4);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rpt1"))
		{
			memcpy (AdParam.AltRPT1Call,(void*)curHTTP.data+6 ,8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"rpt2"))
		{
			memcpy (AdParam.AltRPT2Call,(void*)curHTTP.data+6 ,8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"yrc"))
		{
			memcpy (AdParam.AltYourCall,(void*)curHTTP.data+6 ,8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg0"))
		{
				 sFlag |= 0x01;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg1"))
		{
				 sFlag |= 0x02;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg2"))
		{
				 sFlag |= 0x04;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg3"))
		{
				 sFlag |= 0x08;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg4"))
		{
				 sFlag |= 0x10;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg5"))
		{
				 sFlag |= 0x20;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg6"))
		{
				 sFlag |= 0x40;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"flg7"))
		{
				 sFlag |= 0x80;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"aflg"))
		{
				 sRepMode2 |= 0x01;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"txhr"))
		{
				  AdParam.Mode2 |= 0x04;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"amyc"))
		{
				  aFlag |= 0x08;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"amy2"))
		{
				  aFlag |= 0x10;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ayrc"))
		{
				  aFlag |= 0x20;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"art1"))
		{
				  aFlag |= 0x40;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"art2"))
		{
				  aFlag |= 0x80;
		}
	}

	AdParam.AltFlags[2] = sFlag;
	AdParam.AltFlags[0] = aFlag;
	AdParam.AltFlags[1] = sRepMode2; 

	AdParam.UpdateFlag = 0x02;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config3.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


Ad3ConfigFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/nodeadapter/config3.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}



static HTTP_IO_RESULT HTTPUserPassword(void)
{
	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto UserFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto UserFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto UserFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"user"))
		{
			memcpy ((char *)&AppConfig.UserID, curHTTP.data + 6, 16);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"pwd"))
		{
			memcpy ((char *)&AppConfig.PASSWORD, curHTTP.data + 6, 16);
		}
	}

	// All parsing complete!  Save new settings and force a reboot
	//	SaveAppConfig(&newAppConfig);
	AppConfig.CheckSum = CalcIPChecksum((BYTE*)&AppConfig, sizeof(AppConfig)-2);
	MACReadWriteSW = 2; /* set read */
	while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);

	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "/protect/user.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


UserFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/protect/user.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPRefSelectLink(void)
{
	BYTE	i;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto RefSelectFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto RefSelectFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto RefSelectFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"reff"))
		{
			strcpypgm2ram((char*)curHTTP.data, "/refselect.htm");
			curHTTP.httpStatus = HTTP_REDIRECT;		
			return HTTP_IO_DONE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"Refn"))
		{
			memcpy (RefNameTemp,(void*)curHTTP.data+6,7);
			for (i = 0 ; i < 8 ; i++)
			{
				if (RefNameTemp[i] == 0x00) RefNameTemp[i] = 0x20;
			}
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"lnki") || !strcmppgm2ram((char*)curHTTP.data, (ROM char*)"lnkw"))
		{
			if (NodeDPlusState >= DS_WAIT_LINK_COMMAND)
			{
				memcpy (RefNameSave, RefNameTemp, 8);
				RefNameSet = 2;
				sprintf (text, "%19.19s From PC   Link command accept. (Gateway/Reforector select)\r\n",TimeDate);
				xQueueSend( xDstarLogQueue, &text, 0 );
				if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"lnkw")) VoiceMsg = TRUE;
				else 	VoiceMsg = FALSE;
				strcpypgm2ram((char*)curHTTP.data, "/status.htm");
				curHTTP.httpStatus = HTTP_REDIRECT;		
				return HTTP_IO_DONE;
			}
			else
			{
				memcpy (notice, "Please wait until Reload the Gate/Ref list", 43);
				notice[42] = 0x00;
			}
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"canc"))
		{
			strcpypgm2ram((char*)curHTTP.data, "/status.htm");
			curHTTP.httpStatus = HTTP_REDIRECT;		
			return HTTP_IO_DONE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"band"))
		{// Read new Repeater/Reflector Name Band Info.
			RefNameTemp[7] = *(curHTTP.data+6);
			for (i = 0 ; i < 8 ; i++)
			{
				if (RefNameTemp[i] == 0x00) RefNameTemp[i] = 0x20;
			}
		}
	}

	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "refselect.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


RefSelectFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/refselect.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}


static HTTP_IO_RESULT HTTPAuthCallPin(void)
{
	static	WORD i;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto AuthFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto AuthFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto AuthFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"auth"))
		{
			memset (&AppConfig.DefaultAuthCall, 0x20, 8);
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			i = strlen ((char *)pnt);
			if (i > 8) i = 8;
			if (i) memcpy ((char *)&AppConfig.DefaultAuthCall, pnt, 8);
			for (i = 0 ; i < 8 ; i++)
			{
				AppConfig.DefaultAuthCall[i] = (BYTE)toupper(AppConfig.DefaultAuthCall[i]);
			}
			for (i = 0 ; i < 8 ; i++)
			{
				if (AppConfig.DefaultAuthCall[7-i] == 0x20) AppConfig.DefaultAuthCall[7-i] = 0x00;
				else if (AppConfig.DefaultAuthCall[7-i] == 0x00) continue;
				else goto next00;
			}
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"pin"))
		{
			memset (&AppConfig.DefaultAuthPin, 0x20, 8);
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			i = strlen ((char *)pnt);
			if (i > 8) i = 8;
			if (i) memcpy ((char *)&AppConfig.DefaultAuthPin, curHTTP.data + 6, 8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"node"))
		{
			memset (&AppConfig.DefaultNodeName, 0x20, 8);
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			i = strlen ((char *)pnt);
			if (i > 8) i = 8;
			if (i)	memcpy ((char *)&AppConfig.DefaultNodeName, curHTTP.data + 6, 8);
			for (i = 0 ; i < 8 ; i++)
			{
				AppConfig.DefaultNodeName[i] = (BYTE)toupper(AppConfig.DefaultNodeName[i]);
			}
			for (i = 0 ; i < 8 ; i++)
			{
				if (AppConfig.DefaultNodeName[7-i] == 0x20) AppConfig.DefaultNodeName[7-i] = 0x00;
				else if (AppConfig.DefaultNodeName[7-i] == 0x00) continue;
				else goto next00;
			}
		}
next00:
		continue;
	}
	// All parsing complete!  Save new settings and force a reboot
	//	SaveAppConfig(&newAppConfig);
	AppConfig.CheckSum = CalcIPChecksum((BYTE*)&AppConfig, sizeof(AppConfig)-2);
	MACReadWriteSW = 2; /* set read */
	while (MACReadWriteSW) vTaskDelay (10 / portTICK_RATE_MS);

	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "/protect/gatewayauth.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


AuthFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/protect/gatewayauth.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPLinkUnlink(void)
{
	notice[0] = 0x00;
	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto LinkUnlinkFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto LinkUnlinkFailure;

		// Read a form field name
		if(PTT_On)
			{
				memcpy (notice, "Please wait until PTT off (Still doing other command.)", 54);
				notice[54] = 0x00;
				goto LinkUnlinkFailure;
			}
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto LinkUnlinkFailure;
			
		// Parse the value that was read
		if(!memcmp((char*)curHTTP.data, (ROM char*)"link", 4))
		{
			if (NodeDPlusState >= DS_WAIT_LINK_COMMAND)
			{
				memcpy (RefNameSave, AppConfig.DefaultRefName, 8);
				RefNameSet = 2;
				sprintf (text, "%19.19s From PC   Link command accept. (%8s)\r\n",TimeDate,RefNameSave);
				xQueueSend( xDstarLogQueue, &text, 0 );
				VoiceMsg = FALSE;
			}
			else
			{
				memcpy (notice, "Please wait until Reload the Gate/Ref list", 43);
				notice[42] = 0x00;
			}
		}
		else if(!memcmp((char*)curHTTP.data, (ROM char*)"ulnk", 4))
		{
			if (NodeDPlusState >= DS_WAIT_LINK_COMMAND)
			{
				RefNameSet = 1;
				memset (RefName, 0x20, 8);
				sprintf (text, "%19.19s From PC   UnLink command accept.\r\n",TimeDate);
				xQueueSend( xDstarLogQueue, &text, 0 );
			}
			else
			{
				memcpy (notice, "Please wait until Reload the Gate/Ref list", 43);
				notice[42] = 0x00;
			}
		}
		else if(!memcmp((char*)curHTTP.data, (ROM char*)"reld", 4))
		{
			if (!AppConfig.Flags.ircDDB)
			{
				RefNameSet = 3;
				sprintf (text, "%19.19s From PC   ReLoad command accept.\r\n",TimeDate);
			}
			else
			{
				sprintf (text, "%19.19s From PC   Not Support ReLoad command in ircDDB mode.\r\n",TimeDate);
				memcpy (notice, "Not Support ReLoad command in ircDDB mode", 41);
				notice[41] = 0x00;
			}
			xQueueSend( xDstarLogQueue, &text, 0 );
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"lapr"))
		{
			RefNameSet = 4;
			sprintf (text, "%19.19s From PC   APRS server Link command accept.\r\n",TimeDate);
			xQueueSend( xDstarLogQueue, &text, 0 );
		}
		else if(!memcmp((char*)curHTTP.data, (ROM char*)"uapr", 4))
		{
			RefNameSet = 5;
			sprintf (text, "%19.19s From PC   APRS server Unlink command accept.\r\n",TimeDate);
			xQueueSend( xDstarLogQueue, &text, 0 );
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"ircd"))
		{
			if(curHTTP.data[6] == '1')
				AppConfig.Flags.ircDDB = 1;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"irct"))
		{
			if(curHTTP.data[6] == '0') AppConfig.ircDDBtype = 0;
			else if(curHTTP.data[6] == '1') AppConfig.ircDDBtype = 1;
			else if(curHTTP.data[6] == '2') AppConfig.ircDDBtype = 2;
		}
	}

	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "status.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


LinkUnlinkFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/status.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}


/******** FIRMWARE file upload ***********/
FSFILE	*uploadfile = NULL;

static HTTP_IO_RESULT HTTPUpdate(void)
{
	WORD lenA, lenB;
	
	#define SM_READ_SEPARATOR	(0u)
	#define SM_SKIP_TO_DATA		(1u)
	#define SM_READ_DATA		(2u)
	#define SM_POST_COMPLETE	(3u)
	
	// Don't care about curHTTP.data at this point, so use that for buffer
	switch(curHTTP.smPost)
	{
		// Just started, so try to find the separator string
		case SM_READ_SEPARATOR:
			FWUpdate = TRUE;
			// See if a CRLF is in the buffer
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"\r\n", 2, 0, FALSE);
			if(lenA == 0xffff)
			{//if not, ask for more data
				return HTTP_IO_NEED_DATA;
			}
		
			// If so, figure out where the last byte of data is
			// Data ends at CRLFseparator--CRLF, so 6+len bytes
			curHTTP.byteCount -= lenA + 6;
			
			// Read past the CRLF
			curHTTP.byteCount -= TCPGetArray(sktHTTP, NULL, lenA+2);
			
			// Save the next state (skip to CRLFCRLF)
			curHTTP.smPost = SM_SKIP_TO_DATA;
	

		// No break...continue reading the headers if possible
				
		// Skip the headers
		case SM_SKIP_TO_DATA:
			// Look for the CRLFCRLF
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"\r\n\r\n", 4, 0, FALSE);
	
			if(lenA != 0xffff)
			{// Found it, so remove all data up to and including
				lenA = TCPGetArray(sktHTTP, NULL, lenA+4);
				curHTTP.byteCount -= lenA;
				curHTTP.smPost = SM_READ_DATA;
			}
			else
			{// Otherwise, remove as much as possible
				lenA = TCPGetArray(sktHTTP, NULL, TCPIsGetReady(sktHTTP) - 4);
				curHTTP.byteCount -= lenA;
			
				// Return the need more data flag
				return HTTP_IO_NEED_DATA;
			}
			
			memcpy (Text, "Firmware File   " "  Download Start", 32);
			xMessage.pcMessage = (char *)&Text;
			xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
			xQueueSend( xLCDQueue, &xMessage, 20 / portTICK_RATE_MS );

			uploadfile = FSfopen ("IMAGE.HEX", FS_WRITE);
			vTaskDelete(USBHandle);
			vTaskDelete(FSHandle);
			// No break if we found the header terminator
			
		// Read and hash file data
		case SM_READ_DATA:
			// Find out how many bytes are available to be read
			lenA = TCPIsGetReady(sktHTTP);
			if(lenA > curHTTP.byteCount)
				lenA = curHTTP.byteCount;
			LED1_IO ^= 1;
	
			while(lenA > 0u)
			{// Add up to 64 bytes at a time to the sum
				lenB = TCPGetArray(sktHTTP, curHTTP.data, (lenA < 64u)?lenA:64);
				curHTTP.byteCount -= lenB;
				lenA -= lenB;
				FSfwrite (&curHTTP.data, lenB, 1, uploadfile);
			}
//				memcpy (Text, "Firmware File   ", 16);
//				sprintf ((char *)&Text[16],"Up %7ul Bytes",nbytes);         
//				xMessage.pcMessage = (char *)&Text;
//				xMessage.xMinDisplayTime = 1000 / portTICK_RATE_MS;
//				xQueueSend( xLCDQueue, &xMessage, 0);]
					
			// If we've read all the data
			if(curHTTP.byteCount == 0u)
			{// Calculate and copy result to curHTTP.data for printout
				curHTTP.smPost = SM_POST_COMPLETE;
				if (uploadfile != NULL)
				{
					flushData();
					FSfclose (uploadfile);
					uploadfile = NULL;
				}

				memcpy (Text, "Firmware Update " "           Start", 32);
				xMessage.pcMessage = (char *)&Text;
				xMessage.xMinDisplayTime = 60000 / portTICK_RATE_MS;
				xQueueSend( xLCDQueue, &xMessage, 0 );
				vTaskDelay( 100 / portTICK_RATE_MS );
//				portDISABLE_INTERRUPTS();
//				return HTTP_IO_DONE;
			    // Convert Address to Physical Address

				NVMADDR = KVA_TO_PA((unsigned int)USER_APP_RESET_ADDRESS);

				// Unlock and Erase Page
				NVMemOperation(NVMOP_PAGE_ERASE);
				portDISABLE_INTERRUPTS();
	
				Reset();
				while (1);
			}
				
			// Ask for more data
			return HTTP_IO_NEED_DATA;
	}
	
	return HTTP_IO_DONE;
}

/******** access file upload ***********/
static HTTP_IO_RESULT HTTPAccessFile(void)
{
	static	WORD lenA, lenB;
	static	BYTE	calls[8];
	static	WORD	i, k;
	static	BYTE	callSW;
	
	#define SM_READ_SEPARATOR	(0u)
	#define SM_SKIP_TO_DATA		(1u)
	#define SM_READ_DATA		(2u)
	#define SM_POST_COMPLETE	(3u)
	
	// Don't care about curHTTP.data at this point, so use that for buffer
	switch(curHTTP.smPost)
	{
		// Just started, so try to find the separator string
		case SM_READ_SEPARATOR:
			// See if a CRLF is in the buffer
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"\r\n", 2, 0, FALSE);
			if(lenA == 0xffff)
			{//if not, ask for more data
				return HTTP_IO_NEED_DATA;
			}
		
			// If so, figure out where the last byte of data is
			// Data ends at CRLFseparator--CRLF, so 6+len bytes
			curHTTP.byteCount -= lenA + 6;
			
			// Read past the CRLF
			curHTTP.byteCount -= TCPGetArray(sktHTTP, NULL, lenA+2);
			
			// Save the next state (skip to CRLFCRLF)
			curHTTP.smPost = SM_SKIP_TO_DATA;
			
			uploadfile = NULL;
			i = 0;
			memset (calls, 0x20, 8);
			callSW = 0;	
			// No break...continue reading the headers if possible
				
		// Skip the headers
		case SM_SKIP_TO_DATA:
			// Look for the CRLFCRLF
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"acct\"\r\n\r\n", 9, 0, FALSE);
	
			if(lenA != 0xffff)
			{// Found it, so remove all data up to and including
				lenA = TCPGetArray(sktHTTP, NULL, lenA+9);
				curHTTP.byteCount -= lenA;
				curHTTP.smPost = SM_READ_DATA;
			}
			else
			{// Otherwise, remove as much as possible
				lenA = TCPGetArray(sktHTTP, NULL, TCPIsGetReady(sktHTTP) - 9);
				curHTTP.byteCount -= lenA;
			
				// Return the need more data flag
				return HTTP_IO_NEED_DATA;
			}
			
			// No break if we found the header terminator
			
		// Read and hash file data
		case SM_READ_DATA:
			// Find out how many bytes are available to be read
			lenA = TCPIsGetReady(sktHTTP);
			if(lenA > curHTTP.byteCount)
				lenA = curHTTP.byteCount;

			while(lenA > 0u)
			{// Add up to 64 bytes at a time to the sum
				lenB = TCPGetArray(sktHTTP, curHTTP.data, (lenA < 64u)?lenA:64);
				curHTTP.byteCount -= lenB;
				lenA -= lenB;

				for (k = 0 ; k < lenB; k++)
				{
					switch (callSW)
					{
						case 0:
							if (curHTTP.data[k] == ',')
							{
								i = 0;
								callSW = 1;
							} else {
								if (i < 8)
								{
									calls[i] = curHTTP.data[k];
									i++;
								}
							}
							break;

						case 1:
							if (uploadfile == NULL) uploadfile = FSfopen ("CALLSIGN.TXT", FS_WRITE);
							if (curHTTP.data[k] == 'A')
							{
								FSfprintf (uploadfile, "%8.8sALLOW\n",calls);
							}
							else
							{												
								FSfprintf (uploadfile, "%8.8sDENY\n",calls);
							}
							callSW = 2;
							memset (calls, 0x20,8);
							CallCheck_Skip = FALSE;
							break;
			
						case 2:
							if ((curHTTP.data[k] == ',') || (curHTTP.data[k] == ';')) callSW = 0;
							break;
					}
				}
			}
					
			// If we've read all the data
			if(curHTTP.byteCount == 0u)
			{// Calculate and copy result to curHTTP.data for printout
				curHTTP.smPost = SM_POST_COMPLETE;
				if (uploadfile != NULL)
				{
					flushData();
					FSfclose (uploadfile);
					uploadfile = NULL;
				}
				return HTTP_IO_DONE;
			}
				
			// Ask for more data
			return HTTP_IO_NEED_DATA;
	}
	
	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPDprsFile(void)
{
	static	BYTE	i;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto DprsFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	DPRS.DprsSW = FALSE;
	DPRS.RadioSW = FALSE;
	DPRS.InetSW = FALSE;	
	DPRS.ClientID = FALSE;
	DPRS.AutoLink = FALSE;
	DPRS.AutoReLink = FALSE;
	DPRS.smtpAuth = FALSE;
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto DprsFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto DprsFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpip"))
		{
			i = strlen((char *)curHTTP.data + 6);
			if (i > 20) i = 20;
			memset (&DPRS.IPaddress, 0x00, 20);
			memcpy ((char *)&DPRS.IPaddress, curHTTP.data + 6, i);
			for (i = 0; i < 20 ; i++) if (DPRS.IPaddress[i] <= 0x20) DPRS.IPaddress[i] = 0x00;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dppt"))
		{
			DPRS.Port = atoi ((char*)curHTTP.data + 6);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpcl"))
		{
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			i = strlen ((char *)pnt);
			if (i > 7) i = 7;
			if (i) memcpy ((char *)&DPRS.CallSign, pnt, i);
			else memset ((char *)&DPRS.CallSign, 0x00, 7);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dsid"))
		{
			DPRS.CallSign[7] =  *(curHTTP.data + 6);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"alid"))
		{
			memset(DPRS.AltSSID, 0x20, 2);
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			if (*pnt ==	0x00) goto next00;
			DPRS.AltSSID[0] = *pnt++;
			if (*pnt ==	0x00) goto next00;
			DPRS.AltSSID[1] = *pnt++;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpcd"))
		{
			memcpy ((char *)&DPRS.ValidCode, curHTTP.data + 6, 8);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpsw"))
		{
			if(curHTTP.data[6] == '1') DPRS.DprsSW = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dprd"))
		{
			if(curHTTP.data[6] == '1') DPRS.RadioSW = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpit"))
		{
			if(curHTTP.data[6] == '1') DPRS.InetSW = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"pcmt"))
		{
			i = strlen((char *)curHTTP.data + 6);
			if (i > 20) i = 20;
			memcpy ((char *)&DPRS.Comment, curHTTP.data + 6, i);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"plat"))
		{
			DPRS.Lat = atof ((char*)curHTTP.data + 6) * 10000;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"plng"))
		{
			DPRS.Long = atof ((char*)curHTTP.data + 6) * 10000;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"clid"))
		{
			if(curHTTP.data[6] == '1') DPRS.ClientID = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpat"))
		{
			if(curHTTP.data[6] == '1') DPRS.AutoLink = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dpre"))
		{
			if(curHTTP.data[6] == '1') DPRS.AutoReLink = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"smts"))
		{
			i = strlen((char *)curHTTP.data + 6);
			if (i > 40) i = 40;
			memcpy ((char *)&DPRS.SmtpServer, curHTTP.data + 6, i);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"smtp"))
		{
			DPRS.SmtpPort = atoi ((char*)curHTTP.data + 6);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"smah"))
		{
			if(curHTTP.data[6] == '1') DPRS.smtpAuth = TRUE;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"smus"))
		{
			i = strlen((char *)curHTTP.data + 6);
			if (i > 16) i = 16;
			memset (&DPRS.Password, 0x00, 16);
			memcpy ((char *)&DPRS.User, curHTTP.data + 6, i);
			for (i = 0 ; i < 16 ; i++) if (DPRS.User[i] <= 0x20) DPRS.User[i] = 0x00;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"smps"))
		{
			i = strlen((char *)curHTTP.data + 6);
			if (i > 16 ) i = 16;
			memset (&DPRS.Password, 0x00, 16);
			memcpy ((char *)&DPRS.Password, curHTTP.data + 6, i);
			for (i = 0 ; i < 16 ; i++) if (DPRS.Password[i] <= 0x20) DPRS.Password[i] = 0x00;
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"pint"))
		{
			pnt = curHTTP.data + 6;
			while (*pnt == 0x20) pnt++;
			DPRS.Interval = atoi ((char*)pnt);
		}
next00:
		continue;
	}
	// All parsing complete!  Save new settings and force a reboot

	DPRS_SW = 2;
	while (DPRS_SW) vTaskDelay (50 / portTICK_RATE_MS);
	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "/protect/dprs.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;

DprsFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/protect/dprs.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}
 
/******** Dprs Msg Control file upload ***********/
static HTTP_IO_RESULT HTTPDprsCtrlFile(void)
{
	static	WORD lenA, lenB;
	static	BYTE	temp[70];
	static	WORD	i, j, k;
	static	BYTE	callSW;
	
	#define SM_READ_SEPARATOR	(0u)
	#define SM_SKIP_TO_DATA		(1u)
	#define SM_READ_DATA		(2u)
	#define SM_POST_COMPLETE	(3u)
	
	// Don't care about curHTTP.data at this point, so use that for buffer
	switch(curHTTP.smPost)
	{
		// Just started, so try to find the separator string
		case SM_READ_SEPARATOR:
			// See if a CRLF is in the buffer
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"\r\n", 2, 0, FALSE);
			if(lenA == 0xffff)
			{//if not, ask for more data
				return HTTP_IO_NEED_DATA;
			}
		
			// If so, figure out where the last byte of data is
			// Data ends at CRLFseparator--CRLF, so 6+len bytes
			curHTTP.byteCount -= lenA + 6;
			
			// Read past the CRLF
			curHTTP.byteCount -= TCPGetArray(sktHTTP, NULL, lenA+2);
			
			// Save the next state (skip to CRLFCRLF)
			curHTTP.smPost = SM_SKIP_TO_DATA;
			
			uploadfile = NULL;
			i = 0;
			memset (temp, 0x20, 70);
			callSW = 0;	
			// No break...continue reading the headers if possible
				
		// Skip the headers
		case SM_SKIP_TO_DATA:
			// Look for the CRLFCRLF
			
			lenA = TCPFindROMArray(sktHTTP, (ROM BYTE*)"msgc\"\r\n\r\n", 9, 0, FALSE);

			if(lenA != 0xffff)
			{// Found it, so remove all data up to and including
				lenA = TCPGetArray(sktHTTP, NULL, lenA+9);
				curHTTP.byteCount -= lenA;
				curHTTP.smPost = SM_READ_DATA;
			}
			else
			{// Otherwise, remove as much as possible
				lenA = TCPGetArray(sktHTTP, NULL, TCPIsGetReady(sktHTTP) - 9);
				curHTTP.byteCount -= lenA;
			
				// Return the need more data flag
				return HTTP_IO_NEED_DATA;
			}
			
			// No break if we found the header terminator
			
		// Read and hash file data
		case SM_READ_DATA:
			// Find out how many bytes are available to be read
			lenA = TCPIsGetReady(sktHTTP);
			if(lenA > curHTTP.byteCount)
				lenA = curHTTP.byteCount;

			while(lenA > 0u)
			{// Add up to 64 bytes at a time to the sum
				lenB = TCPGetArray(sktHTTP, curHTTP.data, (lenA < 64u)?lenA:64);
				curHTTP.byteCount -= lenB;
				lenA -= lenB;

				for (k = 0 ; k < lenB ; k++)
				{
					switch (callSW)
					{
						case 0:				/* callsign */
							if (curHTTP.data[k] == ',')
							{
								i = 9;
								callSW = 1;
							} else {
								if (i <= 8)
								{
									temp[i] = curHTTP.data[k];
									i++;
								}
							}
							break;

						case 1:				/* on/off	*/
							if (curHTTP.data[k] == ',')
							{
								i = 12;
								callSW = 2;
							}
							else
							{
								if (i <= 11)
								{												
									temp[i] = curHTTP.data[k];
									i++;
								}
							}
							break;
			
						case 2:			/* e-mail */
							if ((curHTTP.data[k] == ',') || (curHTTP.data[k] == ';'))
							{
								for (j = 0 ; j < 70 ; j++)
								{
									if (temp[69-j] <= 0x20)
									{
										 temp[69-j] = 0x00;
									}
									else
									{
										break;
									}
								}
								if (uploadfile == NULL) uploadfile = FSfopen (DprsMsgRecvCallFile, FS_WRITE);
								FSfprintf (uploadfile, "%s\r\n", temp);
								i = 0;
								callSW = 0;		
								memset (temp, 0x20, 70);
							}
							else
							{												
								temp[i] = curHTTP.data[k];
								i++;
							}
							break;
					}
				}
			}
					
			// If we've read all the data
			if(curHTTP.byteCount == 0u)
			{// Calculate and copy result to curHTTP.data for printout
				curHTTP.smPost = SM_POST_COMPLETE;
				if (uploadfile != NULL)
				{
					flushData();
					FSfclose (uploadfile);
					uploadfile = NULL;
					ReqLoadAprsCallCheck = TRUE;
				}
				return HTTP_IO_DONE;
			}
				
			// Ask for more data
			return HTTP_IO_NEED_DATA;
	}
	
	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPAprsMsgSend(void)
{
	extern	BOOL	AprsMessageSend;
	static	WORD	len;

	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto AprsMsgSendFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto AprsMsgSendFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto AprsMsgSendFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dest"))
		{
			memset (&AprsDest, 0x00, 9);
			len = strlen ((char *)curHTTP.data + 6);
			if (len > 9) len = 9;
			memcpy ((char *)&AprsDest, curHTTP.data + 6, len);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"mesg"))
		{
			memset (&AprsMessage, 0x00, 70);
			len = strlen ((char *)curHTTP.data + 6);
			if (len > 67) len = 67;
			memcpy ((char *)&AprsMessage, curHTTP.data + 6, len);
		}
		else if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"scal"))
		{
			memset (&AprsSenderCall, 0x00, 10);
			len = strlen ((char *)curHTTP.data + 6);
			if (len > 9) len = 9;
			memcpy ((char *)&AprsSenderCall, curHTTP.data + 6, len);
		}
	}

	// Set the board to reboot and display reconnecting information
	ReqAprsSeq = TRUE;
	while (ReqAprsSeq) (10 / portTICK_RATE_MS);
	AprsMessageSend = TRUE;
	RcvAprsSeq = 10000;
	strcpypgm2ram((char*)curHTTP.data, "aprsmsg.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


AprsMsgSendFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "aprsmsg.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}

static HTTP_IO_RESULT HTTPDebugSet(void)
{
	if(curHTTP.byteCount > TCPIsGetReady(sktHTTP) + TCPGetRxFIFOFree(sktHTTP))
		goto DebugFailure;
	
	// Ensure that all data is waiting to be parsed.  If not, keep waiting for 
	// all of it to arrive.
	if(TCPIsGetReady(sktHTTP) < curHTTP.byteCount)
		return HTTP_IO_NEED_DATA;
	
	DebugSW = FALSE;

	// Read all browser POST data
	while(curHTTP.byteCount)
	{
		// Read a form field name
		if(HTTPReadPostName(curHTTP.data, 6) != HTTP_READ_OK)
			goto DebugFailure;
			
		// Read a form field value
		if(HTTPReadPostValue(curHTTP.data + 6, sizeof(curHTTP.data)-6-2) != HTTP_READ_OK)
			goto DebugFailure;
			
		// Parse the value that was read
		if(!strcmppgm2ram((char*)curHTTP.data, (ROM char*)"dbgs"))
		{
			if(curHTTP.data[6] == '1') DebugSW = TRUE;
		}
	}

	// All parsing complete!  Save new settings and force a reboot
	//	SaveAppConfig(&newAppConfig);

	// Set the board to reboot and display reconnecting information
	strcpypgm2ram((char*)curHTTP.data, "/protect/debug.htm?");
	curHTTP.httpStatus = HTTP_REDIRECT;	
	
	return HTTP_IO_DONE;


DebugFailure:
	lastFailure = TRUE;
	strcpypgm2ram((char*)curHTTP.data, "/protect/debug.htm");
	curHTTP.httpStatus = HTTP_REDIRECT;		

	return HTTP_IO_DONE;
}


#endif	// #if defined(STACK_USE_HTTP_APP_RECONFIG)

#endif //(use_post)


void HTTPPrint_hellomsg(void)
{
	static	BYTE *ptr;
	
	ptr = HTTPGetROMArg(curHTTP.data, (ROM BYTE*)"name");
	
	// We omit checking for space because this is the only data being written
	if(ptr != NULL)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"Hello, ");
		TCPPutString(sktHTTP, ptr);
	}

	return;
}

void HTTPPrintIP(IP_ADDR ip)
{
	static	BYTE digits[4];
	static	BYTE i;
	
	for(i = 0; i < 4u; i++)
	{
		if(i)
			TCPPut(sktHTTP, '.');
		uitoa(ip.v[i], digits);
		TCPPutString(sktHTTP, digits);
	}
}


void HTTPPrint_config_dhcpchecked(void)
{
	if(AppConfig.Flags.bIsDHCPEnabled)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_config_ip(void)
{
	HTTPPrintIP(AppConfig.MyIPAddr);
	return;
}

void HTTPPrint_config_gw(void)
{
	HTTPPrintIP(AppConfig.MyGateway);
	return;
}

void HTTPPrint_config_subnet(void)
{
	HTTPPrintIP(AppConfig.MyMask);
	return;
}

void HTTPPrint_config_dns1(void)
{
	HTTPPrintIP(AppConfig.PrimaryDNSServer);
	return;
}

void HTTPPrint_config_dns2(void)
{
	HTTPPrintIP(AppConfig.SecondaryDNSServer);
	return;
}

void HTTPPrint_config_mac(void)
{
	static	BYTE i;
	extern	MAC_ADDR	MyMACAddr;              // Application MAC address

	
	if(TCPIsPutReady(sktHTTP) < 18u)
	{//need 17 bytes to write a MAC
		curHTTP.callbackPos = 0x01;
		return;
	}	
	
	// Write each byte
	for(i = 0; i < 6u; i++)
	{
		if(i)
			TCPPut(sktHTTP, ':');
		TCPPut(sktHTTP, btohexa_high(MyMACAddr.v[i]));
		TCPPut(sktHTTP, btohexa_low(MyMACAddr.v[i]));
	}
	
	// Indicate that we're done
	curHTTP.callbackPos = 0x00;
	return;
}

void HTTPPrint_reboot(void)
{
	// This is not so much a print function, but causes the board to reboot
	// when the configuration is changed.  If called via an AJAX call, this
	// will gracefully reset the board and bring it back online immediately
	Reset();
}

void HTTPPrint_rebootaddr(void)
{// This is the expected address of the board upon rebooting

	HTTPPrintIP(AppConfig.MyIPAddr);
}

void HTTPPrint_status_ok(void)
{
	if(lastSuccess)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"block");
	else
		TCPPutROMString(sktHTTP, (ROM BYTE*)"none");
	lastSuccess = FALSE;
}

void HTTPPrint_status_fail(void)
{
	if(lastFailure)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"block");
	else
		TCPPutROMString(sktHTTP, (ROM BYTE*)"none");
	lastFailure = FALSE;
}

#endif

void HTTPPrint_server_name(void)
{
		TCPPutROMString(sktHTTP, AppConfig.ServerName);
}

//void HTTPPrint_dplus_port(void)
//{
//	static	BYTE digits[10];
//	
//		uitoa(AppConfig.DefaultDPlusPort, digits);
//		TCPPutString(sktHTTP, digits);
//}

void HTTPPrint_JitterSize(void)
{
	sprintf (temp, "%u", AppConfig.JitterBufferWait);
	TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_TimeZone(void)
{
	static	char digits[10];
	
		sprintf (digits,"%+2d",AppConfig.TmZone);
		TCPPutString(sktHTTP, (BYTE *)digits);
}

void HTTPPrint_TimeZoneMinutes(void)
{
	static	char digits[10];
	
		sprintf (digits,"%2u",AppConfig.TmZoneMinutes);
		TCPPutString(sktHTTP, (BYTE *)digits);
}

void HTTPPrint_CurTime(void)
{
		memcpy (temp, &TimeDate[11], 8);
		temp[8] = 0x00;
	
		TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_CurDate(void)
{
		memcpy (temp, &TimeDate, 10);
		temp[10] = 0x00;
	
		TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_CurPTT(void)
{
	if (PTT_On)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE *)"ON");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE *)"OFF");
	}
}

void HTTPPrint_CurCOS(void)
{
	if (COS_On)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE *)"ON");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE *)"OFF");
	}
}

void HTTPPrint_CurUser(void)
{
	static	BYTE	temp[9];
	if (COS_On)
	{
		memcpy (temp, CurUser, 8);
		temp[8] = 0x00;
		TCPPutString(sktHTTP, (BYTE *)temp);
	}
	else if (PTT_On)
	{
		memcpy (temp, MyCallSign, 8);
		temp[8] = 0x00;
		TCPPutString(sktHTTP, (BYTE *)temp);
	} else {
		TCPPutString(sktHTTP, (BYTE *)"  ");
	}
}

void HTTPPrint_CurSSN(void)
{
	static	BYTE temp[5];

	if (COS_On)
	{
		uitoa(SSN, temp);
		TCPPutString(sktHTTP, (BYTE *)temp);
	} else {
		TCPPutString(sktHTTP, (BYTE *)"");
	}
}

void HTTPPrint_RefName(void)
{
	static	BYTE temp[9];
	static	BYTE len;
	
		memset (temp, 0x00, 9);
		len = strlen((char *)AppConfig.DefaultRefName);
		if (len > 8) len = 8;
		memcpy (temp, AppConfig.DefaultRefName, len);
		TCPPutString(sktHTTP, (BYTE *)temp);
}


void HTTPPrint_auth_call(void)
{
	static	BYTE temp[9];
	
		memcpy (temp, AppConfig.DefaultAuthCall, 8);
		temp[8] = 0x00;
		TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_pin(void)
{
	static	BYTE temp[9];
	
		memcpy (temp, AppConfig.DefaultAuthPin, 8);
		temp[8] = 0x00;
		TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_NodeName(void)
{
	static	BYTE temp[9];
	
		memcpy (temp, AppConfig.DefaultNodeName, 8);
		temp[8] = 0x00;
		TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_NtpName(void)
{
		TCPPutROMString(sktHTTP, AppConfig.NTPserver);
}

void HTTPPrint_delay_time(void)
{

	sprintf (temp, "%u0", AdParam.DelayTime);
	TCPPutString(sktHTTP, (BYTE *)temp);
}


void HTTPPrint_time_out()
{
	sprintf (temp, "%u0", AdParam.TimeOut);
	TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_keep_alive()
{
	sprintf (temp, "%u0", AdParam.KeepAlive);
	TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_jittter_size()
{
	sprintf (temp, "%u", AdParam.JitterSize);
	TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_RX_AutoPolarity()
{
	if (AdParam.Invert & 0x80)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}
void HTTPPrint_RX_Polarity()
{
	if (AdParam.Invert & 0x02)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_TX_Polarity()
{
	if (AdParam.Invert & 0x01)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}}

void HTTPPrint_COS_Check()
{
	if (AdParam.Mode & 0x04)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_SN_Squelch()
{
	sprintf (temp, "%u", AdParam.SN_Squelch);
	TCPPutString(sktHTTP, (BYTE *)temp);
}

void HTTPPrint_CRC_Check()
{
	if (AdParam.Mode & 0x02)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_LastFrame()
{
	if (AdParam.Mode & 0x08)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_HeaderGen()
{
	if (AdParam.Mode2 & 0x01)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_HeaderGenType()
{
	if (AdParam.Mode2 & 0x02)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_HeaderRep()
{
	if (AdParam.Mode2 & 0x04)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_HalfFull()
{
	if (AdParam.Mode & 0x80)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_user_id(void)
{
		TCPPutROMString(sktHTTP, AppConfig.UserID);
}

void HTTPPrint_user_pass(void)
{
		TCPPutROMString(sktHTTP, AppConfig.PASSWORD);
}

void HTTPPrint_user_passC(void)
{
		TCPPutROMString(sktHTTP, AppConfig.PASSWORD);
}

void HTTPPrint_BeaconM(void)
{
		memcpy (msg , AdParam.BeaconMessage, 20);
		msg[20] = 0x00;
		TCPPutROMString(sktHTTP, msg);

}

void HTTPPrint_BeaconCall(void)
{
		memcpy (msg , AdParam.BeaconCall, 8);
		msg[8] = 0x00;
		TCPPutROMString(sktHTTP, msg);

}

void HTTPPrint_BeaconIntv(void)
{
	BYTE digits[10];

		uitoa(AdParam.BeaconInterval * 10, digits);
		TCPPutROMString(sktHTTP, digits);
}

void HTTPPrint_alt_mycall(void)
{
		memcpy (msg , AdParam.AltMyCall, 8);
		msg[8] = 0x00;
		TCPPutROMString(sktHTTP, msg);
}

void HTTPPrint_alt_mycall2(void)
{
		memcpy (msg , AdParam.AltMyCall2, 4);
		msg[4] = 0x00;
		TCPPutROMString(sktHTTP, msg);
}

void HTTPPrint_alt_yourcall(void)
{
		memcpy (msg , AdParam.AltYourCall, 8);
		msg[8] = 0x00;
		TCPPutROMString(sktHTTP, msg);

}

void HTTPPrint_alt_rpt1call(void)
{
		memcpy (msg , AdParam.AltRPT1Call, 8);
		msg[8] = 0x00;
		TCPPutROMString(sktHTTP, msg);
}

void HTTPPrint_alt_rpt2call(void)
{
		memcpy (msg , AdParam.AltRPT2Call, 8);
		msg[8] = 0x00;
		TCPPutROMString(sktHTTP, msg);
}

void HTTPPrint_flag0()
{
	if (AdParam.AltFlags[2] & 0x01)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag1()
{
	if (AdParam.AltFlags[2] & 0x02)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag2()
{
	if (AdParam.AltFlags[2] & 0x04)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag3()
{
	if (AdParam.AltFlags[2] & 0x08)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag4()
{
	if (AdParam.AltFlags[2] & 0x10)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag5()
{
	if (AdParam.AltFlags[2] & 0x20)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag6()
{
	if (AdParam.AltFlags[2] & 0x40)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_flag7()
{
	if (AdParam.AltFlags[2] & 0x80)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_Aflag()
{
	if (AdParam.AltFlags[1] & 0x01)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_a_mycall()
{
	if (AdParam.AltFlags[0] & 0x08)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_a_mycall2()
{
	if (AdParam.AltFlags[0] & 0x10)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_a_yourcall()
{
	if (AdParam.AltFlags[0] & 0x20)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_a_rpt1call()
{
	if (AdParam.AltFlags[0] & 0x40)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_a_rpt2call()
{
	if (AdParam.AltFlags[0] & 0x80)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_ircDDB()
{
	if (AppConfig.Flags.ircDDB)
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	} else {
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	}
}

void HTTPPrint_ircDDBtype()
{
	if (AppConfig.Flags.ircDDB)
	{
		if (AppConfig.ircDDBtype == 0)		TCPPutROMString(sktHTTP, (ROM BYTE*)"Reflector");
		else if (AppConfig.ircDDBtype == 1) TCPPutROMString(sktHTTP, (ROM BYTE*)"DCS");
		else if (AppConfig.ircDDBtype == 2) TCPPutROMString(sktHTTP, (ROM BYTE*)"Dxtra");
	}
}

void HTTPPrint_RefSelectFirst()
{
	static	char temp[8];
	if (AppConfig.Flags.ircDDB)
	{
		temp[7] = 0x00;
		TCPPutROMString(sktHTTP, (ROM BYTE*)"<input type=\"text\" name=\"Refn\" class=\"ad1\" size=\"7\" value=\"");
		memcpy (temp, AppConfig.DefaultRefName, 7);
		TCPPutROMString(sktHTTP, (ROM BYTE*)temp);
		TCPPutROMString(sktHTTP, (ROM BYTE*)"\">");
		
	}
	else
	{
		TCPPutROMString(sktHTTP, (ROM BYTE*)"&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
		TCPPutROMString(sktHTTP, (ROM BYTE*)"<select name=\"Refn\">");
		ReqRefList = TRUE;
	}
	curHTTP.callbackPos = 0;
	return;
}


void HTTPPrint_RefSelect()
{
	static	WORD	i, cnt;

	if (AppConfig.Flags.ircDDB)
	{
//		TCPFlush (sktHTTP);
//		curHTTP.callbackPos = 0;
		return;
	}

	cnt = 0;
	while (1)
	{
		xQueueReceive( xDstarHttpQueue, &RefBuf, portMAX_DELAY );
		if (RefBuf[0] == 0xff)
		{
			TCPPutROMString(sktHTTP, (ROM BYTE*)"</select>");
			TCPFlush (sktHTTP);
			curHTTP.callbackPos = 0;
			return;
		}
		TCPPutROMString(sktHTTP, (ROM BYTE*)"<option value=\"");
		RefBuf[7] = 0x00;
		TCPPutROMString(sktHTTP, &RefBuf);
		TCPPut(sktHTTP, '"');
		for (i = 0 ; i < 7 ; i++)
		{
			if (RefBuf[i] == 0x00) RefBuf[i] = 0x20;
		}
		if (!memcmp (AppConfig.DefaultRefName,&RefBuf,7))
							TCPPutROMString(sktHTTP, (ROM BYTE*)" selected");
		TCPPut(sktHTTP, '>');
		TCPPutROMString(sktHTTP, &RefBuf);
		TCPPut(sktHTTP, '\n');
		cnt++;
		if (cnt > 15)
		{
			curHTTP.callbackPos = 1;
			return;
		}
	}
}

void HTTPPrint_Band()
{
		msg[0]= AppConfig.DefaultRefName[7];
		msg[1] = 0x00;

		TCPPutROMString(sktHTTP, msg);
}

void HTTPPrint_BandSelect()
{
	static	WORD	i;
	static	BYTE	band[10] = {"ABCDEFGNP*"};
	static	BYTE	band_dcs[26] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZ"};


	if (!AppConfig.Flags.ircDDB)
	{
		for (i = 0 ; i < 10 ; i++)
		{
			TCPPutROMString(sktHTTP, (ROM BYTE*)"<option value=\"");
			TCPPut(sktHTTP, band[i]);
			TCPPut(sktHTTP, '"');
			if (AppConfig.DefaultRefName[7] == band[i])
							TCPPutROMString(sktHTTP, (ROM BYTE*)" selected");
			TCPPut(sktHTTP, '>');
			TCPPut(sktHTTP, band[i]);
			TCPPut(sktHTTP, '\n');
		}
	}
	else if (AppConfig.ircDDBtype == 1)	/* DCS */
	{
		for (i = 0 ; i < 26 ; i++)
		{
			TCPPutROMString(sktHTTP, (ROM BYTE*)"<option value=\"");
			TCPPut(sktHTTP, band_dcs[i]);
			TCPPut(sktHTTP, '"');
			if (AppConfig.DefaultRefName[7] == band_dcs[i])
							TCPPutROMString(sktHTTP, (ROM BYTE*)" selected");
			TCPPut(sktHTTP, '>');
			TCPPut(sktHTTP, band_dcs[i]);
			TCPPut(sktHTTP, '\n');
		}
	}
	else if (AppConfig.ircDDBtype == 2)		/* Dxtra */
	{
		for (i = 0 ; i < 10 ; i++)
		{
			TCPPutROMString(sktHTTP, (ROM BYTE*)"<option value=\"");
			TCPPut(sktHTTP, band[i]);
			TCPPut(sktHTTP, '"');
			if (AppConfig.DefaultRefName[7] == band[i])
							TCPPutROMString(sktHTTP, (ROM BYTE*)" selected");
			TCPPut(sktHTTP, '>');
			TCPPut(sktHTTP, band[i]);
			TCPPut(sktHTTP, '\n');
		}
	}
}

void HTTPPrint_ReConnect(void)
{
	if(AppConfig.Flags.AutoReConnect)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_ConnectUser(void)
{
	extern	struct	connected_table *first_pnt;
	static	struct	connected_table *pnt;
	static	BYTE	temp[9];

	pnt = first_pnt->f_chain;
	if (!pnt)
	{
		TCPPutROMString(sktHTTP,"<tr><td>NONE</td></tr>");
		return;
	}
	while (pnt)
	{
		memcpy (temp, pnt->MyName, 8);
		temp[8] = 0x00;
		TCPPutROMString(sktHTTP,"<tr><td>");
		TCPPutROMString(sktHTTP,temp);
		TCPPutROMString(sktHTTP,"</td></tr>");
		pnt = pnt->f_chain;
	}
}

void HTTPPrint_AutConnect(void)
{
	if(AppConfig.Flags.AutoConnect)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_Repeater(void)
{
	if(AppConfig.Flags.RepeaterSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_DisableRFcmd(void)
{
	if(AppConfig.Flags.DisableRFcmd)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_VoiceMsg(void)
{
	if(AppConfig.Flags.VoiceMsgSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_CurRef(void)
{
	static	BYTE temp[9];
	
		memcpy (temp, RefName, 8);
		if (!memcmp (temp, "        ", 8) || (strlen((char *)temp) == 0))
		{
			TCPPutROMString(sktHTTP, (ROM BYTE*)"NO LINK ");
		} else {					
			temp[8] = 0x00;
			TCPPutString(sktHTTP, (BYTE *)temp);
		}
}

void HTTPPrint_Notice(void)
{
		TCPPutROMString(sktHTTP, notice);
}

void HTTPPrint_DPRS_checked(void)
{
	if(DPRS.DprsSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_dprsRadio(void)
{
	if(DPRS.RadioSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_dprsInet(void)
{
	if(DPRS.InetSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_dprsip(void)
{
		TCPPutROMString(sktHTTP, DPRS.IPaddress);
}

void HTTPPrint_dprsport(void)
{
	static	char	tmp[10];
		
	sprintf (tmp, "%d", DPRS.Port);
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_dprscall(void)
{
	static	BYTE tmp[8];
		
	memcpy (tmp , DPRS.CallSign, 7);
	tmp[7] = 0x00;

	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_dstarid(void)
{
	TCPPut(sktHTTP,DPRS.CallSign[7]);
}

void HTTPPrint_altdstarid(void)
{
	TCPPut(sktHTTP,DPRS.AltSSID[0]);
	TCPPut(sktHTTP,DPRS.AltSSID[1]);
}

void HTTPPrint_dprsvalid(void)
{
	static	BYTE tmp[8];
		
	memcpy (tmp , DPRS.ValidCode, 8);
	tmp[7] = 0x00;
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_plat(void)
{
	static	BYTE tmp[8];
		
	sprintf ((char *)tmp, "%ld.%04lu", DPRS.Lat/10000, labs(DPRS.Lat%10000));
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_plong(void)
{
	static	BYTE tmp[8];
		
	sprintf ((char *)tmp, "%ld.%04lu", DPRS.Long/10000, labs(DPRS.Long%10000));
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_pcomment(void)
{
	static	BYTE tmp[21];
		
	memcpy (tmp , DPRS.Comment, 20);
	tmp[20] = 0x00;
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_pinterval(void)
{
	static	BYTE tmp[5];
		
	sprintf ((char *)tmp, "%-d", DPRS.Interval);
	tmp[4] = 0x00;
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_clientID (void)
{
	if(DPRS.ClientID)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_dprsAutoLink (void)
{
	if(DPRS.AutoLink)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_dprsAutoReLink (void)
{
	if(DPRS.AutoReLink)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	return;
}

void HTTPPrint_DateTime(void)
{
	extern	struct RefDateTime RDateTime;
	static	BYTE	temp[50];

	sprintf ((char *)temp, "%04d/%02d/%02d %02d:%02d:%02d&nbsp;&nbsp;%4d records",
			RDateTime.year, RDateTime.month, RDateTime.day, RDateTime.hours, RDateTime.minutes, RDateTime.seconds, RDateTime.cnt);
	temp[43] = 0x00;
	TCPPutROMString(sktHTTP, temp);
}

void HTTPPrint_http(void)
{
	static	BYTE temp[9], i;
	
		memcpy (temp, RefName, 7);
		temp[7] = 0x00;
		for (i = 0 ; i < 7 ; i++)
		{
			if (temp[i] == 0x20) temp[i] = 0x00;
		}		

		if (strlen((char *)temp) == 0)
		{
			TCPPutROMString(sktHTTP, "NONE");
			return;
		}

		TCPPutROMString(sktHTTP, "<a href=http://www.d-starusers.org/viewrepeater.php?system=");
		TCPPutROMString(sktHTTP, temp);
		TCPPutROMString(sktHTTP, " target=_blank><b>Repeater Detail for ");
		TCPPutROMString(sktHTTP, temp);
		TCPPutROMString(sktHTTP, "</b></a>");
		TCPPutROMString(sktHTTP, "&nbsp;&nbsp;&nbsp;<a href=http://");
		TCPPutROMString(sktHTTP, temp);
		TCPPutROMString(sktHTTP, ".dstargateway.org/ target=_blank><b>DPLUS Dashboard</b></a>");
}


void HTTPPrint_EmailAccessFirst()
{
	DprsAccessFileReadSW = TRUE;
	curHTTP.callbackPos = 0;
}


void HTTPPrint_EmailAccess()
{
	static	WORD	cnt;
	static	BYTE	acc[60];
	static	BYTE	tmp[10];
	static	char	outs[50];

	cnt = 0;
	while (xQueueReceive( xDstarDprsAccQueue, acc, portMAX_DELAY ) == pdTRUE)
	{
		if (acc[0] == 0xff)
		{
			TCPFlush (sktHTTP);
			curHTTP.callbackPos = 0;
			AccCtlCnt = 0;
			return;
		}
		memcpy (tmp, acc, 9);
		tmp[9] = 0x00;
		sprintf (outs, "callsign[%d] = \"%s\";\naccess[%d] = \"",AccCtlCnt,tmp,AccCtlCnt); 
		TCPPutROMString(sktHTTP, outs);
		if (!memcmp(&acc[9], "ON", 2)) TCPPutROMString(sktHTTP, "ON\";\n");
		else TCPPutROMString(sktHTTP, "OFF\";\n");
		sprintf (outs, "email[%d] = \"%s\";\n",AccCtlCnt,&acc[12]); 
		TCPPutROMString(sktHTTP, outs);
		cnt++;
		AccCtlCnt++;
		if (cnt > 20)
		{
			curHTTP.callbackPos = 1;
			return;
		}
	}
}

void HTTPPrint_accessFirst()
{
	AccessFileReadSW = TRUE;
	curHTTP.callbackPos = 0;
}


void HTTPPrint_access()
{
	static	WORD	cnt;
	static	BYTE	acc[10];
	static	BYTE	tmp[9];
	static	char	outs[50];

	cnt = 0;
	while (xQueueReceive( xDstarAccQueue, acc, portMAX_DELAY ) == pdTRUE)
	{
		if (acc[0] == 0xff)
		{
			TCPFlush (sktHTTP);
			curHTTP.callbackPos = 0;
			AccCtlCnt = 0;
			return;
		}
		memcpy (tmp, acc, 8);
		tmp[8] = 0x00;
		sprintf (outs, "callsign[%d] = \"%s\";\naccess[%d] = \"",AccCtlCnt,tmp,AccCtlCnt); 
		TCPPutROMString(sktHTTP, outs);
		if (acc[8] == 'A') TCPPutROMString(sktHTTP, "ALLOW\";\n");
		else TCPPutROMString(sktHTTP, "DENY\";\n");
		cnt++;
		AccCtlCnt++;
		if (cnt > 20)
		{
			curHTTP.callbackPos = 1;
			return;
		}
	}
}

void HTTPPrint_uploaded(void)
{
	// Set a flag to indicate not finished
	curHTTP.callbackPos = 1;
	
	// Make sure there's enough output space
	if(TCPIsPutReady(sktHTTP) < 32u + 37u + 5u)
		return;

	// Check for flag set in HTTPPostMD5
	if(curHTTP.smPost != SM_POST_COMPLETE)
	{// No file uploaded, so just return
		TCPPut(sktHTTP, 0x00);
		curHTTP.callbackPos = 0;
		return;
	}
	TCPPutROMString(sktHTTP, (ROM BYTE*)"<b> Upload completed<br>Please wait about 20 seconds</b>");
	curHTTP.callbackPos = 0;

	return;
}

void HTTPPrint_smtpserver(void)
{
	static BYTE	i;

	for (i = 0 ; i < 40 ; i++)
	{
		TCPPut (sktHTTP, DPRS.SmtpServer[i]);
	}
}

void HTTPPrint_smtpport(void)
{
	static	BYTE	tmp[10];

	sprintf ((char *)tmp, "%d", DPRS.SmtpPort);

	TCPPutROMString(sktHTTP, tmp);
}


void HTTPPrint_CurAPRS(void)
{
	extern	BOOL	DprsOpen;

	if (DprsOpen) TCPPutROMString(sktHTTP, (BYTE *)"LINK&nbsp;&nbsp;");
	else TCPPutROMString(sktHTTP, (BYTE *)"UNLINK");

	return;
}


void HTTPPrint_smtpuser(void)
{
	TCPPutROMString(sktHTTP, DPRS.User);
}

void HTTPPrint_smtppass(void)
{
	TCPPutROMString(sktHTTP, DPRS.Password);
}

void HTTPPrint_smtp_auth(void)
{
	if(DPRS.smtpAuth)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	else
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	return;
}

void HTTPPrint_ack(void)
{
	if(RcvAprsSeq == AprsSeq)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	else
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	return;
}

void HTTPPrint_DebugSwitch(void)
{
	if(DebugSW)
		TCPPutROMString(sktHTTP, (ROM BYTE*)"checked");
	else
		TCPPutROMString(sktHTTP, (ROM BYTE*)"unchecked");
	return;
}


void HTTPPrint_destination(void)
{
	TCPPutROMString(sktHTTP, AprsDest);
}

void HTTPPrint_mesgBody(void)
{
	TCPPutROMString(sktHTTP, AprsMessage);
}

void HTTPPrint_orgSSID(void)
{
	static	BYTE	tmp[9];
	static	BYTE	i, k;

	k = 0;
	for (i = 0 ; i < 8 ; i++)
	{
		if (DPRS.CallSign[i] != 0x20)
		{
			tmp[k] = DPRS.CallSign[i];
			k++;
		}
		else
		{
			if (k > 0)
			{
				if(tmp[k-1] != '-')
				{
					tmp[k] = '-';
					k++;
				}
			}
		}
	}
	tmp[k] = 0x00;

	TCPPutROMString(sktHTTP, "<option value=\"");
	TCPPutROMString(sktHTTP, tmp);
	TCPPutROMString(sktHTTP, "\" selected>");
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_altSSID(void)
{
	static	BYTE	tmp[10];
	static	BYTE	i, k;

	k = 0;
	for (i = 0 ; i < 7 ; i++)
	{
		if (DPRS.CallSign[i] != 0x20)
		{
			tmp[k] = DPRS.CallSign[i];
			k++;
		}
		else
		{
			if (k > 0)
			{
				if(tmp[k-1] != '-')
				{
					tmp[k] = '-';
					k++;
				}
			}
		}
	}
	if (DPRS.AltSSID[0] != 0x20)
	{
		tmp[k++] = DPRS.AltSSID[0];
		if (DPRS.AltSSID[1] != 0x20) tmp[k++] = DPRS.AltSSID[1];
	}
	if (tmp[k-1] == '-') k--;
	tmp[k] = 0x00;

	TCPPutROMString(sktHTTP, "<option value=\"");
	TCPPutROMString(sktHTTP, tmp);
	TCPPutROMString(sktHTTP, "\">");
	TCPPutROMString(sktHTTP, tmp);
}

void HTTPPrint_sendercall(void)
{
	TCPPutROMString(sktHTTP, EmailTemp);
}

void HTTPPrint_LastMsg(void)
{
	TCPPutROMString(sktHTTP, &EmailTemp[20]);
}

#define NVMemIsError()    (NVMCON & (_NVMCON_WRERR_MASK | _NVMCON_LVDERR_MASK))

/********************************************************************
* Function: 	NVMemOperation()
*
* Precondition: 
*
* Input: 		NV operation
*
* Output:		NV eror
*
* Side Effects:	This function must generate MIPS32 code only and 
*				hence the attribute (nomips16)
*
* Overview:     Performs reuested operation.
*
*			
* Note:		 	None.
********************************************************************/
UINT __attribute__((nomips16)) NVMemOperation(UINT nvmop)
{
    int	int_status;
    int	susp;

    // Disable DMA & Disable Interrupts
	#ifdef _DMAC
	int_status = INTDisableInterrupts();
	susp = DmaSuspend();
	#else
	int_status = INTDisableInterrupts(); 
	#endif	// _DMAC

    // Enable Flash Write/Erase Operations
    NVMCON = NVMCON_WREN | nvmop;
    // Data sheet prescribes 6us delay for LVD to become stable.
    // To be on the safer side, we shall set 7us delay.
    delay_us(7);

    NVMKEY 		= 0xAA996655;
    NVMKEY 		= 0x556699AA;
    NVMCONSET 	= NVMCON_WR;

    // Wait for WR bit to clear
    while(NVMCON & NVMCON_WR);
    
    // Disable Flash Write/Erase operations
    NVMCONCLR = NVMCON_WREN;  

	DelayMs (100);
	Reset();

//
//	// Enable DMA & Enable Interrupts
//	#ifdef _DMAC
//	DmaResume(susp);
//	INTRestoreInterrupts(int_status);
//	#else
//	INTRestoreInterrupts(int_status);
//	#endif // _DMAC
//
//	// Return Error Status
//    return(NVMemIsError());
}



const UINT countPerMicroSec = ((8000000/1000000)/2);

/********************************************************************
* Function: 	delay_us()
*
* Precondition: 
*
* Input: 		Micro second
*
* Output:		None.
*
* Side Effects:	Uses Core timer. This may affect other functions using core timers.
				For example, core timer interrupt may not work, or may loose precision.				
*
* Overview:     Provides Delay in microsecond.
*
*			
* Note:		 	None.
********************************************************************/
void delay_us(UINT us)
{
   
    UINT targetCount;  
    UINT bakupCount; 
    // Assert "us" not zero. This must be caught during debug phase.
//    ASSERT(us!=0);
    // backup current count of the core timer.
    bakupCount = ReadCoreTimer();
    // Core timer increments every 2 sys clock cycles.
    // Calculate the counts required to complete "us". 
    targetCount = countPerMicroSec * us;      
    // Restart core timer.
    WriteCoreTimer(0);    
    // Wait till core timer completes the count.    
    while(ReadCoreTimer() < targetCount);
    
    // Restore count back.
    WriteCoreTimer(bakupCount + targetCount);       	
   
}  

