class ApplicationController < ActionController::Base
  # Prevent CSRF attacks by raising an exception.
  # For APIs, you may want to use :null_session instead.
  protect_from_forgery with: :exception
  respond_to :html, :xml, :json
  responders :flash
  before_action :set_locale, :get_headline

  class ProcessError < StandardError
    def initialize(msg, status = 404)
      @status = status
      super(msg)
    end
    attr_accessor :status
  end

  rescue_from ProcessError, :with => :render_proc_error
  rescue_from ActiveRecord::RecordNotFound, :with => :render_ar_error

  private
  def set_locale
    request.params[:controller].index('rails_admin/') == 0 and
      return I18n.locale = :en
    ua_locale = request.env['HTTP_ACCEPT_LANGUAGE'].to_s.scan(/^[a-z]{2}/).first
    ua_locale or return true
    if Rails.application.config.supported_locales.member? ua_locale.to_sym
      I18n.locale = ua_locale.to_sym
    end
    case I18n.locale
    when :ja
      Time.zone = 'Tokyo'
    end
  end

  def get_headline
    @headline = Rails.cache.fetch("headline", expires_in: 600) do
      day_after = Time.now - Rails.configuration.headline_daylimit.days
      News.where(skip_headline: false).where('created_at > ?', day_after).first
    end
  end

  def raise_error_400(msg = "")
    raise ProcessError.new(msg, 400)
  end

  def raise_error_403(msg = "")
    raise ProcessError.new(msg, 403)
  end

  def raise_error_404(msg = "")
    raise ProcessError.new(msg, 404)
  end

  def render_error(opt)
    @message = opt[:message]
    opt[:status] ||= 400
    opt[:template] ||= "misc/error_#{opt[:status]}"
    respond_to do |format|
      format.html { render opt }
      format.json {
        opt.delete :template
        render :json => opt, :status => opt[:status]
      }
      format.any {
        opt[:template] += ".html"
        opt[:content_type] = 'text/html'
        render opt
      }
    end
  end

  def render_proc_error(e)
    logger.error "Process error [#{request.path}]: (#{e.status}) #{e.message}"
    render_error :message => e.message, :status => e.status
  end

  def render_ar_error(e)
    logger.error "RecordNotFound [#{request.path}]: (404) #{e.message}"
    render_error :message => e.message, :status => 404
  end

end
