/*
 * 全体で共有するコード
 */
(function(window){'use strict'

let GLOBAL = window ? window : {}
GLOBAL.APP = GLOBAL.APP ? GLOBAL.APP : {}
let APP = GLOBAL.APP
GLOBAL.SLASH = String.fromCharCode(47) // Riot のコンパイルエラーを踏むことがある

APP.mixin = {}
APP.util = {}

APP.mixin.appDefault = {
  /**
   * 祖先のカスタムタグを検索する
   * @param tagName タグ名。正規表現可。省略時は、最上位の祖先を返す。
   * @return 祖先のタグ。tagName 指定時で、見つからなかった場合は、null を返す。
   */
  findAncestor: function(tagName) {
    function findParentName(me){
      if (me && me.parent && me.parent.root && me.parent.root.tagName)
        return me.parent.root.tagName
    }
    var self = this, parentTagName
    while(parentTagName = findParentName(self)) {
      self = self.parent
      if (tagName && new RegExp('^' + tagName + '$', 'i').test(parentTagName))
        return self
    }
    if (tagName) return null
    return self
  },

  /**
   * document.body 配下に自身要素があるかどうか
   * @return あればtrue
   */
  isUnderBodyTag: function() {
    var el = this.root
    while (el) {
      if (el.tagName == 'BODY') return true
      el = el.parentNode
    }
    return false
  },

  /**
   * Riot on イベント拡張版。
   * DOM ツリーにマウントされている場合のみfn を実行
   * @return void
   */
  exOn: function(ev, fn) {
    var self = this
    this.on(ev, function() {
      if (! this.isUnderBodyTaｇ()) return
      fn.apply(self, arguments)
    })
  }
}

/**
 * コンテキストパスまでのURL を取得
 * @return url
 */
APP.util.getTopUrl = function() {
  var a = document.createElement('a')
  a.href = APP.session.contextPath
  return a.href
}

/**
 * Specialized function for looping an array-like collection with `each={}`
 * @param   { Array } els - collection of items
 * @param   {Function} fn - callback function
 * @returns { Array } the array looped
 * copied from riot.js
 */
APP.util.each = function(els, fn) {
  var len = els ? els.length : 0

  for (var i = 0, el; i < len; i++) {
    el = els[i]
    // return false -> current item was removed by fn during the loop
    if (el != null && fn(el, i) === false) i--
  }
  return els
}

/**
 * ファイル名から拡張子を取得
 */
APP.util.getExtension = function(s) {
  if (typeof s != 'string') return ''
  var arr = s.split('.')
  return arr.length == 1 ? s : arr[arr.length - 1]
}

/**
 * 日付をフォーマットする
 * @param  {Date}   date     Date 型日付、または数値型。その他の場合はそのものを単純に返却する。
 * @param  {String} [format] フォーマット YYYY-MM-DD hh:mm:ss.SSS
 * @return {String}          フォーマット済み日付
 *
 * thanks to http://qiita.com/osakanafish/items/c64fe8a34e7221e811d0
 */
APP.util.formatDate = function (date, format) {
  if (!format) format = 'YYYY/MM/DD hh:mm:ss'
  if (! (date instanceof Date) && isNaN(date)) return date
  date = new Date(date)
  format = format.replace(/YYYY/g, date.getFullYear())
  format = format.replace(/MM/g, ('0' + (date.getMonth() + 1)).slice(-2))
  format = format.replace(/DD/g, ('0' + date.getDate()).slice(-2))
  format = format.replace(/hh/g, ('0' + date.getHours()).slice(-2))
  format = format.replace(/mm/g, ('0' + date.getMinutes()).slice(-2))
  format = format.replace(/ss/g, ('0' + date.getSeconds()).slice(-2))
  if (format.match(/S/g)) {
    var milliSeconds = ('00' + date.getMilliseconds()).slice(-3),
        i = 0, length = format.match(/S/g).length
    for (; i < length; i++) format = format.replace(/S/, milliSeconds.substring(i, i + 1))
  }
  return format
}

/**
 * バイト表示
 * thanks to https://gist.github.com/thomseddon/3511330
 */
APP.util.bytes = function(bytes, precision) {
  if (bytes === 0) return bytes
  if (isNaN(parseFloat(bytes)) || !isFinite(bytes)) return '-'
  if (typeof precision === 'undefined') precision = 1
  var units = ['bytes', 'kB', 'MB', 'GB', 'TB', 'PB'],
      number = Math.floor(Math.log(bytes) / Math.log(1024))
  return (bytes / Math.pow(1024, Math.floor(number))).toFixed(precision) +  ' ' + units[number]
}


/**
 * エンターキーイベントを判定
 */
APP.util.isEnterKey = function(e) {
  return e.which == 13/*enter key*/
}

/**
 * ファイル名に適切でない記号を使っているかチェック
 * @return boolean trueなら不適切
 */
APP.util.invalidFileName = function(name){
  if (! name){
    return false;
  }
  return 0 <= name.indexOf('\\') ||
    0 <= name.indexOf('/') ||
    0 <= name.indexOf(':') ||
    0 <= name.indexOf('*') ||
    0 <= name.indexOf('?') ||
    0 <= name.indexOf('"') ||
    0 <= name.indexOf('<') ||
    0 <= name.indexOf('>') ||
    0 <= name.indexOf('|');
}

/**
 * Object からクエリーパラメーターへ変換
 * http://stackoverflow.com/questions/1714786/querystring-encoding-of-a-javascript-object
 */
APP.util.serialize = function(obj) {
  var str = [];
  for(var p in obj)
    if (obj.hasOwnProperty(p)) {
      str.push(encodeURIComponent(p) + "=" + encodeURIComponent(obj[p]));
    }
  return str.join("&");
}

APP.util.fetchOfPost = function(callUrl, jsonData, responseFunction){
	return fetch(`http://${window.location.host}${APP.session.contextPath}${callUrl}`, {
			method: 'post',
			headers: {
				'Accept': 'application/json;charset=utf-8',
			    'Content-Type': 'application/json;charset=utf-8'
			},
			body: JSON.stringify(jsonData)
			})
			.then(responseFunction)
			.catch (error => {
                console.log('Request failed', error)
            });
}

APP.util.fetchOfPostForm = function(callUrl, formData, responseFunction){
	return fetch(`http://${window.location.host}${APP.session.contextPath}${callUrl}`, {
				method: 'post',
				body: new FormData(formData)
			})
			.then(responseFunction)
			.catch (error => {
                console.log('Request failed', error)
             });
}

APP.util.fetchOfGet = function(callUrl, responseFunction){
	return fetch(`http://${window.location.host}${APP.session.contextPath}${callUrl}`, {credentials: 'include'})
			.then(responseFunction)
			.catch (error => {
                console.log('Request failed', error)
            });
}

APP.util.isInputNumberKeyCode = k =>{
	if(!((k >= 48 && k <= 57) || // 0～9
            (k >= 96 && k <= 105) || // テンキ―0～9
            k == 8 ||  // backspace
            k == 46 || // delete
            k == 39 || // →
            k == 37 || // ←
            k == 36 || // Home
            k == 35)   // End
            ) {
           return false
       }
	return true
}

APP.util.isKeyUpNumberKeyCode = k =>{
	if(!((k >= 48 && k <= 57) || // 0～9
            (k >= 96 && k <= 105) || // テンキ―0～9
            k == 8 ||  // backspace
            k == 46)   // delete
            ) {
           return false
       }
	return true
}

APP.util.isMaxlengthOver = (val, maxlength) =>{
	if(!isEmpty(maxlength)){
        if(val.length > maxlength){
            return true
        }
    }
	return false;
}

//日付文字列の"YYYY/MM/DD形式想定"
APP.util.toDate = val =>{
    return new Date(val)
}

function tryDecodeURIComponent(value) {
  try {
    return decodeURIComponent(value);
  } catch (e) {
    // Ignore any invalid uri component
  }
}

function isDefined(value) {return typeof value !== 'undefined';}
function isUndefined(value) {return typeof value === 'undefined';}
var isArray = Array.isArray;
function isEmpty(value) { return isUndefined(value) || value === '' || value === null || value !== value;}
function getPrice(price){
    if(price != null){
        return price + '円'
    }
    return ""
}

})(window)