package jp.co.nic.ngware.config;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.security.SecurityProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.authentication.configurers.GlobalAuthenticationConfigurerAdapter;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.access.AccessDeniedHandler;
import org.springframework.security.web.access.AccessDeniedHandlerImpl;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.csrf.MissingCsrfTokenException;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import jp.co.nic.ngware.enums.ApplicationMessages;
import jp.co.nic.ngware.repository.UserDetailsServiceImpl;
import jp.co.nic.ngware.util.ApplicationProperties;
import jp.co.nic.ngware.util.ApplicationUtils;

@Configuration
@EnableWebSecurity // Spring Securityの基本設定
@Order(SecurityProperties.ACCESS_OVERRIDE_ORDER)
public class WebSecurityConfig extends WebSecurityConfigurerAdapter {

	@Autowired
	SuccessHandler successHandler;

	@Autowired
	FailureHandler failureHandler;

	@Autowired
	ApplicationProperties prop;

	@Bean
	SuccessHandler successHander() {
		return new SuccessHandler();
	}

	@Bean
	FailureHandler failureHandler() {
		return new FailureHandler();
	}

	@Override
	public void configure(WebSecurity web) throws Exception {
		// セキュリティ設定を無視するリクエスト設定
		// 静的リソース(images、css、javascript)に対するアクセスはセキュリティ設定を無視する
		web.ignoring().antMatchers(
				"/app/**",
				"/css/**",
				"/js/**",
				"/scripts/**",
				"/tag/**",
				"/bower_components/**",
				"/webjars/**");
	}

	@Override
	protected void configure(HttpSecurity http) throws Exception {
		// 認可の設定
		http.authorizeRequests()
				.antMatchers("/", "/login").permitAll() // indexは全ユーザーアクセス許可
				.anyRequest().authenticated(); // それ以外は全て認証無しの場合アクセス不許可

		http.csrf().disable();
		// ログイン設定
		http.formLogin()
				.loginProcessingUrl("/authentication") // 認証処理のパス
				.loginPage("/") // ログインフォームのパス
				.successHandler(successHandler).failureHandler(failureHandler) // 認証時に呼ばれるハンドラクラス
				.usernameParameter("accountId").passwordParameter("password") // ユーザー名、パスワードのパラメータ名
				.and().exceptionHandling().authenticationEntryPoint(authenticationEntryPoint())
				.accessDeniedHandler(accessDeniedHandler());

		// ログアウト設定
		http.logout()
				.logoutRequestMatcher(new AntPathRequestMatcher("/logout**")) // ログアウト処理のパス
				.logoutSuccessUrl("/") // ログアウト完了時のパス
				.deleteCookies("JSESSIONID")
				.invalidateHttpSession(true).permitAll();

	}

	@Configuration
	protected static class AuthenticationConfiguration extends GlobalAuthenticationConfigurerAdapter {
		@Autowired
		UserDetailsServiceImpl userDetailsService;

		@Override
		public void init(AuthenticationManagerBuilder auth) throws Exception {
			// 認証するユーザーを設定する
			auth.userDetailsService(userDetailsService)
					// 入力値をbcryptでハッシュ化した値でパスワード認証を行う
					.passwordEncoder(new BCryptPasswordEncoder(13));
		}
	}

	public class SuccessHandler implements AuthenticationSuccessHandler {
		@Override
		public void onAuthenticationSuccess(HttpServletRequest req, HttpServletResponse res, Authentication auth)
				throws IOException, ServletException {
			int seconds = prop.getSessionTimeOut();
			req.getSession().setMaxInactiveInterval(seconds);
			// ログイン成功時の処理
			req.getSession(true).removeAttribute("loginError");
			req.getSession(true).setAttribute("loggedUser", ApplicationUtils.getAuthentication());
			res.sendRedirect("/ngware/dashboard");
		}
	}

	public class FailureHandler implements AuthenticationFailureHandler {
		@Override
		public void onAuthenticationFailure(HttpServletRequest request, HttpServletResponse response,
				AuthenticationException authenticationException) throws IOException, ServletException {
			String errorId = "";

			// ExceptionからエラーIDをセットする
			if (authenticationException instanceof BadCredentialsException) {
				errorId = ApplicationMessages.USER_NOT_FOUND.getCode();
			}

			// ログイン画面にリダイレクトする
			request.getSession(true).setAttribute("loginError", ApplicationMessages.USER_NOT_FOUND.getMessage());
			response.sendRedirect(request.getContextPath() + "/login#?error=" + errorId);
		}
	}

	@Bean
	AuthenticationEntryPoint authenticationEntryPoint() {
		return new SessionExpiredDetectingLoginUrlAuthenticationEntryPoint("/login");
	}

	@Bean
	AccessDeniedHandler accessDeniedHandler() {
		return new AccessDeniedHandler() {
			@Override
			public void handle(HttpServletRequest req, HttpServletResponse res,
					AccessDeniedException accessDeniedException) throws IOException, ServletException {
				if (accessDeniedException instanceof MissingCsrfTokenException) {
					authenticationEntryPoint().commence(req, res, null);
				} else {
					new AccessDeniedHandlerImpl().handle(req, res, accessDeniedException);
				}
			}
		};
	}

}