﻿Public Class Board

    Public Const SIZE_X As Integer = 3
    Public Const SIZE_Y As Integer = 4

    ''' <summary>
    ''' 盤枠
    ''' </summary>
    ''' <remarks></remarks>
    Private _cells((SIZE_X + 2) * (SIZE_Y + 2) - 1) As Koma

    ''' <summary>
    ''' 持ち駒
    ''' </summary>
    ''' <remarks></remarks>
    Private _Capured(5) As Integer

    ''' <summary>
    ''' 手番
    ''' </summary>
    ''' <remarks></remarks>
    Private _side As Side

    ''' <summary>
    ''' 手数
    ''' </summary>
    ''' <remarks></remarks>
    Private _turnCount As Integer

    ''' <summary>
    ''' 棋譜ログ
    ''' </summary>
    ''' <remarks></remarks>
    Private _logs As New List(Of Move)

    Public Function GetCells(ByVal pos As Position) As Koma
        Return GetCells(pos.X, pos.Y)
    End Function
    Public Function GetCells(ByVal x As Integer, ByVal y As Integer) As Koma
        Return _cells(GetCellsIndex(x, y))
    End Function

    Private Sub SetCells(ByVal pos As Position, ByVal komaValue As Koma)
        SetCells(pos.X, pos.Y, komaValue)
    End Sub
    Private Sub SetCells(ByVal x As Integer, ByVal y As Integer, ByVal komaValue As Koma)
        _cells(GetCellsIndex(x, y)) = KomaValue
    End Sub

    Private Function GetCellsIndex(ByVal x As Integer, ByVal y As Integer)
        Return y * (SIZE_X + 2) + x
    End Function

    Public Sub New()
        For x As Integer = 0 To SIZE_X + 1
            SetCells(x, 0, Koma.WALL)
            SetCells(x, SIZE_Y + 1, Koma.WALL)
        Next
        For y As Integer = 0 To SIZE_Y + 1
            SetCells(0, y, Koma.WALL)
            SetCells(SIZE_X + 1, y, Koma.WALL)
        Next
        For y As Integer = 1 To SIZE_Y
            For x As Integer = 1 To SIZE_X
                SetCells(x, y, Koma.Empty)
            Next
        Next
        SetCells(1, 1, Koma.GoteKI)
        SetCells(2, 1, Koma.GoteLI)
        SetCells(3, 1, Koma.GoteZO)
        SetCells(2, 2, Koma.GoteHI)
        SetCells(2, 3, Koma.SenteHI)
        SetCells(1, 4, Koma.SenteZO)
        SetCells(2, 4, Koma.SenteLI)
        SetCells(3, 4, Koma.SenteKI)
        '持ち駒はなし
        For Each c As Integer In _Capured
            c = 0
        Next
        Me._side = Side.sente
        Me._turnCount = 0
    End Sub

    ''' <summary>
    ''' コピーコンストラクタ
    ''' </summary>
    ''' <param name="source"></param>
    ''' <remarks></remarks>
    Public Sub New(ByVal source As Board)
        Debug.Assert(source._cells.Count = Me._cells.Count)
        For i As Integer = 0 To Me._cells.Count - 1
            Me._cells(i) = source._cells(i)
        Next
        Debug.Assert(source._Capured.Count = Me._Capured.Count)
        For i As Integer = 0 To Me._Capured.Count - 1
            Me._Capured(i) = source._Capured(i)
        Next
        Me._side = source._side
        Me._turnCount = source._turnCount
        For Each log As Move In source._logs
            _logs.Add(log)
        Next
    End Sub

    ''' <summary>
    ''' init.txtの内容を作成。
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ToString() As String
        Dim s As New System.Text.StringBuilder
        For y As Integer = 1 To SIZE_Y
            For x As Integer = 1 To SIZE_X
                Dim koma As Koma = GetCells(x, y)
                Dim side As Side = KomaUtil.KomaSide(koma)
                s.Append(SideUtil.ToCode(side))
                s.Append(KomaUtil.ToCode(koma))
            Next
            s.AppendLine()
        Next
        For i As Integer = 0 To _Capured.Count - 1
            s.Append(_Capured(i))
        Next
        s.AppendLine()
        s.AppendLine(SideUtil.ToCode(_side))
        Return s.ToString
    End Function

    ''' <summary>
    ''' 取った駒の持ち駒のインデックスを返す
    ''' </summary>
    ''' <param name="komaval"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Shared Function CapuredKomaIndex(ByVal komaval As Koma) As Koma
        '先手のひよこ，ぞう，きりん」，「後手のひよこ，ぞう，きりんの順
        Dim orderCaptured As Koma() = {Koma.SenteHI, Koma.SenteZO, Koma.SenteKI, Koma.GoteHI, Koma.GoteZO, Koma.GoteKI}
        For i As Integer = 0 To orderCaptured.Count - 1
            If orderCaptured(i) = komaval Then
                Return i
            End If
        Next
        Return -1
    End Function

    ''' <summary>
    ''' 持ち駒の枚数を返す。
    ''' </summary>
    ''' <param name="komaVal"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function CapturedKomaCount(ByVal komaVal As Koma) As Integer
        Dim idx As Integer = CapuredKomaIndex(komaVal)
        Debug.Assert(idx <> -1)
        Dim ret As Integer = _Capured(idx)
        Debug.Assert(0 <= ret AndAlso ret <= 2)
        Return ret
    End Function


    ''' <summary>
    ''' 着手を行う。
    ''' </summary>
    ''' <param name="move"></param>
    ''' <remarks></remarks>
    Public Sub DoMove(ByVal move As Move)
        Debug.Assert(CanMove(move))
        DoEdit(move)
        AddLog(move)
        _side = DirectCast(-_side, Side)
    End Sub

    ''' <summary>
    ''' 一手進める
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub DoNext()
        If _turnCount >= _logs.Count Then
            Exit Sub
        End If
        Dim move As Move = _logs(_turnCount)
        DoMove(move)
    End Sub

    Private Sub AddLog(ByVal move As Move)
        If _logs.Count = _turnCount Then
            _logs.Add(move)
        ElseIf _logs.Count > _turnCount Then
            If _logs(_turnCount) <> move Then
                _logs(_turnCount) = move
                For i As Integer = _logs.Count - 1 To _turnCount + 1 Step -1
                    _logs.RemoveAt(i)
                Next
            End If
        End If
        _turnCount += 1
        Debug.Assert(_turnCount <= _logs.Count)
    End Sub

    ''' <summary>
    ''' 元に戻せるか
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function CanUndo() As Boolean
        Return 0 < _turnCount
    End Function

    ''' <summary>
    ''' 初手に戻す
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub UndoAll()
        While CanUndo()
            Undo()
        End While
    End Sub

    ''' <summary>
    ''' 一手戻す。
    ''' </summary>
    ''' <remarks></remarks>
    Public Sub Undo()
        If Not CanUndo() Then
            Exit Sub
        End If
        _turnCount -= 1
        Dim move As Move = _logs(_turnCount)
        If move.FromPos.IsCaptured Then
            _Capured(CapuredKomaIndex(move.Koma)) += 1
            SetCells(move.ToPos, Koma.Empty)
        Else
            If move.CapturedPiece <> Koma.Empty Then
                Dim capturedPiece As Koma = KomaUtil.CaputringKoma(move.CapturedPiece)
                _Capured(CapuredKomaIndex(capturedPiece)) -= 1
            End If
            SetCells(move.ToPos, move.CapturedPiece)
            If move.IsPromote Then
                SetCells(move.FromPos, move.Koma - 1) 'にわとりをヒヨコに変換
            Else
                SetCells(move.FromPos, move.Koma)
            End If

        End If
        _side = -_side
    End Sub

    Public Sub DoEdit(ByVal move As Move)
        If move.FromPos.IsCaptured Then
            _Capured(CapuredKomaIndex(move.Koma)) -= 1
            SetCells(move.ToPos, move.Koma)
        Else
            If Not GetCells(move.ToPos) = Koma.Empty Then
                Dim capturedPiece As Koma = KomaUtil.CaputringKoma(GetCells(move.ToPos))
                If capturedPiece <> Koma.GoteLI AndAlso
                   capturedPiece <> Koma.SenteLI Then
                    _Capured(CapuredKomaIndex(capturedPiece)) += 1
                    move.CapturedPiece = GetCells(move.ToPos)
                End If
            End If
            If move.Koma <> GetCells(move.FromPos) Then
                move.IsPromote = True
            End If
            SetCells(move.ToPos, move.Koma)
            SetCells(move.FromPos, Koma.Empty)
        End If
    End Sub

    ''' <summary>
    ''' 手が着手可能かを調べる
    '''  </summary>
    ''' <param name="move"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function CanMove(ByVal move As Move) As Boolean
        If Not _side = KomaUtil.KomaSide(move.Koma) Then
            Return False
        End If
        Dim fromPos As Position = move.FromPos
        If fromPos.IsCaptured Then
            If GetCells(move.ToPos) = Koma.Empty Then
                Return True
            End If
        Else
            If KomaUtil.KomaSide(move.Koma) = KomaUtil.KomaSide(GetCells(move.ToPos)) Then
                '味方の駒がいる
                Return False
            End If
            If GetCells(move.ToPos) = Koma.WALL Then
                '枠からはみ出ている。
                Return False
            End If
            If CanMoveDirection(move) Then
                Return True
            End If
        End If
        Return False
    End Function

    ''' <summary>
    ''' 移動可能かを方向をチェックして返す。
    ''' </summary>
    ''' <param name="move"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function CanMoveDirection(ByVal move As Move)
        Dim directions() As Direction = GetKomaDirection(move.Koma)
        Dim direction As Direction = move.ToPos - move.FromPos
        For Each d As Direction In directions
            If d = direction Then
                Return True
            End If
        Next
        Return False
    End Function

    ''' <summary>
    ''' それぞれの駒の移動可能方向を返す。
    ''' </summary>
    ''' <param name="komaVal"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function GetKomaDirection(ByVal komaVal As Koma) As Direction()
        Dim directionsList()() As Direction = {New Direction() {},
                                               New Direction() {
                                                   New Direction(-1, -1),
                                                   New Direction(-1, 0),
                                                   New Direction(-1, 1),
                                                   New Direction(0, -1),
                                                   New Direction(0, 1),
                                                   New Direction(1, -1),
                                                   New Direction(1, 0),
                                                   New Direction(1, 1)},
                                               New Direction() {
                                                   New Direction(-1, 0),
                                                   New Direction(0, -1),
                                                   New Direction(0, 1),
                                                   New Direction(1, 0)},
                                               New Direction() {
                                                   New Direction(-1, -1),
                                                   New Direction(-1, 1),
                                                   New Direction(1, -1),
                                                   New Direction(1, 1)},
                                               New Direction() {
                                                   New Direction(0, -1)},
                                               New Direction() {
                                                   New Direction(1, -1),
                                                   New Direction(0, -1),
                                                   New Direction(-1, -1),
                                                   New Direction(1, 0),
                                                   New Direction(-1, 0),
                                                   New Direction(0, 1)},
                                               New Direction() {
                                                   New Direction(-1, -1),
                                                   New Direction(-1, 0),
                                                   New Direction(-1, 1),
                                                   New Direction(0, -1),
                                                   New Direction(0, 1),
                                                   New Direction(1, -1),
                                                   New Direction(1, 0),
                                                   New Direction(1, 1)},
                                              New Direction() {
                                                   New Direction(-1, 0),
                                                   New Direction(0, -1),
                                                   New Direction(0, 1),
                                                   New Direction(1, 0)},
                                               New Direction() {
                                                   New Direction(-1, -1),
                                                   New Direction(-1, 1),
                                                   New Direction(1, -1),
                                                   New Direction(1, 1)},
                                               New Direction() {
                                                   New Direction(0, 1)},
                                               New Direction() {
                                                   New Direction(1, 1),
                                                   New Direction(0, 1),
                                                   New Direction(-1, 1),
                                                   New Direction(1, 0),
                                                   New Direction(-1, 0),
                                                   New Direction(0, -1)}
                                              }

        Return directionsList(komaVal)
    End Function
End Class
