/*
 * OpenI2CRADIO
 * RADIO CHIP AKC6955 Handler
 * (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2
 */

#include <stdarg.h>
#include <stdio.h>
#include <delay.h>
#include <string.h>
#include "akc6955.h"
#include "i2c_io.h"



void akc6955_writecmd(unsigned char reg, unsigned char data)
{
    i2c_idle();
    OPENASMASTER();
    i2c_writechar(AKC6955_ADDRESS);
    i2c_writechar(reg);
    i2c_writechar(data);
    CLOSEASMASTER();
}

unsigned char akc6955_readcmd(unsigned char reg)
{
    unsigned char c;
    i2c_idle();
    OPENASMASTER();
    i2c_writechar(AKC6955_ADDRESS);
    i2c_writechar(reg);
    c = i2c_readchar();
    CLOSEASMASTER();
    return c;
}


void akc6955_chg_fm(unsigned char fm)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_POWER);
    if(fm != 0){
        b |= 0x40;
    } else {
        b &= 0xbf;
    }
    akc6955_writecmd(AKC6955_POWER, b);
}

void akc6955_set_amband(unsigned char band)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b &= 0x07; // extract FM
    b = b | ((band & 0x1f) << 3);
    akc6955_writecmd(AKC6955_BAND, b);
}

void akc6955_set_fmband(unsigned char band)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b &= 0xf8; // extract AM
    b = b | (band & 0x07);
    akc6955_writecmd(AKC6955_BAND, b);
}

unsigned char akc6955_get_amband(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b = (b & 0xf8) >> 3;
    return b;
}

unsigned char akc6955_get_fmband(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b = b & 0x07;
    return b;
}

void akc6955_set_power(unsigned char on)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_POWER);
    if(on != 0){
        b |= 0x80;
    } else {
        b &= 0x7f;
    }
    akc6955_writecmd(AKC6955_POWER, b);
}

void akc6955_do_tune(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_POWER);
    b &= 0xdf; // Tun = '0'
    akc6955_writecmd(AKC6955_POWER, b);
    // Need sleep?
    b |= 0x20; // Tun = '1'
    akc6955_writecmd(AKC6955_POWER, b);
}

unsigned char akc6955_tune(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_RCH_HI) & 0x20;
    if(b == 0x00) {
        return 0;
    } else {
        return 0xff;
    }
}

unsigned int akc6955_mode3k(unsigned char flag)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_CH_HI);
    if(flag != 0){
        b |= 0x20;
    } else {
        b &= 0xdf;
    }
    akc6955_writecmd(AKC6955_CH_HI, b);
//    akc6955_do_tune();
    delay1ktcy(500); // Wait 62.5ms
    return akc6955_get_freq();
}

void akc6955_set_tune(unsigned char mode_3k, unsigned int ch)
{
    unsigned char band;
    unsigned char fm;
    unsigned char b;
    unsigned int i;

    i = ch;
    fm = akc6955_readcmd(AKC6955_POWER) & 0x40;
    if(fm == 0){
        // AM
        band = akc6955_get_amband();
    } else {
        band = 0;
    }
    if(band == 2){
        // BAND=AM && MW2
        i = (i / 3) * 3;
    }
    if(i > 0x1fff) i = 0x1fff;
    //i = ch & 0x1fff;

    if(mode_3k == 0){
        b = ((i >> 8) & 0x1f) | 0x40; // 32.768KHz clock
    } else {
        b = ((i >> 8) & 0x1f) | 0x60;
    }
    akc6955_writecmd(AKC6955_CH_HI, b);
    b = i & 0xff;
    akc6955_writecmd(AKC6955_CH_LO, b);
    akc6955_do_tune();
    do{
        // Use software-delay, but recommands hardware-delay ;-(
        delay1ktcy(100); // Wait 100000 cycles = 12.5ms
    } while(akc6955_tune() == 0);
}

void akc6955_do_scan(unsigned char up)
{
    unsigned char b;
    akc6955_do_tune();
    b = akc6955_readcmd(AKC6955_POWER);
    b &= 0xe3;
    if(up != 0) {
        b |= 0x08;
    }
    akc6955_writecmd(AKC6955_POWER, b); // Falldown seek bit to '0'.
    b |= 0x10;
    akc6955_writecmd(AKC6955_POWER, b); // Raise seek bit to '1'.
}

void akc6955_abort_scan(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_POWER);
    b &= 0xef;
    akc6955_writecmd(AKC6955_POWER, b); // Falldown seek bit to '0'.
}

unsigned char akc6955_chk_donescan(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_RCH_HI) & 0x40;
    if(b == 0) return 0;
    return 0xff;
}


/*
 * Get AM/FM Frequency: ret = KHz at AM, 10KHz @FM.
 */
unsigned int akc6955_get_freq(void)
{
    unsigned char b, h, l;
    unsigned char fm;
    unsigned int i;
    b = akc6955_readcmd(AKC6955_CNR_AM) & 0x80;
    l = akc6955_readcmd(AKC6955_RCH_LO) ;
    h = akc6955_readcmd(AKC6955_RCH_HI) & 0x1f;
    fm = akc6955_readcmd(AKC6955_POWER) & 0x40;
    i = l + h * 256;
    if(fm != 0){ // 25KHz
        i = i * 10; // 25 / 10 = 10/4
        i = i / 4;
        i += 3000; // 30MHz
    } else if(b == 0){ // 5K
        i = i * 5;
    } else { // 3K
        i = i * 3;
    }
    return i;
}

void akc6955_set_freq(unsigned int freq)
{
    unsigned int ch;
    unsigned char fm;
    unsigned char mode3k = 0;

    fm = akc6955_readcmd(AKC6955_POWER) & 0x40;
    if(fm != 0) { // FM
      ch = freq - 3000;
      ch = ch * 4;
      ch = ch / 10; // 10/25 = 0.4
    } else {
        mode3k = akc6955_readcmd(AKC6955_CNR_AM) & 0x80;
        if(mode3k == 0){
            ch = freq / 5;
        } else {
            ch = freq / 3;
        }
    }
    akc6955_set_tune(mode3k, ch);
}


unsigned char akc6955_get_cnr(void)
{
    unsigned char fm;
    unsigned char b;
    fm = akc6955_readcmd(AKC6955_POWER) & 0x40;
    if(fm != 0) { // FM
        b = akc6955_readcmd(AKC6955_CNR_FM);
    } else { // AM
        b = akc6955_readcmd(AKC6955_CNR_AM);
    }
    b &= 0x7f;
    return b;
}

int akc6955_read_level(void)
{
    unsigned char fm;
    unsigned char rflevel, iflevel;
    unsigned char b;
    unsigned int rssi;
    unsigned int freq;
    int level;

    fm = akc6955_readcmd(AKC6955_POWER) & 0x40;
    b =  akc6955_readcmd(AKC6955_PGALEVEL);
    rflevel = (b & 0xe0) >> 5;
    iflevel = (b & 0x1c) >> 2;
    rssi = (unsigned int)(akc6955_readcmd(AKC6955_RSSI) & 0x7f);
    if(fm != 0){
        level = 103 - (rssi + 6 * (rflevel + iflevel));
    } else {
        freq = akc6955_get_freq();
        if(freq > 2560) { // ASSUME SW
            level = 103 - (rssi + 6 * (rflevel + iflevel));
        } else { // ASSUME MW,LW
            level = 123 - (rssi + 6 * (rflevel + iflevel));
        }
    }
    return level;
}

unsigned int akc6955_up_freq(unsigned int step)
{
    unsigned int freq;
    freq = akc6955_get_freq();
    freq += step;
    akc6955_set_freq(freq);
    return akc6955_get_freq();
}


unsigned int akc6955_down_freq(unsigned int step)
{
    unsigned int freq;
    freq = akc6955_get_freq();
    if(freq <= step) return freq;
    freq -= step;
    akc6955_set_freq(freq);
    return akc6955_get_freq();
}
