#include "thlcdoutput.h"

#include <QDebug>

#if DAQDRIVER_INSTALLED
#define DAQmxErrChk(functionCall) if( DAQmxFailed(error=(functionCall)) ) goto Error; else
#endif

int ThLCDOutput::dio_init()
{
    int ret = 0;

#if DAQDRIVER_INSTALLED
    int32       error = 0;
    unsigned char data = 0;
    int32 written;
    char errBuff[2048];
    /*********************************************/
    // DAQmxBase Configure Code
    /*********************************************/
    DAQmxErrChk (DAQmxBaseCreateTask("",&taskHandle));
    DAQmxErrChk (DAQmxBaseCreateDOChan(taskHandle,"Dev1/port0","",DAQmx_Val_ChanForAllLines));

    /*********************************************/
    // DAQmxBase Start Code
    /*********************************************/
    DAQmxErrChk (DAQmxBaseStartTask(taskHandle));

    error = DAQmxBaseWriteDigitalU8(taskHandle, 1, 1, 1.0, DAQmx_Val_GroupByChannel, &data, &written, NULL);
#endif

#if DAQDRIVER_INSTALLED
    return error;
Error:
    if (DAQmxFailed (error)) {
        DAQmxBaseGetExtendedErrorInfo (errBuff, 2048);
    }

    if (taskHandle != 0) {
        DAQmxBaseStopTask (taskHandle);
        DAQmxBaseClearTask (taskHandle);
    }
    if (error) {
        //   printf ("DAQmxBase Error %ld: %s\n", error, errBuff);
        qDebug() << "DAQmxBase Error : " << errBuff;
    }
    ret = error;
#endif

    return ret;
}
int ThLCDOutput::dio_write(unsigned char data)
{
    int ret = 0;
#if DAQDRIVER_INSTALLED
    int32 written;

    DAQmxBaseWriteDigitalU8(taskHandle, 1, 1, 1.0, DAQmx_Val_GroupByChannel, &data, &written, NULL);
    ret = written;
#endif

    return ret;
}

void ThLCDOutput::dio_final()
{
#if DAQDRIVER_INSTALLED
    if( taskHandle != 0 )  {
        unsigned char data = 0;
        int32 written;
        DAQmxBaseWriteDigitalU8(taskHandle, 1, 1, 1.0, DAQmx_Val_GroupByChannel, &data, &written, NULL);
        /*********************************************/
        // DAQmxBase Stop Code
        /*********************************************/
        DAQmxBaseStopTask(taskHandle);
        DAQmxBaseClearTask(taskHandle);
        taskHandle = 0;
    }
#endif
}

int ThLCDOutput::lcd_output(unsigned char d, unsigned char rs)
{
    int ret = 0;

    unsigned char en;
    unsigned char data;

    // output upper 4bit
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | ((d >> 4) & 0x0F);
    dio_write(data);
    usleep(1); // ?
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | ((d >> 4) & 0x0F);
    dio_write(data);
    usleep(20); // 20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | ((d >> 4) & 0x0F);
    dio_write(data);
    usleep(20); // 20us

    // output lower 4bit
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | (d & 0x0F);
    dio_write(data);
    usleep(1); // ?
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | (d & 0x0F);
    dio_write(data);
    usleep(20); // 20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | (d & 0x0F);
    dio_write(data);
    usleep(20); // 20us

    return ret;
}

int ThLCDOutput::lcd_init()
{
    int ret = 0;

    // 0x10; rs
    // 0x20; e
    // 0xFF; data
    unsigned char data = 0;
    unsigned char cmd = 0;
    unsigned char rs = 0;
    unsigned char en = 0;

    rs = 0; // 0 = command
    cmd = 3; // command 8bit connect
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(1); // 200ns
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(20); // wait20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    msleep(5); // wait 5ms


    rs = 0; // 0 = command
    cmd = 3; // command 8bit connect
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(1); // 200ns
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(20); // wait20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(200);  // wait 200us

    rs = 0; // 0 = command
    cmd = 3; // command 8bit connect
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(1); // 200ns
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(20); // wait20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(80);  // wait 40us

    rs = 0; // 0 = command
    cmd = 2; // command 4bit connect
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(1); // 200ns
    en = 1; // E -> High
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(20); // wait20us
    en = 0; // E -> Low
    data = (rs << 5) | (en << 4) | cmd;
    dio_write(data);
    usleep(20); // wait20us

    usleep(2);  // wait 40us

    lcd_output(0x28, 0); // set function = 2line display
    msleep(2);

    lcd_output(0x08, 0); // display off
    msleep(2);

    lcd_output(0x0E, 0); // display on, cursor on, cursor-blinking off
    msleep(2);

    lcd_output(0x06, 0); // set entry-mode = address increment when write
    msleep(2);

    lcd_output(0x01, 0); // clear display
    msleep(2); // wait >= 1.64ms


    return ret;
}

void ThLCDOutput::lcd_clear()
{
    lcd_output(0x01, 0); // clear display
    msleep(2); // wait >= 1.64ms
}
void ThLCDOutput::lcd_locate(unsigned char line, unsigned char col)
{
    unsigned char d;

    d = 0x80;
    if (line) d = 0xC0;
    d |= col;
    lcd_output(d, 0);
}
void ThLCDOutput::lcd_putchar(unsigned char d)
{
    lcd_output(d, 1);
}
void ThLCDOutput::lcd_putstr(unsigned char *str)
{
    qDebug() << "str = " << (char*)str;
    while ( *str != '\0') {
        lcd_putchar( *(str++) );
    }
}


//
ThLCDOutput::ThLCDOutput(QObject *parent) :
    QThread(parent)
{
    stopped = false;

    EvtQue = new QQueue<EventData>();
}

ThLCDOutput::~ThLCDOutput()
{

    delete EvtQue;
}

void ThLCDOutput::stop()
{
    stopped = true;
}

void ThLCDOutput::run()
{
    EventData event;
    int event_id;
    char text[128];
    int err;

    err = dio_init();
    emit notify_dio_ready(err);
    if (err) {
        goto Error;
    }

    lcd_init();

    while (!stopped) {
        if (EvtQue->isEmpty()) {
            msleep(100);
        } else {
            event = EvtQue->dequeue();
            event_id = event.id;
            switch(event_id) {
            case Ev_Init:
                lcd_init();
                break;
            case Ev_Clear:
                lcd_clear();
                break;
            case Ev_OutputString:
                //qDebug() << "event.param = " << event.param;
                lcd_clear();
                lcd_locate(0, 0);
                strcpy(text, event.param.toAscii().data());
                lcd_putstr((unsigned char*)text);
                break;
            default:
                msleep(100);
                break;
            }
        }
    }

    dio_final();
Error:
    ;
}
void ThLCDOutput::sendEvent_Init()
{
    EventData evt;

    evt.id = Ev_Init;
    EvtQue->enqueue(evt);
}

void ThLCDOutput::sendEvent_Clear()
{
    EventData evt;

    evt.id = Ev_Clear;
    EvtQue->enqueue(evt);
}

void ThLCDOutput::sendEvent_OutputText(QString text)
{
    EventData evt;

    evt.id = Ev_OutputString;
    evt.param = text;
    EvtQue->enqueue(evt);
}
