﻿/*
  sted_screen_win32.h
  win32 screen driver

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/
#pragma once

#ifndef STED_SCREEN_WIN32_H
#define STED_SCREEN_WIN32_H

#include <windows.h>
#include <comdef.h>
#include <wincodec.h>
#include <wincodecsdk.h>
#include <d2d1.h>
#include <d2d1helper.h>
#include <dwrite.h>
#include "avrt.h"
#include <mmsystem.h>
#include "sted_screen.h"
#include "exception.h"
#include "sf_windows.h"

/** RGB構造体 */
struct STED_RGB {
  int fRed;
  int fGreen;
  int fBlue;
};

enum {
  EShift = 0,
  EControl,
  EOPT1,
  EOPT2,
  EXF1,
  EXF2,
  EXF3,
  EXF4,
  EXF5,
  EKana,
  EKigo,
  EToroku,
  EInsert,
  EDel,
  EHome,
  EUndo,
  ERollUp,
  ERollDown,
  EF1,
  EF2,
  EF3,
  EF4,
  EF5,
  EF6,
  EF7,
  EF8,
  EF9,
  EF10,
};

#define MASK_ON fBrushMaskOn
#define MASK_OFF fBrushMaskOff

#define IDT_STED2_WAIT_TIMER	1


class CSTedScreenWin32 : public CSTedScreenBase,public sf::base_win32_window<>
{
public:
	typedef boost::shared_ptr<CSTedScreenWin32> ptr_type;

	friend int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR    lpCmdLine,
                     int       nCmdShow);
private:

	CSTedScreenWin32(const std::wstring& menu_name,const std::wstring& name,bool fit_to_display,float width,float height);
	void Construct(int in_screensize = 14);

	static void init_instance();

public:

	virtual LRESULT window_proc(HWND hwnd,boost::uint32_t message, WPARAM wParam, LPARAM lParam);

	static ptr_type& instance()
	{
		return instance_;
	}

  virtual void create();


	//static CSTedScreenWin32* New(int in_screensize=14);
	~CSTedScreenWin32();
	void Tidy(void);

	// window
	void InitWindow(void);
	void CloseWindow(void);
	void RepaintWindow(void);
	void UpdateWindow(void);
	void DisableUpdateWindow(void);

	// color
	int SetGraphicsPalette(int in_pal, int in_color);
	int SetTextPalette(int in_pal, int in_color);
	int SetGraphicsColor(int in_color);
	int SetTextColor(int in_color);
  
	// cursor
	void SetCursor(int in_onoff);

	// text
	void Locate(int in_x, int in_y);
	void Puts(const char* in_str);
	void ClsEol(void);
	void ClsEd(void);
	void SetTextWindow(int in_y, int in_h);
	void GraphicsPuts(int in_x, int in_y, const char* in_str);

	// graphics
	void TextXBox(int in_x0, int in_y0, int in_x1, int in_y1, int in_page);
	void TextXXLine(int in_page, int in_x0, int in_y, int in_x1, int in_linestyle);
	void TextXYLine(int in_page, int in_x, int in_y0, int in_y1, int in_linestyle);
	void TextReverse(int in_x, int in_y, int in_width, int in_page);
	void TextReverseArea(int in_r_ad, int in_r_ln, int in_edit_scr);
	void TextFill(int in_page, int in_x, int in_y, int in_w, int in_h, int in_linestyle);
	void TextRasterCopy(int in_dst, int in_src, int in_line, int in_mode);
	void TextScroll(int in_x0, int in_y0, int in_xs, int in_ys, int in_x1, int in_y1);

	void GraphicsBox(int in_x0, int in_y0, int in_x1, int in_y1, unsigned int in_color, unsigned int in_linestyle);
	int GraphicsPoint(int in_x, int in_y);
	void GraphicsLine(int in_x0, int in_y0, int in_x1, int in_y1, int in_color, int in_linestyle);
	void GraphicsFill(int in_x0, int in_y0, int in_x1, int in_y1, int in_color);

	void CopyTextToGraphics(int in_editscreen);
	void CopyTextToGraphics2(int in_editscreen);

	void GraphicsHome(int in_currenthome);
	void ClsAll(void);
	void GraphicsClear(void);

	// key
	int KeyInit(void);
	int KeyIn(int in_code);
	int KeyInp(void);
	int SftSense(void);
	int BitSense(int in_group);
	int KeySense(void);
	void KeyWait(void);
	void LedMode(int in_code, int in_onoff);
	void ClearKeyBuffer(void);

	// pointer
	void MouseInit(void);
	void SetMouseCursorVisible(int in_onoff);
	int GetMouseButtonState(void);
	int SetMouseMoveLimit(int in_x0, int in_y0, int in_x1, int in_y1);
	int GetMousePos(int *out_x, int *out_y);
	void MouseWait(void);

	// midi related
	BOOL SetMidiPorts(int in_min, int* in_mout);
	int OpenMidiDevice(void);
	void CloseMidiDevice(void);
	int GetMidiData(void);
	int CanMidiRead(void);
	void PutMidiData(int in_data);
	void MidiWait(void);
	void MidiPlayerWait(void);

	// midi player
	int StartMidiPlayer(unsigned char* in_data, int in_len);
	int StopMidiPlayer(void);
	int CloseMidiPlayer(int in_sig);

	// file related
	dosmode_t FileChmod(const char* in_name, dosmode_t in_atr);
	int FileChdir(const char* in_name);
	int FileCurdir(int in_drive, char* out_buff);
	int FileCurdrv(void);
	int FileDelete(const char* in_name);
	int FileRename(const char* in_name, const char* in_newname);
	int FileFiles(struct _x68_filbuf *in_buff, const char *in_name, int in_atr);
	int FileNfiles(struct _x68_filbuf *in_buff);
	FILE* FileFopen(const char* in_name, const char* in_mode);

	// misc
	int OnTime(void);
	int USleep(long in_usec);

public:

	void Paint(RECT *in_rect,HDC dc);
	void NotifyPointerPosition(int in_x, int in_y, int in_wparam);
	void NotifyPointerButtonState(int in_button, int in_state);
	void NotifyKeyPressed(int in_key);
	void NotifyKeyReleased(int in_key);
	void NotifyChar(int in_char);
	void NotifyMidiIn(unsigned long in_data);
	void NotifyTimer(UINT in_timerid);

	LPCTSTR toUCS2(const char* in_str, int *out_len);
	LPCTSTR toUCS2(const char* in_str, int in_len, int *out_len);
	const char* toSJIS(LPCTSTR in_str, int *out_len);
	
  virtual void discard_device();
	virtual void create_device();
  virtual void create_device_independent_resources();
private:

	void DoMessageLoop(void);
	void DrawCursor(void);
	void InvalidateRect(const RECT* in_rect);
	void InvalidateRect(const int in_x, const int in_y, const int in_w, const int in_h);
	int GetTextWidth(LPCTSTR in_str, int in_len);
	BOOL CheckKeyBuffer(int in_key);

	BOOL TextToWindowPos(int *out_x, int *out_y);
	BOOL TextToWindowPos(int in_text_x, int in_text_y, int *out_x, int *out_y);
	BOOL toWindowPos(int in_x, int in_y, int *out_x, int *out_y);

	COLORREF PageToColor(int in_page);

	int ConvertWinKeyToX68Key(int in_key);
	BOOL ConvertWinKeyToX68Key(int in_key, int* out_ascii, int* out_code);

	void AppendMidiData(int in_c);

	// メンバ変数
	
	//HBITMAP fText;
	//HBITMAP fTextMask;
	//HBITMAP fGraphicsWindow[2];
	//HBITMAP fOffScreen;

	ID2D1BitmapRenderTargetPtr text_bitmap_target_;
  ID2D1GdiInteropRenderTargetPtr text_dc_target_;
	//ID2D1BitmapRenderTargetPtr text_mask_bitmap_target_;
	ID2D1BitmapRenderTargetPtr graphics_bitmap_target_[2];
  ID2D1GdiInteropRenderTargetPtr graphics_dc_target_[2];
	//ID2D1BitmapRenderTargetPtr offscreen_bitmap_target_;
//	ID2D1BitmapRenderTargetPtr text_bitmap_target_;

	// HBitMapをそのままID2D1BitmapRenderTargetにする。

	BOOL fOffScreenIsDirty;
	RECT fInvalidRect;
	BOOL fCanUpdateWindow;

	// HWND fWindow;

	// メニューハンドル
	HMENU fMenu;

  
  //HDC fTextDC;
	//HDC fTextMaskDC;
	//HDC fGraphicsWindowDC[2];
	//HDC fOffScreenDC;
	//HDC fWindowDC;

  int fCurrentTextX;
	int fCurrentTextY;
	int fTextOriginY;
	int fTextHeight;

	BOOL fIsCursorOn;

	int fCurrentGraphics;

	int fWindowWidth;
	int fWindowHeight;
	int fTextLineHeight;

  static const int fX68WindowWidth = 768;
	static const int fX68WindowHeight = 512;
	static const int fX68TextWidth = 96;
	static const int fX68TextHeight = 32;

	LPTSTR fUCS2Buf;
	char* fSJISBuf;

	static const int fUCS2BufLen = 1024;
	static const int fSJISBufLen = 1024;

	STED_RGB fTextPalette[4];
	STED_RGB fGraphicsPalette[32];

	static const int fMaxGraphicsColor = 32;
	static const int fMaxTextColor = 4;
	int fCurrentTextColor;
	int fCurrentTextBackColor;
	int fCurrentGraphicsColor;

	//HBRUSH fBrushMaskOff;
	//HBRUSH fBrushMaskOn;
	//HBRUSH fBrushClear;
	//HBRUSH fBrushTextClear;
  IDWriteTextFormatPtr text_fmt_;
  
  ID2D1SolidColorBrushPtr brush_mask_off_;
  ID2D1SolidColorBrushPtr brush_mask_on_;
  ID2D1SolidColorBrushPtr brush_clear_;
  ID2D1SolidColorBrushPtr brush_text_clear_;
  ID2D1SolidColorBrushPtr brush_text_;

	// pointers
	int fPointerMinX;
	int fPointerMaxX;
	int fPointerMinY;
	int fPointerMaxY;
	int fCurrentPointerX;
	int fCurrentPointerY;
	int fCurrentPointerButtonStateL;
	int fCurrentPointerButtonStateR;

	BOOL fPointerEventNotified;

	// key
	static const int fKeyBufferLen = 1024;
	int fKeyBuffer[1024];
	int fKeyBufferStartPtr;
	int fKeyBufferEndPtr;
	int fKeyBufferAvailable;

	BOOL fKeyShiftPressed;
	BOOL fKeyControlPressed;
	BOOL fKeyOPT1Pressed;
	BOOL fKeyOPT2Pressed;
	BOOL fKeyKanaPressed;
	BOOL fKeyInsertPressed;

	BOOL fKeyEventNotified;

	int fKeyConv[1024];

	// midi
	HMIDIIN fMidiInDevice;
	HMIDIOUT fMidiOutDevice[2];
	int fMidiInPort;
	int fMidiOutPort[2];
	static const int fMidiOutPorts=2;
	static const int fMidiBufferLen = 4096;
	int fMidiBuffer[4096];
	int fMidiBufferStartPtr;
	int fMidiBufferEndPtr;
	int fMidiBufferAvailable;
	int fMidiOutPacket[4];
	unsigned char fMidiExclusiveBuf[4096];
	static const int fMidiExclusiveBufLen = 4096;
	int fMidiOutCurrentPort;

	BOOL fTimerEventNotified;

	// file
	HANDLE fFileDirHandle;

	// アクセラレータ
	HACCEL accel_;

	static ptr_type instance_;

};



#endif /* STED_SCREEN_H */
