﻿/*
  ==============================================================================

   Copyright 2005-11 by Satoshi Fujiwara.

   async can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   async is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with async; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/

#include "StdAfx.h"
#if _DEBUG
#define _CRTDBG_MAP_ALLOC
#include <crtdbg.h>
#define new new(_NORMAL_BLOCK, __FILE__, __LINE__)
#endif
#include "sf_memory.h"
#include "audio_base.h"
#include "wasapi.h"

namespace sf{
  wasapi_shared_timer::wasapi_shared_timer(::WAVEFORMATEXTENSIBLE& wfx) 
    : is_enabled_(false),position_(0),is_start_(false)/*,buffer_control_event_(::CreateEventEx(NULL, NULL, 0, EVENT_MODIFY_STATE | SYNCHRONIZE))*/
  {

    try {
      //      thread_priority_.set_priority(AVRT_PRIORITY_NORMAL);

      // WASAPIの初期化処理

      // IMMDeviceEnumeratorの取得
      THROW_IF_ERR(
        CoCreateInstance(
        __uuidof(MMDeviceEnumerator), NULL, CLSCTX_INPROC_SERVER,
        IID_PPV_ARGS(&device_enumerator_)));

      // デフォルトのオーディオデバイスを取得する
      THROW_IF_ERR(
        device_enumerator_
        ->GetDefaultAudioEndpoint(eRender,eMultimedia,&current_device_)
        );

      // オーディオクライアントを取得
      THROW_IF_ERR(
        current_device_
        ->Activate(__uuidof(IAudioClient), CLSCTX_INPROC_SERVER,
        NULL, reinterpret_cast<void **>(&audio_client_))
        );

      // 代替フォーマット定義
      sf::co_task_memory<WAVEFORMATEX>  alt_format;

      // 読みこもうとしているWAVファイルのフォーマットをサポートしているか？
      HRESULT hr = audio_client_->IsFormatSupported(
        AUDCLNT_SHAREMODE_SHARED ,&wfx.Format,&alt_format);
      bool sample_rate_convert = false;
      float sample_rate_backup = 0.0f;
      if(hr != S_OK) //  S_FALSEならそのフォーマットをサポートしていないらしい..
      {
        // サンプルレートのコンバート
        if(alt_format->nSamplesPerSec != wfx.Format.nSamplesPerSec)
        {
          // 本来のサンプルレートをバックアップする
          sample_rate_backup = wfx.Format.nSamplesPerSec;
          // 代替フォーマットのサンプルレートをセットする
          wfx.Format.nSamplesPerSec = alt_format->nSamplesPerSec;
          // 再計算する
          wfx.Format.nAvgBytesPerSec = alt_format->nSamplesPerSec * wfx.Format.nBlockAlign;
          // もう一回チェックする。
          // サンプルレート以外でサポートしていない点があれば例外が発生する。
          THROW_IF_ERR(audio_client_->IsFormatSupported(
            AUDCLNT_SHAREMODE_SHARED ,&wfx.Format,&alt_format));
          // フラグをセットする
          sample_rate_convert = true;
        } else {
          // サンプルレート以外であれば例外を出す。
          throw win32_error_exception(hr);
        }
      }

      // 再生クライアントの初期化

      REFERENCE_TIME buffer_period =  latency_ms_ /* ms */ * 10000 ;

      REFERENCE_TIME buffer_duration = buffer_period * periods_per_buffer_;

      THROW_IF_ERR(audio_client_->Initialize(AUDCLNT_SHAREMODE_SHARED  , 
        AUDCLNT_STREAMFLAGS_NOPERSIST | AUDCLNT_STREAMFLAGS_RATEADJUST, 
        buffer_duration, 
        buffer_period,
        &(wfx.Format), 
        NULL));

      // バッファサイズの取得
      THROW_IF_ERR(audio_client_->GetBufferSize(&buffer_size_));

      // 再生クライアントの取得
      THROW_IF_ERR(audio_client_->GetService(IID_PPV_ARGS(&audio_render_client_)));

      // サンプルレートコンバータ
      if(sample_rate_convert)
      {
        THROW_IF_ERR(audio_client_->GetService(IID_PPV_ARGS(&audio_clock_adjustment_)));
        // 本来のサンプルレートをセットする
        audio_clock_adjustment_->SetSampleRate(sample_rate_backup);
      }

      num_of_frames_ = wfx.Format.nBlockAlign;
      mix_format_ = wfx;
      is_enabled_ = true;
    } catch (win32_error_exception& e)
    {
      exception_holder_.reset(new win32_error_exception(e.hresult()));
      is_enabled_ = false;
    } catch(...) {
      is_enabled_ = false;
    }
  }

  wasapi_shared_timer::~wasapi_shared_timer()
  {
    safe_release(audio_clock_adjustment_);
    safe_release(audio_render_client_);

    // WASAPIの終了処理
    if(audio_client_)
    {
      audio_client_->Stop();
      audio_client_->Reset();
      audio_client_.Release();
    }

    safe_release(current_device_);
    safe_release(device_enumerator_);
  }

  void wasapi_shared_timer::create_wave_data(){
    // サイン波の生成
    boost::uint32_t buffer_size_in_bytes = buffer_size_ * mix_format_.Format.nBlockAlign;
    size_t render_data_length = mix_format_.Format.nSamplesPerSec * 10 /* 秒 */ * mix_format_.Format.nBlockAlign / sizeof(short);
    tone_buffer_.reserve(render_data_length);

    double sampleIncrement = (440 /* Hz */ * (M_PI * 2.0)) / (double)mix_format_.Format.nSamplesPerSec;
    double theta = 0.0;

    for (size_t i = 0 ; i < render_data_length ; i += mix_format_.Format.nChannels)
    {
      double sinValue = sin( theta );
      for(size_t j = 0 ;j < mix_format_.Format.nChannels; j++)
      {
        tone_buffer_.push_back((short)(sinValue * _I16_MAX));
      }
      theta += sampleIncrement;
    }
  };

  void wasapi_shared_timer::play_buffer(BYTE* source_buffer)
  {
    static const size_t inc  = (buffer_size_ * num_of_frames_) / (sizeof(short) * periods_per_buffer_);
    static const size_t buffer_in_periods = buffer_size_ / periods_per_buffer_;
    BYTE* buffer;

    if(!is_start_)
    {

      // 最初にバッファを埋める
      THROW_IF_ERR(audio_render_client_->GetBuffer(buffer_size_,&buffer));
      ::CopyMemory(buffer,source_buffer,get_buffer_byte_size());

      // レイテンシ時間*バッファ数分進める
      THROW_IF_ERR(audio_render_client_->ReleaseBuffer(buffer_size_,0));

      // 再生開始
      THROW_IF_ERR(audio_client_->Start());
      is_start_ = true;
      return;
    }


    // レイテンシ時間だけ待つ
    Sleep(latency_ms_);

    uint32_t padding;
    uint32_t frames_available;
    uint32_t count_period = periods_per_buffer_;
    while(count_period > 0)
    {
      // パディングを求める。
      THROW_IF_ERR(audio_client_->GetCurrentPadding(&padding));
      frames_available = buffer_size_ - padding;

      // パディングを除いた部分のバッファを埋める。
      // パディングを除いた部分のサイズがbuffer_in_periodsより小さい場合はつぎにまわす。
      // パディングを除いた部分を一気に埋めようとしたけどできなかった。。
      while(buffer_in_periods <= frames_available && count_period > 0 ) 
      {
        THROW_IF_ERR(audio_render_client_->GetBuffer(buffer_in_periods,&buffer));
        ::CopyMemory(buffer,source_buffer,buffer_in_periods *  num_of_frames_);
        THROW_IF_ERR(audio_render_client_->ReleaseBuffer(buffer_in_periods,0));
        // レイテンシ時間だけ進める
        source_buffer += buffer_in_periods * num_of_frames_;
        --count_period;
        // パディングを再度求める
        THROW_IF_ERR(audio_client_->GetCurrentPadding(&padding));
        frames_available = buffer_size_ - padding;
      }

      if(count_period > 0)
      {
        Sleep(latency_ms_);
      }
    }
  }
  void wasapi_shared_timer::reset()
  {
    THROW_IF_ERR(audio_client_->Reset());
  }
  void wasapi_shared_timer::stop() {
    //再生停止
    if(is_start_)
    {
      THROW_IF_ERR(audio_client_->Stop());
      reset();
      is_start_ = false;

    };
  }
}
