package org.unitedfront2.domain.communication;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.annotation.Resource;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.SimpleUser;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.test.TransactionalTestCaseWithInitialData;

public class MessageTest extends TransactionalTestCaseWithInitialData {

    private Message message;

    @Autowired private MessageTable messageTable;

    @Autowired private AccessControlTable accessControlTable;

    @Resource private List<Locale> availableLocales;

    @Override
    public void onSetUpInTransaction() throws Exception {
        super.onSetUpInTransaction();
        message = domainFactory.prototype(Message.class);
        Map<Locale, MessageEntry> entryMap
            = new HashMap<Locale, MessageEntry>();
        MessageEntry entryJa = new MessageEntry();
        entryJa.setSubject("^Cg");
        entryJa.setBody("{");
        entryMap.put(Locale.JAPANESE, entryJa);
        MessageEntry entryEn = new MessageEntry();
        entryEn.setSubject("Title");
        entryEn.setBody("Body");
        entryMap.put(Locale.ENGLISH, entryEn);
        message.setCode("test_message");
        message.setEntryMap(entryMap);
        message.setAuthorId(simpleUser1.getId());
        message.setOwnerId(simpleUser1.getId());
        message.setReadAccessControl(domainFactory.prototype(Public.class));
        message.setWriteAccessControl(domainFactory.prototype(OwnerOnly.class));
    }

    @Test
    public void testHashCode() {
        Assert.assertTrue(message.equals(message));
        Assert.assertTrue(message.hashCode() == message.hashCode());
    }

    @Test
    public void testConstructorAvailableLocalesݒ肳OłCX^Xł() {
        Map<Locale, MessageEntry> m = new HashMap<Locale, MessageEntry>();
        m.put(Locale.JAPANESE, new MessageEntry("", "{"));
        new Message(null, m, simpleUser1.getId(), simpleUser1.getId(),
                domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class));
    }

    @Test
    public void testSetAvailableLocalesT|[gꂪǉ폜() {
        Map<Locale, MessageEntry> m = new HashMap<Locale, MessageEntry>();
        m.put(Locale.CHINESE, new MessageEntry("", "{"));
        m.put(Locale.JAPANESE, new MessageEntry("", "{"));
        Message msg = new Message(null, m, simpleUser1.getId(), simpleUser1.getId(),
                domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class));
        msg.setAvailableLocales(availableLocales);
        Assert.assertTrue(msg.getEntryMap().containsKey(Locale.JAPANESE));
        Assert.assertTrue(msg.getEntryMap().containsKey(Locale.ENGLISH));
        Assert.assertFalse(msg.getEntryMap().containsKey(Locale.CHINESE));
    }

    @Test
    public void testGetRequiredEntry() {
        Map<Locale, MessageEntry> entryMap
            = new HashMap<Locale, MessageEntry>();
        MessageEntry entryEn = new MessageEntry();
        entryEn.setSubject("Title");
        entryEn.setBody("Body");
        entryMap.put(Locale.ENGLISH, entryEn);
        message.setEntryMap(entryMap);
        MessageEntry entry = message.getRequiredEntry();
        Assert.assertEquals("Title", entry.getSubject());
    }

    @Test
    public void testGetRequiredEntryȂ̂ŋ() {
        message.setEntryMap(new HashMap<Locale, MessageEntry>());
        MessageEntry entry = message.getRequiredEntry();
        Assert.assertNotNull(entry);
        Assert.assertNull(entry.getSubject());
        Assert.assertNull(entry.getBody());
    }

    @Test
    public void testGetSubject() {
        Assert.assertEquals("^Cg", message.getSubject(Locale.JAPANESE));
    }

    @Test
    public void testGetSubjectw肵P[ȂꍇgetRequiredEntryŕԂGgp() {
        Assert.assertEquals(message.getRequiredEntry().getSubject(), message.getSubject(Locale.CHINESE));
    }

    @Test
    public void testGetSubjectw肵P[̏ꍇgetRequiredEntryŕԂGgp() {
        message.setSubject("", Locale.ENGLISH);
        Assert.assertEquals(message.getRequiredEntry().getSubject(), message.getSubject(Locale.ENGLISH));
    }

    @Test
    public void testGetBody() {
        Assert.assertEquals("{", message.getBody(Locale.JAPANESE));
    }

    @Test
    public void testGetBodyw肵P[ȂꍇgetRequiredEntryŕԂGgp() {
        Assert.assertEquals(message.getRequiredEntry().getBody(), message.getBody(Locale.CHINESE));
    }

    @Test
    public void testStoreo^() throws MessageCodeUsedByOtherException {
        SimpleUser updatedUser = new SimpleUser();
        updatedUser.setId(simpleUser1.getId());
        message.store();
        Assert.assertNotNull(message.getId());
        Message foundMessage = messageTable.find(message.getId());
        message.setAuthorId(simpleUser1.getId());
        Assert.assertEquals(message, foundMessage);
        Assert.assertNotNull(message.getLastUpdateDate());
        Assert.assertNotNull(message.getRegistrationDate());
    }

    @Test(expected=MessageCodeUsedByOtherException.class)
    public void testStoreo^R[hgpĂ() throws MessageCodeUsedByOtherException {
        SimpleUser updatedUser = new SimpleUser();
        updatedUser.setId(simpleUser1.getId());
        message.store();
        Message message2 = domainFactory.prototype(message, Message.class);
        message2.setId(null);
        message2.store();
    }

    @Test
    public void testStoreR[hēo^() throws MessageCodeUsedByOtherException {
        SimpleUser updatedUser = new SimpleUser();
        updatedUser.setId(simpleUser1.getId());
        message.setAuthorId(updatedUser.getId());
        message.setCode(null);
        message.store();
        Assert.assertNotNull(message.getId());
        Message foundMessage = messageTable.find(message.getId());
        message.setAuthorId(simpleUser1.getId());
        Assert.assertEquals(message, foundMessage);
        Assert.assertNotNull(message.getCode());
    }

    @Test
    public void testStoreXV() throws MessageCodeUsedByOtherException, InterruptedException {
        message.store(); // o^
        Message before = domainFactory.prototype(message, Message.class);
        Date oldDate = message.getLastUpdateDate();

        // XV
        SimpleUser updatedUser = new SimpleUser();
        updatedUser.setId(simpleUser1.getId());
        message.setAuthorId(updatedUser.getId());
        Map<Locale, MessageEntry> entryMap = new HashMap<Locale, MessageEntry>();
        MessageEntry entryJa = new MessageEntry();
        entryJa.setSubject("VE" + message.getSubject(Locale.JAPANESE));
        entryJa.setBody("VE" + message.getBody(Locale.JAPANESE));
        entryMap.put(Locale.JAPANESE, entryJa);
        MessageEntry entryEn = new MessageEntry();
        entryEn.setSubject("new_" + message.getSubject(Locale.ENGLISH));
        entryEn.setBody("new_" + message.getBody(Locale.ENGLISH));
        entryMap.put(Locale.ENGLISH, entryEn);
        message.setEntryMap(entryMap);
        java.lang.Thread.sleep(1000);
        message.store();

        Message foundMessage = messageTable.find(message.getId());
        message.setAuthorId(simpleUser1.getId());
        Assert.assertNotSame(before, foundMessage);
        Assert.assertEquals(message, foundMessage);
        Assert.assertTrue(oldDate.before(message.getLastUpdateDate()));
        Assert.assertEquals(oldDate, foundMessage.getRegistrationDate());
    }

    @Test(expected=MessageCodeUsedByOtherException.class)
    public void testStoreXVR[hgpĂ() throws MessageCodeUsedByOtherException {
        SimpleUser updatedUser = new SimpleUser();
        updatedUser.setId(simpleUser1.getId());
        message.store();
        Message message2 = domainFactory.prototype(message, Message.class);
        message2.setId(null);
        message2.setCode(message2.getCode() + "2");
        message2.store();
        message2.setCode(message.getCode());
        message2.store();
    }

    @Test
    public void testSetEntryMapݒ肳ȂɂẴGgi[() {
        MessageEntry entry = new MessageEntry();
        entry.setSubject("eXg");
        entry.setBody("̓eXgłB");
        Map<Locale, MessageEntry> entryMap = new HashMap<Locale, MessageEntry>();
        entryMap.put(Locale.JAPANESE, entry);
        message.setEntryMap(entryMap);
        for (Locale l : availableLocales) {
            Assert.assertTrue(message.getEntryMap().containsKey(l));
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testSetEntryMapΉĂȂ() {
        MessageEntry entry = new MessageEntry();
        entry.setSubject("eXg");
        entry.setBody("̓eXgłB");
        Map<Locale, MessageEntry> entryMap = new HashMap<Locale, MessageEntry>();
        entryMap.put(Locale.FRENCH, entry);
        message.setEntryMap(entryMap);
    }

    @Test
    public void testIsEmptyEntryłȂ() {
        Assert.assertFalse(message.isEmptyEntry());
    }

    @Test
    public void testIsEmptyEntry() {
        message.setEntryMap(null);
        Assert.assertTrue(message.isEmptyEntry());
    }

    @Test
    public void testDelete() throws MessageCodeUsedByOtherException {
        message.store();
        Assert.assertNotNull(messageTable.find(message.getId()));
        message.delete();
        Assert.assertNull(messageTable.find(message.getId()));
        Assert.assertNull(accessControlTable.find(message.getReadAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(message.getWriteAccessControl().getId()));
    }
}
