package org.unitedfront2.dao;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.annotation.Repeat;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.FriendOnly;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.accesscontrol.UserOnly;
import org.unitedfront2.domain.communication.Blog;
import org.unitedfront2.domain.communication.BlogCodeUsedByOtherException;
import org.unitedfront2.domain.communication.BlogEntry;
import org.unitedfront2.domain.communication.Comment;
import org.unitedfront2.domain.communication.CommentTable;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageEntry;
import org.unitedfront2.domain.communication.MessageTable;
import org.unitedfront2.test.DaoTestCaseSupport;

public class BlogDaoTest extends DaoTestCaseSupport<Blog> {

    @Autowired
    private BlogDao blogDao;

    @Autowired
    private DateDao dateDao;

    @Autowired
    private MessageTable messageTable;

    @Autowired
    private AccessControlTable accessControlTable;

    @Autowired
    private CommentTable commentTable;

    @Override
    protected Blog createTestDomain() {
        AccessControl readAccessControl = domainFactory.prototype(Public.class);
        AccessControl writeAccessControl = domainFactory.prototype(Public.class);
        return new Blog(
                "testblog", createOverview(readAccessControl, writeAccessControl),
                simpleUser1.getId(), readAccessControl, writeAccessControl,
                domainFactory.prototype(UserOnly.class));
    }

    private Message createOverview(AccessControl readAccessControl,
            AccessControl writeAccessControl) {
        Message overview = domainFactory.prototype(
                new Message(null, simpleUser1.getId(), simpleUser1.getId(),
                        readAccessControl, writeAccessControl), Message.class);
        overview.setSubject("eXguO", Locale.JAPANESE);
        overview.setBody("̓eXguOłB", Locale.JAPANESE);
        return overview;
    }

    @Override
    protected void editTestDomain(Blog blog) {
        blog.getOverview().setSubject("ҏW̃eXguO", Locale.JAPANESE);
        blog.getOverview().setBody("ҏW̃eXguOłB", Locale.JAPANESE);
        blog.getReadAccessControl().changeTo(UserOnly.class);
        blog.getCommentAccessControl().changeTo(FriendOnly.class);
    }

    @Override
    protected Object getDao() {
        return blogDao;
    }

    @Override
    protected void afterTestDelete(Blog blog) {
        Assert.assertNull(messageTable.find(blog.getOverview().getId()));
        Assert.assertNull(accessControlTable.find(blog.getOverview().getReadAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(blog.getOverview().getWriteAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(blog.getReadAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(blog.getWriteAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(blog.getCommentAccessControl().getId()));
    }

    @Test
    public void testRegisterBlogEntry() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(),
                blog.getWriteAccessControl());
        blogEntry.store();

        blogDao.registerBlogEntry(blog.getId(), blogEntry.getId());
        List<BlogEntry> blogEntries = blogDao.findBlogEntries(blog.getId(), 0, 1);
        Assert.assertSame(1, blogEntries.size());
        Assert.assertEquals(blogEntry, blogEntries.get(0));
    }

    private BlogEntry createBlogEntry(AccessControl readAccessControl, AccessControl writeAccessControl) {
        return domainFactory.prototype(new BlogEntry(createEntry(readAccessControl, writeAccessControl)), BlogEntry.class);
    }

    private Message createEntry(AccessControl readAccessControl, AccessControl writeAccessControl) {
        MessageEntry entry = new MessageEntry();
        entry.setSubject("̋LłB");
        entry.setBody("͍̋LłB");
        Map<Locale, MessageEntry> entryMap = new HashMap<Locale, MessageEntry>();
        entryMap.put(Locale.JAPANESE, entry);
        Message article = domainFactory.prototype(new Message(null, entryMap,
                simpleUser1.getId(), simpleUser1.getId(), readAccessControl,
                writeAccessControl), Message.class);
        return article;
    }

    @Test
    public void testRegisterComment() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry.getId());
        Comment comment = createComment();
        comment.store();
        blogDao.registerComment(blogEntry.getId(), comment.getId());
        List<Comment> comments = blogDao.findComments(blogEntry.getId());
        Assert.assertSame(1, comments.size());
        Assert.assertEquals(comment, comments.get(0));
    }

    private Comment createComment() {
        return domainFactory.prototype(new Comment(null, simpleUser1.getId(),
                "̓Rg̃eXgłB", dateDao.getCurrentDate()), Comment.class);
    }

    @Test
    public void testFindByCode() throws BlogCodeUsedByOtherException {
        Blog expected = createTestDomain();
        blogDao.register(expected);
        Blog actual = blogDao.findByCode(expected.getCode());
        Assert.assertEquals(expected, actual);
    }

    @Test
    public void testFindByCodeȂ() throws BlogCodeUsedByOtherException {
        Blog actual = blogDao.findByCode("-1");
        Assert.assertNull(actual);
    }

    @Test
    public void testFindByOwnerId() {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        List<Blog> list = blogDao.findByOwnerId(blog.getOwnerId());
        Assert.assertSame(1, list.size());
        Assert.assertEquals(blog, list.get(0));
    }

    @Test
    public void testFindPublicProfile() {
        Blog b0 = createTestDomain();
        b0.setCode(b0.getCode() + 0);
        blogDao.register(b0);
        Blog b1 = createTestDomain();
        b1.setCode(b1.getCode() + 1);
        blogDao.register(b1);
        Blog b2 = createTestDomain();
        b2.setCode(b2.getCode() + 2);
        blogDao.register(b2);
        Blog b3 = createTestDomain();
        b3.setCode(b3.getCode() + 3);
        b3.setReadAccessControl(domainFactory.prototype(UserOnly.class));
        blogDao.register(b3);
        List<Blog> list = blogDao.findPublicBlogs();
        Assert.assertSame(3, list.size());
        Assert.assertEquals(list.get(0), b2);
        Assert.assertEquals(list.get(1), b1);
        Assert.assertEquals(list.get(2), b0);
        Assert.assertFalse(list.contains(b3));
    }

    @Test
    @Repeat(value = 3)
    public void testFindPublicProfileRandomly() {
        Blog b0 = createTestDomain();
        b0.setCode(b0.getCode() + 0);
        blogDao.register(b0);
        Blog b1 = createTestDomain();
        b1.setCode(b1.getCode() + 1);
        blogDao.register(b1);
        Blog b2 = createTestDomain();
        b2.setCode(b2.getCode() + 2);
        blogDao.register(b2);
        Blog b3 = createTestDomain();
        b3.setCode(b3.getCode() + 3);
        b3.setReadAccessControl(domainFactory.prototype(UserOnly.class));
        blogDao.register(b3);
        List<Blog> list = blogDao.findPublicBlogsRandomly(2);
        Assert.assertSame(2, list.size());
        Assert.assertFalse(list.contains(b3));
    }

    @Test
    public void testFindBlogEntries͂߂Q() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry0 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry0.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry0.getId());
        BlogEntry blogEntry1 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry1.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry1.getId());
        BlogEntry blogEntry2 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry2.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry2.getId());
        List<BlogEntry> blogEntries = blogDao.findBlogEntries(blog.getId(), 0, 2);
        Assert.assertSame(2, blogEntries.size());
        Assert.assertEquals(blogEntry2, blogEntries.get(0));
        Assert.assertEquals(blogEntry1, blogEntries.get(1));
    }

    @Test
    public void testFindBlogEntriesPQ() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry0 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry0.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry0.getId());
        BlogEntry blogEntry1 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry1.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry1.getId());
        BlogEntry blogEntry2 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry2.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry2.getId());
        BlogEntry blogEntry3 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry3.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry3.getId());
        List<BlogEntry> blogEntries = blogDao.findBlogEntries(blog.getId(), 1, 2);
        Assert.assertSame(2, blogEntries.size());
        Assert.assertEquals(blogEntry2, blogEntries.get(0));
        Assert.assertEquals(blogEntry1, blogEntries.get(1));
    }

    @Test
    public void testFindComments() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry.getId());
        Comment comment0 = createComment();
        comment0.store();
        blogDao.registerComment(blogEntry.getId(), comment0.getId());
        Comment comment1 = createComment();
        comment1.store();
        blogDao.registerComment(blogEntry.getId(), comment1.getId());
        Comment comment2 = createComment();
        comment2.store();
        blogDao.registerComment(blogEntry.getId(), comment2.getId());
        List<Comment> comments = blogDao.findComments(blogEntry.getId());
        Assert.assertSame(3, comments.size());
        Assert.assertEquals(comment0, comments.get(0));
        Assert.assertEquals(comment1, comments.get(1));
        Assert.assertEquals(comment2, comments.get(2));
    }

    @Test
    public void testCountBlogEntry() {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry0 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry0.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry0.getId());
        BlogEntry blogEntry1 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry1.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry1.getId());
        BlogEntry blogEntry2 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry2.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry2.getId());
        Assert.assertSame(3, blogDao.countBlogEntry(blog.getId()));
    }

    @Test
    public void testCountComment() {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry.getId());
        Comment comment0 = createComment();
        comment0.store();
        blogDao.registerComment(blogEntry.getId(), comment0.getId());
        Comment comment1 = createComment();
        comment1.store();
        blogDao.registerComment(blogEntry.getId(), comment1.getId());
        Comment comment2 = createComment();
        comment2.store();
        blogDao.registerComment(blogEntry.getId(), comment2.getId());
        Assert.assertSame(3, blogDao.countComment(blogEntry.getId()));
    }

    @Test
    public void testDeleteBlogEntry() throws BlogCodeUsedByOtherException {
        Blog blog = createTestDomain();
        blogDao.register(blog);
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blogEntry.store();
        blogDao.registerBlogEntry(blog.getId(), blogEntry.getId());
        Comment comment = createComment();
        comment.store();
        blogDao.registerComment(blogEntry.getId(), comment.getId());
        blogDao.deleteBlogEntry(blogEntry.getId());
        List<BlogEntry> actual = blogDao.findBlogEntries(blog.getId(), 0, 1);
        Assert.assertSame(0, actual.size());
        Assert.assertNull(messageTable.find(blogEntry.getId()));
        Assert.assertNull(commentTable.find(comment.getId()));
    }
}
