package org.unitedfront2.domain;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.unitedfront2.util.PropertyUtils;

/**
 * {@link DomainFactory}  Spring Framework DI ReiłB
 *
 * @author kurokkie
 *
 */
public class SpringBeanDomainFactory implements DomainFactory,
        ApplicationContextAware {

    /** O */
    protected final Log logger = LogFactory.getLog(getClass());

    /** {@link ApplicationContext} */
    private ApplicationContext applicationContext;

    @Override
    public <D extends Domain> D prototype(Class<D> domainClass) {
        return (D) applicationContext.getBean(toBeanName(domainClass));
    }

    @Override
    public <T> T prototype(Class<?> domainClass, Class<T> requiredType)
        throws IllegalArgumentException {
        if (!Domain.class.isAssignableFrom(domainClass)) {
            String message = "The domain class '" + domainClass
                + "' is not assignable from the class '" + Domain.class + "'.";
            throw new IllegalArgumentException(message);
        }
        Object o = prototype((Class<? extends Domain>) domainClass);
        if (!requiredType.isAssignableFrom(o.getClass())) {
            String message = "The domain object '" + o
                + "' is not assignable from the class '" + requiredType + "'.";
            throw new IllegalArgumentException(message);
        }
        return (T) o;
    }

    @Override
    public <D> D prototype(D orig) {
        return (D) doPrototype(orig);
    }

    @Override
    public <T> T prototype(Object domainObject, Class<T> clazz)
        throws IllegalArgumentException {
        if (domainObject == null) {
            return null;
        }
        if (!clazz.isAssignableFrom(domainObject.getClass())) {
            String message = "The domain object '" + domainObject
                + "' is not assignable from the class '" + clazz + "'.";
            throw new IllegalArgumentException(message);
        }
        return (T) doPrototype(domainObject);
    }

    private Object doPrototype(Object orig) {
        if (DomainProxy.class.isAssignableFrom(orig.getClass())) {
            return doPrototype(((DomainProxy<?>) orig).getTarget());
        }
        if (LogFactory.getLog(getClass()).isDebugEnabled()) {
            LogFactory.getLog(getClass()).debug("Original Bean = '" + orig
                    + "'");
        }
        Object dest = applicationContext.getBean(toBeanName(orig.getClass()));
        Object target = dest;
        if (DomainProxy.class.isAssignableFrom(target.getClass())) {
            // hCvLV̏ꍇ͂̎Ԃȉ̑̑ΏۂƂȂ
            target = ((DomainProxy<?>) dest).getTarget();
        }

        // 󂢕
        PropertyUtils.copyProperties(target, orig);

        setNewDomainObjects(dest, target);

        return dest;
    }

    private <D> String toBeanName(Class<D> domainClass) {
        String name = domainClass.getSimpleName();
        if (name.length() == 1) {
            return String.valueOf(Character.toLowerCase(name.charAt(0)));
        } else {
            return Character.toLowerCase(name.charAt(0)) + name.substring(1);
        }
    }

    // 錾ꂽtB[hԂ܂BieNX܂ށj
    private Field getDeclaredField(Class<?> clazz, String fieldName)
        throws SecurityException, NoSuchFieldException {
        try {
            return clazz.getDeclaredField(fieldName);
        } catch (NoSuchFieldException e) {
            if (clazz.getSuperclass() != null) {
                return getDeclaredField(clazz.getSuperclass(), fieldName);
            } else {
                throw e;
            }
        }
    }

    // vpeBhCIuWFNg̏ꍇ prototype \bhgĐ
    private void setNewDomainObjects(Object dest, Object target) {
        for (Entry<String, Object> entry
                : PropertyUtils.describe(target).entrySet()) {
            Field field;
            try {
                field = getDeclaredField(target.getClass(), entry.getKey());
            } catch (SecurityException e) {
                if (LogFactory.getLog(getClass()).isDebugEnabled()) {
                    LogFactory.getLog(getClass()).debug(
                            "NoSuchFieldException occured: "
                            + target.getClass().getSimpleName() + "#"
                            + entry.getKey());
                }
                continue;
            } catch (NoSuchFieldException e) {
                if (LogFactory.getLog(getClass()).isDebugEnabled()) {
                    LogFactory.getLog(getClass()).debug(
                            "NoSuchFieldException occured: "
                            + target.getClass().getSimpleName() + "#"
                            + entry.getKey());
                }
                continue;
            }
            if (Modifier.isTransient(field.getModifiers())) {
                if (LogFactory.getLog(getClass()).isDebugEnabled()) {
                    LogFactory.getLog(getClass()).debug("The field '"
                            + target.getClass().getSimpleName() + "#"
                            + entry.getKey()
                            + " is ignored because of transient.");
                }
                continue;
            }
            if (entry.getValue() == null) {
                continue;
            } else if (DomainProxy.class.isAssignableFrom(
                    entry.getValue().getClass())) {
                // hCvLV̏ꍇ̓^[QbggĐ
                DomainProxy<?> dp = (DomainProxy<?>) entry.getValue();
                Domain newValue = prototype(dp.getTarget(), Domain.class);
                PropertyUtils.setProperty(dest, entry.getKey(), newValue);
            } else if (Domain.class.isAssignableFrom(
                    entry.getValue().getClass())) {
                Object value = prototype(entry.getValue(), Domain.class);
                if (LogFactory.getLog(getClass()).isDebugEnabled()) {
                    LogFactory.getLog(getClass()).debug("dest = '" + dest
                            + "', property = '" + entry.getKey()
                            + "', value = '" + value + "'.");
                }
                PropertyUtils.setProperty(dest, entry.getKey(), value);
            }
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext)
        throws BeansException {
        this.applicationContext = applicationContext;
    }
}
