package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.MessageDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageEntry;

/**
 * {@link MessageDao}  JDBC łB
 *
 * @author kurokkie
 *
 */
@Repository(value = "messageDao")
public class MessageDaoImpl extends SimpleDaoSupport<Message>
    implements MessageDao {

    /** bZ[WGg̃}bsONX */
    private static class MessageEntryRowMapper
        implements ParameterizedRowMapper<Entry<Locale, MessageEntry>> {

        @Override
        public Entry<Locale, MessageEntry> mapRow(ResultSet rs, int rowNum)
            throws SQLException {

            Locale locale = new Locale(rs.getString("Locale"));
            MessageEntry entry = new MessageEntry();
            entry.setSubject(rs.getString("Subject"));
            entry.setBody(rs.getString("Body"));
            return new AbstractMap.SimpleEntry<Locale, MessageEntry>(
                locale, entry);
        }
    }

    /** bZ[W̃}bsONX */
    private static class MessageRowMapper
        implements ParameterizedRowMapper<Message> {

        /** ANZXe[u */
        private AccessControlTable accessControlTable;

        /** hCt@Ng */
        private DomainFactory domainFactory;

        /** {@link SimpleJdbcTemplate} */
        private SimpleJdbcTemplate simpleJdbcTemplate;

        public MessageRowMapper(AccessControlTable accessControlTable,
                DomainFactory domainFactory,
                SimpleJdbcTemplate simpleJdbcTemplate) {
            super();
            this.accessControlTable = accessControlTable;
            this.domainFactory = domainFactory;
            this.simpleJdbcTemplate = simpleJdbcTemplate;
        }

        @Override
        public Message mapRow(ResultSet rs, int rowNum) throws SQLException {
            Message message = domainFactory.prototype(Message.class);
            message.setId(rs.getInt("Id"));
            message.setCode(rs.getString("Code"));
            message.setEntryMap(findEntry(message.getId()));
            message.setAuthorId(rs.getInt("AuthorId"));
            message.setOwnerId(rs.getInt("OwnerId"));
            message.setReadAccessControl(accessControlTable.find(
                    rs.getInt("ReadAccessControlId")));
            message.setWriteAccessControl(accessControlTable.find(
                    rs.getInt("WriteAccessControlId")));
            message.setLastUpdateDate(rs.getTimestamp("LastUpdateDate"));
            message.setRegistrationDate(rs.getTimestamp("RegistrationDate"));
            return message;
        }

        private Map<Locale, MessageEntry> findEntry(int id) {
            List<Entry<Locale, MessageEntry>> entries
                = simpleJdbcTemplate.query(
                    "SELECT * FROM MessageEntry WHERE MessageId = ?",
                    new MessageEntryRowMapper(), id);
            Map<Locale, MessageEntry> entryMap
                = new HashMap<Locale, MessageEntry>(entries.size());
            for (Entry<Locale, MessageEntry> entry : entries) {
                entryMap.put(entry.getKey(), entry.getValue());
            }
            return entryMap;
        }
    }

    /** ANZXe[u */
    @Autowired
    private AccessControlTable accessControlTable;

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Message> createRowMapper() {
        return new MessageRowMapper(accessControlTable, getDomainFactory(),
                getSimpleJdbcTemplate());
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Message message) {
        Map<String, Object> args = super.toColumnValueMap(message);
        args.put("ReadAccessControlId", message.getReadAccessControl().getId());
        args.put("WriteAccessControlId",
                message.getWriteAccessControl().getId());
        return args;
    }

    @Override
    protected Class<Message> getDomainClass() {
        return Message.class;
    }

    //**************************************************************************
    // MessageDao
    //**************************************************************************
    @Override
    public Message find(int id) {
        Message message = super.find(id);
        if (message == null) {
            return null;
        }
        return message;
    }

    @Override
    public void register(Message message) {
        message.getReadAccessControl().store();
        message.getWriteAccessControl().store();
        super.register(message);
        registerEntry(message);
    }

    @Override
    public void update(Message message) {
        message.getReadAccessControl().store();
        message.getWriteAccessControl().store();
        super.update(message);
        deleteEntry(message);
        registerEntry(message);
    }

    @Override
    public Message findByCode(String code) {
        Message message;
        try {
            message = getSimpleJdbcTemplate().queryForObject(
                "SELECT * FROM Message WHERE Code = ?", createRowMapper(),
                code);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
        return message;
    }

    @Override
    public void delete(int id) {
        Message m = find(id);
        super.delete(id);
        try {
            m.getReadAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            m.getWriteAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
    }

    private void registerEntry(Message message) {
        for (Entry<Locale, MessageEntry> entry
                : message.getEntryMap().entrySet()) {

            getJdbcTemplate().update(
                "INSERT INTO MessageEntry(MessageId, Locale, Subject, Body)"
                + " VALUES(?, ?, ?, ?)", new Object[] {
                    message.getId(), entry.getKey().getLanguage(),
                    entry.getValue().getSubject(), entry.getValue().getBody(),
                });
        }
    }

    private void deleteEntry(Message message) {
        getJdbcTemplate().update("DELETE FROM MessageEntry WHERE MessageId = ?",
                new Object[] {message.getId()});
    }
}
