package org.unitedfront2.web.controller.account;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.validation.Errors;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;
import org.unitedfront2.domain.Account;
import org.unitedfront2.domain.AccountTable;
import org.unitedfront2.domain.MailAddrAndPassword;
import org.unitedfront2.domain.MailAddrUsedByOtherException;
import org.unitedfront2.validation.Validate;
import org.unitedfront2.validation.ValidationException;
import org.unitedfront2.web.WebUtils;
import org.unitedfront2.web.controller.FormAction;
import org.unitedfront2.web.flow.PageNotFoundException;

/**
 * pX[h̍ĔssANVNXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "account.passwordReissueAction")
public class PasswordReissueAction extends FormAction {

    /** spX[h̍ŏ (4) */
    public static final int PASSWORD_MIN_LENGTH = 4;

    /** spX[h̍ő啶 (8) */
    public static final int PASSWORD_MAX_LENGTH = 8;

    /** pX[hĔs̊mF[ev[g (account.PasswordReissueConfirm-mail) */
    public static final String PASSWORD_REISSUE_CONFIRM_MAIL_TEMPLATE_NAME
        = "account.PasswordReissueConfirm-mail";

    /** pX[hĔs̃[ev[g (account.PasswordReissue-mail) */
    public static final String PASSWORD_REISSUE_MAIL_TEMPLATE_NAME
        = "account.PasswordReissue-mail";

    /** AJEge[u */
    @Autowired private AccountTable accountTable;

    @Override
    protected void initAction() {
        super.initAction();
        setMailTemplateNames(new String[] {
            PASSWORD_REISSUE_CONFIRM_MAIL_TEMPLATE_NAME,
            PASSWORD_REISSUE_MAIL_TEMPLATE_NAME,
        });
        setFormObjectClass(Account.class);
    }

    @Override
    public Event bindAndValidate(RequestContext context) throws Exception {
        super.bindAndValidate(context);
        Errors errors = getFormErrors(context);
        Account account = (Account) getFormObject(context);
        try {
            Validate.notBlank(account.getMailAddr());
            Validate.mailAddr(account.getMailAddr());
        } catch (ValidationException e) {
            errors.rejectValue("mailAddr", e.getCode());
        }
        if (errors.hasErrors()) {
            return error();
        } else {
            return success();
        }
    }

    /**
     * ꎞIɔsF؃L[𐶐܂B
     *
     * @param context {@link RequestContext}
     * @return Cxg
     * @throws AccountNotFoundException AJEg݂Ȃ
     */
    public Event generateTemporaryAuthKey(RequestContext context)
        throws AccountNotFoundException {
        Account account = (Account) getFormObject(context);
        Account found = findAccountWithTemporaryAuthKeyByMailAddr(
                    account.getMailAddr());
        if (found == null) {
            String message = "The account [mailAddr='"
                + account.getMailAddr() + "'] not found.";
            logger.warn(message);
            throw new AccountNotFoundException(message);
        }
        context.getRequestScope().put("account", found);
        doSendMail(context, PASSWORD_REISSUE_CONFIRM_MAIL_TEMPLATE_NAME);
        return success();
    }

    private Account findAccountWithTemporaryAuthKeyByMailAddr(String mailAddr) {
        Account account = accountTable.findByMailAddr(mailAddr);
        if (account == null) {
            return null;
        } else {
            account.retrieveTemporaryAuthKey();
            return account;
        }
    }

    /**
     * pX[h̍Ĕss܂B
     *
     * @param context NGXgReLXg
     * @return Cxg
     * @throws PageNotFoundException F؃L[s
     */
    public Event reissuePassword(RequestContext context)
        throws PageNotFoundException {

        String temporaryAuthKey = context.getRequestParameters().get("key");
        if (StringUtils.isEmpty(temporaryAuthKey)) {
            String message = "The temporary auth key is null. It must be set.";
            logger.warn(message);
            throw new PageNotFoundException(context, this, message);
        }
        try {
            MailAddrAndPassword mailAddrAndPassword
                = reissuePasswordByTemporaryAuthKey(temporaryAuthKey);
            context.getRequestScope().put("mailAddrAndPassword",
                mailAddrAndPassword);
            doSendMail(context, PASSWORD_REISSUE_MAIL_TEMPLATE_NAME);
            return success();
        } catch (AccountNotFoundException e) {
            String message = "The temporary auth key '" + temporaryAuthKey
                + "'is not found.";
            logger.warn(message);
            throw new PageNotFoundException(context, this, message, e);
        }
    }

    private MailAddrAndPassword reissuePasswordByTemporaryAuthKey(
            String temporaryAuthKey) throws AccountNotFoundException {
        Account account = accountTable.findByTemporaryAuthKey(temporaryAuthKey);
        if (account == null) {
            String message = "The temporary auth key '" + temporaryAuthKey
                + "' not found.";
            if (logger.isDebugEnabled()) {
                logger.debug(message);
            }
            throw new AccountNotFoundException(message);
        }
        MailAddrAndPassword map = reissuePassword(account);
        account.deleteTemporaryAuthKey();
        return map;
    }

    private MailAddrAndPassword reissuePassword(Account account) {
        String newPassword = Account.createRandomPassword(
                PASSWORD_MIN_LENGTH, PASSWORD_MAX_LENGTH);
        account.setPlainPassword(newPassword);
        account.encrypt();
        try {
            account.store();
        } catch (MailAddrUsedByOtherException e) {
            logger.error(e);
            throw new IllegalStateException(e);
        }

        return new MailAddrAndPassword(account.getMailAddr(), newPassword);
    }

    /**
     * Ǘ҂pX[h̍Ĕss܂B̓[MpłȂ
     * ɂāApX[hĔs邽߂̎iłB
     *
     * @param context NGXgReLXg
     * @return Cxg
     * @throws AccountNotFoundException AJEg݂Ȃ
     */
    public Event reissuePasswordByAdmin(RequestContext context)
        throws AccountNotFoundException {

        Account account = (Account) getFormObject(context);
        try {
            MailAddrAndPassword mailAddrAndPassword
                = reissuePassword(account.getMailAddr());
            context.getRequestScope().put("mailAddrAndPassword",
                    mailAddrAndPassword);
            return success(mailAddrAndPassword);
        } catch (AccountNotFoundException e) {
            logger.warn(e.getMessage());
            Errors errors = getBindingErrors(context, account, "account");
            errors.rejectValue("mailAddr",
                    "account.validation.mailAddrNotFound",
                    null, e.getMessage());
            throw e;
        }
    }

    private MailAddrAndPassword reissuePassword(String mailAddr)
        throws AccountNotFoundException {

        if (!accountTable.existByMailAddr(mailAddr)) {
            String message = "The mail address '" + mailAddr + "' not found.";
            if (logger.isWarnEnabled()) {
                logger.warn(message);
            }
            throw new AccountNotFoundException(message);
        }

        Account foundAccount = accountTable.findByMailAddr(mailAddr);
        return reissuePassword(foundAccount);
    }

    /**
     * 蓮pX[hĔs̏s܂B{t[XR[vɐݒ肵Ă܂B
     *
     * @param context {@link RequestContext}
     * @return Cxg
     */
    public Event prepareManual(RequestContext context) {
        String subject = WebUtils.getMessage("account.PasswordReissue", null,
            context);
        context.getFlowScope().put("subject", subject);
        return success();
    }
}
