package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcInsert;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.ProfileDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.SimpleUser;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.communication.Profile;

/**
 * {@link ProfileDao} ̎łB
 *
 * @author kurokkie
 *
 */
@Repository(value = "profileDao")
public class ProfileDaoImpl extends SimpleDaoSupport<Profile>
    implements ProfileDao {

    /** vtB[̃}bsONX */
    private static class ProfileRowMapper
        implements ParameterizedRowMapper<Profile> {

        /** hCt@Ng */
        private DomainFactory domainFactory;

        /** ANZXe[u */
        private AccessControlTable accessControlTable;

        public ProfileRowMapper(DomainFactory domainFactory,
                AccessControlTable accessControlTable) {
            super();
            this.domainFactory = domainFactory;
            this.accessControlTable = accessControlTable;
        }

        @Override
        public Profile mapRow(final ResultSet rs, int rowNum)
            throws SQLException {

            Profile profile = domainFactory.prototype(Profile.class);
            int ownerId = rs.getInt("OwnerId");
            profile.setOwnerId(ownerId);
            profile.setReadAccessControl(accessControlTable.find(rs.getInt(
                    "ReadAccessControlId")));
            profile.setWriteAccessControl(accessControlTable.find(rs.getInt(
                    "WriteAccessControlId")));
            return profile;
        }
    }

    /** ANZXe[u */
    @Autowired
    private AccessControlTable accessControlTable;

    /** ANZX}bp[t@Ng */
    @Resource(name = "accessControlDao")
    private RowMapperFactory<AccessControl> accessControlRowMapperFactory;

    /** [U}bp[t@Ng */
    @Resource(name = "simpleUserDao")
    private RowMapperFactory<SimpleUser> simpleUserRowMapperFactory;

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Profile> createRowMapper() {
        return new ProfileRowMapper(getDomainFactory(), accessControlTable);
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Profile profile) {
        Map<String, Object> args = super.toColumnValueMap(profile);
        args.put("ReadAccessControlId", profile.getReadAccessControl().getId());
        args.put("WriteAccessControlId",
                profile.getWriteAccessControl().getId());
        return args;
    }

    @Override
    protected String getPrimaryKeyColumnName() {
        return "OwnerId";
    }

    @Override
    protected SimpleJdbcInsert createSimpleJdbcInsert(DataSource dataSource) {
        return new SimpleJdbcInsert(dataSource).withTableName(getTableName());
    }

    @Override
    protected Class<Profile> getDomainClass() {
        return Profile.class;
    }

    //**************************************************************************
    // ProfileDao
    //**************************************************************************
    @Override
    public void register(Profile profile) {
        profile.getReadAccessControl().store();
        profile.getWriteAccessControl().store();
        super.registerNoReturnKey(profile);
    }

    @Override
    public Profile findByOwnerCode(String userCode) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT * FROM Profile p, User u WHERE u.Code = ? AND "
                    + "p.OwnerId = u.Id", createRowMapper(), userCode);
        } catch (DataAccessException e) {
            if (logger.isInfoEnabled()) {
                logger.info(e.getMessage());
            }
            return null;
        }
    }

    @Override
    public List<SimpleUser> findPublicProfileOwnersRandomly() {
        return getSimpleJdbcTemplate().query(
            "SELECT u.* FROM User u, Profile p, AccessControl a WHERE "
            + "u.Id = p.OwnerId AND p.ReadAccessControlId = a.Id AND "
            + "a.Class = '" + Public.class.getName() + "' ORDER BY RAND()",
            simpleUserRowMapperFactory.createRowMapper());
    }

    @Override
    public List<SimpleUser> findPublicProfileOwnersRandomly(int max) {
        return getSimpleJdbcTemplate().query(
                "SELECT u.* FROM User u, Profile p, AccessControl a WHERE "
                + "u.Id = p.OwnerId AND p.ReadAccessControlId = a.Id AND "
                + "a.Class = '" + Public.class.getName()
                + "' ORDER BY RAND() LIMIT 0, ?",
                simpleUserRowMapperFactory.createRowMapper(), max);
    }

    @Override
    public AccessControl findReadAccessControl(int ownerId) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT * FROM AccessControl, Profile WHERE "
                    + "Profile.OwnerId = ? AND "
                    + "Profile.ReadAccessControlId = AccessControl.Id",
                    accessControlRowMapperFactory.createRowMapper(), ownerId);
        } catch (DataAccessException e) {
            if (logger.isInfoEnabled()) {
                logger.info(e.getMessage());
            }
            return null;
        }
    }

    @Override
    public void update(Profile profile) {
        profile.getReadAccessControl().store();
        profile.getWriteAccessControl().store();
        super.update(profile);
    }
}
