package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.MailDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.communication.Mail;

/**
 * {@link MailDao}  JDBC łB
 *
 * @author kurokkie
 *
 */
@Repository(value = "mailDao")
public class MailDaoImpl extends SimpleDaoSupport<Mail>
    implements MailDao {

    /** [}bsONX */
    private static class MailRowMapper implements ParameterizedRowMapper<Mail> {

        /** hCt@Ng */
        private DomainFactory domainFactory;

        public MailRowMapper(DomainFactory domainFactory) {
            super();
            this.domainFactory = domainFactory;
        }

        @Override
        public Mail mapRow(ResultSet rs, int rowNum) throws SQLException {
            Mail mail = domainFactory.prototype(Mail.class);
            mail.setId(rs.getInt("Id"));
            mail.setCode(rs.getString("Code"));
            mail.setToId(rs.getInt("ToId"));
            int fromId = rs.getInt("FromId");
            if (fromId == 0) {
                mail.setFromId(null);
            } else {
                mail.setFromId(fromId);
            }
            mail.setSentDate(rs.getTimestamp("SentDate"));
            mail.setSubject(rs.getString("Subject"));
            mail.setBody(rs.getString("Body"));
            mail.setRead(rs.getBoolean("IsRead"));

            int nextId = rs.getInt("NextId");
            if (nextId != 0) {
                Mail next = domainFactory.prototype(Mail.class);
                next.setId(nextId);
                mail.setNext(next);
            }
            return mail;
        }
    }

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Mail> createRowMapper() {
        return new MailRowMapper(getDomainFactory());
    }

    @Override
    protected Class<Mail> getDomainClass() {
        return Mail.class;
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Mail mail) {
        Map<String, Object> map = super.toColumnValueMap(mail);
        map.put("IsRead", mail.isRead());
        return map;
    }

    //**************************************************************************
    // MailDao
    //**************************************************************************
    @Override
    public void register(Mail mail, int parentId) {
        register(mail);
        Mail thread = findThread(parentId);
        Mail tail = thread.getTail();
        getJdbcTemplate().update("UPDATE Mail SET NextId = ? WHERE Id = ?",
            new Object[] {mail.getId(), tail.getId()});
    }

    @Override
    public void updateRead(int id, boolean read) {
        getJdbcTemplate().update("UPDATE Mail SET IsRead = ? WHERE Id = ?",
            new Object[] {read, id});
    }

    @Override
    public int countThreads(int ownerId) {
        return getJdbcTemplate().queryForInt(
            "SELECT COUNT(*) FROM Mail WHERE (ToId = ? OR FromId = ?) "
            + "AND NextId IS NULL", new Object[] {ownerId, ownerId});
    }

    @Override
    public int countUnreadMail(int ownerId) {
        return getJdbcTemplate().queryForInt(
            "SELECT COUNT(*) FROM Mail WHERE ToId = ? AND IsRead = FALSE",
            new Object[] {ownerId});
    }

    @Override
    public Mail findThread(int id) {
        Mail mail = find(id);
        if (mail == null) {
            return null;
        }
        setChild(mail);
        return getParent(mail);
    }

    @Override
    public Mail findThreadByCode(String code) {
        Mail mail = findByCode(code);
        if (mail == null) {
            return null;
        }
        setChild(mail);
        return getParent(mail);
    }

    @Override
    public Mail findByCode(String code) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                "SELECT * FROM Mail WHERE Code = ?", createRowMapper(), code);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    private void setChild(Mail mail) {
        if (mail.getNext() == null || mail.getNext().getId() == null) {
            return;
        }
        Mail child = getSimpleJdbcTemplate().queryForObject(
            "SELECT * From Mail WHERE Id = ?", createRowMapper(),
            mail.getNext().getId());
        mail.setNext(child);
        setChild(child);
    }

    private Mail getParent(Mail mail) {
        Mail parent;
        try {
            parent = getSimpleJdbcTemplate().queryForObject(
                "SELECT * From Mail WHERE NextId = ?", createRowMapper(),
                mail.getId());
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return mail;
        }
        parent.setNext(mail);
        return getParent(parent);
    }

    @Override
    public List<Mail> findThreads(int ownerId, int no, int num) {
        List<Mail> mails = getSimpleJdbcTemplate().query(
            "SELECT * From Mail WHERE (ToId = ? OR FromId = ?) AND "
            + "NextId IS NULL ORDER BY Id DESC LIMIT ?, ?",
            createRowMapper(), ownerId, ownerId, no, num);
        List<Mail> threads = new ArrayList<Mail>(mails.size());
        for (Mail mail : mails) {
            Mail thread = findThread(mail.getId());
            threads.add(thread);
        }
        return threads;
    }
}
