package org.unitedfront2.domain.communication;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.FriendOnly;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.accesscontrol.UserOnly;
import org.unitedfront2.test.TransactionalTestCaseWithInitialData;

public class BlogTest extends TransactionalTestCaseWithInitialData {

    @Autowired private BlogTable blogTable;

    @Test
    public void testPost() throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(),
                blog.getWriteAccessControl());

        blog.retrieveEntries(0, 1);
        Assert.assertSame(0, blog.getEntries().size());
        blog.post(blogEntry);
        blog.retrieveEntries(0, 1);
        Assert.assertSame(1, blog.getEntries().size());
        Assert.assertEquals(blogEntry, blog.getEntries().get(0));
    }

    private Blog createBlog() {
        AccessControl readAccessControl = domainFactory.prototype(Public.class);
        AccessControl writeAccessControl = domainFactory.prototype(OwnerOnly.class);
        return domainFactory.prototype(new Blog("test",
                createOverview(readAccessControl, writeAccessControl),
                simpleUser1.getId(), readAccessControl, writeAccessControl,
                domainFactory.prototype(UserOnly.class)));
    }

    private Message createOverview(AccessControl readAccessControl,
            AccessControl writeAccessControl) {
        MessageEntry entry = new MessageEntry();
        entry.setSubject("eXguO");
        entry.setBody("̓eXguOłB");
        Map<Locale, MessageEntry> entryMap
            = new HashMap<Locale, MessageEntry>();
        entryMap.put(Locale.JAPANESE, entry);
        Message overview = domainFactory.prototype(new Message(null, entryMap,
                simpleUser1.getId(), simpleUser1.getId(), readAccessControl,
                writeAccessControl));
        return overview;
    }

    private BlogEntry createBlogEntry(AccessControl readAccessControl,
            AccessControl writeAccessControl) {
        MessageEntry entry = new MessageEntry();
        entry.setSubject("̋LłB");
        entry.setBody("͍̋LłB");
        Map<Locale, MessageEntry> entryMap
            = new HashMap<Locale, MessageEntry>();
        entryMap.put(Locale.JAPANESE, entry);
        Message article = domainFactory.prototype(new Message(null, entryMap,
                simpleUser1.getId(), simpleUser1.getId(), readAccessControl,
                writeAccessControl));
        return domainFactory.prototype(new BlogEntry(article));
    }

    @Test
    public void testStoreo^() throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        Blog actual = blogTable.find(blog.getId());
        Assert.assertEquals(blog, actual);
    }

    @Test(expected=BlogCodeUsedByOtherException.class)
    public void testStoreo^݂邪R[hd()
        throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        Blog blog2 = domainFactory.prototype(blog);
        blog2.setId(null);
        blog2.store();
    }

    @Test
    public void testStoreXV() throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        blog.setCode("new_" + blog.getCode());
        blog.getReadAccessControl().changeTo(FriendOnly.class);
        blog.store();
        Blog actual = blogTable.find(blog.getId());
        Assert.assertEquals(blog, actual);
    }

    @Test(expected=BlogCodeUsedByOtherException.class)
    public void testStoreXV݂邪R[hd()
        throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        Blog blog2 = domainFactory.prototype(blog);
        blog2.setId(null);
        blog2.setCode(blog2.getCode() + "2");
        blog2.store();
        blog2.setCode(blog.getCode());
        blog2.store();
    }

    @Test
    public void testRetrieveEntriesAll() throws BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        BlogEntry entry1 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blog.post(entry1);
        BlogEntry entry2 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blog.post(entry2);
        BlogEntry entry3 = createBlogEntry(blog.getReadAccessControl(), blog.getWriteAccessControl());
        blog.post(entry3);
        blog.retrieveEntries();
        Assert.assertNotNull(blog.getEntries());
        Assert.assertSame(3, blog.getEntries().size());
        Assert.assertEquals(entry3, blog.getEntries().get(0));
        Assert.assertEquals(entry2, blog.getEntries().get(1));
        Assert.assertEquals(entry1, blog.getEntries().get(2));
    }

    @Test
    public void testDelete() throws BlogCodeUsedByOtherException,
        BlogEntryExistException {
        Blog blog = createBlog();
        blog.store();
        blog.delete();
        Assert.assertNull(blogTable.find(blog.getId()));
    }

    @Test(expected=BlogEntryExistException.class)
    public void testDeleteuOL݂() throws BlogEntryExistException,
        BlogCodeUsedByOtherException {
        Blog blog = createBlog();
        blog.store();
        BlogEntry blogEntry = createBlogEntry(blog.getReadAccessControl(),
                blog.getWriteAccessControl());
        blog.post(blogEntry);
        blog.delete();
    }
}
