package org.unitedfront2.domain.communication;

import java.io.Serializable;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.unitedfront2.validation.HtmlValidate;
import org.unitedfront2.validation.Validate;
import org.unitedfront2.validation.ValidationException;

/**
 * bZ[W̌؃NXłB
 *
 * @author kurokkie
 *
 */
public class MessageValidator implements Serializable {

    /** R[h̐K\̃ftHg ([0-9a-z]+) */
    public static final String DEFAULT_CODE_REGEX = "[0-9a-z]+";

    /** R[hő啶񒷂̃ftHg (32) */
    public static final int DEFAULT_CODE_MAX_LENGTH = 32;

    /** ̍ő啶񒷂̃ftHg (64) */
    public static final int DEFAULT_SUBJECT_MAX_LENGTH = 64;

    /** VAԍ */
    private static final long serialVersionUID = -606068948905259522L;

    /** R[h̐K\ */
    private String codeRegex = DEFAULT_CODE_REGEX;

    /** R[h̍ő啶 */
    private int codeMaxLength = DEFAULT_CODE_MAX_LENGTH;

    /** ̍ő啶 */
    private int subjectMaxLength = DEFAULT_SUBJECT_MAX_LENGTH;

    /** bZ[We[u */
    private transient MessageTable messageTable;

    @Override
    public String toString() {
        return new ToStringBuilder(this)
            .append("subjectMaxLength", subjectMaxLength).toString();
    }

    @Override
    public boolean equals(final Object other) {
        if (!(other instanceof MessageValidator)) {
            return false;
        }
        MessageValidator castOther = (MessageValidator) other;
        return new EqualsBuilder()
            .append(subjectMaxLength, castOther.subjectMaxLength).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
            .append(subjectMaxLength).toHashCode();
    }

    /**
     * R[h؂܂B<p>
     *
     * ؓe
     * <ul>
     *   <li>K{</li>
     *   <li>ő啶</li>
     *   <li>K\</li>
     * </ul>
     *
     * @param message bZ[W
     * @throws ValidationException {@link ValidationException}
     */
    public void validateCode(Message message) throws ValidationException {
        String code = message.getCode();
        Validate.notBlank(code);
        Validate.maxLength(code, codeMaxLength);
        Validate.match(code, codeRegex);
        if (!message.identify(messageTable.findByCode(code))) {
            throw new ValidationException(
                    "message.MessageCodeUsedByOtherException",
                    new Object[] {code},
                    "The code '" + code + "' is used by other.");
        }
    }

    /**
     * Gg}bv̑SĂ̗vf؂܂B<p>
     *
     * ؓe
     * <ul>
     *   <li>̍ő啶</li>
     *   <li>{̍ő啶</li>
     *   <li>{ HTML ̏ƂȂĂ</li>
     * </ul>
     *
     * @param message bZ[W
     * @throws ValidationException ؗO
     * @see HtmlValidate#htmlText(String)
     */
    public void validateEntryMap(Message message)
        throws ValidationException {
        Map<Locale, MessageEntry> entryMap = message.getEntryMap();
        Validate.notNull(entryMap);
        for (MessageEntry entry : entryMap.values()) {
            validateSubject(entry.getSubject());
            validateBody(entry.getBody());
        }
    }

    /**
     * ؂܂B<p>
     *
     * ؓe: ̍ő啶
     *
     * @param subject 
     * @throws ValidationException ؗO
     */
    public void validateSubject(String subject) throws ValidationException {
        if (subject == null) {
            return;
        }
        Validate.maxLength(subject, subjectMaxLength);
    }

    /**
     * {؂܂B<p>
     *
     * ؓe
     * <ul>
     *   <li>{̍ő啶</li>
     *   <li>{ HTML ̏ƂȂĂ</li>
     * </ul>
     *
     * @param body {
     * @throws ValidationException ؗO
     */
    public void validateBody(String body) throws ValidationException {
        if (body == null) {
            return;
        }
        HtmlValidate.htmlText(body);
    }

    public String getCodeRegex() {
        return codeRegex;
    }

    public void setCodeRegex(String codeRegex) {
        this.codeRegex = codeRegex;
    }

    public int getCodeMaxLength() {
        return codeMaxLength;
    }

    public void setCodeMaxLength(int codeMaxLength) {
        this.codeMaxLength = codeMaxLength;
    }

    public int getSubjectMaxLength() {
        return subjectMaxLength;
    }

    public void setSubjectMaxLength(int subjectMaxLength) {
        this.subjectMaxLength = subjectMaxLength;
    }

    public void setMessageTable(MessageTable messageTable) {
        this.messageTable = messageTable;
    }
}
