package org.unitedfront2.web.controller.blog;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.validation.DataBinder;
import org.springframework.validation.Errors;
import org.springframework.web.util.HtmlUtils;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;
import org.unitedfront2.domain.AccountTable;
import org.unitedfront2.domain.SimpleUser;
import org.unitedfront2.domain.User;
import org.unitedfront2.domain.communication.Blog;
import org.unitedfront2.domain.communication.BlogEntry;
import org.unitedfront2.domain.communication.BlogEntryTable;
import org.unitedfront2.domain.communication.BlogTable;
import org.unitedfront2.domain.communication.Comment;
import org.unitedfront2.web.WebUtils;
import org.unitedfront2.web.controller.FormAction;
import org.unitedfront2.web.flow.PageNotFoundException;
import org.unitedfront2.web.handler.TitleInterceptor;

/**
 * uOLƂ̃Rg̉{ƁARgeANVNXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "blog.entryViewAction")
public class EntryViewAction extends FormAction {

    /** AJEge[u */
    @Autowired private AccountTable accountTable;

    /** uOe[u */
    @Autowired private BlogTable blogTable;

    /** uOLe[u */
    @Autowired private BlogEntryTable blogEntryTable;

    @Override
    protected void initAction() {
        super.initAction();
        setFormObjectClass(Comment.class);
        setMailTemplateName("blog.BlogComment-mail");
    }

    @Override
    protected void initBinder(RequestContext context, DataBinder binder) {
        super.initBinder(context, binder);
        binder.setAllowedFields(new String[] {"text"});
    }

    /**
     * uOLƃRge\܂B
     *
     * @param context {@link RequestContext}
     * @return Cxg
     * @throws Exception {@link Exception}
     * @throws PageNotFoundException uO܂̓uOL݂Ȃ
     */
    @Override
    public Event setupForm(RequestContext context)
        throws PageNotFoundException, Exception {
        super.setupForm(context);

        // uO̎擾
        Blog blog = Utils.findAndReadAccessByCode(context, this, blogTable,
                "blogCode");
        blog.retrieveOwner();
        blog.getOwner().retrieveProfile();

        // Rge
        SimpleUser user = WebUtils.findUser(context);
        boolean canComment;
        if (user == null) {
            canComment = blog.canComment();
        } else {
            canComment = blog.canComment(user.getId());
        }

        // uOL擾
        BlogEntry entry = Utils.findAndReadAccessByCode(context, this,
                blogEntryTable, "entryCode");
        entry.retrieveOwner();
        entry.retrieveCommenterProfiles();

        // 
        if (user != null) {
            blog.getOwner().getProfile().footmark(user.getId());
        }

        context.getFlowScope().put("blog", blog);
        context.getFlowScope().put("entry", entry);

        context.getRequestScope().put("canComment", canComment);
        context.getRequestScope().put("blog", blog);
        context.getRequestScope().put("entry", entry);
        context.getRequestScope().put(TitleInterceptor.TITLE_CODE_PARAM_NAME,
                "blog.EntryView");
        context.getRequestScope().put(TitleInterceptor.TITLE_ARGS_PARAM_NAME,
                new Object[] {
                    HtmlUtils.htmlEscape(entry.getEntry().getSubject(
                            WebUtils.getLocale(context))),
                });
        return success();
    }

    @Override
    protected void doValidate(RequestContext context, Object formObject,
            Errors errors) throws Exception {
        super.doValidate(context, formObject, errors);

        // `FbN
        Blog blog = (Blog) context.getFlowScope().get("blog");
        User user = WebUtils.findUser(context);
        if (user == null) {
            blog.commentAccess();
        } else {
            blog.commentAccess(user.getId());
        }
    }

    /**
     * [Uł邩ǂ肵܂B
     *
     * @param context {@link RequestContext}
     * @return Cxg
     */
    public Event isAnonymous(RequestContext context) {
        context.getFlowScope().put("isAnonymous",
                WebUtils.findUser(context) == null);
        return success();
    }

    /**
     * Rg𓊍e܂B
     *
     * @param context {@link RequestContext}
     * @return Cxg
     */
    public Event postComment(RequestContext context) {
        Comment comment = (Comment) getFormObject(context);

        // e҂̐ݒB[Ủ\
        User user = WebUtils.findUser(context);
        if (user != null) {
            comment.setCommenterId(user.getId());
        }

        BlogEntry entry = (BlogEntry) context.getFlowScope().get("entry");
        entry.post(comment);

        Blog blog = (Blog) context.getFlowScope().get("blog");
        if (blogTable.isBlogCommentForwarding(blog.getOwnerId())) {
            // []L
            String commenterName;
            if (comment.getCommenter() == null) {
                // 
                commenterName = WebUtils.getMessage("anonymous", null, context);
            } else {
                commenterName = comment.getCommenter().getName();
            }

            context.getRequestScope().put("ownerAccount",
                    accountTable.get(blog.getOwnerId()));
            context.getRequestScope().put("blog", blog);
            context.getRequestScope().put("entry", entry);
            context.getRequestScope().put("comment", comment);
            context.getRequestScope().put("commenterName", commenterName);
            context.getRequestScope().put("locale",
                    WebUtils.getLocale(context));

            doSendMail(context);
        }

        context.getFlowScope().put("comment", comment);
        return success();
    }
}
