﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Text;
using System.Collections;
using System.Runtime.InteropServices;
using System.Diagnostics;

namespace TouchToKey
{
    static class Program
    {
        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        [STAThread]
        static void Main()
        {
            //Application.EnableVisualStyles();
            //Application.SetCompatibleTextRenderingDefault(false);
            Application.Run(new Form1());
        }
    }

    // 設定ファイルの読み込み
    public class Config
    {
        private string iniFileName;
        public struct AREA
        {
            public string name;
            public string processName;
            public string className;
            public string left;
            public string top;
            public string right;
            public string bottom;
            public string action;
            public string key;
            public string breakEvent;
        }
        public ArrayList areas;
        public string minClickLength;
        public string maxClickLength;
        public string minDragLength;
        public string maxDragLength;
        public string minLongDragLength;
        public string maxLongDragLength;
        public string swipingStepLength;

        #region Windows API

        [DllImport("KERNEL32.DLL")]
        public static extern uint GetPrivateProfileString(string lpAppName, string lpKeyName, string lpDefault, StringBuilder lpReturnedString, uint nSize, string lpFileName);

        [DllImport("KERNEL32.DLL", EntryPoint = "GetPrivateProfileStringA")]
        public static extern uint GetPrivateProfileStringByByteArray(string lpAppName, string lpKeyName, string lpDefault, byte[] lpReturnedString, uint nSize, string lpFileName);

        [DllImport("KERNEL32.DLL")]
        public static extern uint GetPrivateProfileInt(string lpAppName, string lpKeyName, int nDefault, string lpFileName);

        #endregion

        // iniファイルから指定セクションの指定キーの値を取得する
        private string GetValue(string appName, string keyName, string defaultValue)
        {
            StringBuilder sb = new StringBuilder(1024);
            GetPrivateProfileString(appName, keyName, defaultValue, sb, (uint)sb.Capacity, iniFileName);
            return sb.ToString();
        }
        // iniファイルからセクションの一覧を取得する
        private string[] GetSections()
        {
            byte[] arr = new byte[1024];
            uint resultSize = GetPrivateProfileStringByByteArray(null, null, "-", arr, (uint)arr.Length, iniFileName);
            string result = Encoding.Default.GetString(arr, 0, (int)resultSize - 1);
            return result.Split('\0');
        }

        // コンストラクタ：iniファイルを読み込み
        public Config()
        {
            iniFileName = AppDomain.CurrentDomain.BaseDirectory + "TouchToKey.ini";

            // 全体設定
            minClickLength = GetValue("Common", "MinClickLength", "0");
            maxClickLength = GetValue("Common", "MaxClickLength", "8");
            minDragLength = GetValue("Common", "MinDragLength", "32");
            maxDragLength = GetValue("Common", "MaxDragLength", "50%");
            minLongDragLength = GetValue("Common", "MinLongDragLength", maxDragLength);
            maxLongDragLength = GetValue("Common", "MaxLongDragLength", "100%");
            swipingStepLength = GetValue("Common", "SwipingStepLength", "48");

            // 領域設定
            areas = new ArrayList();
            string[] sections = GetSections();
            foreach (string section in sections)
            {
                if (section == "Common") continue;

                AREA area = new AREA();
                area.name = section;
                Debug.WriteLine("Section: " + section);

                area.processName = GetValue(section, "Process", "");
                Debug.WriteLine(section + ".Process: " + area.processName);

                area.className = GetValue(section, "Class", "");
                Debug.WriteLine(section + ".Class: " + area.className);

                area.left = GetValue(section, "Left", "");
                Debug.WriteLine(section + ".Left: " + area.left);

                area.top = GetValue(section, "Top", "");
                Debug.WriteLine(section + ".Top: " + area.top);

                area.bottom = GetValue(section, "Bottom", "");
                Debug.WriteLine(section + ".Bottom: " + area.bottom);

                area.right = GetValue(section, "Right", "");
                Debug.WriteLine(section + ".Right: " + area.right);

                area.action = GetValue(section, "Action", "");
                Debug.WriteLine(section + ".Action: " + area.action);

                area.key = GetValue(section, "Key", "");
                area.key = area.key.ToUpper().Replace("{SPACE}", " ");
                Debug.WriteLine(section + ".Key: " + area.key);

                area.breakEvent = GetValue(section, "Break", "FALSE").ToUpper();
                Debug.WriteLine(section + ".Break: " + area.breakEvent);

                // 設定が有効か確認する
                if (area.processName.Length > 0 && area.className.Length >= 0 &&
                    area.left.Length > 0 && area.top.Length > 0 &&
                    area.right.Length > 0 && area.bottom.Length > 0 &&
                    area.action.Length > 0 && area.key.Length > 0)
                {
                    areas.Add(area);
                    Debug.WriteLine("Section " + section + " is valid.");
                }
                else
                {
                    Debug.WriteLine("Section " + section + " is invalid.");
                }
            }
        }
    }

    // 別プロセスのウィンドウ関連処理
    public class Window
    {
        #region WindowsAPI

        [StructLayout(LayoutKind.Sequential)]
        public struct RECT
        {
            public int left;
            public int top;
            public int right;
            public int bottom;
        }
        [DllImport("User32.Dll")]
        static extern int GetWindowRect(IntPtr hWnd, out RECT rect);

        [DllImport("User32.Dll")]
        static extern IntPtr GetDesktopWindow();

        [DllImport("user32.dll")]
        static extern IntPtr GetForegroundWindow();

        [DllImport("User32.Dll", CharSet = CharSet.Unicode)]
        static extern int GetClassName(IntPtr hWnd, StringBuilder s, int nMaxCount);

        [DllImport("user32.dll")]
        static extern uint GetWindowThreadProcessId(IntPtr hWnd, ref uint lpdwProcessId);

        [DllImport("user32")]
        static extern bool IsWindowVisible(IntPtr hWnd);

        #endregion

        public struct INFO
        {
            public string processName;
            public string className;
            public RECT rect;
        }
        public INFO info;

        // コンストラクタ：ウィンドウ情報を格納する
        public Window(IntPtr hWnd)
        {
            uint pid = 0;
            GetWindowThreadProcessId(hWnd, ref pid);
            Process p = Process.GetProcessById((int)pid);
            // プロセス名
            info.processName = p.ProcessName;
            // クラス名
            StringBuilder sb = new StringBuilder(1024);
            GetClassName(hWnd, sb, sb.Capacity);
            info.className = sb.ToString();
            // 位置
            RECT rect = new RECT();
            GetWindowRect(hWnd, out info.rect);
        }

        // フォアグラウンドウィンドウのウィンドウハンドルを取得する
        public static IntPtr GetForegroundWindowHandle()
        {
            return GetForegroundWindow();
        }

        // 設定で指定された値をピクセルに変換する
        public static int ToPixcel(string strValue, int intLength)
        {
            int len = strValue.Length;
            int px;
            double per;
            // "px" または 単位なし = ピクセル
            strValue = strValue.ToLower().Replace("px", "");
            if (int.TryParse(strValue, out px))
            {
                return px;
            }
            // "%" = パーセント
            if (len >= 2 && strValue.Substring(len - 1) == "%" &&
                double.TryParse(strValue.Substring(0, len - 1), out per))
            {
                // 最大値を100%までに制限する
                if (per > 100) per = 100;
                if (per < -100) per = -100;
                return (int)((double)(intLength) * (per / 100));
            }
            // 不正な値の場合は0を返す
            return 0;
        }
        public static int ToScreenX(string strValue, int intLeft, int intRight)
        {
            int width = ToPixcel(strValue, intRight - intLeft);
            // 数値がマイナスの場合は右側基準
            return (width >= 0) ? intLeft + width : intRight + width;
        }
        public static int ToScreenY(string strValue, int intTop, int intBottom)
        {
            int height = ToPixcel(strValue, intBottom - intTop);
            // 数値がマイナスの場合は下側基準
            return (height >= 0) ? intTop + height : intBottom + height;
        }
    }
}
