/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/IntraDayTradeRef.cs#10 $
 * $DateTime: 2008/03/13 13:20:43 $
 * 
 */
using System;
using System.Diagnostics;

using IntraDayTrade=Bellagio.Data.IntraDayTrade;
using DateBasedQuoteSet=Bellagio.Data.DateBasedQuoteSet;
using CurrentPriceInfo=Bellagio.Data.CurrentPriceInfo;
using AbstractStockProfile=Bellagio.Data.AbstractStockProfile;
using RealtimeInfoStringI=Bellagio.Data.RealtimeInfoString<int>;

namespace Bellagio.Values {
    public class IntraDayTradeRef : BV {
        private IntraDayTrade _IntraDayTrade;

        public override bool IsNil {
            get {
                return _IntraDayTrade==null;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.IntraDayTrade(this);
        }
        public override void Let(BV value) {
            _IntraDayTrade = ((IntraDayTradeRef)value)._IntraDayTrade;
        }
        public override BT BT {
            get { return IntraDayTradeType.instance; }
        }

        public IntraDayTrade Value {
            get {
                return _IntraDayTrade;
            }
            set {
                _IntraDayTrade = value;
            }
        }
    }

    public class IntraDayTradeType : BObjectT {
        private IntraDayTradeType()
            : base("IntraDayTrade", new IntraDayTradeRef()) {
        }

        public static IntraDayTradeType instance;
        static IntraDayTradeType() {
            instance = new IntraDayTradeType();
        }
        public override BV CreateInstance() {
            return new IntraDayTradeRef();
        }

        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT isIntraDayTrade = instance;
            BT[] noArg = BuiltInFunctionLibrary.noArg;
            lib.DefineInternalFunction("price", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetLastPrice));
            lib.DefineInternalFunction("open", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetOpenPrice));
            lib.DefineInternalFunction("high", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetHighPrice));
            lib.DefineInternalFunction("low", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetLowPrice));
            lib.DefineInternalFunction("creditlong", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetLastCreditLong));
            lib.DefineInternalFunction("creditshort", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetLastCreditShort));
            lib.DefineInternalFunction("lastclose", isIntraDayTrade, BT.Double, noArg, new BInternalExecution(GetLastClose));
            lib.DefineInternalFunction("minquotes", isIntraDayTrade, QuoteType.instance, noArg, new BInternalExecution(GetMinutelyIntraDayTrades));
            lib.DefineInternalFunction("ticks", isIntraDayTrade, TimeAndSalesType.instance, noArg, new BInternalExecution(GetTicks));
            lib.DefineInternalFunction("dayquotes", isIntraDayTrade, QuoteType.instance, noArg, new BInternalExecution(GetDayQuotes));

            //̊֐͂ł̂
            lib.DefineInternalFunction("code", null, isIntraDayTrade, new BT[] { BT.String }, new BInternalExecution(GetIntraDayTrade));
        }
        private static ExecResult GetLastPrice(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef IntraDayTrade = (IntraDayTradeRef)target;
            //TODO 򂫂Ȃ IntraDayTradeɂ̃R[h͈ړI
            IntraDayTrade rs = IntraDayTrade.Value as IntraDayTrade;
            if(rs==null)
                return ExecResult.Nil;
            else if(extra.CurrentTime==null) { //\Ȍ̍ŐV
                ((BDouble)result).Value = rs.CurrentPriceInfo.CurrentPrice.Value;
            }
            else { //w
                Quote q = rs.Minutely;
                if(rs.IsLastMinuteCandleAvailable && extra.CurrentTime.AsMinutelyInt()==BTime.AsMinutelyInt(rs.LastMinuteCandle.Time)) { //CurrentS{lƓƂAIntraDayTradeɂ͓R܂܂ĂȂ
                    ((BDouble)result).Value = rs.LastMinuteCandle.Open;
                }
                else if(q.IsNil)
                    return ExecResult.Nil;
                else {
                    int index = q.FindExactly(extra.CurrentTime.AsInt());
                    if(index==-1)
                        return ExecResult.Nil;
                    else
                        ((BDouble)result).Value = q.CandleAt(index).Open; //Y̎nl
                }
            }

            return ExecResult.OK;
        }
        private static ExecResult GetLastClose(BV target, EvalContext extra, BV[] args, BV result) {
            return GetCurrentPriceValue(target, delegate(CurrentPriceInfo cp) { return cp.LastClose; }, result);
        }
        private static ExecResult GetOpenPrice(BV target, EvalContext extra, BV[] args, BV result) {
            return GetCurrentPriceValue(target, delegate(CurrentPriceInfo cp) { return cp.OpenPrice; }, result);
        }
        private static ExecResult GetHighPrice(BV target, EvalContext extra, BV[] args, BV result) {
            return GetCurrentPriceValue(target, delegate(CurrentPriceInfo cp) { return cp.HighPrice; }, result);
        }
        private static ExecResult GetLowPrice(BV target, EvalContext extra, BV[] args, BV result) {
            return GetCurrentPriceValue(target, delegate(CurrentPriceInfo cp) { return cp.LowPrice; }, result);
        }
        private delegate RealtimeInfoStringI CurrentPriceValueDelegate(CurrentPriceInfo cp);
        private static ExecResult GetCurrentPriceValue(BV target, CurrentPriceValueDelegate getter, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade dt =  dtref.Value;
            if(dt==null)
                return ExecResult.Nil;
            else {
                RealtimeInfoStringI v = getter(dt.CurrentPriceInfo);
                if(v.IsEmpty)
                    return ExecResult.Nil;
                else
                    ((BDouble)result).Value = v.Value;
            }
            return ExecResult.OK;
        }

        private static ExecResult GetLastCreditLong(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade dt =  dtref.Value;
            if(dt==null)
                return ExecResult.Nil;
            else {
                CurrentPriceInfo cp = dt.CurrentPriceInfo;
                if(cp.CreditLongVolume.IsEmpty)
                    return ExecResult.Nil;
                else
                    ((BDouble)result).Value = cp.CreditLongVolume.Value;
            }
            return ExecResult.OK;
        }
        private static ExecResult GetLastCreditShort(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade dt =  dtref.Value;
            if(dt==null)
                return ExecResult.Nil;
            else {
                CurrentPriceInfo cp = dt.CurrentPriceInfo;
                if(cp.CreditShortVolume.IsEmpty)
                    return ExecResult.Nil;
                else
                    ((BDouble)result).Value = cp.CreditShortVolume.Value;
            }
            return ExecResult.OK;
        }
        private static ExecResult GetMinutelyIntraDayTrades(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade rs = dtref.Value as IntraDayTrade;
            if(rs==null)
                return ExecResult.Nil;
            else if(extra.CurrentTime==null) //\Ȍ̍ŐV
                ((SubseqQuote)result).Set(rs.Minutely);
            else if(rs.IsLastMinuteCandleAvailable && extra.CurrentTime.AsMinutelyInt()==BTime.AsMinutelyInt(rs.LastMinuteCandle.Time)) //JgS{l̎̂Ƃ
                ((SubseqQuote)result).Set(rs.Minutely);
            else {
                Quote q = rs.Minutely;
                if(q.IsNil || q.Count==0)
                    return ExecResult.Nil;
                else if(q.LastCandle.Time < extra.CurrentTime.AsInt())
                    ((SubseqQuote)result).Set(rs.Minutely);
                else {
                    int index = q.FindExactly(extra.CurrentTime.AsMinutelyInt());
                    if(index==-1)
                        return ExecResult.Nil;
                    else
                        ((SubseqQuote)result).Set(rs.Minutely, 0, index+1);
                }
            }
            return ExecResult.OK;
        }
        //ReLXgԂ݂ăTuZbgT&SԂ悤ɂȂĂׂڂ
        private static ExecResult GetTicks(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade rs = dtref.Value as IntraDayTrade;
            if(rs==null)
                return ExecResult.Nil;
            else
                result = rs.TimeAndSales;
            return ExecResult.OK;
        }
        private static ExecResult GetDayQuotes(BV target, EvalContext extra, BV[] args, BV result) {
            IntraDayTradeRef dtref = (IntraDayTradeRef)target;
            IntraDayTrade rs = dtref.Value as IntraDayTrade;
            if(rs==null)
                return ExecResult.Nil;
            else {
                DateBasedQuoteSet qs = BellagioRoot.DateBasedQuoteProvider.Lookup(rs.Stock);
                result = qs==null? QuoteType.instance.Nil : qs.GetOrCreate(Quote.QuoteUnit.Daily, Bellagio.Data.SplitReflection.Reflected);
            }
            return ExecResult.OK;
        }


        //ǂ̃R[hQƂĂ邩𒲂ׂP[X̂EvalContext甲
        private static ExecResult GetIntraDayTrade(BV target, EvalContext extra, BV[] args, BV result) {
            BString code = args[0] as BString;
            Debug.Assert(code!=null);
            EvalSpecial_CodeCollector c = extra.EvalSpecial_CodeCollector;
            if(c!=null) { //R[hW
                c.AddStock(code.Value);
                //NilԂĂ܂ƕ]̃V[gJbĝߑ̖QƂȂB
                IntraDayTrade dt = new IntraDayTrade(null);
                dt.Reset();
                ((IntraDayTradeRef)result).Value = dt;
            }
            else {
                AbstractStockProfile stock = BellagioRoot.GlobalStockCollection.FindExact(code.Value);
                Debug.Assert(stock!=null);
                IntraDayTrade v = BellagioRoot.IntraDayTradeProvider.Lookup(stock.Primary);
                if(v==null)
                    return ExecResult.Nil;
                else
                    ((IntraDayTradeRef)result).Value = v;
            }
            return ExecResult.OK;
        }
    }
}
