/*
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Travis/OriginReportableText.cs#10 $
 * $DateTime: 2008/01/18 19:36:59 $
 * 
 * ؍\eLXg(XMLȂ)IuWFNgւ̃}bsOtNVɂčsB
 * eLXgɃG[Ƃɒʒmo悤ɂȂĂ̂B
 * 
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Xml;
using System.IO;
using System.Diagnostics;
using System.Drawing;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Travis.ORT {
    //ORTextIuWFNgƂĎQƂ
    public interface IORText {
        string Value { get; }
        int Line { get; }
        int Column { get; }
        ObjectOrigin ContainerLocation { get; }
        bool IsOmitted { get; }
        FieldInfo Field { get; } //ڂFieldInfo͂ƋC邢Ȃ

        string LocationString { get; }

        //Color ParseColorInORText(string elem_value);

    }

    //Ńt@C̃|WVQƂł@\AėpIݒt@CǂݏpNX
    //OriginReportableObject̔hNX̃oƂēoꂷBRXgN^newȂĂ悤struct
    public struct ORText : IORText {
        private FieldInfo _field;
        private ObjectOrigin _containerLocation;
        private int _line;
        private int _column;
        private string _value;

        public ORText(FieldInfo field, ObjectOrigin parent, int line, int col, string value) {
            _field = field;
            _containerLocation = parent;
            _line = line;
            _column = col;
            _value = value;
        }
        public ORText(FieldInfo field, ObjectOrigin parent) { //ȗĂƂ̃RXgN^
            _field = field;
            _containerLocation = parent;
            _line = 0;
            _column = 0;
            _value = null;
        }
        public int Line {
            get {
                return _line;
            }
        }
        public int Column {
            get {
                return _column;
            }
        }
        public string Value {
            get {
                return _value;
            }
        }
        public ObjectOrigin ContainerLocation {
            get {
                return _containerLocation;
            }
        }
        public FieldInfo Field {
            get {
                return _field;
            }
        }
        public bool IsOmitted {
            get {
                return _value==null;
            }
        }
        public string LocationString {
            get {
                if(this.IsOmitted)
                    return "";
                else if(_containerLocation==null)
                    return String.Format("{0}s {1}", _line, _column);
                else
                    return String.Format("t@C {0} {1}s {2}", _containerLocation.FileName, _line, _column);
            }
        }

        #region parsing
        public string ParseMandatoryString() {
            if(this.IsOmitted)
                throw new FormatException(FormatMissingElementMessage(this));
            else
                return this.Value;
        }
        public string ParseOptionalString(string defaultvalue) {
            if(this.IsOmitted)
                return defaultvalue;
            else
                return this.Value;
        }

        //p[Xł邩A̓G[Ƃ^Cv
        public int ParseMandatoryInt() {
            Debug.Assert(!this.IsOmitted);
            return ParseIntInORText(this.Value);
        }
        public int ParseOptionalInt(int defaultvalue) {
            if(this.IsOmitted || _value.Length==0)
                return defaultvalue;
            else
                return ParseMandatoryInt();
        }

        public int ParseIntInORText(string elem_value) {
            int x;
            if(Int32.TryParse(elem_value, out x))
                return x;
            else
                throw new FormatException(FormatMulformedElementMessage(this, ""));
        }
        public double ParseMandatoryDouble() {
            double x;
            if(Double.TryParse(this.Value, out x))
                return x;
            else
                throw new FormatException(FormatMulformedElementMessage(this, "l"));
        }
        public double ParseOptionalDouble(double def) {
            if(this.IsOmitted || _value.Length==0)
                return def;
            else
                return ParseMandatoryDouble();
        }
        public long ParseMandatoryLong() {
            long x;
            if(Int64.TryParse(this.Value, out x))
                return x;
            else
                throw new FormatException(FormatMulformedElementMessage(this, ""));
        }
        public long ParseOptionalLong(long def) {
            if(this.IsOmitted || _value.Length==0)
                return def;
            else
                return ParseMandatoryLong();
        }
        public bool ParseOptionalBool(bool def) {
            string t = ParseOptionalString("");
            if(t=="true")
                return true;
            else if(t=="false")
                return false;
            else
                return def;
        }
        public Color ParseOptionalColor(Color defaultvalue) {
            if(this.IsOmitted)
                return defaultvalue;
            else
                return ParseColorOrException();
        }
        public Color ParseOptionalColor(string defaultvalue) {
            if(this.IsOmitted)
                return ParseColorInORText(defaultvalue);
            else
                return ParseColorInORText(this.Value);
        }
        public Color ParseColorOrException() {
            Debug.Assert(!this.IsOmitted);
            return ParseColorInORText(this.Value);
        }
        public Color ParseColorInORText(string elem_value) {
            Color result;
            if(Bellagio.Drawing.DrawingUtil.TryParseColor(elem_value, out result))
                return result;
            else
                throw new FormatException(FormatMulformedElementMessage(this, elem_value, "F"));
        }

        //G[bZ[W 
        public static string FormatMissingElementMessage(IORText text) {
            Debug.Assert(text.IsOmitted);
            ObjectOrigin or = text.ContainerLocation;
            //K{vf݂Ȃ
            return String.Format("t@C {0}\ns {1}  {2}\nK{̗vf {3} ܂B",
                or.FileName, or.Line, or.Column, text.Field.Name);
        }
        //݂`s
        public static string FormatMulformedElementMessage(IORText text, string typename) {
            Debug.Assert(!text.IsOmitted);
            if(text.Value.Length > 0)
                return FormatMulformedElementMessage(text, text.Value, typename);
            else
                return String.Format("t@C {0}\ns {1}  {2}\n{3}ƂĐ`^Kv܂B", text.ContainerLocation.FileName, text.Line, text.Column, typename);
        }
        public static string FormatMulformedElementMessage(IORText text, string elem_value, string typename) {
            Debug.Assert(!text.IsOmitted);
            return String.Format("t@C {0}\ns {1}  {2}\n{3} {4}ƂĕsȌ`łB", text.ContainerLocation.FileName, text.Line, text.Column, elem_value, typename);
        }

        #endregion
    }

    //m[h̊NX
    public class OriginReportableObject {
        public ObjectOrigin _location;
    }

    //tB[hɃAgr[g錾悤ɂĂANXɂPƂ邵
    [AttributeUsage(AttributeTargets.Class, AllowMultiple=false)]
    public class ORTextValueFieldNameAttribute : Attribute {
        private string _name;
        public ORTextValueFieldNameAttribute(string fn) {
            _name = fn;
        }
        public string Name {
            get {
                return _name;
            }
        }
    }

    //ʒu
    public sealed class ObjectOrigin {
        private string _filename;
        private int _line;
        private int _column;
        public ObjectOrigin(string filename, int line, int col) {
            _filename = filename;
            _line = line;
            _column = col;
        }
        public string FileName {
            get {
                return _filename;
            }
        }
        public int Line {
            get {
                return _line;
            }
        }
        public int Column {
            get {
                return _column;
            }
        }
    }
    //RNVvfׂC^[tF[X
    public interface IORCollectionMember : IEnumerable {
        void Clear();
        void Add(object item);
        int Count { get; }
    }
    public interface IORCollectionMemberT<T> : IORCollectionMember {
        void Add(T item);
        T this[int index] { get; }
    }

    //̃C^tF[X̕W@
    public class ORCollectionMember<T> : IORCollectionMemberT<T> {
        protected List<T> _data;
        public ORCollectionMember() {
            _data = new List<T>();
        }
        public int Count {
            get {
                return _data.Count;
            }
        }
        public void Clear() {
            _data.Clear();
        }
        public void Add(object item) {
            _data.Add((T)item);
        }
        public void Add(T item) {
            _data.Add(item);
        }
        public T this[int index] {
            get {
                return _data[index];
            }
        }
        public bool Contains(T elem) {
            return _data.Contains(elem);
        }
        public IEnumerator GetEnumerator() {
            return _data.GetEnumerator();
        }
    }

    //ConfigTypeInfõRNV
    public class ORTypeInfoCollection {
        private Dictionary<Type, ORTypeInfo> _data;
        public ORTypeInfoCollection() {
            _data = new Dictionary<Type, ORTypeInfo>();
        }
        //ǉ͂ǂłOk
        public void Add(Type k) {
            ORTypeInfo ti = new ORTypeInfo(this, k);
            Debug.Assert(!_data.ContainsKey(k)); //do^͂
            _data.Add(k, ti);
        }
        public void Add(ORTypeInfo ti) {
            Type k = ti.TargetType;
            Debug.Assert(!_data.ContainsKey(k)); //do^͂
            _data.Add(k, ti);
        }
        public ORTypeInfo Find(Type t) {
            ORTypeInfo s = null;
            _data.TryGetValue(t, out s);
            return s;
        }
        public int Count {
            get {
                return _data.Count;
            }
        }
    }

    //^̎^Bpublic^ConfigItemł郁óAvfƃöv݂ĎŃZbg.
    //܂AConfigNodeLocation^̃oɂ͗vf̎n܂t@CL^
    public class ORTypeInfo {
        private Type _targetType;
        private FieldInfo _textValueField; //XMLGg̃eLXgZbgtB[hBeLXgsvȂƂnull
        private Dictionary<string, FieldInfo> _fields; //OGgւ̃}bv RNVłĂRNVvf͎
        private Dictionary<string, ORTypeInfo> _childElements;
        private Dictionary<string, ORTypeInfo> _collections;

        public ORTypeInfo(ORTypeInfoCollection childCollection, Type t) {
            Debug.Assert(typeof(OriginReportableObject).IsAssignableFrom(t));
            _targetType = t;
            _fields = new Dictionary<string, FieldInfo>();
            _childElements = new Dictionary<string, ORTypeInfo>();
            _collections = new Dictionary<string, ORTypeInfo>();

            Type element = null;
            foreach(FieldInfo fi in t.GetFields()) {
                if(fi.FieldType==typeof(ORText))
                    _fields.Add(fi.Name, fi);
                else if(childCollection!=null && (element = FindCollectionElement(fi.FieldType))!=null && childCollection.Find(element)!=null) {
                    _collections.Add(fi.Name, childCollection.Find(element));
                    _fields.Add(fi.Name, fi);
                }
                else if(childCollection!=null && childCollection.Find(fi.FieldType)!=null) {
                    _childElements.Add(fi.Name, childCollection.Find(fi.FieldType));
                    _fields.Add(fi.Name, fi);
                }
            }

            object[] tvf = t.GetCustomAttributes(typeof(ORTextValueFieldNameAttribute), false);
            if(tvf.Length > 0) {
                _textValueField = t.GetField(((ORTextValueFieldNameAttribute)tvf[0]).Name);
            }

        }
        public FieldInfo FindField(string name) {
            FieldInfo r = null;
            bool f = _fields.TryGetValue(name, out r);
            return r;
        }
        public ORTypeInfo FindCollectionField(string name) {
            ORTypeInfo s = null;
            _collections.TryGetValue(name, out s);
            return s;
        }
        public ORTypeInfo FindElementField(string name) {
            ORTypeInfo s = null;
            _childElements.TryGetValue(name, out s);
            return s;
        }
        public object CreateInstance() {
            return _targetType.Assembly.CreateInstance(_targetType.FullName);
        }
        public Type TargetType {
            get {
                return _targetType;
            }
        }
        public FieldInfo TextValueField {
            get {
                return _textValueField;
            }
        }
        //GenericRNV^vf𔲂oB: fieldtype List<T> ̂ƂATԂ
        private static Type FindCollectionElement(Type fieldtype) {
            if(!typeof(IORCollectionMember).IsAssignableFrom(fieldtype)) return null;

            Type[] ifs = fieldtype.GetInterfaces();
            Type ior = Type.GetType("Travis.ORT.IORCollectionMemberT`1");
            Debug.Assert(ior!=null);
            foreach(Type t in ifs) {
                if(t.IsGenericType && t.GetGenericTypeDefinition()==ior) { //ZI
                    return t.GetGenericArguments()[0];
                }
            }
            return null;
        }
    }

    public abstract class ORReader {
        protected string _filename;
        protected bool _ignoreUnknownElement;
        
        public abstract void Read(ORTypeInfo schema, OriginReportableObject target);

        public abstract void Close();
        protected abstract int CurrentLine { get; }
        protected abstract int CurrentColumn { get; }

        public bool IgnoreUnknownElement {
            get {
                return _ignoreUnknownElement;
            }
            set {
                _ignoreUnknownElement = value;
            }
        }

        protected void ThrowUnexpectedEOF() {
            throw new IOException(FormatEOFMessage());
        }
        protected void ThrowUnexpectedElement(string actual) {
            throw new IOException(FormatUnexpectedElementMessage(actual));
        }
        protected void ThrowDuplicatedElement(string actual) {
            throw new IOException(FormatDuplicatedElementMessage(actual));
        }
        public string FormatEOFMessage() {
            return String.Format("t@C{0} s{1} {2} \ȂI[ł", _filename, CurrentLine, CurrentColumn);
        }
        public string FormatUnexpectedElementMessage(string actual) {
            return String.Format("t@C{0} s{1} {2} \Ȃvf{3}܂", _filename, CurrentLine, CurrentColumn, actual);
        }
        public string FormatDuplicatedElementMessage(string actual) {
            return String.Format("t@C{0} s{1} {2} dvf{3}܂", _filename, CurrentLine, CurrentColumn, actual);
        }
    }

    public class XmlORReader : ORReader {
        private XmlReader _reader;
        private IXmlLineInfo _lineInfo;

        public XmlORReader(string filename, TextReader reader)
            : this(filename, XmlReader.Create(reader)) {
        }
        public XmlORReader(string filename, StreamReader reader) : this(filename, XmlReader.Create(reader)) {
        }
        public XmlORReader(string filename, XmlReader reader) {
            _filename = filename;
            _reader = reader;
            _lineInfo = _reader as IXmlLineInfo;
            Debug.Assert(_lineInfo!=null);
        }
        public override void Close() {
            _reader.Close();
        }

        protected override int CurrentLine {
            get {
                return _lineInfo.LineNumber;
            }
        }
        protected override int CurrentColumn {
            get {
                return _lineInfo.LinePosition;
            }
        }

        //PIuWFNg𖄂߂BśA_reader͂̃IuWFNgɑΉGg̎̈ʒuwĂB
        public override void Read(ORTypeInfo schema, OriginReportableObject target) {
            Debug.Assert(schema!=null);
            Debug.Assert(target!=null);

            //check element
            ReadToElement();
            if(_reader.EOF)
                ThrowUnexpectedEOF();
          
            //m[ḧʒuL^
            target._location = new ObjectOrigin(_filename, _lineInfo.LineNumber, _lineInfo.LinePosition);

            //attr
            if(_reader.HasAttributes) {
                while(_reader.MoveToNextAttribute())
                    SetValue(_reader.LocalName, schema, target, _lineInfo.LineNumber, _lineInfo.LinePosition, _reader.Value);
                _reader.MoveToElement();
            }

            if(_reader.IsEmptyElement) {
                _reader.Read();
            }
            else {
                //qvf()
                _reader.Read();
                while(!_reader.EOF && _reader.NodeType!=XmlNodeType.EndElement) {
                    //NOTE قƂLocalNameႾ߂
                    if(_reader.NodeType==XmlNodeType.Element) {
                        string n = _reader.LocalName;
                        ORTypeInfo child = null;
                        if((child = schema.FindCollectionField(n))!=null)
                            ReadChildCollectionElement(child, n, schema, target);
                        else if((child = schema.FindElementField(n))!=null)
                            ReadChildElement(child, n, schema, target);
                        else
                            ReadValue(n, schema, target);
                    }
                    else if(_reader.NodeType==XmlNodeType.Text && schema.TextValueField!=null) //ڂ̃eLXgZbgtB[hw肳Ăꍇ
                        ReadTextValue(schema.TextValueField.Name, schema, target);
                    else
                        _reader.Read();
                }
                _reader.Read(); //I[̎vf
            }
        }

        //ElementɂƂAɑΉchildschemaǂ݁ARNVɒǉ
        private void ReadChildCollectionElement(ORTypeInfo childschema, string name, ORTypeInfo schema, OriginReportableObject target) {
            Debug.Assert(_reader.NodeType==XmlNodeType.Element);
            OriginReportableObject child = (OriginReportableObject)childschema.CreateInstance();
            Read(childschema, child);
            
            //쐬łǉ
            IORCollectionMember coll = (IORCollectionMember)schema.FindField(name).GetValue(target);
            coll.Add(child);
        }
        //ElementɂƂAq̃GgɑΉIuWFNg쐬ăZbgBRNV
        private void ReadChildElement(ORTypeInfo childschema, string name, ORTypeInfo schema, OriginReportableObject target) {
            Debug.Assert(_reader.NodeType==XmlNodeType.Element);
            OriginReportableObject child = (OriginReportableObject)childschema.CreateInstance();
            Read(childschema, child);

            //쐬łǉ
            schema.FindField(name).SetValue(target, child);
        }

        //ElementɂƂÃ^ÕeLXgǂŃtB[hɐݒ
        private void ReadValue(string name, ORTypeInfo schema, OriginReportableObject target) {
            Debug.Assert(_reader.NodeType==XmlNodeType.Element);
            if(_reader.IsEmptyElement) {
                SetValue(name, schema, target, _lineInfo.LineNumber, _lineInfo.LinePosition, "");
            }
            else {
                _reader.Read();
                int line = _lineInfo.LineNumber;
                int column = _lineInfo.LinePosition;
                string value = ReadElementValue();
                if(value!=null)
                    SetValue(name, schema, target, line, column, value);
                Debug.Assert(_reader.NodeType==XmlNodeType.EndElement);
            }

            _reader.Read();
        }
        private void ReadToElement() {
            while(_reader.NodeType!=XmlNodeType.Element) _reader.Read();
        }
        private void ReadToEndElement() {
            if(!_reader.IsEmptyElement) {
                while(_reader.NodeType!=XmlNodeType.EndElement) _reader.Read();
            }
        }
        private string ReadElementValue() {
            //Element̎Äʒuł邱ƂOɒlǂށBqvf͋Ȃ
            string result = null;
            while(true) {
                switch(_reader.NodeType) {
                    case XmlNodeType.Text:
                    case XmlNodeType.CDATA:
                    case XmlNodeType.Whitespace:
                        result = result==null? _reader.Value : result + _reader.Value;
                        break;
                    case XmlNodeType.Element:
                        ThrowUnexpectedElement(_reader.LocalName);
                        break;
                    case XmlNodeType.EndElement:
                        return result;
                }
                _reader.Read();
            }
        }
        //TextɂƂA̒lǂŃtB[hɐݒ肵Aɂ
        private void ReadTextValue(string name, ORTypeInfo schema, OriginReportableObject target) {
            Debug.Assert(_reader.NodeType==XmlNodeType.Text);
            int line = _lineInfo.LineNumber;
            int column = _lineInfo.LinePosition;
            string value = _reader.Value;
            if(value!=null)
                SetValue(name, schema, target, line, column, value);

            _reader.Read();
        }

        private void SetValue(string membername, ORTypeInfo schema, OriginReportableObject target, int line, int col, string value) {
            FieldInfo f = schema.FindField(membername);
            if(f==null) {
                if(!_ignoreUnknownElement) //ݒɂȂĂ΃X[
                    ThrowUnexpectedElement(membername);
            }
            else {
                ORText t = (ORText)f.GetValue(target);
                if(!t.IsOmitted)
                    ThrowDuplicatedElement(membername);
                else
                    f.SetValue(target, new ORText(f, target._location, line, col, value));
            }
        }
    }

    //eLXg->EnumϊȂǁAyAw肵Ăăp[X
    public class ORTextEnumParser<T>  {
        //قǐ͑ȂnbVe[uł͂ȂBpx̍ɒǉƋg
        private List<string> _texts;
        private List<T> _values;
        private string _typename;

        public ORTextEnumParser(string typename) {
            _texts = new List<string>();
            _values = new List<T>();
            _typename = typename;
        }
        public ORTextEnumParser<T> Def(string text, T value) {
            _texts.Add(text);
            _values.Add(value);
            return this;
        }
        public T Parse(ref ORText value) {
            string t = value.ParseMandatoryString();
            for(int i=0; i<_texts.Count; i++)
                if(t==_texts[i]) return _values[i];
            throw new FormatException(ORText.FormatMulformedElementMessage(value, _typename));
        }
        public T Parse(string value) {
            for(int i=0; i<_texts.Count; i++)
                if(value==_texts[i]) return _values[i];
            throw new FormatException(String.Format("{0} is invalid for {1}", value, _typename));
        }
        public string Format(T value) {
            for(int i=0; i<_texts.Count; i++)
                if(value.Equals(_values[i])) return _texts[i];
            throw new FormatException(String.Format("Unexpected value {0} for {1}", value, _typename));
        }
    }


#if UNITTEST

    [TestFixture]
    public class XmlConfigReadingTests {
        public class Setting1 : OriginReportableObject {
            public ORText a;
            public ORText b;
        }
        public class Setting2 : OriginReportableObject {
            public ORText a;
            //Xg
            public ORCollectionMember<Setting1> s;

            //RXgN^ŃRNV͍쐬Ă邱Ƃ
            public Setting2() {
                s = new ORCollectionMember<Setting1>();
            }
        }

        [ORTextValueFieldName("text")]
        public class TextValueFieldTest : OriginReportableObject {
            public ORText a;
            public ORText text;
        }
        public class NotCollectionMember : OriginReportableObject {
            public TextValueFieldTest t;
        }

        private ORTypeInfo _setting1;
        private ORTypeInfo _setting2;
        private ORTypeInfo _textValue;
        private ORTypeInfo _notCollectionMember;

        [TestFixtureSetUp]
        public void CreateSchema() {
            ORTypeInfoCollection sc = new ORTypeInfoCollection();
            _setting1 = new ORTypeInfo(sc, typeof(Setting1));
            sc.Add(_setting1);
            _setting2 = new ORTypeInfo(sc, typeof(Setting2));
            sc.Add(_setting2);
            _textValue = new ORTypeInfo(sc, typeof(TextValueFieldTest));
            sc.Add(_textValue);
            _notCollectionMember = new ORTypeInfo(sc, typeof(NotCollectionMember));
            sc.Add(_notCollectionMember);
        }
        [Test]
        public void Simple1() {
            XmlORReader r = new XmlORReader("f", new StringReader("<s><a>123</a><b>456</b>  </s>"));
            Setting1 s = new Setting1();

            r.Read(_setting1, s);

            Assert.IsNotNull(s._location);
            Assert.AreEqual("f", s._location.FileName);
            Assert.AreEqual(1, s._location.Line);
            Assert.AreEqual(2, s._location.Column);

            Assert.IsNotNull(s.a);
            Assert.IsNotNull(s.b);
            Assert.AreEqual("123", s.a.Value);
            Assert.AreEqual(1, s.a.Line);
            Assert.AreEqual(7, s.a.Column);
            Assert.AreEqual("456", s.b.Value);
            Assert.AreEqual(1, s.b.Line);
            Assert.AreEqual(17, s.b.Column);
        }
        [Test]
        public void Simple2() {
            //Agr[gx[X
            XmlORReader r = new XmlORReader("f", new StringReader("<s a=\"123\" b=\"456\"/>"));
            Setting1 s = new Setting1();

            r.Read(_setting1, s);

            Assert.IsNotNull(s._location);
            Assert.AreEqual("f", s._location.FileName);
            Assert.AreEqual(1, s._location.Line);
            Assert.AreEqual(2, s._location.Column);

            Assert.IsNotNull(s.a);
            Assert.IsNotNull(s.b);
            Assert.AreEqual("123", s.a.Value);
            Assert.AreEqual(1, s.a.Line);
            Assert.AreEqual(4, s.a.Column);
            Assert.AreEqual("456", s.b.Value);
            Assert.AreEqual(1, s.b.Line);
            Assert.AreEqual(12, s.b.Column);
        }
        [Test]
        public void EmptyElement() {
            //vf
            XmlORReader r = new XmlORReader("f", new StringReader("<s a=\"\"><b/></s>"));
            Setting1 s = new Setting1();

            r.Read(_setting1, s);

            //MandatoryłO̕񂪂Ƃ͂
            Assert.IsNotNull(s._location);
            Assert.AreEqual(s.a.ParseMandatoryString(), "");
            Assert.AreEqual(s.b.ParseMandatoryString(), "");
        }
        [Test]
        public void Formats() {
            //Rg͖邱ƂmF
            XmlORReader r = new XmlORReader("f", new StringReader("<s><!-- aaa -->\r\n<a>\r\n</a>\r\n<b><![CDATA[456]]></b>  </s>"));
            Setting1 s = new Setting1();

            r.Read(_setting1, s);

            Assert.IsNotNull(s._location);
            Assert.AreEqual("f", s._location.FileName);
            Assert.AreEqual(1, s._location.Line);
            Assert.AreEqual(2, s._location.Column);

            Assert.IsNotNull(s.a);
            Assert.IsNotNull(s.b);
            Assert.AreEqual("\n", s.a.Value);
            Assert.AreEqual(2, s.a.Line);
            Assert.AreEqual(4, s.a.Column);
            Assert.AreEqual("456", s.b.Value);
            Assert.AreEqual(4, s.b.Line);
            Assert.AreEqual(13, s.b.Column); //CDATÂƂ͒g|CgĂ悤
        }
        [Test]
        public void Error() {
            //G[bZ[W̏o邱ƂmF
            bool err = false;

            //`łȂ
            XmlORReader r = new XmlORReader("f", new StringReader("<s>"));
            Setting1 s = new Setting1();
            try {
                r.Read(_setting1, s);
            }
            catch(Exception ) {
                err= true;
            }
            Assert.IsTrue(err);
            
            //svf
            err = false;
            try {
                s = new Setting1();
                r = new XmlORReader("f", new StringReader("<s><a>0</a><b>0</b><c>0</c></s>"));
                r.Read(_setting1, s);
            }
            catch(Exception ex) {
                Assert.AreEqual(r.FormatUnexpectedElementMessage("c"), ex.Message);
                err= true;
            }
            Assert.IsTrue(err);

            //svfłݒȂpX
            err = false;
            try {
                s = new Setting1();
                r = new XmlORReader("f", new StringReader("<s><a>0</a><b>0</b><c>0</c></s>"));
                r.IgnoreUnknownElement = true;
                r.Read(_setting1, s);
            }
            catch(Exception) {
                err= true;
            }
            Assert.IsFalse(err);

            //dvf
            err = false;
            try {
                s = new Setting1();
                r = new XmlORReader("f", new StringReader("<s><a>0</a><a>0</a></s>"));
                r.Read(_setting1, s);
            }
            catch(Exception ex) {
                Assert.AreEqual(r.FormatDuplicatedElementMessage("a"), ex.Message);
                err= true;
            }
            Assert.IsTrue(err);
        }
        [Test]
        public void Collection1() {
            XmlORReader r = new XmlORReader("f", new StringReader("<t><a>A</a><s><a>AA</a></s><s><a>BB</a></s></t>"));
            Setting2 t = new Setting2();
            r.Read(_setting2, t);

            Assert.AreEqual("A", t.a.Value); //͕ʂ̃tB[h
            Assert.AreEqual(2, t.s.Count); //RNV͂ł
            Setting1 s1 = t.s[0];
            Setting1 s2 = t.s[1];
            Assert.AreEqual("AA", s1.a.Value);
            Assert.AreEqual(18, s1.a.Column);
            Assert.AreEqual("BB", s2.a.Value);
            Assert.AreEqual(34, s2.a.Column);
        }
        [Test]
        public void Collection2() { //Agr[gx[X̓ǂݍ
            XmlORReader r = new XmlORReader("f", new StringReader("<t><a>A</a><s a=\"AA\"/><s a=\"BB\"/></t>"));
            Setting2 t = new Setting2();
            r.Read(_setting2, t);

            Assert.AreEqual("A", t.a.Value); //͕ʂ̃tB[h
            Assert.AreEqual(2, t.s.Count); //RNV͂ł
            Setting1 s1 = t.s[0];
            Setting1 s2 = t.s[1];
            Assert.AreEqual("AA", s1.a.Value);
            Assert.AreEqual(15, s1.a.Column);
            Assert.AreEqual("BB", s2.a.Value);
            Assert.AreEqual(26, s2.a.Column);
        }

        [Test]
        public void NotCollection() {
            XmlORReader r = new XmlORReader("f", new StringReader("<t><t a='10'>11</t></t>"));
            NotCollectionMember t = new NotCollectionMember();
            r.Read(_notCollectionMember, t);

            Assert.AreEqual("10", t.t.a.Value);
            Assert.AreEqual("11", t.t.text.Value);
        }

        [Test]
        public void AutoTextValue() {
            XmlORReader r = new XmlORReader("f", new StringReader("<t a='1'>2</t>"));
            TextValueFieldTest t = new TextValueFieldTest();
            r.Read(_textValue, t);

            Assert.AreEqual("1", t.a.Value);
            Assert.AreEqual("2", t.text.Value);

            r = new XmlORReader("f", new StringReader("<t a='2'/>"));
            t = new TextValueFieldTest();
            r.Read(_textValue, t);

            Assert.AreEqual("2", t.a.Value);
            Assert.IsTrue(t.text.IsOmitted);
        }

    }
#endif

#if UNITTEST
    [TestFixture]
    public class ParseTests {
        public class C : OriginReportableObject {
            public ORText a;
        }
        private ORTypeInfoCollection _lib;
        private FieldInfo _fieldA;

        [TestFixtureSetUp]
        public void Setup() {
            _lib = new ORTypeInfoCollection();
            _lib.Add(typeof(C));
            _fieldA = _lib.Find(typeof(C)).FindField("a");
        }

        [Test]
        public void MissingMandatory() {
            bool err = false;
            C c = new C();
            c._location = new ObjectOrigin("f", 3, 4);
            c.a = new ORText(_fieldA, c._location);
            try {
                c.a.ParseMandatoryString();
            }
            catch(Exception ex) {
                err = true;
                Assert.AreEqual(ORText.FormatMissingElementMessage(c.a), ex.Message);
            }
            Assert.IsTrue(err);
        }
        [Test]
        public void IntSuccess() {
            C c1 = new C();
            Assert.AreEqual(1, c1.a.ParseOptionalInt(1));
            c1.a = new ORText(_fieldA, c1._location, 2, 3, "4");
            Assert.AreEqual(4, c1.a.ParseOptionalInt(1));

        }
        [Test]
        public void IntFail() {
            C c = new C();
            c._location = new ObjectOrigin("f", 3, 4);

            bool err = false;
            c.a = new ORText(_fieldA, c._location, 1, 2, "x");
            try {
                c.a.ParseMandatoryInt();
            }
            catch(Exception ex) {
                err = true;
                Assert.AreEqual(ORText.FormatMulformedElementMessage(c.a, ""), ex.Message);
            }
            Assert.IsTrue(err);

            err = false;
            c.a = new ORText(_fieldA, c._location, 1, 2, "");
            try {
                c.a.ParseMandatoryInt();
            }
            catch(Exception ex) {
                err = true;
                Assert.AreEqual(ORText.FormatMulformedElementMessage(c.a, ""), ex.Message); //lȂƂ
            }
            Assert.IsTrue(err);
        }
        [Test]
        public void ColorTest() {
            C c1 = new C();
            c1._location = new ObjectOrigin("f", 3, 4);
            c1.a = new ORText(_fieldA, c1._location, 2, 3, "Green");
            Assert.AreEqual(Color.Green, c1.a.ParseColorOrException());

            c1.a = new ORText(_fieldA, c1._location);
            Assert.AreEqual(Color.Red, c1.a.ParseOptionalColor("Red"));
            Assert.AreEqual(Color.FromArgb(0x12, 0xA4, 0xB6), c1.a.ParseOptionalColor("#12a4B6"));
            Assert.AreEqual(Color.Red, c1.a.ParseOptionalColor("red"));

            bool err = false;
            c1.a = new ORText(_fieldA, c1._location, 1, 2, "qqq");
            try {
                c1.a.ParseColorOrException();
            }
            catch(Exception ex) {
                err = true;
                Assert.AreEqual(ORText.FormatMulformedElementMessage(c1.a, "F"), ex.Message);
            }
            Assert.IsTrue(err);
        }
    }
#endif

}
