/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Script/ExpressionParserTests.cs#6 $
 * $DateTime: 2007/10/25 12:56:12 $
 * Expressionp[X邱ƂɊւeXg
 */
#if UNITTEST
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using antlr;
using NUnit.Framework;

using Bellagio.Script.Parser;
using Bellagio.UnitTests;

namespace Bellagio.Script {
    [TestFixture]
    public class ExpressionLexTests {
        [Test]
        public void Lex1() {
            IToken[] ts = Parse("t+1");
            Assert.AreEqual(3, ts.Length);
            Assert.AreEqual(ExpressionParserTokenTypes.IDENTIFIER, ts[0].Type);
            Assert.AreEqual(ExpressionParserTokenTypes.ARITH_OP_ADD, ts[1].Type);
            Assert.AreEqual(ExpressionParserTokenTypes.NUMBER, ts[2].Type);
        }
        [Test]
        public void Lex2() {
            IToken[] ts = Parse("a.b");
            Assert.AreEqual(3, ts.Length);
            Assert.AreEqual(ExpressionParserTokenTypes.IDENTIFIER, ts[0].Type);
            Assert.AreEqual(7, ts[1].Type); //ExpressionParserTokenTypesȂ̂ŕ@ύXΕω\
            Assert.AreEqual(ExpressionParserTokenTypes.IDENTIFIER, ts[2].Type);
        }

        private IToken[] Parse(string text) {
            ExpressionLexer l = new ExpressionLexer(new StringReader(text));
            List<IToken> r = new List<IToken>();
            do {
                IToken t = l.nextToken();
                if(t.Type==ExpressionParserTokenTypes.EOF) break;
                r.Add(t);
            } while(true);
            return r.ToArray();
        }
    }

    [TestFixture]
    public class ExpressionParserTests {

        [Test]
        public void expr00_Basic1() {
            Assert.AreEqual("(1 + 2)", Reformat("1+2"));
        }
        [Test]
        public void expr01_Ref1() {
            Assert.AreEqual("(a * d)", Reformat("a*d"));
        }
        [Test]
        public void expr03_Literals() {
            Expression t1 = Expression.ParseExpression("@t900");
            Expression d1 = Expression.ParseExpression("@d1231");
            Expression b1 = Expression.ParseExpression("true");
            Expression n1 = Expression.ParseExpression("3");
            Expression n2 = Expression.ParseExpression("1.2");
            Expression p1 = Expression.ParseExpression("$a");
            Expression s1 = Expression.ParseExpression("a");
            Expression st = Expression.ParseExpression("\"xxx\"");

            Assert.IsTrue(t1 is LiteralTimeExpression);
            Assert.AreEqual("900", ((LiteralTimeExpression)t1).Value);
            Assert.IsTrue(d1 is LiteralDateExpression);
            Assert.AreEqual("1231", ((LiteralDateExpression)d1).Value);
            Assert.IsTrue(b1 is LiteralBooleanExpression);
            Assert.IsTrue(((LiteralBooleanExpression)b1).Value);
            Assert.IsTrue(n1 is LiteralNumberExpression);
            Assert.AreEqual("3", ((LiteralNumberExpression)n1).Value);
            Assert.IsTrue(n2 is LiteralNumberExpression);
            Assert.IsTrue(p1 is ParameterExpression);
            Assert.AreEqual("a", ((ParameterExpression)p1).Symbol);
            Assert.IsTrue(s1 is SymbolExpression);
            Assert.AreEqual("a", ((SymbolExpression)s1).Symbol);
            Assert.IsTrue(st is LiteralStringExpression);
            Assert.AreEqual("xxx", ((LiteralStringExpression)st).Value);
        }
        [Test]
        public void expr07_Cond() {
            ConditionalExpression e = (ConditionalExpression)Expression.ParseExpression("a? (1+1) : 0");
            Assert.IsTrue(e.Condition is SymbolExpression);
            Assert.IsTrue(e.TrueCase is BinaryOpExpression);
            Assert.IsTrue(e.FalseCase is LiteralNumberExpression);

        }
        [Test]
        public void expr08_Operators1() {
            Assert.AreEqual("((10 - 3) - 2)", Reformat("10-3-2"));
        }
        [Test]
        public void expr09_Operators2() {
            Assert.AreEqual("(((2 * 3) + (4 / 5)) + (6 % 7))", Reformat("2*3+4/5+6%7"));
        }
        [Test]
        public void expr10_Operators3() {
            Assert.AreEqual("((0 && (1 < (2 + 3)))? 4 : 5)", Reformat("0 && 1 < 2 + 3? 4 : 5"));
        }
        [Test]
        public void expr11_UnaryOperators1() {
            Assert.AreEqual("((-a) + 4)", Reformat("-a+4"));
            Assert.AreEqual("(4 + (-a))", Reformat("4+-a"));
        }
        [Test]
        public void expr12_Ref1() {
            Assert.AreEqual("(a.b)", Reformat("a.b"));
            Assert.AreEqual("((a.b) + (c.d))", Reformat("a.b + c.d"));
            Assert.AreEqual("(((a.b).c).d)", Reformat("a.b.c.d"));
        }
        [Test]
        public void expr13_Inv1() {
            Assert.AreEqual("(a(0, 1))", Reformat("a(0,1)"));
        }
        [Test]
        public void expr14_array1() {
            Assert.AreEqual("(a.(b[c]))", Reformat("a.b[c]"));
            Assert.AreEqual("((a[0])[0])", Reformat("a[0][0]"));
            Assert.AreEqual("((1 + $a)[0])", Reformat("(1+$a)[0]"));
            Assert.AreEqual("((f())[0])", Reformat("f()[0]"));
        }
        [Test]
        public void expr15_lambda() {
            Assert.AreEqual("(lambda ((int a, double d) (i * d)))", Reformat("lambda ((int a, double d) i * d)"));
            Assert.AreEqual("(lambda ((int[] i) (i.length)))", Reformat("lambda ((int[] i) i.length)"));
        }
        [Test]
        public void expr16_let() {
            Assert.AreEqual("(let ((a=0, b=2) (a + b)))", Reformat("let ((a=0, b=2) a+b)"));
        }

        [Test]
        public void type_descriptions() {
            TypeDesc td = Expression.ParseTypeDesc("int");
            Assert.AreEqual("int", ((NamedTypeDesc)td).Name);

            td = Expression.ParseTypeDesc("double[][]");
            TypeDesc td2 = ((ArrayTypeDesc)td).Element;
            TypeDesc td3 = ((ArrayTypeDesc)td2).Element;
            Assert.AreEqual("double", ((NamedTypeDesc)td3).Name);

            td = Expression.ParseTypeDesc("(double,int[] -> int)");
            LambdaTypeDesc lt = (LambdaTypeDesc)td;
            Assert.AreEqual("int", ((NamedTypeDesc)lt.RetType).Name);
            Assert.AreEqual(2, lt.ArgCount);
            Assert.AreEqual("double", ((NamedTypeDesc)lt.ArgAt(0)).Name);
            Assert.AreEqual("int[]", lt.ArgAt(1).Format());
        }

        [Test]
        public void Errors() {
            //ł̓t@CɏoOKBg̑Ó͖ڎB
            //string fn = UnitTestUtil.GetUnitTestConfig("nunit.errorMessageFile");
            //Console.Out.WriteLine(fn);
            ErrorMessageWriter w = new  ErrorMessageWriter("C:\\IOPort\\bellagio_errors.csv");
            ErrorMessageWriter.DoString px = new ErrorMessageWriter.DoString(Expression.ParseExpression);
            w.ConfirmError("parse", px, "(");
            w.Close();
        }

        private string Reformat(string text) {
            try {
                Expression e = Expression.ParseExpression(text);
                ExpressionFormatter f = new ExpressionFormatter();
                f.ExcessiveBracket = true;
                f.SpaceAtBinaryOp = true;
                return f.Format(e);
            }
            catch(BellagioException ex) {
                Console.Out.WriteLine(ex.Message);
                throw ex;
            }
        }
    }
}

#endif
