/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/FreeLine.cs#6 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * R
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using System.Diagnostics;

using Poderosa;

using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Forms;

namespace Bellagio.Chart {
    /* {\
       start, end̂Q_PivotŎw肷ԂƁẢ`悷B
    @Nbv̈ɑ΂āAQPivot́@(1)Q_Ƃ (2)P_̂ݓ (3)ǂȂ@̂R̏󋵂邪A
    @Sŕ`悷ׂ̐3, 2, 1ƂȂB
    @̂ƂAPivotQԒƃNbv̈̌_߂Kv邪AŝGetClipEdgeł
    */

    public class FreeLine {
        public enum LineDrawMode {
            Normal, Hover
        }
        public struct Pivot {
            //`悷Ƃ͂Wɕϊ
            public int datetime; //t܂͎  CHARTPOSITIONɂ悤ƂlAVACYɓ
            public double price; //li liȊOɑ΂Đ͖̂T|[gBɓ`[gł͂PeBbN̕ł̂ŒlidoubleɂăX[WO

        }
        private Pivot _start;
        private Pivot _end;

        public FreeLine(Pivot start) {
            _start = start;
            _end = start;
        }
        public void SetEnd(Pivot p) {
            _end = p;
        }
        public bool IsCollapsed {
            get {
                //start/end܂ɋ߂ƂԂĂƂ݂Ȃ
                return _start.datetime==_end.datetime && Math.Abs(_start.price - _end.price)<=2; //liŔ͖ɂ鍷ăC
            }
        }
        /*
        public Rectangle GetInclusion(Rectangle rect) {
            if(_pivot==_destination)
                return new Rectangle(_pivot, new Size(0, 0));
            Point[] es = GetEdge(rect, _pivot, _destination);
            return new Rectangle(Math.Min(es[0].X, es[1].X), Math.Min(es[0].Y, es[1].Y), Math.Abs(es[0].X-es[1].X), Math.Abs(es[0].Y-es[1].Y));
        }
        //\ɗʒułȂƐm肳Ȃ悤ɂ
        public bool PivotHasEnoughDistanceTo(Point pt) {
            return Math.Abs(_pivot.X-pt.X)>10 || Math.Abs(_pivot.Y-pt.Y)>10;
        }
        */
        public Pivot Start {
            get {
                return _start;
            }
        }
        public Pivot End {
            get {
                return _end;
            }
        }

        public void AppendTo(StringBuilder bld) {
            bld.Append(_start.datetime).Append(',');
            bld.Append(_start.price.ToString("F0")).Append(',');
            bld.Append(_end.datetime).Append(',');
            bld.Append(_end.price.ToString("F0"));
        }
        public static FreeLine Parse(string value) {
            string[] e = value.Split(',');
            if(e.Length!=4) throw new FormatException("FreeLine parse error " + value);
            Pivot start, end;
            start.datetime = Int32.Parse(e[0]);
            start.price = Double.Parse(e[1]);
            end.datetime = Int32.Parse(e[2]);
            end.price = Double.Parse(e[3]);
            FreeLine r = new FreeLine(start);
            r.SetEnd(end);
            return r;
        }


        public bool StartAsDisplayPoint(IChartDataSource data, ChartDrawing drawing, out Point result) {
            return PivotToDisplayPoint(_start, data, drawing, out result);
        }
        public bool EndAsDisplayPoint(IChartDataSource data, ChartDrawing drawing, out Point result) {
            return PivotToDisplayPoint(_end, data, drawing, out result);
        }
        private static bool PivotToDisplayPoint(Pivot p, IChartDataSource data, ChartDrawing drawing, out Point result) {
            CHARTPOSITION pos;
            if(data.GetChartPositionAt(p.datetime, out pos)) {
                int di = pos.ToDisplayIndex();
                if(di!=-1) {
                    int x = drawing.MainLocation.Right - drawing.ChartDrawingSettings.candlePitch/2 - (drawing.DisplayPosition.ToDisplayIndex() - di) * drawing.ChartDrawingSettings.candlePitch;
                    int y = (int)drawing.ChartScaleInfo.PriceTrnanslator.Translate(p.price);
                    result = new Point(x, y);
                    return true;
                }
            }
            result = new Point(0, 0);
            return false;
        }
        public LinearTranslator CreateScaleTranslator(IChartDataSource data, ChartDrawing drawing) {
            Point p1, p2;
            if(!StartAsDisplayPoint(data, drawing, out p1) || !EndAsDisplayPoint(data, drawing, out p2)) return null; //`s

            return LinearTranslator.Solve(p1.X, p1.Y, p2.X, p2.Y);
        }

        public void Draw(Graphics g, Rectangle clip, IChartDataSource data, ChartDrawing drawing, LineDrawMode mode) {
            Point p1, p2;
            Point edge1, edge2;
            if(!StartAsDisplayPoint(data, drawing, out p1) || !EndAsDisplayPoint(data, drawing, out p2)) return; //`s
            //Debug.WriteLine(String.Format("{0},{1} - {2},{3}", _start.datetime, _start.price, _end.datetime, _end.price));

            clip.Intersect(drawing.MainLocation.Value);
            GetCrossEdge(clip, p1, p2, out edge1, out edge2);
            bool c1 = clip.Contains(p1);
            bool c2 = clip.Contains(p2);
            
            if(p1==p2 || (!clip.Contains(edge1) && !clip.Contains(edge2)))
                return; //`悵Ȃ悢
            else if(mode==LineDrawMode.Hover) {
                using(Pen pen = new Pen(drawing.ChartDrawingSettings.hoveredFreeLineColor.col, 2)) {
                    g.DrawLine(pen, edge1, edge2);
                }
            }
            else if(!c1 && !c2) { //ǂ܂܂ĂȂƂ
                using(Pen pen = new Pen(drawing.ChartDrawingSettings.hoveredFreeLineColor.col, 2)) {
                    pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;
                    g.DrawLine(pen, edge1, edge2);
                }
            }
            else if(c1 && c2) {
                //̓_܂܂ĂƂ́AP{izo[jorR{iʏ펞j̐ƂȂ
                Pen pen = new Pen(drawing.ChartDrawingSettings.freeLineColor.col);
                g.DrawLine(pen, p1, p2);
                pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;
                g.DrawLine(pen, edge1, p1);
                g.DrawLine(pen, p2, edge2);
                pen.Dispose();
            }
            else { //Е̂݊܂܂ĂƂ
                Pen pen = new Pen(drawing.ChartDrawingSettings.freeLineColor.col);
                if(c1)
                    g.DrawLine(pen, p1, edge2);
                else
                    g.DrawLine(pen, p2, edge1);
                pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;
                if(c1)
                    g.DrawLine(pen, p1, edge1);
                else
                    g.DrawLine(pen, p2, edge2);
                pen.Dispose();
            }

        }

        //p1p2ʂrectƌ_ԂBp1->p2̌edge1->edge2͓̌ł邱
        private static void GetCrossEdge(Rectangle rect, Point p1, Point p2, out Point edge1, out Point edge2) {
            if(p1.X==p2.X) {
                edge1 = new Point(p1.X, p1.Y<p2.Y? rect.Top : rect.Bottom-1);
                edge2 = new Point(p1.X, p1.Y<p2.Y? rect.Bottom-1 : rect.Top);
                return;
            }
            else if(p1.Y==p2.Y) {
                edge1 = new Point(p1.X<p2.X? rect.Left : rect.Right-1, p1.Y);
                edge2 = new Point(p1.X<p2.X? rect.Right-1 : rect.Left, p1.Y);
                return;
            }

            AbstractValueTranslator lt = LinearTranslator.Solve(p1.X, p1.Y, p2.X, p2.Y);
            List<Point> ar = new List<Point>();
            ar.Add(new Point(rect.Left, (int)lt.Translate(rect.Left)));
            ar.Add(new Point(rect.Right-1, (int)lt.Translate(rect.Right-1)));
            ar.Add(new Point((int)lt.Inverse(rect.Top), rect.Top));
            ar.Add(new Point((int)lt.Inverse(rect.Bottom-1), rect.Bottom-1));
            //̂S_XW̏ɕׁAQԖڂƂRԖڂ
            ar.Sort(new PointComparer(p1.X < p2.X));
            edge1 = ar[1];
            edge2 = ar[2];
        }
        private class PointComparer : IComparer<Point> {
            private int _sign;
            public PointComparer(bool s) {
                _sign = s? 1 : -1;
            }
            public int Compare(Point a, Point b) {
                return _sign * (a.X - b.X);
            }
        }
    }

    //PʂɂPpӂRNVƕ`̏ԊǗ
    public class FreeLineSet : IEnumerable<FreeLine> {
        private List<FreeLine> _lines;
        private FreeLine _newDrawing;
        private FreeLine _focusedLine;

        public FreeLineSet() {
            _lines = new List<FreeLine>();
        }

        public bool IsDrawingNewLine {
            get {
                return _newDrawing!=null;
            }
        }
        public FreeLine FocusedLine {
            get {
                return _focusedLine;
            }
        }
        public void Clear() {
            _lines.Clear();
            _newDrawing = null;
            _focusedLine = null;
        }

        public void AddFreeLine(FreeLine line) {
            _lines.Add(line);
        }
        public void RemoveFocusedLine() {
            Debug.Assert(_focusedLine!=null);
            _lines.Remove(_focusedLine);
            _focusedLine = null;
        }

        public void Draw(Graphics g, Rectangle clip, IChartDataSource data, ChartDrawing drawing) {
            foreach(FreeLine l in _lines)
                l.Draw(g, clip, data, drawing, l==_focusedLine? FreeLine.LineDrawMode.Hover : FreeLine.LineDrawMode.Normal);
            if(_newDrawing!=null && !_newDrawing.IsCollapsed)
                _newDrawing.Draw(g, clip, data, drawing, FreeLine.LineDrawMode.Normal);
        }

        public IEnumerator<FreeLine> GetEnumerator() {
            return _lines.GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator() {
            return _lines.GetEnumerator();
        }

        public void StartDrawingNewLine(IChartDataSource data, ChartDrawing drawing, Point mousePos) {
            FreeLine.Pivot p;
            if(GetPivot(data, drawing, mousePos, out p)) {
                _newDrawing = new FreeLine(p);
            }
        }
        public void EndDrawingNewLine(IChartDataSource data, ChartDrawing drawing, Point mousePos) {
            Debug.Assert(_newDrawing!=null);
            FreeLine.Pivot p;
            if(GetPivot(data, drawing, mousePos, out p)) {
                if(_newDrawing.Start.datetime!=p.datetime) {
                    _newDrawing.SetEnd(p);
                    _lines.Add(_newDrawing);
                }
            }
            _newDrawing = null;
        }
        //InvalidateKv邩ǂԂ
        public bool OnMouseMoveNewLine(IChartDataSource data, ChartDrawing drawing, Point mousePos) {
            Debug.Assert(_newDrawing!=null);
            FreeLine.Pivot p;
            if(GetPivot(data, drawing, mousePos, out p)) {
                //Debug.WriteLine(String.Format("NEwLine {0},{1}", p.datetime, p.price));
                _newDrawing.SetEnd(p);
            }
            return true;
        }
        public bool OnMouseMoveFocuedLine(IChartDataSource data, ChartDrawing drawing, Point mousePos) {
            bool inv = _focusedLine!=null;
            _focusedLine = null;
            foreach(FreeLine line in _lines) {
                LinearTranslator tr = line.CreateScaleTranslator(data, drawing);
                if(tr!=null && Math.Abs(tr.Translate(mousePos.X) - mousePos.Y) <= 2) {
                    _focusedLine = line;
                    return true;
                }
            }
            return inv;
        }

        private bool GetPivot(IChartDataSource data, ChartDrawing drawing, Point mousePos, out FreeLine.Pivot result) {
            result = new FreeLine.Pivot();
            if(drawing.DisplayPosition.IsEmpty) return false;

            CHARTPOSITION pos = data.SectionInfo.GetChartPosition(drawing.DisplayPosition.ToDisplayIndex() - (drawing.MainLocation.Right-mousePos.X)/drawing.ChartDrawingSettings.candlePitch);
            if(pos.IsEmpty) return false;

            result.datetime = data.GetDateOrTimeAt(pos);
            result.price = drawing.ChartScaleInfo.PriceTrnanslator.Inverse(mousePos.Y);
            return true;
        }

        //VACY
        public override string ToString() {
            StringBuilder bld = new StringBuilder();
            foreach(FreeLine fl in _lines) {
                if(bld.Length > 0) bld.Append(';');
                fl.AppendTo(bld);
            }
            return bld.ToString();
        }
        public void FillByString(string value) {
            this.Clear();
            foreach(string e in value.Split(';')) {
                _lines.Add(FreeLine.Parse(e));
            }
        }

    }

    //Preferenceւ̕ۑ
    public static class FreeLineSerializeUtil {
        private class RootNode : IDynamicPreferenceNode {
            public string PreferenceID {
                get { return "freeline"; }
            }
            public IDynamicPreferenceNode ParentNode {
                get { return null; }
            }
        }

        private class Element : IDynamicPreferenceItem {
            public Element(string id) {
                _id = id;
            }
            private string _id;

            public string  DefaultValue {
	            get { return ""; }
            }

            public string  Value {
	            get { 
		            return GetDynamicPreference().InternalGetValue(this); 
	            }
	            set {
                    GetDynamicPreference().InternalSetValue(this, value);
	            }
            }


            public string  PreferenceID {
                get { return _id; }
            }

            public IDynamicPreferenceNode ParentNode {
                get { return _rootNode; }
            }

        }

        private static RootNode _rootNode;
        private static Dictionary<string, Element> _elements = new Dictionary<string,Element>();

        private static DynamicPreference GetDynamicPreference() {
            DynamicPreference d = BellagioRoot.FixedPreferences.Chart.UserOperation;
            return d;
        }
        public static void Init() {
            if(_rootNode!=null) return;

            _rootNode = new RootNode();
            DynamicPreference d = BellagioRoot.FixedPreferences.Chart.UserOperation;
            //ŏStructuredText̒ǂPrefItem\zĂȂƁADynamicPrefɃ[hƂɐǂ܂Ȃ
            if(d.UnderlyingData!=null) {
                StructuredText content = d.UnderlyingData.GetChildOrNull(0);
                if(content!=null) {
                    foreach(StructuredText.Entry item in content.Children) {
                        string name = item.name;
                        Element e = new Element(name);
                        _elements.Add(name, e);
                        d.RegisterItem(e);
                    }
                }
            }
        }

        public static string GetValue(Stock stock, Quote.QuoteUnit unit) {
            if(unit==Quote.QuoteUnit.Minutely) return ""; //̕ۑ͂Ȃ

            string key = GetKey(stock, unit);
            Element e;
            if(!_elements.TryGetValue(key, out e)) { //ȂRegisterItemēǂݍ݂Ƀ`W
                e = new Element(key);
                _elements.Add(key, e);
                
                GetDynamicPreference().RegisterItem(e);
            }
            return e.Value;
        }
        public static void SetValue(Stock stock, Quote.QuoteUnit unit, string value) {
            if(unit==Quote.QuoteUnit.Minutely) return; //̕ۑ͂Ȃ

            string key = GetKey(stock, unit);
            Element e;
            if(_elements.TryGetValue(key, out e))
                e.Value = value;
            else if(value.Length > 0) {
                e = new Element(key);
                _elements.Add(key, e);
                GetDynamicPreference().RegisterItem(e);
                e.Value = value;
            }

        }

        private static string GetKey(Stock stock, Quote.QuoteUnit unit) {
            return new StringBuilder().Append(stock.Profile.Code).Append('.').Append(unit==Quote.QuoteUnit.Daily? "D" : unit==Quote.QuoteUnit.Weekly? "W" : "M").ToString();
        }
    }


#if false
    public class FreeLineCollection {
        private ArrayList _data;
        public FreeLineCollection() {
            _data = new ArrayList();
        }
        public int Count {
            get {
                return _data.Count;
            }
        }

        public void Add(AbstractBrand br, ChartFormat format, bool logScale, SolidFreeLine fl) {
            fl._code = br.Code;
            fl._targetFormat = format;
            fl._logScale = logScale;
            _data.Add(fl);
        }

        public SolidFreeLine[] Find(AbstractBrand br, ChartFormat format, bool logScale) {
            ArrayList t = new ArrayList();
            foreach(SolidFreeLine fl in _data) {
                if(fl._code==br.Code && fl._targetFormat==format && fl._logScale==logScale)
                    t.Add(fl);
            }
            return (SolidFreeLine[])t.ToArray(typeof(SolidFreeLine));
        }

        public void ClearAll() {
            _data.Clear();
        }
        public void Clear(AbstractBrand br, ChartFormat format, bool logScale) {
            ArrayList temp = new ArrayList();
            IEnumerator ie = _data.GetEnumerator();
            while(ie.MoveNext()) {
                SolidFreeLine fl = (SolidFreeLine)ie.Current;
                if(!(fl._code==br.Code && fl._targetFormat==format && fl._logScale==logScale))
                    temp.Add(fl);
            }
            _data = temp;
        }
        public void Remove(int id) {
            for(int i=0; i<_data.Count; i++) {
                SolidFreeLine l = (SolidFreeLine)_data[i];
                if(l._id==id) {
                    _data.RemoveAt(i);
                    break;
                }
            }
        }


        public void Load(StorageNode parent) {
            string t = parent["free-lines"];
            if(t==null) return;
            foreach(string ee in t.Split(',')) {
                SolidFreeLine fl = new SolidFreeLine();
                string[] e = ee.Split(':');
                fl._code = Int32.Parse(e[0]);
                switch(e[1]) {
                    case "D":
                        fl._targetFormat = ChartFormat.Daily;
                        fl._logScale = false;
                        break;
                    case "W":
                        fl._targetFormat = ChartFormat.Weekly;
                        fl._logScale = false;
                        break;
                    case "M":
                        fl._targetFormat = ChartFormat.Monthly;
                        fl._logScale = false;
                        break;
                    case "DL":
                        fl._targetFormat = ChartFormat.Daily;
                        fl._logScale = true;
                        break;
                    case "WL":
                        fl._targetFormat = ChartFormat.Weekly;
                        fl._logScale = true;
                        break;
                    case "ML":
                        fl._targetFormat = ChartFormat.Monthly;
                        fl._logScale = true;
                        break;
                }
                fl._date1 = Int32.Parse(e[2]);
                fl._value1 = Double.Parse(e[3]);
                fl._date2 = Int32.Parse(e[4]);
                fl._value2 = Double.Parse(e[5]);
                fl._id = SolidFreeLine.NextID++;
                _data.Add(fl);
            }
        }
        public void SaveTo(StorageNode parent) {
            StringBuilder bld = new StringBuilder();
            foreach(SolidFreeLine sl in _data) {
                if(bld.Length>0) bld.Append(",");
                String format;
                switch(sl._targetFormat) {
                    case ChartFormat.Daily:
                    default:
                        format = "D";
                        break;
                    case ChartFormat.Weekly:
                        format = "W";
                        break;
                    case ChartFormat.Monthly:
                        format = "M";
                        break;
                }
                if(sl._logScale) {
                    format += "L";
                }
                bld.Append(String.Format("{0}:{1}:{2}:{3:F2}:{4}:{5:F2}", sl._code, format, sl._date1, sl._value1, sl._date2, sl._value2));
            }
            parent["free-lines"] = bld.ToString();
        }
    }
#endif
}
