/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/TimeAndSales.cs#11 $
 * $DateTime: 2008/02/19 10:18:47 $
 * 
 * TickTime & SalesIuWFNg
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Bellagio.Data;

#if UNITTEST
using NUnit.Framework;
#endif


namespace Bellagio.Values {

    //Jp
    public interface ITickData {
        int Price { get; }
        int Time { get; }
        int Volume { get; }
    }
    public interface ITimeAndSales {
        int Count { get; }
        ITickData TickAt(int index);
    }


    //TickDataƔ̊֌W
    public enum TickItaRelation {
        Unknown, //s\
        Ask,     //ɂԂ
        Bid,     //ɂԂ
        Yose     //
    }

    //藚
    public class TickData : BV, ITickData {
        private int _time; //BTimeȂ̒lƓATickData̓CX^X̂ł邽߂ɐint
        private int _price;
        private int _volume;
        private TickItaRelation _itaRelation;

        public TickData(int time, int price, int volume, TickItaRelation rel) {
            _time = time;
            _price = price;
            _volume = volume;
            _itaRelation = rel;
        }
        public TickData(TickData t) {
            Let(t);
        }
        public override bool IsNil {
            get {
                return _price==-1 && _volume==-1; //nil̎dlƂBTickData̓CX^X̂ŃtOpӂɃ
            }
        }
        public override void Let(BV value) {
            TickData t = (TickData)value;
            _time = t._time;
            _price = t._price;
            _volume = t._volume;
            _itaRelation = t._itaRelation;
        }
        public int Time {
            get {
                return _time;
            }
        }
        public string TimeAsString {
            get {
                return BTime.DefaultFormat(new BTime(_time));
            }
        }
        public int Price {
            get {
                return _price;
            }
        }
        public int Volume {
            get {
                return _volume;
            }
        }
        public TickItaRelation TickItaRelation {
            get {
                return _itaRelation;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.TickData(this);
        }
        public override BT BT {
            get {
                return TickDataType.instance;
            }
        }

        //Time̕ύX͏Ԃ󂷋ꂪ̂ŐTdɁBvpeBZbgɂ͂Ȃ
        public void SetTime(int time) {
            _time = time;
        }
    }

    public class TickDataType : BObjectT {
        private TickDataType()
            : base("TickData", new TickData(0, -1, -1, TickItaRelation.Unknown)) {
        }
        public override BV CreateInstance() {
            return new TickData(0, 0, 0, TickItaRelation.Unknown);
        }

        public static TickDataType instance;
        static TickDataType() {
            instance =  new TickDataType();
        }

        //̒ł܂Ƃ߂ēo^Ⴄ
        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT isTickData = instance;
            BT[] noArg = FunctionLibrary.noArg;

            lib.DefineInternalFunction("time", isTickData, BT.Time, noArg, new BInternalExecution(GetTime));
            lib.DefineInternalFunction("price", isTickData, BT.Int, noArg, new BInternalExecution(GetPrice));
            lib.DefineInternalFunction("volume", isTickData, BT.Int, noArg, new BInternalExecution(GetVolume));
            lib.DefineInternalFunction("buy", isTickData, BT.Bool, noArg, new BInternalExecution(IsBuyOrder));
            lib.DefineInternalFunction("sell", isTickData, BT.Bool, noArg, new BInternalExecution(IsSellOrder));
        }

        private static ExecResult GetTime(BV target, EvalContext extra, BV[] args, BV result) {
            ((BTime)result).LetInt(ToTickData(target).Time);
            return ExecResult.OK;
        }
        private static ExecResult GetPrice(BV target, EvalContext extra, BV[] args, BV result) {
            ((BInt)result).Value = ToTickData(target).Price;
            return ExecResult.OK;
        }
        private static ExecResult GetVolume(BV target, EvalContext extra, BV[] args, BV result) {
            ((BInt)result).Value = ToTickData(target).Volume;
            return ExecResult.OK;
        }
        private static ExecResult IsBuyOrder(BV target, EvalContext extra, BV[] args, BV result) {
            ((BBoolean)result).Value = ToTickData(target).TickItaRelation==TickItaRelation.Ask;
            return ExecResult.OK;
        }
        private static ExecResult IsSellOrder(BV target, EvalContext extra, BV[] args, BV result) {
            ((BBoolean)result).Value = ToTickData(target).TickItaRelation==TickItaRelation.Bid;
            return ExecResult.OK;
        }
        private static TickData ToTickData(BV target) {
            Debug.Assert(target.BT==instance);
            return (TickData)target;
        }
    }


    //TickDataRNVAփRo[g@\肷邾낤
    public class TimeAndSales : BArray , ITimeAndSales {
        private List<TickData> _data;

        public TimeAndSales() : base(TickDataType.instance.ArrayType()) {
            _data = new List<TickData>(); //f[^ʂƂ͈Hv邩
        }
        public override void Format(BVFormatter formatter) {
            formatter.TimeAndSales(this);
        }
        public override int Count {
            get {
                return _data.Count;
            }
        }
        public override BV this[int index] {
            get {
                return _data[index];
            }
        }
        public override void LetAt(int index, BV value) {
            Debug.Assert(false, "unimplemented");
        }
        public override BT BT {
            get {
                return TimeAndSalesType.instance;
            }
        }
        public override void Let(BV value) {
            TimeAndSales ts = (TimeAndSales)value;
            if(_data.Count > ts.Count) _data.RemoveRange(ts.Count, _data.Count - ts.Count);
            Debug.Assert(_data.Count <= ts.Count);
            for(int i=0; i<ts.Count; i++) {
                TickData s = ts.TickAt(i);
                Debug.Assert(s!=null);
                if(i < _data.Count)
                    _data[i].Let(s);
                else
                    _data.Add(new TickData(s));
            }
            Debug.Assert(_data.Count==ts.Count);
        }
        internal TickData TickAt(int index) {
            return _data[index];
        }
        ITickData ITimeAndSales.TickAt(int index) {
            return _data[index];
        }
        internal TickData TickAtOrNull(int index) {
            if(index<0 || index>=_data.Count)
                return null;
            else
                return _data[index];
        }
        internal TickData LastTick {
            get {
                if(_data.Count==0)
                    return null;
                else
                    return _data[_data.Count-1];
            }
        }
        public int LastPrice {
            get {
                TickData td = this.LastTick;
                return td==null? -1 : td.Price;
            }
        }

        public TimeAndSales Subsequence(int index, int length) {
            TimeAndSales ts = new TimeAndSales();
            for(int i=index; i<index+length; i++) ts._data.Add(_data[i]);
            return ts;
        }

        public int PreDifferentPrice {
            get {
                int t = this.LastPrice;
                for(int i=_data.Count-2; i>=0; i--) {
                    if(_data[i].Price!=t) return _data[i].Price;
                }
                return -1;
            }
        }


        internal void Add(int time, int price, int volume, TickItaRelation rel) {
            if(_data.Count>0 && time < _data[_data.Count-1].Time) {
                throw new BellagioException(String.Format("sȏTickf[^M܂B\n({0}->{1})", BTime.FormatHHMM(_data[_data.Count-1].Time), BTime.FormatHHMM(time)));
            }
            if(volume==0) return; //QUICK̃Az

            Debug.Assert(volume>0);
            _data.Add(new TickData(time, price, volume, rel));
        }
        internal void Add(TickData td) {
            //Debug.Assert(_data.Count==0 || td.Time >= _data[_data.Count-1].Time);
            Debug.Assert(td!=null);
            Debug.Assert(td.Volume>0);
            _data.Add(td);
        }
        public void Clear() {
            _data.Clear();
        }
       

        //timeȏɂȂŏ̃GgԂ
        //TODO QTɂ
        public int FindByTime(int time) {
            for(int i=0; i<_data.Count; i++)
                if(_data[i].Time >= time) return i;
            return -1;
        }

        //͈͌BI[exclusiveBȂ-1Zbg
        public void FindByTimeRangeBackward(int start_time, int end_time, out int start_index, out int end_index) {
            Debug.Assert(start_time <= end_time);
            start_index = -1;
            end_index = -1;

            bool found = false;
            for(int i = _data.Count-1; i>=0; i--) {
                int t = _data[i].Time;
                if(end_index==-1 && t < end_time) {
                    end_index = i+1; //͂߂Ĕ͈͂ɓ
                    found = true;
                }

                if(found) {
                    if(t >= start_time) //͈͂ɓĂ
                        start_index = i;
                    else
                        break; //{I
                }
            }

            //邽
            if(start_index==-1) end_index = -1;
        }

        //ɕϊ
        public ConcreteQuote ConvertToQuotes(int open_time, int end_time, StockExchange exch) {
            int index = 0;
            int last_price = 0;
            ConcreteQuote q = new ConcreteQuote(Quote.QuoteUnit.Minutely);

            int zenba_hike = MarketUtil.GetZenbaCloseTime(exch).AsInt();
            int goba_hike  = MarketUtil.GetGobaCloseTime(exch).AsInt();

            int time = open_time;
            while(time < end_time) {
                int next_minute = time+60;
                int minute_end_index = FindUntil(index, time, next_minute, (next_minute==zenba_hike || next_minute==goba_hike)); //indexȍ~ŁAtime+60܂ł̎ԂɓĂ̂Ԃ
                if(minute_end_index==index) { //ŶȂƂ
                    q.Add(new Candle(q, q.Count, time, last_price, 0));
                }
                else {
                    Debug.Assert(index < minute_end_index);
                    Candle c = new Candle(q, q.Count, time, _data[index].Price, _data[index].Volume);
                    index++;
                    while(index < minute_end_index) {
                        c.Update(_data[index].Price, _data[index].Volume);
                        index++;
                    }
                    q.Add(c);
                    index = minute_end_index;
                    last_price = c.Close;
                }

                time += 60;
                if(time==zenba_hike) time = MarketUtil.GetGobaOpenTime(exch).AsInt();
            }

            return q;
        }

        //startȌŁAԂend_time܂ł̂̂̎̃CfbNXԂBend_time܂߂邩ǂIvV
        //܂߂̂ȂƂindex̂܂ܕԂ
        private int FindUntil(int index, int start_time, int end_time, bool inclusive) {
            if(index >= _data.Count) return index;

            while(true) {
                TickData td = _data[index];
                if(td.Time < start_time)
                    return index;
                else if(td.Time > end_time || (td.Time==end_time && !inclusive))
                    return index;

                index++;
                if(index==_data.Count) return _data.Count;
            }
        }

    }

    public class TimeAndSalesType : BArrayObjectT {
        private TimeAndSalesType()
            : base("TimeAndSales", TickDataType.instance) {
            _nilInstance = new TimeAndSales().MakeNil();
            Debug.Assert(_nilInstance.IsNil);
        }
        public override BV CreateInstance() {
            return new TimeAndSales();
        }

        public static TimeAndSalesType instance = new TimeAndSalesType();

        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT isTimeAndSales = instance;
            BT[] noArg = FunctionLibrary.noArg;

            lib.DefineInternalFunction("first", isTimeAndSales, TickDataType.instance, noArg, new BInternalExecution(FirstTickData));
            lib.DefineInternalFunction("first", isTimeAndSales, TickDataType.instance, FunctionLibrary.oneIntArg, new BInternalExecution(FirstTickDataN));
            lib.DefineInternalFunction("last", isTimeAndSales, TickDataType.instance, noArg, new BInternalExecution(LastTickData));
            lib.DefineInternalFunction("last", isTimeAndSales, TickDataType.instance, FunctionLibrary.oneIntArg, new BInternalExecution(LastTickDataN));
            lib.DefineInternalFunction("tail", isTimeAndSales, instance, FunctionLibrary.oneIntArg, new BInternalExecution(TailTickDataArray));
            lib.DefineInternalFunction("head", isTimeAndSales, instance, FunctionLibrary.oneIntArg, new BInternalExecution(HeadTickDataArray));
        }

        private static bool IsTimeAndSales(BT type) {
            return type==instance;
        }
        private static TimeAndSales ToTimeAndSales(BV target) {
            Debug.Assert(target.BT==instance);
            return (TimeAndSales)target;
        }

        private static ExecResult FirstTickData(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            ((TickData)result).Let(ts.TickAt(0));
            return ExecResult.OK;
        }
        private static ExecResult FirstTickDataN(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            BInt index = (BInt)args[0];
            ((TickData)result).Let(ts.TickAt(index.Value));
            return ExecResult.OK;
        }
        private static ExecResult LastTickData(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            ((TickData)result).Let(ts.TickAt(ts.Count - 1));
            return ExecResult.OK;
        }
        private static ExecResult LastTickDataN(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            BInt index = (BInt)args[0];
            ((TickData)result).Let(ts.TickAt(ts.Count - 1 - index.Value));
            return ExecResult.OK;
        }
        private static ExecResult TailTickDataArray(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            BInt length = (BInt)args[0];
            result.Let(ts.Subsequence(ts.Count-length.Value, length.Value));
            return ExecResult.OK;
        }
        private static ExecResult HeadTickDataArray(BV target, EvalContext extra, BV[] args, BV result) {
            TimeAndSales ts = ToTimeAndSales(target);
            BInt length = (BInt)args[0];
            result.Let(ts.Subsequence(0, length.Value));
            return ExecResult.OK;
        }
    }

#if UNITTEST
    [TestFixture]
    public class TimeAndSalesTest {
        [Test]
        public void Basic0() {
            TimeAndSales t = new TimeAndSales();
            t.Add(0, 100, 1, TickItaRelation.Unknown);
            t.Add(0, 100, 1, TickItaRelation.Unknown);

            ConcreteQuote ya = t.ConvertToQuotes(0, 60, StockExchange.T);
            Assert.AreEqual(1, ya.Count);
            Candle t1 = ya.CandleAt(0);
            Assert.AreEqual(100, t1.Open);
            Assert.AreEqual(100, t1.Close);
            Assert.AreEqual(2, t1.Volume);
        }
        [Test]
        public void Basic1() {
            TimeAndSales t = new TimeAndSales();
            t.Add(0, 100, 1, TickItaRelation.Unknown);
            t.Add(0, 101, 1, TickItaRelation.Unknown);

            ConcreteQuote ya = t.ConvertToQuotes(0, 120, StockExchange.T);
            Assert.AreEqual(2, ya.Count);
            Candle t1 = ya.CandleAt(0);
            Candle t2 = ya.CandleAt(1);
            Assert.AreEqual(100, t1.Open);
            Assert.AreEqual(101, t1.Close);
            Assert.AreEqual(2, t1.Volume);
            Assert.AreEqual(101, t2.Open);
            Assert.AreEqual(101, t2.Close);
            Assert.AreEqual(0, t2.Volume);
        }
        [Test]
        public void Basic2() {
            TimeAndSales t = new TimeAndSales();
            t.Add(120, 100, 1, TickItaRelation.Unknown);
            t.Add(121, 101, 1, TickItaRelation.Unknown);
            t.Add(240, 103, 3, TickItaRelation.Unknown);
            ConcreteQuote ya = t.ConvertToQuotes(0, 300, StockExchange.T);
            Assert.AreEqual(5, ya.Count);
            Candle t1 = ya.CandleAt(0);
            Candle t2 = ya.CandleAt(1);
            Candle t3 = ya.CandleAt(2);
            Candle t4 = ya.CandleAt(3);
            Candle t5 = ya.CandleAt(4);
            Assert.AreEqual(0, t1.Time);
            Assert.AreEqual(0, t1.Open);
            Assert.AreEqual(0, t1.Close);
            Assert.AreEqual(0, t1.Volume);
            Assert.AreEqual(60, t2.Time);
            Assert.AreEqual(0, t2.Open);
            Assert.AreEqual(0, t2.Close);
            Assert.AreEqual(0, t2.Volume);
            Assert.AreEqual(120, t3.Time);
            Assert.AreEqual(100, t3.Open);
            Assert.AreEqual(101, t3.Close);
            Assert.AreEqual(2, t3.Volume);
            Assert.AreEqual(180, t4.Time);
            Assert.AreEqual(101, t4.Open);
            Assert.AreEqual(101, t4.Close);
            Assert.AreEqual(0, t4.Volume);
            Assert.AreEqual(240, t5.Time);
            Assert.AreEqual(103, t5.Open);
            Assert.AreEqual(103, t5.Close);
            Assert.AreEqual(3, t5.Volume);

            Assert.AreEqual(103, t.LastPrice);
            Assert.AreEqual(101, t.PreDifferentPrice);
        }
        [Test]
        public void Hike() {
            TimeAndSales t = new TimeAndSales();
            int zc = 3600*11; //11:00AM
            t.Add(zc-60, 100, 1, TickItaRelation.Unknown);
            t.Add(zc,  101, 1, TickItaRelation.Unknown);

            ConcreteQuote ya = t.ConvertToQuotes(zc-60, zc, StockExchange.T);

            Assert.AreEqual(1, ya.Count);
            Candle t1 = ya.CandleAt(0);
            Assert.AreEqual(zc-60, t1.Time);
            Assert.AreEqual(100, t1.Open);
            Assert.AreEqual(101, t1.Close);
            Assert.AreEqual(2, t1.Volume);
            
        }

        [Test]
        public void FindRange() {
            TimeAndSales t = new TimeAndSales();
            t.Add(100, 100, 1, TickItaRelation.Unknown);
            t.Add(101, 101, 1, TickItaRelation.Unknown);
            t.Add(102, 102, 1, TickItaRelation.Unknown);
            t.Add(103, 103, 1, TickItaRelation.Unknown);
            t.Add(104, 104, 1, TickItaRelation.Unknown);

            int start, end;
            //S
            t.FindByTimeRangeBackward(100, 105, out start, out end);
            Assert.AreEqual(0, start);
            Assert.AreEqual(5, end);

            //
            t.FindByTimeRangeBackward(100, 103, out start, out end);
            Assert.AreEqual(0, start);
            Assert.AreEqual(3, end);

            t.FindByTimeRangeBackward(101, 102, out start, out end);
            Assert.AreEqual(1, start);
            Assert.AreEqual(2, end);

            //E
            t.FindByTimeRangeBackward(103, 106, out start, out end);
            Assert.AreEqual(3, start);
            Assert.AreEqual(5, end);

            t.FindByTimeRangeBackward(105, 106, out start, out end);
            Assert.AreEqual(-1, start);

            t.FindByTimeRangeBackward(99, 100, out start, out end);
            Assert.AreEqual(-1, start);

            t.FindByTimeRangeBackward(99, 103, out start, out end);
            Assert.AreEqual(0, start);
            Assert.AreEqual(3, end);

        }
    }
#endif
}
