/*
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Travis/LocationOperation.cs#13 $
 * $DateTime: 2008/03/13 13:17:01 $
 * 
 * Rectanglëˑ֌W
 */
using System;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;

#if UNITTEST
using NUnit.Framework;
#endif

using Travis.LocationOperation.Internal;

namespace Travis.LocationOperation {

    //Rectangleɂ傢Ƌ@\ǉNX
    public class BRect {
        private Rectangle _value;
        private NotifyRectangleOperator.D _handler;

        public BRect() {
            _handler = new NotifyRectangleOperator.D(Notify);
        }
        public BRect(Rectangle rect) {
            _value = rect;
            _handler = new NotifyRectangleOperator.D(Notify);
        }
        public BRect(Point topleft, Size size) {
            _value = new Rectangle(topleft, size);
            _handler = new NotifyRectangleOperator.D(Notify);
        }
        public override string ToString() {
            return _value.ToString();
        }
        private void Notify(Rectangle r) {
            _value = r;
        }
        public Rectangle Value {
            get {
                return _value;
            }
        }
        public bool IsEmpty {
            get {
                return _value.IsEmpty;
            }
        }
        public NotifyRectangleOperator.D Handler {
            get {
                return _handler;
            }
        }
        public void Set(int left, int top, int width, int height) {
            _value.X = left;
            _value.Y = top;
            _value.Width = width;
            _value.Height = height;
        }
        public void Set(Rectangle rect) {
            _value = rect;
        }

        //l͂Ƃ悤
        public int Left {
            get {
                return _value.Left;
            }
            set {
                _value.X = value;
            }
        }
        public int Right {
            get {
                return _value.Right;
            }
        }
        public int Top {
            get {
                return _value.Top;
            }
            set {
                _value.Y = value;
            }
        }
        public int Bottom {
            get {
                return _value.Bottom;
            }
        }
        public int Width {
            get {
                return _value.Width;
            }
            set {
                _value.Width = value;
            }
        }
        public int Height {
            get {
                return _value.Height;
            }
            set {
                _value.Height = value;
            }
        }
        public int CenterX {
            get {
                return _value.Left + _value.Width/2;
            }
        }
        public int CenterY {
            get {
                return _value.Top + _value.Height/2;
            }
        }
        public bool Contains(Point p) {
            return _value.Contains(p);
        }

        public bool ContainsX(int x) {
            return _value.Left <= x && x < _value.Right;
        }
        public bool ContainsY(int y) {
            return _value.Top <= y && y < _value.Bottom;
        }

        public bool IntersectsWith(ref Rectangle o) {
            return _value.IntersectsWith(o);
        }
        //Fłق
        public bool IntersectsWith(ref RectangleF o) {
            if(_value.Right < o.Left)  return false;
            if(_value.Left  > o.Right) return false;
            if(_value.Bottom < o.Top) return false;
            if(_value.Top > o.Bottom) return false;
            return true;
        }
        public Point GetCorner(RectCorner corner) {
            switch(corner) {
                case RectCorner.TopLeft:
                    return new Point(_value.Left, _value.Top);
                case RectCorner.TopRight:
                    return new Point(_value.Right, _value.Top);
                case RectCorner.BottomLeft:
                    return new Point(_value.Left, _value.Bottom);
                case RectCorner.BottomRight:
                    return new Point(_value.Right, _value.Bottom);
            }
            Debug.Assert(false);
            return new Point(0, 0);
        }
    }

    //
    public enum SplitDir {
        Top, Bottom, Left, Right
    }
    public enum RectCorner {
        TopLeft, TopRight, BottomLeft, BottomRight
    }
    
    /**
     * RectOpƂ
     * @ǂɕ\邩̈ʒu֌W錾Iɒ`@\B`߂ΉZRect()łĂB
     * @\ȃTCYȂƂɔ\ɂA㉺邢͍E̕䗦ȂŒsNZōsAƂƂłB
     * @_CAO{bNX̃Rg[̈ʒu֌WLq̂ɂgB
     */

    public abstract class RectOp {

        //Œp[^łȂςɂƂ͂gBS̃Iy[^ɔĂ킯ł͂Ȃ̂ŕKvɉĒǉĂ
        public delegate int RuntimeInt();
        public delegate bool RuntimeBool();

        public abstract void Rect(Rectangle rect);

        //util
        protected static Rectangle EmptyUnlessPositive(Rectangle rect) {
            if(rect.X>=0 && rect.Y>=0 && rect.Width>0 && rect.Height>0) //X,YOȂ̂͋Ƃɒ
                return rect;
            else
                return Rectangle.Empty;
        }
    }

    namespace Internal {
        //`dł
        public abstract class TrunkRectOp : RectOp {
            protected RectOp _trunk;
            public virtual void Bind(RectOp op) {
                Debug.Assert(_trunk==null); //P̂݃oCh
                _trunk = op;
            }

            public RectOp Trunk {
                get {
                    return _trunk;
                }
            }
        }


        //PZ
        public abstract class UnaryRectOp : TrunkRectOp {

            protected abstract Rectangle Operate(Rectangle rect);

            public override void Rect(Rectangle rect) {
                if(_trunk!=null) {
                    if(rect.IsEmpty)
                        _trunk.Rect(rect);
                    else
                        _trunk.Rect(EmptyUnlessPositive(Operate(rect)));
                }
            }

        }

        //H͂Ȃ
        public abstract class NOPRectOp : UnaryRectOp {
            protected override Rectangle Operate(Rectangle rect) {
                return rect;
            }
        }

        //̊{NX
        public abstract class BranchRectOp : TrunkRectOp {
            private RectOp _branch;

            public override void Bind(RectOp br) {
                Debug.Assert(br!=null);
                if(_branch==null) //Q̂݃oChBBranch->Trunk̏
                    _branch = br;
                else
                    base.Bind(br);
            }

            public override void Rect(Rectangle rect) {
                Rectangle branch, trunk;
                if(rect.IsEmpty) {
                    branch = Rectangle.Empty;
                    trunk = Rectangle.Empty;
                }
                else
                    OperateBranch(rect, out branch, out trunk);

                if(_branch!=null) _branch.Rect(EmptyUnlessPositive(branch));
                if(_trunk!=null) _trunk.Rect(EmptyUnlessPositive(trunk));
            }
            protected abstract void OperateBranch(Rectangle rect, out Rectangle branch, out Rectangle trunk);

            public RectOp Branch {
                get {
                    return _branch;
                }
            }

        }


        //Z͂ʒm
        public class NotifyRectangleOperator : NOPRectOp {
            //Zʂ̃Lb`|Cg
            public delegate void D(Rectangle rect);
            protected D _d;
            public NotifyRectangleOperator(D d) {
                _d = d;
            }
            public D Notify {
                get {
                    return _d;
                }
                set {
                    _d = value;
                }
            }
            public override void Rect(Rectangle rect) {
                if(_d!=null) _d(rect);
                base.Rect(rect);
            }

        }

        //Rg[̃TCY
        public class ControlBindOperator : NOPRectOp {
            private Control _target;
            private bool _autoVisible; //rectemptŷƂIVisible𒲐

            public ControlBindOperator(Control target, bool autovisible) {
                _target = target;
                _autoVisible = autovisible;
            }

            public override void Rect(Rectangle rect) {
                _target.Location = rect.Location;
                _target.Size = rect.Size;
                if(_autoVisible) _target.Visible = !rect.IsEmpty;
                base.Rect(rect);
            }
        }

        //BRectɒʒm
        public class BRectBindOperator : NOPRectOp {
            private BRect _target;
            public BRectBindOperator(BRect target) {
                _target = target;
            }
            public override void Rect(Rectangle rect) {
                _target.Set(rect);
                base.Rect(rect);
            }
        }


        //ꂼZ
        public sealed class ShiftRectangleOperator : UnaryRectOp {
            private int _x;
            private int _y;
            private int _width;
            private int _height;
            public ShiftRectangleOperator(int x, int y, int width, int height) {
                _x = x;
                _y = y;
                _width = width;
                _height = height;
            }
            protected override Rectangle Operate(Rectangle rect) {
                return new Rectangle(rect.X+_x, rect.Y+_y, rect.Width+_width, rect.Height+_height);
            }
        }

        //ӂJbg
        public sealed class CutRectangleOperator : UnaryRectOp {
            private SplitDir _dir;
            private int _length;
            public CutRectangleOperator(SplitDir dir, int length) {
                _dir = dir;
                _length = length;
            }
            protected override Rectangle Operate(Rectangle rect) {
                switch(_dir) {
                    case SplitDir.Top:
                        return new Rectangle(rect.X, rect.Y+_length, rect.Width, rect.Height-_length);
                    case SplitDir.Bottom:
                        return new Rectangle(rect.X, rect.Y, rect.Width, rect.Height-_length);
                    case SplitDir.Left:
                        return new Rectangle(rect.X+_length, rect.Y, rect.Width-_length, rect.Height);
                    case SplitDir.Right:
                        return new Rectangle(rect.X, rect.Y, rect.Width-_length, rect.Height);
                }
                Debug.Assert(false); //L̂ꂩ
                return Rectangle.Empty;
            }
        }
        public sealed class CenteringRectangleOperator : UnaryRectOp {
            private bool _horizontal;
            private int _size;

            public CenteringRectangleOperator(bool horizontal, int size) {
                _horizontal =  horizontal;
                _size = size;
            }
            protected override Rectangle Operate(Rectangle rect) {
                if(_horizontal)
                    return new Rectangle(rect.X+rect.Width/2-_size/2, rect.Y, _size, rect.Height);
                else
                    return new Rectangle(rect.X, rect.Y+rect.Height/2-_size/2, rect.Width, _size);
            }
        }

        //TChɔzuBÊƂY,㉺̂ƂXW킹BTCY͌Œl
        public sealed class SideRectangleOperator : UnaryRectOp {
            private SplitDir _dir; //SplitDir𗬗p
            private int _margin;
            private int _width;
            private int _height;
            public SideRectangleOperator(SplitDir dir, int margin, int width, int height) {
                _dir = dir;
                _margin = margin;
                _width = width;
                _height = height;
            }
            protected override Rectangle Operate(Rectangle rect) {
                switch(_dir) {
                    case SplitDir.Top:
                        return new Rectangle(rect.X, rect.Y - _margin - _height, _width, _height);
                    case SplitDir.Bottom:
                        return new Rectangle(rect.X, rect.Bottom + _margin, _width, _height);
                    case SplitDir.Left:
                        return new Rectangle(rect.X - _margin - _width, rect.Y, _width, _height);
                    case SplitDir.Right:
                        return new Rectangle(rect.Right + _margin, rect.Y, _width, _height);
                }
                Debug.Assert(false); //L̂ꂩ
                return Rectangle.Empty;
            }
        }
        //K̃TCYmۂłĂȂEmpty
        public sealed class MinSizeRectangleOperator : UnaryRectOp {
            private int _minWidth;
            private int _minHeight;
            public MinSizeRectangleOperator(int minwidth, int minheight) {
                _minWidth = minwidth;
                _minHeight = minheight;
            }
            protected override Rectangle Operate(Rectangle rect) {
                if(rect.Width < _minWidth || rect.Height < _minHeight)
                    return Rectangle.Empty;
                else
                    return rect;
            }
        }
        //p珊̃ItZbgAуTCY
        public sealed class CornerOffsetRectangleOperator : UnaryRectOp {
            private RectCorner _corner;
            private int _offsetX;
            private int _offsetY;
            private int _width;
            private int _height;
            public CornerOffsetRectangleOperator(RectCorner corner, int offsetX, int offsetY, int width, int height) {
                _corner = corner;
                _offsetX = offsetX;
                _offsetY = offsetY;
                _width = width;
                _height = height;
            }
            protected override Rectangle Operate(Rectangle rect) {
                int x = (_corner==RectCorner.TopLeft || _corner==RectCorner.BottomLeft)? rect.X : rect.Right;
                int y = (_corner==RectCorner.TopLeft || _corner==RectCorner.TopRight)? rect.Y : rect.Bottom;
                return new Rectangle(x + _offsetX, y + _offsetY, _width, _height);
            }
        }
        //ContentAlignmentɂēɔzuBTCY͏Ԃł܂Ȃ
        public sealed class ContentAlignmentRectangleOperator : UnaryRectOp {
            private ContentAlignment _align;
            private int _margin;
            private int _width;
            private int _height;

            public ContentAlignmentRectangleOperator(ContentAlignment align, int margin, int width, int height) {
                _align = align;
                _margin = margin;
                _width = width;
                _height = height;
            }

            protected override Rectangle Operate(Rectangle rect) {
                //ContentAlignment̐\mFASQׂ̂ňقȂ鐔lȂ̂if
                int x, y;
                if(_align==ContentAlignment.TopLeft || _align==ContentAlignment.MiddleLeft || _align==ContentAlignment.BottomLeft)
                    x = rect.X + _margin;
                else if(_align==ContentAlignment.TopCenter || _align==ContentAlignment.MiddleCenter || _align==ContentAlignment.BottomCenter)
                    x = rect.X + (rect.Width - _width) / 2;
                else
                    x = rect.Right - _margin - _width;

                if(_align==ContentAlignment.TopLeft || _align==ContentAlignment.TopCenter || _align==ContentAlignment.TopRight)
                    y = rect.Y + _margin;
                else if(_align==ContentAlignment.MiddleLeft || _align==ContentAlignment.MiddleCenter || _align==ContentAlignment.MiddleRight)
                    y = rect.Y + (rect.Height - _height) / 2;
                else
                    y = rect.Bottom - _margin - _height;

                return new Rectangle(x, y, _width, _height);
            }
        }

        //Z͂ɂQɕ򂷂Iy[^
        public class TeeRectangleOperator : BranchRectOp {
            protected override void OperateBranch(Rectangle rect, out Rectangle branch, out Rectangle trunk) {
                branch = rect;
                trunk = rect;
            }
        }


        //㉺ÊꂩɂQ邽߂̕Iy[^
        public abstract class SplitBranchRectangleOperatorBase : BranchRectOp {
            protected SplitDir _branchDir;
            protected override void OperateBranch(Rectangle rect, out Rectangle branch, out Rectangle trunk) {
                int sz = BranchSize(rect);
                switch(_branchDir) {
                    case SplitDir.Top:
                        branch = new Rectangle(rect.X, rect.Y, rect.Width, sz);
                        trunk = new Rectangle(rect.X, rect.Y+sz, rect.Width, rect.Height-sz);
                        break;
                    case SplitDir.Bottom:
                        branch = new Rectangle(rect.X, rect.Bottom-sz, rect.Width, sz);
                        trunk = new Rectangle(rect.X, rect.Y, rect.Width, rect.Height-sz);
                        break;
                    case SplitDir.Left:
                        branch = new Rectangle(rect.X, rect.Y, sz, rect.Height);
                        trunk = new Rectangle(rect.X+sz, rect.Y, rect.Width-sz, rect.Height);
                        break;
                    case SplitDir.Right:
                        branch = new Rectangle(rect.Right-sz, rect.Y, sz, rect.Height);
                        trunk = new Rectangle(rect.X, rect.Y, rect.Width-sz, rect.Height);
                        break;
                    default:
                        Debug.Assert(false); //L̂ꂩ
                        branch = Rectangle.Empty;
                        trunk = Rectangle.Empty;
                        break;
                }
            }
            protected abstract int BranchSize(Rectangle rect); //hNXŕTCY
        }

        //_splitRatioŕ邪A}_branchMinSizemۂłȂ΂߂đSʂ
        public sealed class SplitRatioBranchRectangleOperator : SplitBranchRectangleOperatorBase {
            private float _splitRatio;
            private int _branchMinSize;

            public SplitRatioBranchRectangleOperator(SplitDir dir, float ratio, int minsize) {
                Debug.Assert(ratio>0 && ratio<1.0F);
                _branchDir = dir;
                _splitRatio = ratio;
                _branchMinSize = minsize;
            }
            protected override int BranchSize(Rectangle rect) {
                int s = (_branchDir==SplitDir.Top || _branchDir==SplitDir.Bottom)? rect.Height : rect.Width;
                int v = (int)(s * _splitRatio);
                return v<_branchMinSize? 0 : v;
            }
        }

        //size݂̕邪AŜ̃TCYminWholeSizeȏłȂ΂߂
        public sealed class SplitFixedSizeBranchRectangleOperator : SplitBranchRectangleOperatorBase {
            private int _splitSize;
            private int _minWholeSize;
            public SplitFixedSizeBranchRectangleOperator(SplitDir dir, int split_size, int min_whole_size) {
                _branchDir = dir;
                _splitSize = split_size;
                _minWholeSize = min_whole_size;
            }
            protected override int BranchSize(Rectangle rect) {
                int s = (_branchDir==SplitDir.Top || _branchDir==SplitDir.Bottom)? rect.Height : rect.Width;
                return s<_minWholeSize? 0 : _splitSize;
            }
        }
        //RuntimeInt
        public sealed class SplitLateBindFixedSizeBranchRectangleOperator : SplitBranchRectangleOperatorBase {
            private RuntimeInt _splitSize;
            private RuntimeInt _minWholeSize;
            public SplitLateBindFixedSizeBranchRectangleOperator(SplitDir dir, RuntimeInt split_size, RuntimeInt min_whole_size) {
                _branchDir = dir;
                _splitSize = split_size;
                _minWholeSize = min_whole_size;
            }
            protected override int BranchSize(Rectangle rect) {
                int s = (_branchDir==SplitDir.Top || _branchDir==SplitDir.Bottom)? rect.Height : rect.Width;
                return (_minWholeSize!=null && s<_minWholeSize())? 0 : _splitSize();
            }
        }

        public sealed class DynamicBranchOperator : BranchRectOp {
            private RuntimeBool _condition;
            public DynamicBranchOperator(RuntimeBool cond) {
                _condition = cond;
            }
            protected override void OperateBranch(Rectangle rect, out Rectangle branch, out Rectangle trunk) {
                if(_condition()) {
                    branch = rect;
                    trunk = Rectangle.Empty;
                }
                else {
                    branch = Rectangle.Empty;
                    trunk = rect;
                }
            }
        }
    }


    //RectangleOperator邽߂̃NX
    public class RectangleOperatorBuilder {
        private RectOp _root;
        private TrunkRectOp _trunk; //I[Zbgnull
        private Stack<BranchRectOp> _branchNodes; //򂵂Ė̃m[h
        private int _margin;

        public RectangleOperatorBuilder() {
            _branchNodes = new Stack<BranchRectOp>();
        }
        public RectangleOperatorBuilder(int margin) : this() {
            _margin = margin;
        }
        public int Margin {
            get {
                return _margin;
            }
        }
        public RectangleOperatorBuilder SetMargin(int margin) {
            _margin = margin;
            return this;
        }

        public RectOp Finish() {
            Debug.Assert(_branchNodes.Count==0); //ʂoɂ͑Sς݂łȂ
            
            return _root;
        }


        private void AppendTrunkInternal(TrunkRectOp op) {
            Debug.Assert(op!=null);
            if(_root==null)
                _root = op; //ŏAppendŃ[g쐬
            else {
                if(_trunk==null) throw new InvalidOperationException("RectangleOp\zG["); //ĂяoԈႢ
                _trunk.Bind(op);
            }
            _trunk = op;
        }
        private void AppendBranchInternal(BranchRectOp op) {
            _branchNodes.Push(op);
            AppendTrunkInternal(op);
        }

        //Appendł͊Ɛ錾
        public RectangleOperatorBuilder CloseBranch() {
            BranchRectOp b = _branchNodes.Pop();
            _trunk = b;
            return this;
        }

        //悭}[N
        public RectangleOperatorBuilder ProbeBRect(BRect r) {
            Debug.Assert(r!=null);
            AppendTrunkInternal(new BRectBindOperator(r));
            return this;
        }
        public RectangleOperatorBuilder ProbeControlAutoVisible(Control c) {
            Debug.Assert(c!=null);
            AppendTrunkInternal(new ControlBindOperator(c, true));
            return this;
        }
        public RectangleOperatorBuilder ProbeControl(Control c) {
            Debug.Assert(c!=null);
            AppendTrunkInternal(new ControlBindOperator(c, false));
            return this;
        }

        //ڔ
        public RectangleOperatorBuilder Append(RectOp op) {
            Debug.Assert(op is TrunkRectOp);
            AppendTrunkInternal((TrunkRectOp)op);
            return this;
        }
        public RectangleOperatorBuilder AppendBranch(BranchRectOp op) {
            AppendBranchInternal(op);
            return this;
        }
        //I
        public RectangleOperatorBuilder Condition(RectOp.RuntimeBool cond) {
            AppendBranchInternal(new DynamicBranchOperator(cond));
            return this;
        }
        //w肵Ē
        public RectangleOperatorBuilder HCenter(int width) {
            AppendTrunkInternal(new CenteringRectangleOperator(true, width));
            return this;
        }
        //w肵Ē
        public RectangleOperatorBuilder VCenter(int height) {
            AppendTrunkInternal(new CenteringRectangleOperator(false, height));
            return this;
        }

        //Sk
        public RectangleOperatorBuilder Shrink(int border_width) {
            ShiftRectangleOperator nc = new ShiftRectangleOperator(border_width, border_width, -border_width*2, -border_width*2);
            AppendTrunkInternal(nc);
            return this;
        }
        //ړ
        public RectangleOperatorBuilder Move(int dx, int dy) {
            ShiftRectangleOperator nc = new ShiftRectangleOperator(dx, dy, 0, 0);
            AppendTrunkInternal(nc);
            return this;
        }
        //Jbg
        public RectangleOperatorBuilder Cut(SplitDir dir, int length) {
            CutRectangleOperator nc = new CutRectangleOperator(dir, length);
            AppendTrunkInternal(nc);
            return this;
        }

        //inopj
        public RectangleOperatorBuilder Tee() {
            AppendBranchInternal(new TeeRectangleOperator());
            return this;
        }

        // (䗦)
        public RectangleOperatorBuilder SplitRatio(SplitDir branch, float ratio, int minsize) {
            SplitRatioBranchRectangleOperator nc = new SplitRatioBranchRectangleOperator(branch, ratio, minsize);
            AppendBranchInternal(nc);
            return this;
        }
        // (ŒTCY)
        public RectangleOperatorBuilder SplitFixed(SplitDir branch, int size, int minWhole) {
            SplitFixedSizeBranchRectangleOperator nc = new SplitFixedSizeBranchRectangleOperator(branch, size, minWhole);
            AppendBranchInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder SplitFixed(SplitDir branch, RectOp.RuntimeInt size, RectOp.RuntimeInt minWhole) {
            SplitLateBindFixedSizeBranchRectangleOperator nc = new SplitLateBindFixedSizeBranchRectangleOperator(branch, size, minWhole);
            AppendBranchInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder SplitFixed(SplitDir branch, int size) {
            SplitFixedSizeBranchRectangleOperator nc = new SplitFixedSizeBranchRectangleOperator(branch, size, 0);
            AppendBranchInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder SplitFixed(SplitDir branch, RectOp.RuntimeInt size) {
            SplitLateBindFixedSizeBranchRectangleOperator nc = new SplitLateBindFixedSizeBranchRectangleOperator(branch, size, null);
            AppendBranchInternal(nc);
            return this;
        }
        //SplitFixed̊ȗ
        public RectangleOperatorBuilder SplitTop(int size) {
            return SplitFixed(SplitDir.Top, size);
        }
        public RectangleOperatorBuilder SplitLeft(int size) {
            return SplitFixed(SplitDir.Left, size);
        }

        //ftHg}[Wg㉺E
        public RectangleOperatorBuilder Right(int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Right, _margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder Left(int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Left, _margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder Above(int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Top, _margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder Bottom(int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Bottom, _margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        //}[WwŎg㉺E MM = Manual Margin
        public RectangleOperatorBuilder RightMM(int margin, int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Right, margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder LeftMM(int margin, int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Left, margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder AboveMM(int margin, int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Top, margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        public RectangleOperatorBuilder BottomMM(int margin, int width, int height) {
            SideRectangleOperator nc = new SideRectangleOperator(SplitDir.Bottom, margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }

        //CornetOffset
        public RectangleOperatorBuilder CornerOffset(RectCorner corner, int offsetX, int offsetY, int width, int height) {
            CornerOffsetRectangleOperator nc = new CornerOffsetRectangleOperator(corner, offsetX, offsetY, width, height);
            AppendTrunkInternal(nc);
            return this;
        }
        //ContentAlignment
        public RectangleOperatorBuilder ContentAlignment(ContentAlignment align, int width, int height) {
            ContentAlignmentRectangleOperator nc = new ContentAlignmentRectangleOperator(align, _margin, width, height);
            AppendTrunkInternal(nc);
            return this;
        }

        //ŒTCỸRg[zuRrl[V
        public RectangleOperatorBuilder PlaceControl(ContentAlignment align, Control control) {
            return PlaceControl(align, control, control.Width, control.Height);
        }
        public RectangleOperatorBuilder PlaceControl(ContentAlignment align, Control control, int width, int height) {
            ContentAlignmentRectangleOperator nc = new ContentAlignmentRectangleOperator(align, _margin, width, height);
            AppendTrunkInternal(nc);
            return ProbeControl(control);
        }
        public RectangleOperatorBuilder LeftControl(Control control) {
            return LeftControl(control, control.Width, control.Height);
        }
        public RectangleOperatorBuilder LeftControl(Control control, int width, int height) {
            Left(width, height);
            return ProbeControl(control);
        }
        public RectangleOperatorBuilder RightControl(Control control) {
            return RightControl(control, control.Width, control.Height);
        }
        public RectangleOperatorBuilder RightControlMM(Control control, int margin) {
            return RightControlMM(control, margin, control.Width, control.Height);
        }
        public RectangleOperatorBuilder RightControl(Control control, int width, int height) {
            Right(width, height);
            return ProbeControl(control);
        }
        public RectangleOperatorBuilder RightControlMM(Control control, int margin, int width, int height) {
            RightMM(margin, width, height);
            return ProbeControl(control);
        }

        public RectangleOperatorBuilder BottomControl(Control control) {
            return BottomControl(control, control.Width, control.Height);
        }
        public RectangleOperatorBuilder BottomControl(Control control, int width, int height) {
            Bottom(width, height);
            return ProbeControl(control);
        }
        public RectangleOperatorBuilder AboveControl(Control control) {
            return AboveControl(control, control.Width, control.Height);
        }
        public RectangleOperatorBuilder AboveControl(Control control, int width, int height) {
            Above(width, height);
            return ProbeControl(control);
        }
    }

#if UNITTEST
    [TestFixture]
    public class LocationOperationTests {
        [Test]
        public void RectangleBasic() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder().Shrink(10).ProbeBRect(h1).Move(30, 40).ProbeBRect(h2).Finish();

            r.Rect(new Rectangle(0, 0, 100, 100));
            Assert.AreEqual(new Rectangle(10, 10, 80, 80), h1.Value);
            Assert.AreEqual(new Rectangle(40, 50, 80, 80), h2.Value);

            r.Rect(new Rectangle(0, 0, 15, 15));
            Assert.AreEqual(new Rectangle(0,0,0,0), h1.Value);
            Assert.AreEqual(new Rectangle(0, 0, 0, 0), h2.Value);
        }

        [Test]
        public void SplitRatio() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder().SplitRatio(SplitDir.Left, 0.2F, 0).ProbeBRect(h1).CloseBranch().ProbeBRect(h2).Finish();

            r.Rect(new Rectangle(100, 100, 200, 100));
            Assert.AreEqual(new Rectangle(100, 100,  40, 100), h1.Value);
            Assert.AreEqual(new Rectangle(140, 100, 160, 100), h2.Value);

            r = new RectangleOperatorBuilder().SplitRatio(SplitDir.Right, 0.2F, 0).ProbeBRect(h1).CloseBranch().ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(100, 100, 200, 100));
            Assert.AreEqual(new Rectangle(260, 100,  40, 100), h1.Value);
            Assert.AreEqual(new Rectangle(100, 100, 160, 100), h2.Value);

            r = new RectangleOperatorBuilder().SplitRatio(SplitDir.Top, 0.2F, 0).ProbeBRect(h1).CloseBranch().ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(100, 100, 200, 100));
            Assert.AreEqual(new Rectangle(100, 100, 200, 20), h1.Value);
            Assert.AreEqual(new Rectangle(100, 120, 200, 80), h2.Value);

            r = new RectangleOperatorBuilder().SplitRatio(SplitDir.Bottom, 0.2F, 0).ProbeBRect(h1).CloseBranch().ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(100, 100, 200, 100));
            Assert.AreEqual(new Rectangle(100, 180, 200, 20), h1.Value);
            Assert.AreEqual(new Rectangle(100, 100, 200, 80), h2.Value);

            r = new RectangleOperatorBuilder().SplitRatio(SplitDir.Bottom, 0.2F, 30).ProbeBRect(h1).CloseBranch().ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(100, 100, 200, 100));
            Assert.AreEqual(new Rectangle(0,0,0, 0), h1.Value);
            Assert.AreEqual(new Rectangle(100, 100, 200, 100), h2.Value);
            
            //SplitFixed͂ĂȂ
        }


        private class RT {
            public int _minWhole;
            public int _size;
            public int minWhole() { return _minWhole; }
            public int size() { return _size; }
        }
        [Test]
        public void SplitRuntimeFixed() {
            BRect h1 = new BRect();
            RT size = new RT();
            RectOp r = new RectangleOperatorBuilder().SplitFixed(SplitDir.Top, new RectOp.RuntimeInt(size.size), new RectOp.RuntimeInt(size.minWhole)).ProbeBRect(h1).CloseBranch().Finish();

            size._size = 10;
            size._minWhole = 0;

            r.Rect(new Rectangle(0, 0, 100, 100));
            Assert.AreEqual(new Rectangle(0, 0, 100, 10), h1.Value);

            size._size = 30;
            r.Rect(new Rectangle(0, 0, 100, 100));
            Assert.AreEqual(new Rectangle(0, 0, 100, 30), h1.Value);


        }
        

        [Test]
        public void Side() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder().SetMargin(8).Right(24,16).ProbeBRect(h1).Bottom(32,32).ProbeBRect(h2).Finish();

            r.Rect(new Rectangle(0,0,16,16));
            Assert.AreEqual(new Rectangle(24,  0, 24,16), h1.Value);
            Assert.AreEqual(new Rectangle(24, 24, 32,32), h2.Value);
        }

        [Test]
        public void Tee() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder(8).Tee()
                .Bottom(32, 32).ProbeBRect(h1).CloseBranch()
                .Right(24,24).ProbeBRect(h2)
                .Finish();
            r.Rect(new Rectangle(0, 0, 16, 16));
            Assert.AreEqual(new Rectangle(0, 24, 32, 32), h1.Value);
            Assert.AreEqual(new Rectangle(24, 0, 24, 24), h2.Value);

        }

        [Test]
        public void Move() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder().Move(32, 32).ProbeBRect(h1).Finish();

            r.Rect(new Rectangle(0, 0, 16, 32));
            Assert.AreEqual(new Rectangle(32, 32, 16, 32), h1.Value);
            Assert.AreEqual(new Rectangle(0, 0, 0, 0), h2.Value);
        }

        [Test]
        public void CornerOffset() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder().CornerOffset(RectCorner.BottomRight, -10, 10, 20, 30).ProbeBRect(h1).CornerOffset(RectCorner.TopRight, 10, 10, 20, 30).ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(0, 0, 16, 32));
            Assert.AreEqual(new Rectangle(6, 42, 20, 30), h1.Value);
            Assert.AreEqual(new Rectangle(36, 52, 20, 30), h2.Value);
        }

        [Test]
        public void ContentAlignmentTest() {
            BRect h1 = new BRect();
            BRect h2 = new BRect();
            RectOp r = new RectangleOperatorBuilder(4).ContentAlignment(ContentAlignment.TopLeft, 10, 20).ProbeBRect(h1)
                .Right(5, 10).ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(0, 0, 100, 100));
            Assert.AreEqual(new Rectangle(4, 4, 10, 20), h1.Value);
            Assert.AreEqual(new Rectangle(18, 4, 5, 10), h2.Value);

            r = new RectangleOperatorBuilder(4).ContentAlignment(ContentAlignment.MiddleRight, 10, 20).ProbeBRect(h1)
                .Bottom(5, 10).ProbeBRect(h2).Finish();
            r.Rect(new Rectangle(0, 0, 100, 100));
            Assert.AreEqual(new Rectangle(86, 40, 10, 20), h1.Value);
            Assert.AreEqual(new Rectangle(86, 64, 5, 10), h2.Value);

        }
    }
#endif
}
