/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/Stock.cs#19 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * IuWFNg
 * ƂĂ̏̂StockProfileŁAΏۂ̖ListedStockB
 * Ƃ΁A؁E؂̗ɏꂵĂƁAListedStock͂ꂼɂĂAStockProfilêQƂɂȂB
 * [gIuWFNg̎Qƍ\́ABellagioRoot.GlobalStockCollection..StockProfile.ListedStock ƂȂB
 */
using System;
using System.Diagnostics;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using Poderosa.Util.Collections;
using Travis.Collections;

using Poderosa;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {
    //
    public class SplitInfo {
        private int _date;
        private double _ratio;

        public int Date {
            get {
                return _date;
            }
        }
        public double Ratio {
            get {
                return _ratio;
            }
        }
        public SplitInfo(int date, double ratio) {
            _date = date;
            _ratio = ratio;
        }
        public SplitInfo(string txt) {
            //S:ddd:value̍\z
            string[] t = txt.Split(':');
            _date = Int32.Parse(t[1]);
            _ratio = Double.Parse(t[2]);
        }
    }

    //̊{
    public abstract class AbstractStockProfile : IAdaptable {
        protected string _name;
        protected string _alternativeName; //ʖ
        protected string _code;
        protected int _volumeUnit; //Pʊ ꂪĂ͂܂Ȃł͂O
        protected Stock _primary;
        protected Stock _secondary; //DłȂs
        protected SplitInfo[] _splitInfo;
        protected StockProfileFlags _flags;

        public AbstractStockProfile(string name, string code, StockProfileFlags f, int volumeUnit) {
            _name = name;
            _code = code;
            _flags = f;
            _volumeUnit = volumeUnit;
        }
        public string Name {
            get {
                return _name;
            }
        }
        public string Code {
            get {
                return _code;
            }
        }
        public Stock Primary {
            get {
                return _primary;
            }
        }
        public Stock Secondary {
            get {
                return _secondary;
            }
        }
        public StockProfileFlags Flags {
            get {
                return _flags;
            }
            set {
                _flags = value;
            }
        }
        public int VolumeUnit {
            get {
                return _volumeUnit;
            }
        }
        //potOn
        public bool IsDerivative { 
            get {
                return (_flags & StockProfileFlags.DerivativeMask)!=StockProfileFlags.None;
            }
        }
        public bool IsTransient {
            get {
                return (_flags & StockProfileFlags.Transient)!=StockProfileFlags.None;
            }
        }
        public bool IsHidden {
            get {
                return (_flags & StockProfileFlags.Hidden)!=StockProfileFlags.None;
            }
        }

        public string AlternativeName {
            get {
                return _alternativeName;
            }
            set {
                _alternativeName = value;
            }
        }
        public SplitInfo[] SplitInfo {
            get {
                return _splitInfo;
            }
            set {
                _splitInfo = value;
            }
        }

        public Stock CreatePrimary(StockExchange exch, StockExchangeSubType subtype, StockFlags flags) {
            _primary = new Stock(this, exch, subtype, flags);
            return _primary;
        }
        public Stock CreateSecondary(StockExchange exch, StockExchangeSubType subtype, StockFlags flags) {
            _secondary = new Stock(this, exch, subtype, flags);
            return _secondary;
        }

        //ۑĂf[^ւ̕ϊɉ΂悢
		public virtual double PriceScale {
			get {
                StockProfileFlags f = _flags & StockProfileFlags.FigureMask;

                if(f==StockProfileFlags.F1)
                    return 0.1;
                else if(f==StockProfileFlags.F2)
                    return 0.01;
                else
                    return 1;
			}
		}
        public double AdjustPriceD(double value) {
            StockProfileFlags f = _flags & StockProfileFlags.FigureMask;
            if(f==StockProfileFlags.None)
                return value; //̃P[Xő
            else
                return value * this.PriceScale;
        }
        public string PriceFormatString {
            get {
                StockProfileFlags f = _flags & StockProfileFlags.FigureMask;

                if(f==StockProfileFlags.F1)
                    return "F1";
                else if(f==StockProfileFlags.F2)
                    return "F2";
                else
                    return "F0";
            }
        }
        public string FormatPriceD(double price) {
            StockProfileFlags f = _flags & StockProfileFlags.FigureMask;

            if(f==StockProfileFlags.F1)
                return (price*0.1).ToString("F1");
            else if(f==StockProfileFlags.F2)
                return (price*0.01).ToString("F2");
            else
                return price.ToString("F0");
        }
        public string FormatPrice(int price) {
            StockProfileFlags f = _flags & StockProfileFlags.FigureMask;

            if(f==StockProfileFlags.F1)
                return String.Format("{0}.{1:D1}", price/10, (price % 10));
            else if(f==StockProfileFlags.F2)
                return String.Format("{0}.{1:D2}", price/100, (price % 100));
            else
                return price.ToString();
        }

        //PʂB"",""̂ǂ炩
        public abstract string VolumeUnitString {
            get;
        }
        //ȂǂvZƂ̈Öق̌WB 敨EIvVł1000
        public abstract int ImplicitPriceFactor {
            get;
        }
        //f[^\[XŎgpR[h@GMOȊOɑΉƂɂ͂̂̎dg݂G邾낤
        public abstract string CodeForDataSource();

        //basis̓t̂Pdate̓ɉ{ɂȂĂ邩𕪊f[^vZ
        public double CalcSplitRatio(int date, int basis) {
            Debug.Assert(basis >= date); //ߋɌf[^
            if(_splitInfo==null) return 1;

            double r = 1;
            for(int i=_splitInfo.Length-1; i>=0; i--) {
                SplitInfo si = _splitInfo[i];
                if(si.Date > basis)
                    continue;
                else if(date < si.Date)
                    r *= si.Ratio;
                else
                    break;
            }
            return r;
        }
        //Sv̂݌
        public SplitInfo FindSplitInfo(int date) {
            if(_splitInfo==null) return null;
            for(int i=_splitInfo.Length-1; i>=0; i--) {
                SplitInfo si = _splitInfo[i];
                if(si.Date==date) return si;
            }
            return null;
        }
        //͈͌
        public SplitInfo FindSplitInfo(int start, int end) {
            if(_splitInfo==null) return null;
            for(int i=_splitInfo.Length-1; i>=0; i--) {
                SplitInfo si = _splitInfo[i];
                if(start <= si.Date && si.Date < end) return si;
            }
            return null;
        }

        public StockExchange[] GetStockExhcnageArray() {
            if(_secondary==null)
                return new StockExchange[] { _primary.Market };
            else
                return new StockExchange[] { _primary.Market , _secondary.Market };
        }
        public Stock MatchMarket(StockExchange ex) {
            if(_primary.Market==ex)
                return _primary;
            else if(_secondary!=null && _secondary.Market==ex)
                return _secondary;
            else
                return null;
        }


        protected Stock CreateStock(StockExchange exch, StockExchangeSubType subtype, StockFlags flags) {
            Stock s = new Stock(this, exch, subtype, flags);
            return s;
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

    //ʂ̂P
    public class BasicStockProfile : AbstractStockProfile {
        private MarketSectors _sector;

        public BasicStockProfile(string name, string code, StockProfileFlags f, int volumeUnit)
            : base(name, code, f, volumeUnit) {
        }

        public override string VolumeUnitString {
            get { return ""; }
        }
        public override int ImplicitPriceFactor {
            get { return 1; }
        }
        public override string CodeForDataSource() {
            return _code;
        }

        public MarketSectors Sector {
            get {
                return _sector;
            }
            set {
                _sector = value;
            }
        }

        public static BasicStockProfile CreateTransient(string code, string name, StockExchange ex) {
            BasicStockProfile p = new BasicStockProfile(name, code, StockProfileFlags.Transient|StockProfileFlags.Hidden, 1);
            p.CreatePrimary(ex, StockExchangeSubType.None, StockFlags.None);
            return p;
        }
    }


    //݁Af[^\[XłDefault, DaytimêQނT|[gƂɂȂĂ
    public enum EveningSessionAware {
        Preference, //PreferenceɈˑČ
        Default,    //̒̕WBCujO͗Ɋ܂
        Daytime,    //݂̂̃f[^
        //Today       //Ɋ܂߂
    }

    //foeBu ̓IvVŕ\ł悤
    
    //،Ƌ̂XR[h
    //肠ă[ http://www.tse.or.jp/sicc/code/cd_rule.html

    public class DerivativeStockProfile : AbstractStockProfile {
        public enum BrandT {
            NikkeiFuture,    //o敨
            NikkeiFutureMini //o敨~j
        }
        private BrandT _brand;
        private EveningSessionAware _eveningSession;
        private string _nineDigitCode;
        private bool _concatinatedFuture; //SԒʎZ

        //`CreateElementsōs
        internal DerivativeStockProfile(string name, string code, string ninecode, BrandT br, StockExchange exch, StockProfileFlags flags, EveningSessionAware evening, StockFlags stockFlags) 
            : base(name, code, flags, 1) {
            Debug.Assert(ninecode.Length==0 || ninecode.Length==9);
            _nineDigitCode = ninecode;
            _eveningSession = evening;

            _brand = br;
            _volumeUnit = 1;
            _primary = new Stock(this, exch, StockExchangeSubType.None, stockFlags);

        }
        internal DerivativeStockProfile(string name, string code, string ninecode, BrandT br, StockExchange exch, StockProfileFlags flags, EveningSessionAware evening)
            : this(name, code, ninecode, br, exch, flags, evening, StockFlags.None) {
        }
        public BrandT Brand {
            get {
                return _brand;
            }
        }
        public EveningSessionAware EveningSessionAware {
            get {
                return _eveningSession;
            }
        }
        public bool ConcatinatedFuture {
            get {
                return _concatinatedFuture;
            }
        }

        public override string VolumeUnitString {
            get { return ""; }
        }
        public override int ImplicitPriceFactor {
            get { return _brand==BrandT.NikkeiFuture? 1000 : 100; }
        }
        public override string CodeForDataSource() {
            return _nineDigitCode;
        }

        //̔r  thisp̌̂Ƃtrue
        public bool IsPrecedingTo(DerivativeStockProfile p) {
            int y1, m1, y2, m2;
            GetYearMonth(_nineDigitCode, out y1, out m1);
            GetYearMonth(p._nineDigitCode, out y2, out m2);

            return y1==y2? (m1 > m2) : (y1 > y2);
        }
        public void GetYearMonth(out int year, out int month) {
            DerivativeStockProfile.GetYearMonth(_nineDigitCode, out year, out month);
        }

        public bool IsGengetsuCombined {
            get {
                return _code=="NFX" || _code=="NMX";
            }
        }

        public static string FormatCode(string ninecode, EveningSessionAware evening) {
            Debug.Assert(ninecode.Length==9);
            BrandT br = GetBrandT(ninecode);
            StringBuilder bld = new StringBuilder();
            bld.Append(br==BrandT.NikkeiFuture? "NF" : "NM");
            if(evening==EveningSessionAware.Default)
                bld.Append("E");
            else if(evening==EveningSessionAware.Daytime)
                bld.Append("D");

            int year, month;
            GetYearMonth(ninecode, out year, out month);
            bld.Append((year-2000).ToString("D2"));
            bld.Append(month.ToString("D2"));
            return bld.ToString();
        }
        public static string FormatName(string ninecode, EveningSessionAware evening) {
            Debug.Assert(ninecode.Length==9);
            int year, month;
            GetYearMonth(ninecode, out year, out month);
            BrandT br = GetBrandT(ninecode);
            string e = "";
            if(evening==EveningSessionAware.Default) //ŵƂ
                e = "()";
            else if(evening==EveningSessionAware.Daytime)
                e = "()";
            return String.Format("o225{0}{1:D2}N{2:D2}{3}", br==BrandT.NikkeiFuture? "敨" : "mini", year-2000, month, e);
        }
        //͉QŕԂ
        public static void GetYearMonth(string ninecode, out int year, out int month) {
            year = (ninecode[2] - '0') + 2005; // 2015Nɂ̔ł͂߂ɂȂ
            month = (ninecode[3] - '0')*10 + (ninecode[4] - '0');
        }
        public static BrandT GetBrandT(string ninecode) {
            return ninecode.EndsWith("19")? BrandT.NikkeiFutureMini : BrandT.NikkeiFuture;
        }
        public static string FormatNineCode(BrandT brand, int year, int month) {
            if(year > 2000) year -= 2000;
            return String.Format("16{0}{1:D2}00{2}", (year + 5) % 10, month, brand==BrandT.NikkeiFuture? "18" : "19");
        }

        //敨PɂACujOΉŘʂł
        public static DerivativeStockProfile[] CreateElements(string ninecode, BrandT br, StockExchange exch, bool invalid) {
            DerivativeStockProfile[] ps = new DerivativeStockProfile[3];
            StockProfileFlags flag = StockProfileFlags.Future;
            if(invalid) flag |= StockProfileFlags.Hidden;
            EveningSessionAware pr = EveningSessionAware.Preference;
            EveningSessionAware da = EveningSessionAware.Daytime;
            EveningSessionAware df = EveningSessionAware.Default; //悭̂Ń[JϐłQ
            ps[0] = new DerivativeStockProfile(FormatName(ninecode, pr), FormatCode(ninecode, pr), ninecode, br, exch, flag|StockProfileFlags.Combined, pr);
            ps[1] = new DerivativeStockProfile(FormatName(ninecode, da), FormatCode(ninecode, da), ninecode, br, exch, flag|StockProfileFlags.Hidden, da);
            ps[2] = new DerivativeStockProfile(FormatName(ninecode, df), FormatCode(ninecode, df), ninecode, br, exch, flag|StockProfileFlags.Hidden, df);
            return ps;
        }

        //Ao[W쐬
        public static DerivativeStockProfile CreateConcatinatedElements(BrandT br) {
            bool large = br==BrandT.NikkeiFuture;
            DerivativeStockProfile p = new DerivativeStockProfile(large? "o225敨()" : "o225敨~j()", large? "NFX" : "NMX", "", br, StockExchange.O, StockProfileFlags.Future|StockProfileFlags.Combined, EveningSessionAware.Preference);
            p._concatinatedFuture = true;
            return p;
        }
        //w肵HiddenBA̍ޗp
        public static DerivativeStockProfile CreateClosedFuture(BrandT br, int y, int m, EveningSessionAware evening) {
            string ninecode = FormatNineCode(br, y, m);
            DerivativeStockProfile p = new DerivativeStockProfile(FormatName(ninecode, evening), FormatCode(ninecode, evening), ninecode, br, StockExchange.O, StockProfileFlags.Future|StockProfileFlags.Hidden, evening, StockFlags.Obsolete);
            return p;
        }
        public static DerivativeStockProfile CreateTransient(string ninecode) {
            BrandT br = GetBrandT(ninecode);
            DerivativeStockProfile p = new DerivativeStockProfile(FormatName(ninecode, EveningSessionAware.Preference), FormatCode(ninecode, EveningSessionAware.Default), ninecode, br, StockExchange.O, StockProfileFlags.Hidden|StockProfileFlags.Future|StockProfileFlags.Transient, EveningSessionAware.Preference);
            return p;
        }

    }

    //wށB͂łȂwWƂĂ͎gp\
    public class MarketIndex : AbstractStockProfile {
        private string _dataSourceCode; //[Uɂ͏oAf[^\[XTChł̂ݎgpR[h
        private Builtins _enumCode;

        public MarketIndex(string name, Builtins code, string dataSourceCode)
            : base(name, code.ToString(), StockProfileFlags.F2, 0) {
            _dataSourceCode = dataSourceCode;
            _enumCode = code;
            _primary = new Stock(this, StockExchange.B, StockExchangeSubType.None, StockFlags.None);
            _volumeUnit = 1;
        }
        public MarketIndex(string name, string code, string dataSourceCode) 
            : base(name, code, StockProfileFlags.None, 0) { //F0Œ͂炵
            _dataSourceCode = dataSourceCode;
            _enumCode = Builtins.None;
            _primary = new Stock(this, StockExchange.B, StockExchangeSubType.None, StockFlags.None);
            _volumeUnit = 1;
        }

        public string DataSourceCode {
            get {
                return _dataSourceCode;
            }
        }
        public Builtins EnumCode {
            get {
                return _enumCode;
            }
        }
        public override string VolumeUnitString {
            get { return ""; }
        }
        public override int ImplicitPriceFactor {
            get { return 1; }
        }
        public override string CodeForDataSource() {
            return _dataSourceCode;
        }

        //悭gR[h͂Ɏ^Bf[^\[XTChł͂̃TuZbggƂ낤Benumvf̂܂܃R[hƂĒʗp
        public enum Builtins {
            None, //YȂB̓JX^̎wɑΉ

            NK225,    NK300, NK500, NKJQ, TOPIX, T2INDEX, MOTHERS, TREIT, 
            //o225, 300 500, oJASDAQ, TOPIX, 2w, ϻްގw, REITw,
            
            TLARGE,   TMID,   TSMALL,
            //؋K͕ʁi^j,  ؋K͕ʁi^j,  ؋K͕ʁi^j

            HCSTANDARD, HCGROWTH, HERCULES,
            //׸ڽwiްށj, ׸ڽwi۰j, ׸ڽwij, 

            JASDAQ,
            //JASDAQ INDEX

            END = JASDAQ //pɍŌ̂̂͒`ȂƂȂ
            
        }
    }


    //s ̒Pʂɑ
    public class Stock : IAdaptable {
        private AbstractStockProfile _profile;
        private StockExchange _market;
        private StockExchangeSubType _subtype;
        private StockFlags _flags;
        private int _obsoleteDate; //p~t(ꒆ0)

        internal Stock(AbstractStockProfile prof, StockExchange exch, StockExchangeSubType subtype, StockFlags flags) {
            _profile = prof;
            _market = exch;
            _subtype = subtype;
            _flags = flags;
        }

        public AbstractStockProfile Profile {
            get {
                Debug.Assert(_profile!=null);
                return _profile;
            }
        }

        public StockExchange Market {
            get {
                return _market;
            }
        }
        public StockExchangeSubType MarketSubType {
            get {
                return _subtype;
            }
        }
        public StockFlags StockFlags {
            get {
                return _flags;
            }
        }
        public int VolumeUnit {
            get {
                return _profile.VolumeUnit;
            }
        }
        public bool IsObsolete {
            get {
                return (_flags  & StockFlags.Obsolete)!=StockFlags.None;
            }
        }
        public int ObsoleteDate {
            get {
                return _obsoleteDate;
            }
            set {
                _obsoleteDate = value;
            }
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }


    public class StockProfileCollection : IEnumerable<AbstractStockProfile> {
        protected bool _dirty; //\[g̕Kv
        protected List<AbstractStockProfile> _data;

        private static BinarySearch<AbstractStockProfile, string>.Comparator comparator =
            delegate(AbstractStockProfile stock, string key) { return String.CompareOrdinal(stock.Code, key); };

        public StockProfileCollection() {
            _data= new List<AbstractStockProfile>();
            _dirty = false;
        }
        public void Clear() {
            _data.Clear();
        }
        public void Add(AbstractStockProfile stock) {
            if(!_dirty && _data.Count>0)
                _dirty = _data[_data.Count-1].Code.CompareTo(stock.Code)>0;
            _data.Add(stock);
        }
        public void Remove(AbstractStockProfile stock) {
            Debug.Assert(stock!=null);
            _data.Remove(stock);
        }
        public AbstractStockProfile GetAt(int index) {
            return _data[index];
        }
        public int Count {
            get {
                return _data.Count;
            }
        }
        public IEnumerator GetEnumerator() {
            if(_dirty) Sort();
            return _data.GetEnumerator();
        }
        IEnumerator<AbstractStockProfile> IEnumerable<AbstractStockProfile>.GetEnumerator() {
            if(_dirty) Sort();
            return _data.GetEnumerator();
        }

        public AbstractStockProfile FindExact(string code) {
            if(_dirty) Sort();
            int t = BinarySearch<AbstractStockProfile, string>.FindExact(_data, code, comparator);
            return t==-1? null : _data[t];
        }
        //ŁAtextȏɂȂőindexԂ
        public int SearchDictionary(string code) {
            if(_dirty) Sort();
            return BinarySearch<AbstractStockProfile, string>.FindLowerBound(_data, code, comparator);
        }

        protected virtual void Sort() {
            _data.Sort(delegate(AbstractStockProfile s1, AbstractStockProfile s2) {
                return String.CompareOrdinal(s1.Code, s2.Code);
            });
            _dirty = false;
        }
    }
    public class GlobalStockCollection : StockProfileCollection {
        //Iv
        private List<DerivativeStockProfile> _derivatives;
        //rgCłȂw
        private List<MarketIndex> _customMarketIndices;

        //ߌ
        private DerivativeStockProfile _currentLarge;
        private DerivativeStockProfile _currentMini;

        public GlobalStockCollection() {
            _derivatives = new List<DerivativeStockProfile>();
            _customMarketIndices = new List<MarketIndex>();
        }

        public DerivativeStockProfile CurrentLarge {
            get {
                return _currentLarge;
            }
        }
        public DerivativeStockProfile CurrentMini {
            get {
                return _currentMini;
            }
        }

        //ʂo^IƂɍsANV
        public void AfterRegistrationAction() {
            //敨̓o^
            Add(DerivativeStockProfile.CreateConcatinatedElements(DerivativeStockProfile.BrandT.NikkeiFuture));
            Add(DerivativeStockProfile.CreateConcatinatedElements(DerivativeStockProfile.BrandT.NikkeiFutureMini));

            Sort(); //Currentݒ肳

            int count = BellagioRoot.FixedPreferences.Chart.ConcatinateFutureCount;
            int y, m;
            _currentLarge.GetYearMonth(out y, out m);
            for(int i=0; i<count; i++) {
                MarketUtil.GetPrevFutureGengetsu(ref y, ref m);
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFuture, y, m, EveningSessionAware.Preference));
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFuture, y, m, EveningSessionAware.Daytime));
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFuture, y, m, EveningSessionAware.Default));
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFutureMini, y, m, EveningSessionAware.Preference));
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFutureMini, y, m, EveningSessionAware.Daytime));
                this.Add(DerivativeStockProfile.CreateClosedFuture(DerivativeStockProfile.BrandT.NikkeiFutureMini, y, m, EveningSessionAware.Default));
            }
        }

        public DerivativeStockProfile FindDerivativeStockProfile(string code, bool includes_hidden_stocks) {
            return FindDerivativeStockProfileInternal(code, includes_hidden_stocks, false, EveningSessionAware.Preference);
        }
        public DerivativeStockProfile FindDerivativeStockProfile(string code, bool includes_hidden_stocks, EveningSessionAware evening) {
            return FindDerivativeStockProfileInternal(code, includes_hidden_stocks, true, evening);
        }
        private DerivativeStockProfile FindDerivativeStockProfileInternal(string code, bool includes_hidden_stocks, bool check_evening, EveningSessionAware evening) {
            if(_dirty) Sort();

            //敨Ȃ炱ł\ȂIvV܂ŎLƔ
            foreach(DerivativeStockProfile d in _derivatives) {
                if(!includes_hidden_stocks && (d.Flags & StockProfileFlags.Hidden)!=StockProfileFlags.None) continue;
                if(check_evening && d.EveningSessionAware!=evening) continue;

                if(d.Code==code || d.CodeForDataSource()==code) return d;
            }
            return null;
        }

        public IEnumerable<DerivativeStockProfile> Derivatives {
            get {
                if(_dirty) Sort();
                return _derivatives;
            }
        }
        public IEnumerable<MarketIndex> CustomMarketIndices {
            get {
                if(_dirty) Sort();
                return _customMarketIndices;
            }
        }

        protected override void Sort() {
            base.Sort();

            _derivatives.Clear();
            _customMarketIndices.Clear();
            //_dataT̂͌
            foreach(AbstractStockProfile p in _data) {
                DerivativeStockProfile d = p as DerivativeStockProfile;
                if(d!=null) {
                    _derivatives.Add(d);

                    if((d.Flags & StockProfileFlags.Hidden)!=StockProfileFlags.None || d.CodeForDataSource().Length==0) continue;

                    if(d.Brand==DerivativeStockProfile.BrandT.NikkeiFuture) {
                        if(_currentLarge==null || _currentLarge.IsPrecedingTo(d)) _currentLarge = d;
                    }
                    if(d.Brand==DerivativeStockProfile.BrandT.NikkeiFutureMini) {
                        if(_currentMini==null || _currentMini.IsPrecedingTo(d)) _currentMini = d;
                    }

                }

                MarketIndex mi = p as MarketIndex;
                if(mi!=null && mi.EnumCode==MarketIndex.Builtins.None) {
                    _customMarketIndices.Add(mi);
                }
            }
        }

        //O̖ 敨͂܂ΉȂ
        public AbstractStockProfile GetNextStock(AbstractStockProfile p) {
            if(p is BasicStockProfile) {
                int code;
                if(!Int32.TryParse(p.Code, out code)) return null;
                code++;
                do {
                    p = FindExact(code.ToString());
                    if(p!=null) return p;
                    code++;
                    if(code==10000) code = 1000;
                } while(true);
            }
            else if(p is MarketIndex) {
                MarketIndex mi = (MarketIndex)p;
                MarketIndex.Builtins bc = mi.EnumCode;
                if(bc==MarketIndex.Builtins.None) {
                    int index = _customMarketIndices.IndexOf(mi);
                    Debug.Assert(index!=-1);
                    index++;
                    if(index==_customMarketIndices.Count) return FindExact(MarketIndex.Builtins.NK225.ToString()); //ŏɖ߂
                    else return _customMarketIndices[index];
                }
                else { //builtin̂ǂꂩ
                    do {
                        if(bc==MarketIndex.Builtins.END) return _customMarketIndices[0];
                        bc = (MarketIndex.Builtins)(bc+1);
                        p = FindExact(bc.ToString());
                        if(p!=null) return p;
                    } while(true);
                }
            }
            return null;
        }
        public AbstractStockProfile GetPrevStock(AbstractStockProfile p) {
            if(p is BasicStockProfile) {
                int code;
                if(!Int32.TryParse(p.Code, out code)) return null;
                code--;
                do {
                    p = FindExact(code.ToString());
                    if(p!=null) return p;
                    code--;
                    if(code==999) code = 9999;
                } while(true);
            }
            else if(p is MarketIndex) {
                MarketIndex mi = (MarketIndex)p;
                MarketIndex.Builtins bc = mi.EnumCode;
                if(bc==MarketIndex.Builtins.None) {
                    int index = _customMarketIndices.IndexOf(mi);
                    Debug.Assert(index!=-1);
                    index--;
                    if(index==-1) return FindExact(MarketIndex.Builtins.NK225.ToString()); //ŏɖ߂
                    else return _customMarketIndices[index];
                }
                else { //builtin̂ǂꂩ
                    do {
                        if(bc==MarketIndex.Builtins.NK225) return _customMarketIndices[_customMarketIndices.Count-1];
                        bc = (MarketIndex.Builtins)(bc-1);
                        p = FindExact(bc.ToString());
                        if(p!=null) return p;
                    } while(true);
                }
            }
            return null;
        }
    }

    //̃RNV
    public abstract class ListedStockCollection : IEnumerable {
        public abstract int Count { get; }
        public abstract IEnumerator GetEnumerator();
    }
    //TypedHashtablê߂GenericIEnumerable͕s
    public class HashListedStockCollection : ListedStockCollection {
        private TypedHashtable<string, AbstractStockProfile> _data;

        public HashListedStockCollection() {
            _data = new TypedHashtable<string, AbstractStockProfile>();

        }

        public override int Count {
            get {
                return _data.Count;
            }
        }
        public void Append(AbstractStockProfile stock) {
            _data[stock.Code] = stock;
        }
        public void Clear() {
            _data.Clear();
        }
        public AbstractStockProfile Find(string code) {
            return _data[code];
        }

        public override IEnumerator GetEnumerator() {
            return _data.Values.GetEnumerator();
        }
    }

    public class ListListedStockCollection : ListedStockCollection {
        private string _stringFormat;
        private List<Stock> _stocks; //ꂪnullǂŃp[Xς݂

        //Ƃp[X\ȃX^C
        public ListListedStockCollection(string format) {
            _stringFormat = format;
        }
        public ListListedStockCollection() {
            _stocks = new List<Stock>();
        }
        private void Bind() {
            //TODO swł悤ɂ
            _stocks = new List<Stock>();
            string[] t = _stringFormat.Split(new char[] { ' ', ',' }, StringSplitOptions.RemoveEmptyEntries);
            foreach(string code in t) {
                AbstractStockProfile stock = BellagioRoot.GlobalStockCollection.FindExact(code);
                if(stock==null) throw new BellagioException(String.Format("R[h{0}̖܂", code));
                _stocks.Add(stock.Primary);
            }
        }

        public void Add(Stock stock) {
            _stocks.Add(stock);
        }

        //xĕ]
        private void CheckBind() {
            if(_stocks==null) Bind();
        }

        public override int Count {
            get { CheckBind(); return _stocks.Count; }
        }
        public override IEnumerator GetEnumerator() {
            CheckBind();
            return _stocks.GetEnumerator();
        }

        public Stock[] ToArray() {
            return _stocks.ToArray();
        }
    }

    //P琬
    public class SingleListedStockCollection : ListedStockCollection {
        private Stock _stock;
        public SingleListedStockCollection(Stock stock) {
            Debug.Assert(stock!=null);
            _stock = stock;
        }
        public override int Count {
            get { return 1; }
        }
        public override IEnumerator GetEnumerator() {
            return new E(this);
        }

        private class E : IEnumerator {
            private SingleListedStockCollection _parent;
            private bool _moved;
            public E(SingleListedStockCollection parent) {
                _parent = parent;
            }
            public object Current {
                get { return _parent._stock; }
            }

            public bool MoveNext() {
                if(_moved)
                    return false;
                else {
                    _moved = true;
                    return true;
                }
            }

            public void Reset() {
                _moved = false;
            }
        }

        public Stock Content {
            get {
                return _stock;
            }
            set {
                _stock = value;
            }
        }
    }

    //g|Cgp
    public interface IStockListConstructor : IAdaptable {
        void AddStock(AbstractStockProfile stock);
    }



#if UNITTEST
    [TestFixture]
    public class StockCollectionTests {
        [TestFixtureSetUp]
        public void Prepare() {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.UNITTEST_STOCK_COUNT = 5;
            BellagioRoot.Init(p);
            BellagioRoot.BootForTest();
        }

        [Test]
        public void ListTest() {
            ListListedStockCollection ls = new ListListedStockCollection("1  2 , 3");
            Assert.AreEqual(3, ls.Count);
            Stock[] t = ls.ToArray();
            Assert.AreEqual("1", t[0].Profile.Code);
            Assert.AreEqual("2", t[1].Profile.Code);
            Assert.AreEqual("3", t[2].Profile.Code);
        }

        [Test]
        public void SingleStockCollectionTest() {
            BasicStockProfile b = new BasicStockProfile("1", "1", StockProfileFlags.None, 1);
            b.CreatePrimary(StockExchange.T, StockExchangeSubType.Ichibu, StockFlags.None);
            SingleListedStockCollection ssc = new SingleListedStockCollection(b.Primary);
            int count = 0;
            foreach(Stock s in ssc) {
                Assert.AreEqual(b.Primary, s);
                count++;
            }
            Assert.AreEqual(1, count);
        }


        [Test]
        public void SortedStockCollectionTest() {
            StockProfileCollection sc = new StockProfileCollection();
            for(int c=0; c<100; c++) {
                BasicStockProfile sp = new BasicStockProfile(c.ToString(), (1000+c).ToString(), StockProfileFlags.None, 1000);
                sp.CreatePrimary(StockExchange.T, StockExchangeSubType.Ichibu, StockFlags.None);
                sc.Add(sp);
            }

            //FindExact
            AbstractStockProfile s = sc.FindExact("1000");
            Assert.AreEqual("0", s.Name);
            s = sc.FindExact("1032");
            Assert.AreEqual("32", s.Name);
            s = sc.FindExact("10000");
            Assert.IsNull(s);

            int i = sc.SearchDictionary("10");
            Assert.AreEqual(0, i);
            i = sc.SearchDictionary("1001");
            Assert.AreEqual(1, i);
            i = sc.SearchDictionary("105");
            Assert.AreEqual(50, i);
            i = sc.SearchDictionary("100:");
            Assert.AreEqual(10, i);
            i = sc.SearchDictionary("1099");
            Assert.AreEqual(99, i);
            i = sc.SearchDictionary("11");
            Assert.AreEqual(-1, i);
        }
    }

    [TestFixture]
    public class DerivativeStockTests {
        [Test]
        public void Future() {
#if false
            //l[~O[ς̂ŕۗ
            DerivativeStockProfile f1 = new DerivativeStockProfile(20070919, DerivativeStockProfile.BrandT.NikkeiFuture, StockExchange.O, 0);
            Assert.AreEqual("o敨07N12", f1.Name);
            Assert.AreEqual("NF", f1.Code);
            Assert.AreEqual("162120018", f1.GetShokenNineCode(20070919));

            DerivativeStockProfile f2 = new DerivativeStockProfile(20070919, DerivativeStockProfile.BrandT.NikkeiFuture, StockExchange.O, 2);
            Assert.AreEqual("o敨08N06", f2.Name);
            Assert.AreEqual("163060018", f2.GetShokenNineCode(20070919));
            Assert.AreEqual("NF3", f2.Code);

            DerivativeStockProfile f3 = new DerivativeStockProfile(20070913, DerivativeStockProfile.BrandT.NikkeiFutureMini, StockExchange.O, 0); //SQO
            Assert.AreEqual("o敨~j07N09", f3.Name);
            Assert.AreEqual("162090019", f3.GetShokenNineCode(20070913));
            Assert.AreEqual("NM", f3.Code);

            DerivativeStockProfile f4 = new DerivativeStockProfile(20070914, DerivativeStockProfile.BrandT.NikkeiFutureMini, StockExchange.O, 0); //SQɐ؂ւ
            Assert.AreEqual("o敨~j07N12", f4.Name);
            Assert.AreEqual("162120019", f4.GetShokenNineCode(20070914));
            Assert.AreEqual("NM", f4.Code);
#endif
        }
    }
#endif

}
