/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/ChartCommands.cs#10 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * `[gɂR}hꎮ
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

using Poderosa;
using Poderosa.Commands;
using Poderosa.Sessions;
using Poderosa.Forms;

using Bellagio.Data;
using Bellagio.Environment;
using Bellagio.Ordering;
using Bellagio.Forms;
using Bellagio.Bookmark;

using Travis.Collections;

namespace Bellagio.Chart {
    public class ChartCommands {
        private BellagioCommandCategory _chartCategory;

        public void Register(ICommandManager cm) {
            ICommandCategory dlg = cm.CommandCategories.Dialogs;
            _chartCategory = new BellagioCommandCategory("`[gݒ");
            
            CanExecuteDelegate chartavailable = new CanExecuteDelegate(IsChartAvailable);

            showTechinalIndicatorsDialog = new GeneralCommandImpl("tactico.chart.showTechinalIndicatorDialog", "eNjJwWݒ_CAO\", dlg, new ExecuteDelegate(ShowTechnicalIndicatorsDialog), chartavailable);
            showNewOrderDialog = new GeneralCommandImpl("tactico.chart.showNewOrderDialog", "_CAO\", dlg, new ExecuteDelegate(ShowNewOrderDialog), new CanExecuteDelegate(CanOrder));
            selectPrimaryMarket = new GeneralCommandImpl("tactico.chart.selectMarket.primary", "s", _chartCategory, new ExecuteDelegate(SelectPrimaryMarket), new CanExecuteDelegate(NotIndices));
            selectSecondaryMarket = new GeneralCommandImpl("tactico.chart.selectMarket.secondary", "s", _chartCategory, new ExecuteDelegate(SelectSecondaryMarket), new CanExecuteDelegate(SecondaryMarketExists));
            clearFreeLine = new GeneralCommandImpl("tactico.chart.clearFreeLine", "R̃NA", _chartCategory, new ExecuteDelegate(ClearFreeLine), chartavailable);
            toggleReflectSplit = new GeneralCommandImpl("tactico.chart.toggleReflectSplit", "̔f؂ւ", _chartCategory, new ExecuteDelegate(ToggleReflectSplit), new CanExecuteDelegate(IsSplitCommandEnabled));
            showAllTickData = new GeneralCommandImpl("tactico.chart.showAllTickData", "Sݒl̕\", dlg, new ExecuteDelegate(ShowAllTickDataDialog), chartavailable);
            addToBookmark = new GeneralCommandImpl("tactico.chart.addToBookmark", "ubN}[N֒ǉ", dlg, new ExecuteDelegate(AddToBookmark), chartavailable);
            reloadDailyDataBasedSessions = new GeneralCommandImpl("tactico.chart.reloadDailyDataBasedSessions", "f[^[h", cm.CommandCategories.File, new ExecuteDelegate(ReloadDailyDataBasedSessions), null);
            toggleIndicatorToolTip = new GeneralCommandImpl("tactico.chart.toggleIndicatorToolTip", "J[\ʒu̎wWҏWc[`bv\؂ւ", _chartCategory, new ExecuteDelegate(ToggleIndicatorToolTip), chartavailable);
            toggleLockIndicatorFigures = new GeneralCommandImpl("tactico.chart.toggleLockIndicatorFigures", "wW̐l\bN", _chartCategory, new ExecuteDelegate(ToggleLockIndicatorFigures), chartavailable);
            cm.Register(showTechinalIndicatorsDialog);
            cm.Register(showNewOrderDialog);
            cm.Register(selectPrimaryMarket);
            cm.Register(selectSecondaryMarket);
            cm.Register(clearFreeLine);
            cm.Register(toggleReflectSplit);
            cm.Register(addToBookmark);
            cm.Register(reloadDailyDataBasedSessions);
            cm.Register(toggleIndicatorToolTip);
            cm.Register(toggleLockIndicatorFigures);
        }
        
        public IGeneralCommand showTechinalIndicatorsDialog;
        public IGeneralCommand showNewOrderDialog;
        public IGeneralCommand selectPrimaryMarket;
        public IGeneralCommand selectSecondaryMarket;
        public IGeneralCommand clearFreeLine;
        public IGeneralCommand toggleIndicatorToolTip;
        public IGeneralCommand toggleLockIndicatorFigures;

        public IGeneralCommand toggleReflectSplit;
        public IGeneralCommand showAllTickData;
        public IGeneralCommand addToBookmark;
        public IGeneralCommand reloadDailyDataBasedSessions;

        public static bool IsChartAvailable(ICommandTarget target) {
            return AsViewContent(target)!=null;
        }
        public static ChartViewContentBase AsViewContent(ICommandTarget target) {
            return target==null? null : (ChartViewContentBase)target.GetAdapter(typeof(ChartViewContentBase));
        }
        public static RealTimeChartViewContent AsRealTimeViewContent(ICommandTarget target) {
            return AsViewContent(target) as RealTimeChartViewContent;
        }
        
        //ȍ~eR}h{
        private static CommandResult ShowTechnicalIndicatorsDialog(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            if(c==null) return CommandResult.Ignored;

            MultipleIndicatorEditor dlg = new MultipleIndicatorEditor(c.ChartSessionBase.GetSourceSchema(), BellagioRoot.FixedPreferences.Chart.MultipleIndicatorEditorSettings);
            dlg.ShowDialog();
            dlg.Dispose();
            return CommandResult.Succeeded;
        }
        private static CommandResult ShowNewOrderDialog(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            Stock stock = c.ChartSessionBase.CurrentStock;
            //NOTE {OrderingPluginւ̈ˑ͂ł͋Ȃ
            Bellagio.Ordering.OrderFormParameter p = Bellagio.Ordering.OrderFormParameter.ForNewPosition(stock.Profile);
            return Ordering.OrderingPlugin.Instance.Commands.OpenOrderFormCommand(target, p);
        }
        private static CommandResult SelectPrimaryMarket(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            Stock s = c.ChartSessionBase.CurrentStock.Profile.Primary;
            if(s==c.ChartSessionBase.CurrentStock) return CommandResult.Ignored;

            c.ChartSessionBase.SetCurrentStock(s);
            return CommandResult.Succeeded;
        }
        private static CommandResult SelectSecondaryMarket(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            Stock s = c.ChartSessionBase.CurrentStock.Profile.Secondary;
            if(s==null) return CommandResult.Failed;
            if(s==c.ChartSessionBase.CurrentStock) return CommandResult.Ignored;

            c.ChartSessionBase.SetCurrentStock(s);
            return CommandResult.Succeeded;
        }
        private static bool SecondaryMarketExists(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            Stock s = c.ChartSessionBase.CurrentStock;
            return s!=null && s.Profile.Secondary!=null;
        }
        private static CommandResult ClearFreeLine(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            c.FreeLineSet.Clear();
            if(c.ContainerControl!=null) c.ContainerControl.Invalidate();
            return CommandResult.Succeeded;
        }
        private static CommandResult ToggleReflectSplit(ICommandTarget target) {
            SplitReflection r = BellagioRoot.FixedPreferences.Chart.ReflectSplit;
            BellagioRoot.FixedPreferences.Chart.ReflectSplit = r==SplitReflection.Reflected? SplitReflection.NotRefrected : SplitReflection.Reflected;

            ChartPreferenceUpdateInfo info = new ChartPreferenceUpdateInfo();
            info.UpdatedAllSchema = true;
            info.UpdatedParameters = true; //p[^ωƂɂ΃`[g̒l̓NA
            info.UpdatedAllIndicators = true;
            return BellagioPlugin.Instance.Commands.ReloadPreference.InternalExecute(null, info);
        }
        private static CommandResult ShowAllTickDataDialog(ICommandTarget target) {
            Stock stock = AsStock(target);
            if(stock==null) return CommandResult.Failed;
            IntraDayTrade dt = BellagioRoot.IntraDayTradeProvider.Lookup(stock);
            if(dt==null) return CommandResult.Failed;

            AllTickListForm dlg = new AllTickListForm(stock, dt.TimeAndSales);
            FormUtil.AdjustStyleForModelessSizableDialog(dlg, BellagioPlugin.Instance.ActivePoderosaWindow.AsForm());
            dlg.FormBorderStyle = FormBorderStyle.SizableToolWindow;
            dlg.Show();
            return CommandResult.Succeeded;
        }
        private static CommandResult AddToBookmark(ICommandTarget target) {
            Stock stock = AsStock(target);
            if(stock==null) return CommandResult.Failed;

            using(SelectBookmarkFolderDialog dlg = new SelectBookmarkFolderDialog()) {
                if(dlg.ShowDialog()==DialogResult.OK) {
                    BookmarkFolder f = dlg.ResultFolder;
                    //ɂΒǉȂ
                    foreach(IBookmarkItem item in f.Items) {
                        StockBookmark sb = item as StockBookmark;
                        if(sb!=null && sb.Value==stock) {
                            BUtil.ShowWarningMessageBox("ɓǉĂ܂");
                            return CommandResult.Failed;
                        }
                    }

                    f.AddItem(new StockBookmark(stock));
                    BellagioRoot.Bookmark.FireEvent.RefreshFolder(f);
                    return CommandResult.Succeeded;
                }
                else
                    return CommandResult.Cancelled;
            }
        }
        private static CommandResult ReloadDailyDataBasedSessions(ICommandTarget target) {
            //f[^̍XV̏BXN[jOɊ֌WƂɂeĂĂ܂X}[głȂ...
            Screening.ScreeningPlugin.Instance.DailyDataCache.Clear();
            BellagioRoot.DateBasedQuoteProvider.ClearAll();
            foreach(ISession session in BellagioPlugin.Instance.SessionManager.AllSessions) {
                StaticChartSession sb = session as StaticChartSession;
                if(sb!=null) {
                    sb.ReloadData();
                }
            }
            return CommandResult.Succeeded;
        }
        private static CommandResult ToggleIndicatorToolTip(ICommandTarget target) {
            ChartViewContentBase v = ChartCommands.AsViewContent(target);
            v.EnableIndicatorToolTip = !v.EnableIndicatorToolTip;
            return CommandResult.Succeeded;
        }
        private static CommandResult ToggleLockIndicatorFigures(ICommandTarget target) {
            ChartViewContentBase v = ChartCommands.AsViewContent(target);
            ChartDrawing.LocalSettings ls = v.ChartDrawing.ChartLocalSettings;
            if(ls.lastLockedIndicatorFigurePos.IsEmpty) { //unlocked
                v.ChartDrawing.LockIndicatorFigurePos();
            }
            else
                v.ChartDrawing.UnlockIndicatorFigurePos();
            return CommandResult.Succeeded;
        }



        public static Stock AsStock(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            if(c==null) return null;
            return c.ChartSessionBase.CurrentStock;
        }
        private static bool NotIndices(ICommandTarget target) {
            Stock s = AsStock(target);
            return s!=null && !(s.Profile is MarketIndex);
        }
        private static bool IsStaticChart(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            return c!=null && c.ChartSessionBase is StaticChartSession;
        }
        private static bool CanOrder(ICommandTarget target) {
            //ʊ܂͐敨\ĂāAOCς݂ł邱Ƃ
            if(!BellagioRoot.DataSourceHost.IsConnected) return false;

            Stock s = AsStock(target);
            return s!=null && CanOrder(s);

        }
        public static bool CanOrder(Stock stock) {
            if(stock==null || (stock.Profile is MarketIndex) || (stock.StockFlags & StockFlags.Obsolete)!=StockFlags.None) return false;
            
            DerivativeStockProfile dp = stock.Profile as DerivativeStockProfile;
            if(dp!=null)
                return !dp.IsGengetsuCombined;
            else
                return true;
        }
        private static bool IsSplitCommandEnabled(ICommandTarget target) {
            ChartViewContentBase c = AsViewContent(target);
            if(c!=null && c.ChartSessionBase is StaticChartSession) {
                Stock s = c.ChartSessionBase.CurrentStock;
                return s!=null && s.Profile is BasicStockProfile;
            }

            return false;
        }
    }



    //W\͈ȉ̒ʂ
    //1. {ݒ_CAOAeNjJwW_CAO
    //2. VK
    //3. s؂ւAl̗L(n̂)
    //4. R̃NA
    //@ΐ\AubN}[N݁AƂƂǉɂȂ邩

    public class ChartMenuGroups  {
        //悭QƂ
        protected static ChartCommands Cmds {
            get {
                return ChartPlugin.Instance.ChartCommands;
            }
        }

        public class ChartSettingsMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    if(Bookmark.BookmarkPlugin.Instance!=null)
                        return new IPoderosaMenu[] {
                            new PoderosaMenuItemImpl(Cmds.showTechinalIndicatorsDialog, "eNjJwW̐ݒ(&T)..."),
                            new PoderosaMenuItemImpl(Cmds.addToBookmark, "ubN}[Nւ̒ǉ(&B)...")
                        };
                    else
                        return new IPoderosaMenu[] {
                            new PoderosaMenuItemImpl(Cmds.showTechinalIndicatorsDialog, "eNjJwW̐ݒ(&T)..."),
                        };
                }
            }
        }
        //A^C`[gł͔ݒ
        public class RealtimeChartSettingsMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    List<IPoderosaMenu> c = new List<IPoderosaMenu>();
                    c.Add(new PoderosaMenuItemImpl(Cmds.showTechinalIndicatorsDialog, "eNjJwW̐ݒ(&T)..."));
                    if(Bookmark.BookmarkPlugin.Instance!=null)
                        c.Add(new PoderosaMenuItemImpl(Cmds.addToBookmark, "ubN}[Nւ̒ǉ(&B)..."));
                    c.Add(new BellagioMenuFolder("`[g̕\(&C)", new IPoderosaMenuGroup[] { new ChartStyleMenuGroup() }));
                    c.Add(new BellagioMenuFolder("ݒl̕\(&P)", new IPoderosaMenuGroup[] { new CurrentPriceStyleMenuGroup() }));
                    c.Add(new BellagioMenuFolder("̕\(&I)", new IPoderosaMenuGroup[] { new ItaStyleMenuGroup() }));
                    c.Add(new BellagioMenuFolder("ݒl̕\(&K)", new IPoderosaMenuGroup[] { new TickStyleMenuGroup() }));
                    c.Add(new PoderosaMenuItemImpl(Cmds.showAllTickData, "Sݒl̕\(&Y)..."));
                    return c.ToArray();
                }
            }
        }

        public class ChartOrderMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new PoderosaMenuItemImpl(Cmds.showNewOrderDialog, "(&O)...")
                    };
                }
            }
        }
        public class DataSettingMenuGroup : PoderosaMenuGroupImpl {
            private IPoderosaCommand _setStockCommand;
            private bool _includesIndices;
            public DataSettingMenuGroup(IPoderosaCommand set_stock_command, bool includes_indices) {
                _setStockCommand = set_stock_command;
                _includesIndices = includes_indices;
            }

            public override IPoderosaMenu[] ChildMenus {
                get {
                    List<IPoderosaMenu> menus = new List<IPoderosaMenu>();
                    if(BellagioVersionInfo.EDITION==BellagioEditions.HatchukunTX)
                        menus.Add(new BellagioMenuFolder("s(&M)", new IPoderosaMenuGroup[] { new MarketMenuGroup() }).SetEnabled(new CanExecuteDelegate(IsMarketSelectable)));

                    menus.Add(new BellagioMenuFolder("̃^uŊJ(&O)",   ChartSchemaMenuGroup.GroupArrayInThisTab()));
                    menus.Add(new BellagioMenuFolder("V^uŊJ(&T)", ChartSchemaMenuGroup.GroupArrayInNewTab()));
                    menus.Add(new BellagioMenuItemImpl(Cmds.toggleReflectSplit, "̔f(&S)",
                            delegate(ICommandTarget t) { return BellagioRoot.FixedPreferences.Chart.ReflectSplit==SplitReflection.Reflected; }));
                    menus.Add(new BellagioMenuFolder(_includesIndices? "敨Ew(&F)" : "敨(&F)", DerivativeAndIndicesGroups()));
                    return menus.ToArray();
                }
            }

            private bool IsMarketSelectable(ICommandTarget target) {
                ChartViewContentBase v = ChartCommands.AsViewContent(target);
                if(v==null)
                    return false;
                else {
                    Stock s = v.ChartSessionBase.CurrentStock;
                    MarketMenuGroup.SetLastStock(s==null? null : s.Profile); //qj[̃eLXg𓮓IɕύX@\Ȃ߂̗Fj[͓ɂЂƂI[vłȂ_ɈˑĂ̂ł悭Ȃ
                    return s!=null && s.Profile.Secondary!=null;
                }
            }

            private IPoderosaMenuGroup[] DerivativeAndIndicesGroups() {
                if(_includesIndices)
                    return new IPoderosaMenuGroup[] { new FutureMenuGroup(_setStockCommand, true), new IndicesMenuGroup(_setStockCommand, IndicesMenuGroup.Category.Major) };
                else
                    return new IPoderosaMenuGroup[] { new FutureMenuGroup(_setStockCommand, false) };
            }
        }
        public class ChartUIMenuGroup : PoderosaMenuGroupImpl {

            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new BellagioMenuFolder("o/IV[^̍(&O)", new IPoderosaMenuGroup[] { new OscillatorHeightMenuGroup() }).SetEnabled(new CanExecuteDelegate(CanExecuteConfigOscillators)),
                        new BellagioMenuItemImpl(Cmds.toggleIndicatorToolTip, "wWҏWc[`bvL(&T)",
                            delegate(ICommandTarget t) { return AsChartViewContentBase(t).EnableIndicatorToolTip; }),
                        new BellagioMenuItemImpl(Cmds.toggleLockIndicatorFigures, "J[\ʒu̎wW\ŃbN(&C)",
                            delegate(ICommandTarget t) { return !AsChartViewContentBase(t).ChartDrawing.ChartLocalSettings.lastLockedIndicatorFigurePos.IsEmpty; }),
                        new PoderosaMenuItemImpl(Cmds.clearFreeLine, "R̃NA(&L)"),
                    };
                }
            }
            private bool CanExecuteConfigOscillators(ICommandTarget target) {
                ChartViewContentBase v = ChartCommands.AsViewContent(target);
                if(v==null)
                    return false;
                else {
                    ChartDocumentBase doc = v.ChartSessionBase.ChartDocument;
                    OscillatorHeightMenuGroup.SetCurrentChartDocument(doc);
                    return doc!=null && doc.OscillatorGroupSet.Count>0;
                }
            }

            private static ChartViewContentBase AsChartViewContentBase(ICommandTarget t) {
                return ChartCommands.AsViewContent(t);
            }
        }
        public class OscillatorHeightMenuGroup : PoderosaMenuGroupImpl {
            private static ChartDocumentBase _currentChartDocument;
            //eNjbN
            public static void SetCurrentChartDocument(ChartDocumentBase doc) {
                _currentChartDocument = doc;
            }

            public override IPoderosaMenu[] ChildMenus {
                get {
                    //o擪
                    List<IPoderosaMenu> menus = new List<IPoderosaMenu>();
                    menus.Add(new BellagioMenuFolder("o", new IPoderosaMenuGroup[] { new VolumeHeightGroup(_currentChartDocument.SourceSchema) }));
                    for(int i=0; i<_currentChartDocument.OscillatorGroupSet.Count; i++) {
                        RuntimeOscillatorGroup gr = _currentChartDocument.OscillatorGroupSet[i];
                        menus.Add(new GroupHeightMenu(gr));
                    }
                    return menus.ToArray();
                }
            }

            private class GroupHeightMenu : BellagioMenuFolder {
                public GroupHeightMenu(RuntimeOscillatorGroup gr)
                    : base(gr.Text, gr.VisibleAny?
                        new IPoderosaMenuGroup[] { new GroupHeightGroup(gr) } : new IPoderosaMenuGroup[0]) {
                    //z̃CWP[^\ݒɂȂĂꍇ̓j[LɂȂB
                    //łȂƁAFȂǂƂȂĂ̂ɐݒ肪Ȃ悤ɌĂ܂
                    this.SetEnabled(delegate(ICommandTarget t) { return gr.VisibleAny; });
                }
            }
            private class GroupHeightGroup : PoderosaMenuGroupImpl {
                private RuntimeOscillatorGroup _runtimeOscillatorGroup;
                public GroupHeightGroup(RuntimeOscillatorGroup gr) {
                    _runtimeOscillatorGroup = gr;
                }
                public override IPoderosaMenu[] ChildMenus {
                    get {
                        RuntimeOscillatorGroup g = _runtimeOscillatorGroup;
                        int dh = BellagioRoot.FixedPreferences.Chart.ChartDrawingSettings.oscillatorDefaultHeight.V;
                        return new IPoderosaMenu[] {
                            new GroupHeightItem(new SetOscillatorHeightCommand(g, 0), "Ȃ"),
                            new GroupHeightItem(new SetOscillatorHeightCommand(g, dh/2), ""), //
                            new GroupHeightItem(new SetOscillatorHeightCommand(g, dh), ""),   //W
                            new GroupHeightItem(new SetOscillatorHeightCommand(g, dh*3/2), "") //1.5{
                        };
                    }
                }
            }
            private class GroupHeightItem : PoderosaMenuItemImpl {
                private SetOscillatorHeightCommand _cmd;
                public GroupHeightItem(SetOscillatorHeightCommand cmd, string text)
                    : base(cmd, text) {
                    _cmd = cmd;
                    _checked = delegate(ICommandTarget target) {
                        int height = ParseUtil.ParseInt(_cmd._group.GetHeightPreferenceItem().Value, BellagioRoot.FixedPreferences.Chart.ChartDrawingSettings.oscillatorDefaultHeight.V);
                        return height == _cmd._height;
                    };
                }
            }
            private class SetOscillatorHeightCommand : IPoderosaCommand {
                internal RuntimeOscillatorGroup _group;
                internal int _height;
                public SetOscillatorHeightCommand(RuntimeOscillatorGroup gr, int height) {
                    _group = gr;
                    _height = height;
                }

                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    _group.GetHeightPreferenceItem().Value = _height.ToString();

                    ChartPreferenceUpdateInfo info = new ChartPreferenceUpdateInfo();
                    info.UpdatedAllSchema = true; //{͈̃XL[}ωĂ邪蔲
                    info.UpdatedAppearance = true;
                    return BellagioPlugin.Instance.Commands.ReloadPreference.InternalExecute(null, info);

                }

                public bool CanExecute(ICommandTarget target) {
                    return true;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }

            ///////////////////////////////////
            private class VolumeHeightGroup : PoderosaMenuGroupImpl {
                private ChartSchema _schema;
                public VolumeHeightGroup(ChartSchema sc) {
                    _schema = sc;
                }
                public override IPoderosaMenu[] ChildMenus {
                    get {
                        int dh = BellagioRoot.FixedPreferences.Chart.ChartDrawingSettings.volumeAreaHeight.V;
                        return new IPoderosaMenu[] {
                            new VolumeHeightItem(new SetVolumeHeightCommand(_schema, 0), "Ȃ"),
                            new VolumeHeightItem(new SetVolumeHeightCommand(_schema, dh/2), ""), //
                            new VolumeHeightItem(new SetVolumeHeightCommand(_schema, dh), ""),   //W
                            new VolumeHeightItem(new SetVolumeHeightCommand(_schema, dh*3/2), "") //1.5{
                        };
                    }
                }
            }
            private class VolumeHeightItem : PoderosaMenuItemImpl {
                private SetVolumeHeightCommand _cmd;
                public VolumeHeightItem(SetVolumeHeightCommand cmd, string text)
                    : base(cmd, text) {
                    _cmd = cmd;
                    _checked = delegate(ICommandTarget target) {
                        int height = ParseUtil.ParseInt(_cmd._schema.VolumeHeightInfo.Value, BellagioRoot.FixedPreferences.Chart.ChartDrawingSettings.volumeAreaHeight.V);
                        return height == _cmd._height;
                    };
                }
            }
            private class SetVolumeHeightCommand : IPoderosaCommand {
                internal ChartSchema _schema;
                internal int _height;
                public SetVolumeHeightCommand(ChartSchema sc, int height) {
                    _schema = sc;
                    _height = height;
                }

                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    _schema.VolumeHeightInfo.Value = _height.ToString();

                    ChartPreferenceUpdateInfo info = new ChartPreferenceUpdateInfo();
                    info.UpdatedAllSchema = true; //{͈̃XL[}ωĂ邪蔲
                    info.UpdatedAppearance = true;
                    return BellagioPlugin.Instance.Commands.ReloadPreference.InternalExecute(null, info);

                }

                public bool CanExecute(ICommandTarget target) {
                    return true;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }
        }

        //Tuj[
        public class MarketMenuGroup : PoderosaMenuGroupImpl {
            private static AbstractStockProfile _lastStock;

            //eLXg𓮓IɕύX邽߂̋ꂵdg
            public static void SetLastStock(AbstractStockProfile s) {
                _lastStock = s;
            }


            public override IPoderosaMenu[] ChildMenus {
                get {
                    StockExchange e1 = _lastStock==null? StockExchange.None : _lastStock.Primary.Market;
                    StockExchange e2 = (_lastStock==null || _lastStock.Secondary==null)? StockExchange.None : _lastStock.Secondary.Market;
                    IOrderItemFormatter f = BellagioRoot.OrderDataProvider.OrderItemFormatter;

                    return new IPoderosaMenu[] {
                        new BellagioMenuItemImpl(Cmds.selectPrimaryMarket, f.FormatStockExchange(e1), new CheckedDelegate(new CheckMark(true).Check)),
                        new BellagioMenuItemImpl(Cmds.selectSecondaryMarket, f.FormatStockExchange(e2), new CheckedDelegate(new CheckMark(false).Check))
                    };
                }
            }
            public override bool IsVolatileContent {
                get {
                    return true;
                }
            }
            private class CheckMark {
                private bool _primary;
                public CheckMark(bool primary) {
                    _primary = primary;
                }
                public bool Check(ICommandTarget target) {
                    ChartViewContentBase v = ChartCommands.AsViewContent(target);
                    if(v==null)
                        return false;
                    else {
                        Stock s = v.ChartSessionBase.CurrentStock;
                        return _primary ^ s!=s.Profile.Primary;
                    }
                        
                }
            }
        }

        public class ChartStyleMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new ChartStyleMenu("Ȃ(&N)", new ChartStyleCommand(ChartDrawingSettings.LayoutStyle.None)),
                        new ChartStyleMenu("(&E)", new ChartStyleCommand(ChartDrawingSettings.LayoutStyle.Normal)),
                    };
                }
            }

            public class ChartStyleMenu : PoderosaMenuItemImpl {
                public ChartStyleMenu(string text, ChartStyleCommand cmd)
                    : base(cmd, text) {
                    _checked = delegate(ICommandTarget target) {
                        RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                        return c!=null && c.ChartLayout==((ChartStyleCommand)_command)._layout;
                    };
                }
            }
            public class ChartStyleCommand : IPoderosaCommand {
                internal ChartDrawingSettings.LayoutStyle _layout;
                public ChartStyleCommand(ChartDrawingSettings.LayoutStyle layout) {
                    _layout = layout;
                }
                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                    if(c!=null) {
                        c.ChartLayout = _layout;
                        BellagioRoot.FixedPreferences.Chart.ChartDrawingSettings.layoutStyle = _layout;
                        //GraphicsSettings͑SɘAĂ̂ōĕ`悪Kv
                        BellagioPlugin.Instance.Commands.DoLayoutAllViews();
                    }
                    return CommandResult.Succeeded;
                }

                public bool CanExecute(ICommandTarget target) {
                    return ChartCommands.AsRealTimeViewContent(target)!=null;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }
        }

        public class CurrentPriceStyleMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new CurrentPriceInfoStyleMenu("Ȃ(&N)", new CurrentPriceInfoStyleCommand(CurrentPriceInfoGraphicsSettings.LayoutStyle.None)),
                        new CurrentPriceInfoStyleMenu("(&E)", new CurrentPriceInfoStyleCommand(CurrentPriceInfoGraphicsSettings.LayoutStyle.Normal)),
                    };
                }
            }

            public class CurrentPriceInfoStyleMenu : PoderosaMenuItemImpl {
                public CurrentPriceInfoStyleMenu(string text, CurrentPriceInfoStyleCommand cmd)
                    : base(cmd, text) {
                    _checked = delegate(ICommandTarget target) {
                        RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                        return c!=null && c.CurrentPriceLayout==((CurrentPriceInfoStyleCommand)_command)._layout;
                    };
                }
            }
            public class CurrentPriceInfoStyleCommand : IPoderosaCommand {
                internal CurrentPriceInfoGraphicsSettings.LayoutStyle _layout;
                public CurrentPriceInfoStyleCommand(CurrentPriceInfoGraphicsSettings.LayoutStyle layout) {
                    _layout = layout;
                }
                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                    if(c!=null) {
                        c.CurrentPriceLayout = _layout;
                        BellagioRoot.FixedPreferences.Chart.CurrentPriceInfoGraphicsSettings.layoutStyle = _layout;
                        //GraphicsSettings͑SɘAĂ̂ōĕ`悪Kv
                        BellagioPlugin.Instance.Commands.DoLayoutAllViews();
                    }
                    return CommandResult.Succeeded;
                }

                public bool CanExecute(ICommandTarget target) {
                    return ChartCommands.AsRealTimeViewContent(target)!=null;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }
        }

        public class ItaStyleMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new ItaStyleMenu("Ȃ(&N)", new ItaStyleCommand(ItaGraphicsSettings.LayoutStyle.None)),
                        new ItaStyleMenu("X^C&1", new ItaStyleCommand(ItaGraphicsSettings.LayoutStyle.Legacy)),
                        new ItaStyleMenu("X^C&2", new ItaStyleCommand(ItaGraphicsSettings.LayoutStyle.Plain)),
                        new ItaStyleMenu("X^C&3", new ItaStyleCommand(ItaGraphicsSettings.LayoutStyle.FixPricetoLeft))
                    };
                }
            }
            private static ItaGraphicsSettings GetSettings(ICommandTarget target) {
                RealTimeChartViewContent v = ChartCommands.AsViewContent(target) as RealTimeChartViewContent;
                return v==null? null : BellagioRoot.FixedPreferences.Chart.ItaGraphicsSettings;
            }
            public class ItaStyleMenu : PoderosaMenuItemImpl {
                public ItaStyleMenu(string text, ItaStyleCommand cmd)
                    : base(cmd, text) {
                    _checked = delegate(ICommandTarget target) {
                        RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                        return c!=null && c.ItaLayout.LayoutStyle==((ItaStyleCommand)_command)._layout;
                    };
                }
            }
            public class ItaStyleCommand : IPoderosaCommand {
                internal ItaGraphicsSettings.LayoutStyle _layout;
                public ItaStyleCommand(ItaGraphicsSettings.LayoutStyle layout)  {
                    _layout = layout;
                }
                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                    if(c!=null) {
                        c.ItaLayout.LayoutStyle = _layout;
                        BellagioRoot.FixedPreferences.Chart.ItaGraphicsSettings.layoutStyle = _layout;
                        //GraphicsSettings͑SɘAĂ̂ōĕ`悪Kv
                        BellagioPlugin.Instance.Commands.DoLayoutAllViews();
                    }
                    return CommandResult.Succeeded;
                }

                public bool CanExecute(ICommandTarget target) {
                    return GetSettings(target)!=null;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }
        }

        public class TickStyleMenuGroup : PoderosaMenuGroupImpl {
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return new IPoderosaMenu[] {
                        new TickStyleMenu("Ȃ(&N)", new TickStyleCommand(TickGraphicsSettings.LayoutStyle.None)),
                        new TickStyleMenu("Xg^(&L)", new TickStyleCommand(TickGraphicsSettings.LayoutStyle.List)),
                        new TickStyleMenu("Ot^(&G)", new TickStyleCommand(TickGraphicsSettings.LayoutStyle.Graph)),
                    };
                }
            }
            public class TickStyleMenu : PoderosaMenuItemImpl {
                public TickStyleMenu(string text, TickStyleCommand cmd)
                    : base(cmd, text) {
                    _checked = delegate(ICommandTarget target) {
                        RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                        return c!=null && c.TickLayout.LayoutStyle==((TickStyleCommand)_command)._layout;
                    };
                }
            }
            public class TickStyleCommand : IPoderosaCommand {
                internal TickGraphicsSettings.LayoutStyle _layout;
                public TickStyleCommand(TickGraphicsSettings.LayoutStyle layout) {
                    _layout = layout;
                }
                public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                    RealTimeChartViewContent c = ChartCommands.AsRealTimeViewContent(target);
                    if(c!=null) {
                        c.TickLayout.LayoutStyle = _layout;
                        BellagioRoot.FixedPreferences.Chart.TickGraphicsSettings.layoutStyle = _layout;
                        //GraphicsSettings͑SɘAĂ̂ōĕ`悪Kv
                        BellagioPlugin.Instance.Commands.DoLayoutAllViews();
                    }
                    return CommandResult.Succeeded;
                }

                public bool CanExecute(ICommandTarget target) {
                    return ChartCommands.AsRealTimeViewContent(target)!=null;
                }

                public IAdaptable GetAdapter(Type adapter) {
                    return BUtil.DefaultGetAdapter(this, adapter);
                }
            }
        }

        //wE敨̖ws
        //ĝ̓`[gƔŁAwwł͓̂n`[ĝ݁B
        public class SetStockMenuItem : PoderosaMenuItemImpl, IPoderosaMenuItemWithArgs {
            private AbstractStockProfile _prof;
            public SetStockMenuItem(AbstractStockProfile prof, IPoderosaCommand cmd)
                : base(cmd, FormatText(prof)) {
                _prof = prof;
            }

            public IAdaptable[] AdditionalArgs {
                get {
                    return new IAdaptable[] { _prof };
                }
            }
            private static string FormatText(AbstractStockProfile prof) {
                return String.Format("{0} - {1}", prof.Code, prof.Name);
            }

        }
        public class FutureMenuGroup : PoderosaMenuGroupImpl {
            private IPoderosaCommand _command;
            private bool _allowCombinedBrand;

            public FutureMenuGroup(IPoderosaCommand cmd, bool allowCombined) {
                _showSeparator = true;
                _command = cmd;
                _allowCombinedBrand = allowCombined;
            }

            public override IPoderosaMenu[] ChildMenus {
                get {
                    List<SetStockMenuItem> list = new List<SetStockMenuItem>();
                    foreach(DerivativeStockProfile p in BellagioRoot.GlobalStockCollection.Derivatives) {
                        if((p.Flags & StockProfileFlags.Hidden)==StockProfileFlags.None && //P@Hiddenł͂Ȃ
                           (_allowCombinedBrand || !p.IsGengetsuCombined)) //Q@Combined̂Ƃ́A_allowCombinedK{
                            list.Add(new SetStockMenuItem(p, _command));
                    }
                    return ArrayUtil.ConvertArray<SetStockMenuItem, IPoderosaMenu>(list.ToArray());
                }
            }
        }
        public class IndicesMenuGroup : PoderosaMenuGroupImpl {
            public enum Category {
                Major, Minor, Custom
            }
            private IPoderosaCommand _command;
            private Category _category;
            private static MarketIndex.Builtins[] major_indices = new MarketIndex.Builtins[] {
                MarketIndex.Builtins.NK225, MarketIndex.Builtins.TOPIX,  MarketIndex.Builtins.MOTHERS, MarketIndex.Builtins.HERCULES, MarketIndex.Builtins.JASDAQ
            };
            private static MarketIndex.Builtins[] minor_indices = new MarketIndex.Builtins[] {
                MarketIndex.Builtins.NK300, MarketIndex.Builtins.NK500, MarketIndex.Builtins.NKJQ, MarketIndex.Builtins.T2INDEX, MarketIndex.Builtins.TREIT, 
                MarketIndex.Builtins.TLARGE,  MarketIndex.Builtins.TMID,   MarketIndex.Builtins.TSMALL,
                MarketIndex.Builtins.HCSTANDARD, MarketIndex.Builtins.HCGROWTH, 
            };
            public IndicesMenuGroup(IPoderosaCommand cmd, Category cat) {
                _command = cmd;
                _category = cat;
            }
            public override IPoderosaMenu[] ChildMenus {
                get {
                    List<IPoderosaMenu> list = new List<IPoderosaMenu>();
                    if(_category==Category.Custom) {
                        foreach(MarketIndex mi in BellagioRoot.GlobalStockCollection.CustomMarketIndices) {
                            list.Add(new SetStockMenuItem(mi, _command));
                        }
                    }
                    else {
                        MarketIndex.Builtins[] t = _category==Category.Major? major_indices : minor_indices;
                        foreach(MarketIndex.Builtins e in t) {
                            AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact(e.ToString());
                            if(prof!=null)
                                list.Add(new SetStockMenuItem(prof, _command));
                        }
                    }

                    //W[O[v́Aqɂ̑̃O[v𓋍ڂ
                    if(_category==Category.Major) {
                        Category cc = BellagioVersionInfo.EDITION==BellagioEditions.HatchukunTX? Category.Minor : Category.Custom;
                        list.Add(new BellagioMenuFolder("̑(&O)", new IPoderosaMenuGroup[] { new IndicesMenuGroup(_command, cc) }));
                    }

                    return list.ToArray();
                }
            }
        }

        public class ChartSchemaMenuGroup : PoderosaMenuGroupImpl {
            
            public ChartSchemaMenuGroup(IPoderosaMenu[] childMenus) {
                _childMenus = childMenus;
            }
            
            public override IPoderosaMenu[] ChildMenus {
                get {
                    return _childMenus;
                }
            }

            public static IPoderosaMenuGroup[] GroupArrayInThisTab() {
                return CreateGroupArray(ChartOpenCommand.OpenStyleT.SameSession);
            }
            public static IPoderosaMenuGroup[] GroupArrayInNewTab() {
                return CreateGroupArray(ChartOpenCommand.OpenStyleT.NewTab);
            }
            private static IPoderosaMenuGroup[] CreateGroupArray(ChartOpenCommand.OpenStyleT style) {
                List<ExtensionKitItem> coll_static = new List<ExtensionKitItem>();
                List<ExtensionKitItem> coll_intraday = new List<ExtensionKitItem>();
                BellagioRoot.SchemaRoot.CollectStaticChartSchema(coll_static);
                BellagioRoot.SchemaRoot.CollectRealTimeChartSchema(coll_intraday);

                IPoderosaMenuGroup group_static = null;
                if(coll_static.Count > 0) {
                    IPoderosaMenuItem[] menus_static = new IPoderosaMenuItem[coll_static.Count];
                    for(int i=0; i<menus_static.Length; i++) {
                        menus_static[i] = CreateChartOpenMenu((ChartSchema)coll_static[i], style);
                    }
                    group_static = new ChartSchemaMenuGroup(menus_static);
                }
                IPoderosaMenuGroup group_intraday = null;
                if(coll_intraday.Count > 0) {
                    IPoderosaMenuItem[] menus_intraday = new IPoderosaMenuItem[coll_intraday.Count];
                    for(int i=0; i<menus_intraday.Length; i++) {
                        menus_intraday[i] = CreateChartOpenMenu((ChartSchema)coll_intraday[i], style);
                    }
                    group_intraday = new ChartSchemaMenuGroup(menus_intraday);
                }

                if(group_intraday==null)
                    return new IPoderosaMenuGroup[] { group_static };
                else if(group_static==null)
                    return new IPoderosaMenuGroup[] { group_intraday };
                else
                    return new IPoderosaMenuGroup[] { group_intraday, group_static };
            }

            private static PoderosaMenuItemImpl CreateChartOpenMenu(ChartSchema k, ChartOpenCommand.OpenStyleT style) {
                ChartOpenCommand cmd = new ChartOpenCommand(k, null, style);
                cmd.UseTargetSessionStock = true;
                StringBuilder bld = new StringBuilder();
                bld.Append(k.description.ParseMandatoryString());
                if(!k.shortcutKey.IsOmitted) {
                    bld.Append("  ");
                    Keys key = Poderosa.Util.WinFormsUtil.ParseKey(k.shortcutKey.ParseMandatoryString());
                    if(key!=Keys.None) {
                        if(style==ChartOpenCommand.OpenStyleT.NewTab) key |= Keys.Shift;
                        bld.Append(Poderosa.Util.WinFormsUtil.FormatShortcut(key));
                    }
                }
                PoderosaMenuItemImpl mi = new PoderosaMenuItemImpl(cmd, bld.ToString());
                
                return mi;
            }
        }
    }

    //`[gJ߂̃R}hBZbVɑ΂ĖύXw邾̃IvVȂǂ
    //TODO ChartPlugin.csOpenChartCommandƎĂ̂œ͂ł͂
    public class ChartOpenCommand : IPoderosaCommand {
        public enum OpenStyleT {
            NewTab, //_schema,_stockŐV^uŊJ
            SameSchema, //_schemaƓXL[}̃ZbV΂gĖύXAłȂΐV^u
            SameSession //CanExecuteœ_targetSession_schema, _stockŊJ
        }
        private ChartSchema _schema;
        private bool _useTargetSessionStock; //_stockZbVɊÂĎݒ肷
        private Stock _stock;
        private ChartSessionBase _targetSession;
        private OpenStyleT _style;
        private ChartSessionBase _resultSession;
        private bool _forStatic;

        public ChartOpenCommand(ChartSchema sc, Stock stock, OpenStyleT style) {
            _schema = sc;
            _stock = stock;
            _style = style;
            _forStatic = sc.ChartUnit!=Bellagio.Values.Quote.QuoteUnit.Minutely;
        }
        public OpenStyleT OpenStyle {
            get {
                return _style;
            }
        }

        public bool UseTargetSessionStock {
            get {
                return _useTargetSessionStock;
            }
            set {
                _useTargetSessionStock = value;
            }
        }

        public Stock CommandTargetStock {
            get {
                return _stock; //͂ƂŃoCh邱Ƃ
            }
            set {
                _stock = value;
            }
        }

        public ChartSessionBase ResultSession {
            get {
                return _resultSession;
            }
        }

        public CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
            if(_style==OpenStyleT.NewTab) {
                return OpenNew(target);
            }
            else if(_style==OpenStyleT.SameSchema) {
                ChartSessionBase sess = FindSession();
                if(sess==null)
                    return OpenNew(target);
                else {
                    //[h͊댯Ȃ̂ŃLZ
                    if(sess.ChartDocument.DocumentStatus==BellagioDocumentStatus.Loading) {
                        Debug.WriteLine("Cancel for loading status");
                        return CommandResult.Cancelled;
                    }

                    //ɃANeBuhLĝƂ͌Ă΂ȂBr[ɃtH[JXsĂ܂AXN[jOʂȂǂ̃EBhEANeBułȂȂĂ܂
                    if(BellagioPlugin.Instance.ActivePoderosaWindow.DocumentTabFeature.ActiveDocument!=sess.IPoderosaDocument)
                        BellagioPlugin.Instance.SessionManager.ActivateDocument(sess.IPoderosaDocument, ActivateReason.InternalAction);
                    sess.SetCurrentStock(_stock);
                    _resultSession = sess;
                    return CommandResult.Succeeded;
                }
            }
            else {
                //SameSession
                ChartSessionBase sess = _targetSession;
                if(sess.ChartDocument.DocumentStatus==BellagioDocumentStatus.Loading) {
                    Debug.WriteLine("Cancel for loading status");
                    return CommandResult.Cancelled;
                }
                sess.ReloadSchema(_schema.FullID, sess.CurrentStock);

                //ɃANeBuhLĝƂ͌Ă΂ȂBr[ɃtH[JXsĂ܂AXN[jOʂȂǂ̃EBhEANeBułȂȂĂ܂
                if(BellagioPlugin.Instance.ActivePoderosaWindow.DocumentTabFeature.ActiveDocument!=sess.IPoderosaDocument)
                    BellagioPlugin.Instance.SessionManager.ActivateDocument(sess.IPoderosaDocument, ActivateReason.InternalAction);
                _resultSession = sess;
                return CommandResult.Succeeded;
            }
        }
        private CommandResult OpenNew(ICommandTarget target) {

            ChartSessionBase rs = CreateSession();
            IPoderosaMainWindow window = (IPoderosaMainWindow)target.GetAdapter(typeof(IPoderosaMainWindow));
            if(window==null) {
                window = BellagioPlugin.Instance.ActivePoderosaWindow;
            }
            _resultSession = rs;
            ISessionManager sm = BellagioPlugin.Instance.SessionManager; 
            sm.StartNewSession(rs, window.ViewManager.GetCandidateViewForNewDocument());
            sm.ActivateDocument(rs.IPoderosaDocument, ActivateReason.InternalAction);

            return CommandResult.Succeeded;
        }
        private ChartSessionBase CreateSession() {
            if(_forStatic) {
                StaticChartSessionCreateParam cp = new StaticChartSessionCreateParam(_schema, _stock);
                cp.Unit = _schema.ChartUnit;
                return new StaticChartSession(cp);
            }
            else {
                RealTimeChartSessionCreateParam cp = new RealTimeChartSessionCreateParam(_schema, _stock);
                cp.Scale = _schema.ChartScale;
                return new RealTimeChartSession(cp);
            }
        }

        public bool CanExecute(ICommandTarget target) {
            ChartViewContentBase vc = ChartCommands.AsViewContent(target);
            if(vc!=null) {
                _targetSession = vc.ChartSessionBase;
                if(_useTargetSessionStock) _stock = _targetSession.CurrentStock;
            }

            if(_stock==null) return true;
            else if(_forStatic) return BellagioRoot.DateBasedQuoteProvider.CanOpenDailyData(_stock);
            else return BellagioRoot.DataSourceHost.CanOpenRealtimeData(_stock);
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        private ChartSessionBase FindSession() {
            foreach(ISession s in BellagioPlugin.Instance.SessionManager.AllSessions) {
                ChartSessionBase sb = s as ChartSessionBase;
                if(sb!=null && sb.GetSourceSchema()==_schema) return sb;
            }
            return null;
        }
    }

    //IGeneralCommand^CṽZbVI[v
    //ChartOpenCommandgĎ邪A̓_قȂ
    // * ANeBuhLg̃XL[}ŎĂ̂ƓȂV^u
    // * ȊOȂANeBuhLgXL[}ύX
    public class ChartOpenGeneralCommand : IGeneralCommand {
        private bool _forStatic;
        private string _schemaID;
        private Keys _defaultShortcutKey;

        public ChartOpenGeneralCommand(string schemaID, Keys shortcut, bool forStatic) {
            _forStatic = forStatic;
            _schemaID = schemaID;
            _defaultShortcutKey = shortcut;
            BellagioPlugin.Instance.CommandManager.Register(this);
        }

        public CommandResult InternalExecute(ICommandTarget target, IAdaptable[] args) {
            IPoderosaMainWindow window = (IPoderosaMainWindow)target.GetAdapter(typeof(IPoderosaMainWindow));
            IPoderosaDocument doc = window.DocumentTabFeature.ActiveDocument;
            ChartSessionBase sess = doc==null? null : doc.OwnerSession as ChartSessionBase;

            ChartOpenCommand cmd;
            if(sess!=null && sess.GetSourceSchema().FullID!=_schemaID) {
                cmd = new ChartOpenCommand(FindSchema(), sess.CurrentStock, ChartOpenCommand.OpenStyleT.SameSession);
                cmd.CanExecute(sess.BellagioViewContent); //Ń^[QbgZbV܂
            }
            else {
                cmd = new ChartOpenCommand(FindSchema(), sess==null? BellagioRoot.GlobalStockCollection.FindExact("NK225").Primary : sess.CurrentStock, ChartOpenCommand.OpenStyleT.NewTab);
            }
            return cmd.InternalExecute(target, args);
        }
        private ChartSchema FindSchema() {
            if(_forStatic) 
                return BellagioRoot.SchemaRoot.FindStaticChartSchema(_schemaID);
            else
                return BellagioRoot.SchemaRoot.FindRealTimeChartSchema(_schemaID);
        }
        public bool CanExecute(ICommandTarget target) {
            return true;
        }
        public string CommandID {
            get {
                return "tactico.openchart."+_schemaID;
            }
        }
        public string Description {
            get {
                return null;
            }
        }
        public Keys DefaultShortcutKey {
            get {
                return _defaultShortcutKey;
            }
        }
        public ICommandCategory CommandCategory {
            get {
                return null;
            }
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        /*
        protected override BellagioSessionBase CreateSession(IAdaptable target) {
            Stock stock = AsStock(target);

            if(_forStatic) {
                ChartSchema cs = BellagioRoot.SchemaRoot.FindStaticChartSchema(_schemaID);
                if(cs!=null) {
                    string sn = cs.chartsession.ParseMandatoryString();
                    Quote.QuoteUnit pru;
                    if(TryParseStaticSession(sn, out pru)) {
                        StaticChartSessionCreateParam cp = new StaticChartSessionCreateParam(cs, stock);
                        cp.Unit = pru;
                        return new StaticChartSession(cp);
                    }
                    else
                        throw new BellagioException("unknown session type"); //TODO errormessage
                }
            }
            else {
                ChartSchema cs = BellagioRoot.SchemaRoot.FindRealTimeChartSchema(_schemaID);
                if(cs!=null) {
                    string sn = cs.chartsession.ParseMandatoryString();
                    int minutely_scale;
                    if(TryParseRealTimeSession(sn, out minutely_scale)) {
                        RealTimeChartSessionCreateParam cp = new RealTimeChartSessionCreateParam(cs, stock);
                        cp.Scale = minutely_scale;
                        return new RealTimeChartSession(cp);
                    }
                    else
                        throw new BellagioException("unknown session type"); //TODO errormessage
                }
            }

            throw new BellagioException(String.Format("`[gXL[} {0} ܂", _schemaID));
        }
        private Stock AsStock(IAdaptable target) {
            if(target==null) return null;

            Stock stock = (Stock)target.GetAdapter(typeof(Stock));
            if(stock!=null) return stock;

            if(target is ICommandTarget)
                return ChartCommands.AsStock((ICommandTarget)target);
            else
                return stock;
        }

        private static bool TryParseStaticSession(string sn, out Quote.QuoteUnit pru) {
            if(sn=="daily")
                pru = Quote.QuoteUnit.Daily;
            else if(sn=="weekly")
                pru = Quote.QuoteUnit.Weekly;
            else if(sn=="monthly")
                pru = Quote.QuoteUnit.Monthly;
            else {
                //߂lɈӖ͂Ȃ
                pru = Quote.QuoteUnit.Daily;
                return false;
            }

            return true; //ok
        }
        private static bool TryParseRealTimeSession(string sn, out int minutely_scale) {
            if(sn=="realtime")
                minutely_scale = 1;
            else if(sn=="realtime3")
                minutely_scale = 3;
            else if(sn=="realtime5")
                minutely_scale = 5;
            else {
                minutely_scale = 1;
                return false;
            }

            return true;
        }
        */
    }

}
