﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using CSCore;

namespace FDK.メディア.サウンド.WASAPI
{
	/// <summary>
	///		オーディオミキサー。
	///		自身が ISampleSource であり、そのまま AudioClient のレンダリングターゲットに指定する。
	/// </summary>
	internal class Mixer : CSCore.ISampleSource
	{
		/// <summary>
		///		音量。0.0(無音)～1.0(原音)。
		/// </summary>
		public float Volume
		{
			get { return this._Volume; }
			set
			{
				if( ( 0.0f > value ) || ( 1.0f < value ) )
					throw new ArgumentOutOfRangeException();

				this._Volume = value;
			}
		}

		/// <summary>
		///		ミキサーのフォーマット。
		/// </summary>
		public CSCore.WaveFormat WaveFormat
		{
			get { return _WaveFormat; }
		}

		/// <summary>
		///		ミキサーはループするので、Position には 非対応。
		/// </summary>
		public long Position
		{
			get { return 0; }
			set { throw new NotSupportedException(); }
		}

		/// <summary>
		///		ミキサーはシークできない。
		/// </summary>
		public bool CanSeek => ( false );

		/// <summary>
		///		ミキサーはループするので、長さの概念はない。
		/// </summary>
		public long Length => ( 0 );

		/// <summary>
		///		指定したフォーマットを持つミキサーを生成する。
		/// </summary>
		public Mixer( CSCore.WaveFormat deviceWaveFormat )
		{
			this._WaveFormat = deviceWaveFormat;
		}

		/// <summary>
		///		ミキサに登録されているサウンドをすべて停止し解放する。
		/// </summary>
		public void Dispose()
		{
			lock( this._スレッド間同期 )
			{
				// すべての Sound を解放する。
				foreach( var sampleSource in this._Sounds )
					sampleSource.Dispose();

				// Sound リストをクリアする。
				this._Sounds.Clear();
			}
		}

		/// <summary>
		///		Sound をミキサーに追加する。
		///		追加されると同時に、Sound の再生が開始される。
		/// </summary>
		public void AddSound( Sound sound )
		{
			if( null == sound )
				throw new ArgumentNullException();

			if( ( sound.SampleSource.WaveFormat.Channels != this._WaveFormat.Channels ) ||		// 同じチャンネル数、
				( sound.SampleSource.WaveFormat.SampleRate != this._WaveFormat.SampleRate ) ||	// 同じ周波数、
				( sound.SampleSource.WaveFormat.WaveFormatTag != AudioEncoding.IeeeFloat ) )	// 常に 32bit-float であること。
			{
				// 違った場合の変換はサポートしない。
				throw new ArgumentException( "ミキサーと同じチャンネル数、サンプルレート、かつ 32bit float 型である必要があります。" );
			}

			lock( this._スレッド間同期 )
			{
				if( !this.Contains( sound ) )
					this._Sounds.Add( sound );
			}
		}

		/// <summary>
		///		Sound をミキサーから除外する。
		///		除外されると同時に、Sound の再生は終了する。
		/// </summary>
		public void RemoveSound( Sound sound )
		{
			lock( this._スレッド間同期 )
			{
				if( this.Contains( sound ) )
					this._Sounds.Remove( sound );
			}
		}

		/// <summary>
		///		Sound がミキサーに登録されているかを調べる。
		/// </summary>
		/// <returns>
		///		Sound がミキサーに追加済みなら true 。
		///	</returns>
		public bool Contains( Sound sound )
		{
			if( null == sound )
				return false;

			return this._Sounds.Contains( sound );
		}

		/// <summary>
		///		バッファにサウンドデータを出力する。
		/// </summary>
		/// <returns>
		///		実際に出力したサンプル数。
		///	</returns>
		public int Read( float[] バッファ, int バッファの出力開始位置, int 出力サンプル数 )
		{
			// ミキサに登録されている Sound の入力とこのメソッドが出力するデータはいずれも常に 32bit-float であり、
			// これは this.WaveFormat.WaveFormatTag とは無関係なので注意。（this.WaveFormat は、チャンネル数とサンプルレートしか見てない。）

			if( 0 < 出力サンプル数 )
			{
				lock( this._スレッド間同期 )
				{
					// 中間バッファが十分あることを確認する。足りなければ新しく確保して戻ってくる。
					this._中間バッファ = this._中間バッファ.CheckBuffer( 出力サンプル数 ); // サンプル数であり、フレーム数（サンプル数×チャンネル数）ではない。

					// まずは無音で埋める。
					Array.Clear( バッファ, 0, 出力サンプル数 );

					// その上に、ミキサに登録されているすべての Sound を加算合成する。
					if( 0 < this._Sounds.Count )
					{
						for( int m = this._Sounds.Count - 1; m >= 0; m-- ) // リストから Remove する場合があるので、リストの後ろから進める。
						{
							var sound = this._Sounds[ m ];

							// 中間バッファにサウンドデータを受け取る。
							int 受け取ったサンプル数 = sound.SampleSource.Read( this._中間バッファ, 0, 出力サンプル数 );

							// 中間バッファから出力バッファへ合成する。
							for( int i = バッファの出力開始位置, n = 0; n < 受け取ったサンプル数; i++, n++ )
							{
								float data = this._中間バッファ[ n ] // 原音
									* sound.Volume                  // 個別音量（Sound）
									* this._Volume;                 // マスタ音量（ミキサ）

								// 先に無音を出力済みなので、上書きかどうかを気にしないで常に加算。
								バッファ[ i ] += data;
							}

							if( 0 == 受け取ったサンプル数 )
							{
								// 再生終了。リストから削除。
								this.RemoveSound( sound );
							}
						}
					}
				}
			}

			return 出力サンプル数;
		}

		private float _Volume = 1.0f;

		private CSCore.WaveFormat _WaveFormat = null;

		private readonly List<Sound> _Sounds = new List<Sound>();

		private float[] _中間バッファ = null;

		private readonly object _スレッド間同期 = new object();
	}
}
