﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using SharpDX;
using SharpDX.Direct2D1;
using SharpDX.DirectWrite;
using FDK;

namespace FDK.メディア
{
	/// <summary>
	///		DirectWrite を使った Direct2D1ビットマップ。
	/// </summary>
	/// <remarks>
	///		「表示文字列」メンバを設定/更新すれば、次回の描画時にビットマップが生成される。
	/// </remarks>
	public class 文字列画像 : Activity
	{
		/// <summary>
		///		このメンバを set すれば、次回の進行描画時に画像が更新される。
		/// </summary>
		public string 表示文字列
		{
			get;
			set;
		} = null;

		public string フォント名
		{
			get;
			set;
		} = "メイリオ";

		public float フォントサイズpt
		{
			get;
			set;
		} = 20.0f;

		public FontWeight フォント幅
		{
			get;
			set;
		} = FontWeight.Normal;

		public FontStyle フォントスタイル
		{
			get;
			set;
		} = FontStyle.Normal;

		public InterpolationMode 補正モード
		{
			get;
			set;
		} = InterpolationMode.Linear;

		public RectangleF? 転送元矩形dpx
		{
			get;
			set;
		} = null;

		public bool 加算合成
		{
			get;
			set;
		} = false;

		public Size2F レイアウトサイズdpx
		{
			get;
			set;
		} = new Size2F( -1f, -1f );

		public bool 下詰め
		{
			get;
			set;
		} = false;


		public 文字列画像()
		{
		}

		public 文字列画像( string 文字列, float フォントサイズpt = 20.0f, string フォント名 = "メイリオ", FontWeight フォント幅 = FontWeight.Normal, FontStyle フォントスタイル = FontStyle.Normal )
			: this()
		{
			this.表示文字列 = 文字列;
			this.フォント名 = フォント名;
			this.フォントサイズpt = フォントサイズpt;
			this.フォント幅 = フォント幅;
			this.フォントスタイル = フォントスタイル;
		}

		protected override void Onデバイス依存リソースの作成( デバイスリソース dr )
		{
			this._前回の表示文字列 = null;

			if( this.表示文字列.Nullでも空でもない() )
			{
				this.ビットマップレンダーターゲットを生成する( dr );
				this._前回の表示文字列 = this.表示文字列; // 最初の構築完了。
			}
		}

		protected override void Onデバイス依存リソースの解放( デバイスリソース dr )
		{
			Utilities.解放する( ref this._黒ブラシ );
			Utilities.解放する( ref this._白ブラシ );
			Utilities.解放する( ref this.ビットマップレンダーターゲット );
			Utilities.解放する( ref this._テキストレイアウト );
			Utilities.解放する( ref this._テキストフォーマット );
		}

		public void 進行描画する(
			デバイスリソース dr,
			float 左位置dpx,
			float 上位置dpx,
			float 不透明度0to1 = 1.0f,
			float X方向拡大率 = 1.0f,
			float Y方向拡大率 = 1.0f,
			Matrix? 変換行列3Dpx = null )
		{
			var 変換行列2Dpx =
				dr.拡大行列DPXtoPX	// スケーリング(1) DPX → PX
				* Matrix3x2.Scaling( X方向拡大率, Y方向拡大率 )   // スケーリング(2)
				* Matrix3x2.Translation( 左位置dpx * dr.拡大率DPXtoPX横方向, 上位置dpx * dr.拡大率DPXtoPX縦方向 );  // 平行移動（物理単位）。

			this.進行描画する( dr, 変換行列2Dpx, 変換行列3Dpx, 不透明度0to1 );
		}

		public void 進行描画する(
			デバイスリソース dr,
			Matrix3x2? 変換行列2Dpx = null,
			Matrix? 変換行列3Dpx = null,
			float 不透明度0to1 = 1.0f )
		{
			Debug.Assert( this.活性化している );

			if( this.表示文字列.Nullまたは空である() )
				return;

			// 表示文字列が変更されているなら、ここで表示ビットマップレンダーターゲットの再構築を行う。
			if( false == string.Equals( this.表示文字列, this._前回の表示文字列 ) )
			{
				this.ビットマップレンダーターゲットを生成する( dr );
			}

			if( null == this.ビットマップレンダーターゲット )
				return;

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				dr.D2DContextの設定をリセットする( dr.D2DContext1 );

				// 変換行列とブレンドモードをD2Dレンダーターゲットに設定する。
				dr.D2DContext1.Transform = 変換行列2Dpx ?? Matrix3x2.Identity;
				dr.D2DContext1.PrimitiveBlend = ( 加算合成 ) ? PrimitiveBlend.Add : PrimitiveBlend.SourceOver;

				// D2Dレンダーターゲットに this.Bitmap を描画する。
				using( var bmp = this.ビットマップレンダーターゲット.Bitmap )
				{
					dr.D2DContext1.DrawBitmap(
						bitmap: bmp,
						destinationRectangle: null,
						opacity: 不透明度0to1,
						interpolationMode: this.補正モード,
						sourceRectangle: this.転送元矩形dpx,
						erspectiveTransformRef: 変換行列3Dpx );
				}
			} );
		}


		protected SharpDX.Direct2D1.BitmapRenderTarget ビットマップレンダーターゲット = null;

		protected void ビットマップレンダーターゲットを生成する( デバイスリソース dr )
		{
			this._前回の表示文字列 = this.表示文字列;

			// テキストフォーマット／レイアウトを作成し、表示ビットマップのサイズを計算する。
			if( null == this._テキストフォーマット )
			{
				this._テキストフォーマット = new TextFormat( dr.DWriteFactory, this.フォント名, this.フォント幅, this.フォントスタイル, this.フォントサイズpt ) {
					TextAlignment = TextAlignment.Leading,
				};
			}

			this.レイアウトサイズdpx = new Size2F( dr.設計画面サイズdpx.Width, dr.設計画面サイズdpx.Height );

			this._テキストレイアウト?.Dispose();
			this._テキストレイアウト = new TextLayout(
				dr.DWriteFactory,
				this.表示文字列,
				this._テキストフォーマット,
				this.レイアウトサイズdpx.Width,
				this.レイアウトサイズdpx.Height );

			var 表示ビットマップのサイズdpx = new Size2F();
			var 上マージン = 0.0f;

			if( this.下詰め )
			{
				表示ビットマップのサイズdpx = new Size2F(
					this._テキストレイアウト.Metrics.WidthIncludingTrailingWhitespace,
					this.レイアウトサイズdpx.Height );       // レイアウトの最大高

				上マージン = this.レイアウトサイズdpx.Height - this._テキストレイアウト.Metrics.Height;
			}
			else
			{
				表示ビットマップのサイズdpx = new Size2F(
					this._テキストレイアウト.Metrics.WidthIncludingTrailingWhitespace,
					this._テキストレイアウト.Metrics.Height );
			}

			// 実際のビットマップサイズに更新。
			this.レイアウトサイズdpx = 表示ビットマップのサイズdpx;

			// ビットマップレンダーターゲットを生成する。
			using( var target = dr.D2DContext1.Target )	// Target を get すると COM参照カウンタが増えるので注意。
			{
				// D2DContext1.Target が設定済みであること。さもなきゃ例外も出さずに落ちる。
				Debug.Assert( null != target );
			}
			this.ビットマップレンダーターゲット?.Dispose();
			this.ビットマップレンダーターゲット = new SharpDX.Direct2D1.BitmapRenderTarget(
				dr.D2DContext1,
				CompatibleRenderTargetOptions.None,
				表示ビットマップのサイズdpx );

			// ブラシの作成がまだなら行う。
			if( null == this._白ブラシ )
			{
				this._白ブラシ = new SolidColorBrush( this.ビットマップレンダーターゲット, Color.LightGray );
			}
			if( null == this._黒ブラシ )
			{
				this._黒ブラシ = new SolidColorBrush( this.ビットマップレンダーターゲット, Color.Black );
			}

			// ビットマップレンダーターゲットにテキストを描画する。
			Utilities.D2DBatchDraw( this.ビットマップレンダーターゲット, () => {

				this.ビットマップレンダーターゲット.Clear( Color.Transparent );

				this.ビットマップレンダーターゲット.DrawTextLayout(    // ドロップシャドウ
					new Vector2( 1.0f, 上マージン + 1.0f ),
					this._テキストレイアウト,
					this._黒ブラシ,
					DrawTextOptions.Clip );

				this.ビットマップレンダーターゲット.DrawTextLayout(    // 本体
					new Vector2( 0.0f, 上マージン ),
					this._テキストレイアウト,
					this._白ブラシ,
					DrawTextOptions.Clip );
			} );
		}


		private string _前回の表示文字列 = null;

		private TextFormat _テキストフォーマット = null;

		private TextLayout _テキストレイアウト = null;

		private SolidColorBrush _白ブラシ = null;

		private SolidColorBrush _黒ブラシ = null;
	}
}
