package org.sqlite.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.List;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.sqlite.schema.DatabaseList;
import org.sqlite.schema.ForeignKeyList;
import org.sqlite.schema.IndexInfo;
import org.sqlite.schema.IndexList;
import org.sqlite.schema.TableInfo;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class JdbcDatabaseMetaDataTest {

    public JdbcDatabaseMetaDataTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private static final String DRIVER_CLASS = "org.sqlite.Driver";
    private static final String DATABASE = System.getProperty("user.dir") + "/test/unittest.db";
    
    private static Connection newConnection() throws ClassNotFoundException, SQLException {
        Class.forName(DRIVER_CLASS);
        return DriverManager.getConnection("jdbc:sqlite:file:" + DATABASE);
    }
    
//    @Test(expected = java.sql.SQLException.class)
//    public void getTableInfoThrowSQLException() throws ClassNotFoundException, SQLException {
//        final Connection conn = newConnection();
//        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
//
//        List<TableInfo> dl = meta.getTableInfo("temp_tbl_1");
//        assertEquals(0, dl.size());
//        
//        conn.close();
//    }
    
    @Test
    public void getTableInfo() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
        List<TableInfo> ti = null;
        
        ti = meta.getTableInfo("temp_tbl_1");
        assertEquals(0, ti.size());
        
        final Statement stmt = conn.createStatement();
        final String tVALUE_DEFAULT = "'_blank''_'";
        final String tVALUE2_DEFAULT = "'_blank''_blank''''_blank_''''''_'";
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT " + tVALUE_DEFAULT
                    + ", tVALUE2 TEXT NOT NULL DEFAULT " + tVALUE2_DEFAULT
                + ")";
        stmt.executeUpdate(sql);

        ti = meta.getTableInfo("temp_tbl_1");
        assertNotNull(ti);
        assertEquals(4, ti.size());
        
        assertEquals(0, ti.get(0).columnIndex);
        assertEquals("ROWID", ti.get(0).columnName);
        assertEquals("INTEGER", ti.get(0).dataType);
        assertNull(ti.get(0).defaultValue);
        assertFalse(ti.get(0).isNotNull);
        assertTrue(ti.get(0).isPrimaryKey);
        
        assertEquals(1, ti.get(1).columnIndex);
        assertEquals("ID", ti.get(1).columnName);
        assertEquals("INTEGER", ti.get(1).dataType);
        assertNull(ti.get(1).defaultValue);
        assertTrue(ti.get(1).isNotNull);
        assertFalse(ti.get(1).isPrimaryKey);
        
        assertEquals(2, ti.get(2).columnIndex);
        assertEquals("tVALUE", ti.get(2).columnName);
        assertEquals("TEXT", ti.get(2).dataType);
        assertEquals(tVALUE_DEFAULT, ti.get(2).defaultValue);
        assertFalse(ti.get(2).isNotNull);
        assertFalse(ti.get(2).isPrimaryKey);
        
        assertEquals(3, ti.get(3).columnIndex);
        assertEquals("tVALUE2", ti.get(3).columnName);
        assertEquals("TEXT", ti.get(3).dataType);
        assertEquals(tVALUE2_DEFAULT, ti.get(3).defaultValue);
        assertTrue(ti.get(3).isNotNull);
        assertFalse(ti.get(3).isPrimaryKey);
        
        conn.close();
    }
    
    @Test
    public void getIndexList() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", rVALUE REAL"
                + ")";
        stmt.executeUpdate(sql);

        List<IndexList> ix = meta.getIndexList("temp_tbl_1");
        assertNotNull(ix);
        assertEquals(0, ix.size());
        
        sql = "CREATE UNIQUE INDEX IF NOT EXISTS ix_temp_tbl_1 "
                + "ON temp_tbl_1(ID COLLATE BINARY DESC)";
        stmt.executeUpdate(sql);
        
        ix = meta.getIndexList("temp_tbl_1");
        assertNotNull(ix);
        assertEquals(1, ix.size());
        assertEquals(0, ix.get(0).seq);
        assertEquals("ix_temp_tbl_1", ix.get(0).indexName);
        assertTrue(ix.get(0).isUnique);
        
        sql = "CREATE INDEX IF NOT EXISTS ix_temp_tbl_2 "
                + "ON temp_tbl_1(tVALUE COLLATE NOCASE)";
        stmt.executeUpdate(sql);
        
        ix = meta.getIndexList("temp_tbl_1");
        assertNotNull(ix);
        assertEquals(2, ix.size());
        assertEquals(0, ix.get(0).seq);
        assertEquals("ix_temp_tbl_2", ix.get(0).indexName);
        assertFalse(ix.get(0).isUnique);
        assertEquals(1, ix.get(1).seq);
        assertEquals("ix_temp_tbl_1", ix.get(1).indexName);
        assertTrue(ix.get(1).isUnique);

        sql = "CREATE INDEX IF NOT EXISTS ix_temp_tbl_3 "
                + "ON temp_tbl_1(rVALUE COLLATE NOCASE)";
        stmt.executeUpdate(sql);
        
        ix = meta.getIndexList("temp_tbl_1");
        assertNotNull(ix);
        assertEquals(3, ix.size());
        assertEquals(0, ix.get(0).seq);
        assertEquals("ix_temp_tbl_3", ix.get(0).indexName);
        assertFalse(ix.get(0).isUnique);
        assertEquals(1, ix.get(1).seq);
        assertEquals("ix_temp_tbl_2", ix.get(1).indexName);
        assertFalse(ix.get(1).isUnique);
        assertEquals(2, ix.get(2).seq);
        assertEquals("ix_temp_tbl_1", ix.get(2).indexName);
        assertTrue(ix.get(2).isUnique);
        
        conn.close();
    }
    
    @Test
    public void getIndexInfo_() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", rVALUE REAL"
                + ")";
        stmt.executeUpdate(sql);

        List<IndexInfo> ix = meta.getIndexInfo("ix_temp_tbl_1");
        assertNotNull(ix);
        assertEquals(0, ix.size());
        
        sql = "CREATE UNIQUE INDEX IF NOT EXISTS ix_temp_tbl_1 "
                + "ON temp_tbl_1(ID COLLATE BINARY DESC)";
        stmt.executeUpdate(sql);
        
        ix = meta.getIndexInfo("ix_temp_tbl_1");
        assertNotNull(ix);
        assertEquals(1, ix.size());
        assertEquals("ID", ix.get(0).columnName);

        sql = "CREATE UNIQUE INDEX IF NOT EXISTS ix_temp_tbl_2 "
                + "ON temp_tbl_1(ID COLLATE BINARY DESC, rVALUE COLLATE BINARY ASC)";
        stmt.executeUpdate(sql);
        
        ix = meta.getIndexInfo("ix_temp_tbl_2");
        assertNotNull(ix);
        assertEquals(2, ix.size());
        assertEquals("ID", ix.get(0).columnName);
        assertEquals("rVALUE", ix.get(1).columnName);
        
        conn.close();
    }
    @Test
    public void getIndexInfo() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final DatabaseMetaData meta = conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", rVALUE REAL"
                + ")";
        stmt.executeUpdate(sql);
        ResultSet rs = null;
        
        rs = meta.getIndexInfo(null, null, "temp_tbl_1", false, true);
        assertFalse(rs.next());
        ResultSetMetaData rsMeta = rs.getMetaData();
        assertEquals("TABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("TABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("TABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("NON_UNIQUE", rsMeta.getColumnLabel(4));
        assertEquals("INDEX_QUALIFIER", rsMeta.getColumnLabel(5));
        assertEquals("INDEX_NAME", rsMeta.getColumnLabel(6));
        assertEquals("TYPE", rsMeta.getColumnLabel(7));
        assertEquals("ORDINAL_POSITION", rsMeta.getColumnLabel(8));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(9));
        assertEquals("ASC_OR_DESC", rsMeta.getColumnLabel(10));
        assertEquals("CARDINALITY", rsMeta.getColumnLabel(11));
        assertEquals("PAGES", rsMeta.getColumnLabel(12));
        assertEquals("FILTER_CONDITION", rsMeta.getColumnLabel(13));
        rs.close();
        
        sql = "CREATE UNIQUE INDEX IF NOT EXISTS ix_temp_tbl_1 "
                + "ON temp_tbl_1(ID COLLATE BINARY DESC)";
        stmt.executeUpdate(sql);
        
        rs = meta.getIndexInfo(null, null, "temp_tbl_1", false, true);
        assertTrue(rs.next());
        assertNull(rs.getString(1));
        assertNull(rs.getString(2));
        assertEquals("temp_tbl_1", rs.getString(3));
        assertTrue(rs.getBoolean(4));
        assertNull(rs.getString(5));
        assertEquals("ix_temp_tbl_1", rs.getString(6));
        assertEquals(DatabaseMetaData.tableIndexOther, rs.getShort(7));
        assertEquals(1, rs.getInt(8));
        assertEquals("ID", rs.getString(9));
        assertEquals("A", rs.getString(10));
        assertNull(rs.getString(11));
        assertNull(rs.getString(12));
        assertNull(rs.getString(13));
        
        rs.close();

        
        sql = "CREATE INDEX IF NOT EXISTS ix_temp_tbl_2 "
                + "ON temp_tbl_1(tVALUE, rVALUE COLLATE BINARY ASC)";
        stmt.executeUpdate(sql);
        
        rs = meta.getIndexInfo(null, null, "temp_tbl_1", false, true);

        assertTrue(rs.next());
        assertNull(rs.getString(1));
        assertNull(rs.getString(2));
        assertEquals("temp_tbl_1", rs.getString(3));
        assertFalse(rs.getBoolean(4));
        assertNull(rs.getString(5));
        assertEquals("ix_temp_tbl_2", rs.getString(6));
        assertEquals(DatabaseMetaData.tableIndexOther, rs.getShort(7));
        assertEquals(1, rs.getInt(8));
        assertEquals("tVALUE", rs.getString(9));
        assertEquals("A", rs.getString(10));
        assertNull(rs.getString(11));
        assertNull(rs.getString(12));
        assertNull(rs.getString(13));

        assertTrue(rs.next());
        assertNull(rs.getString(1));
        assertNull(rs.getString(2));
        assertEquals("temp_tbl_1", rs.getString(3));
        assertFalse(rs.getBoolean(4));
        assertNull(rs.getString(5));
        assertEquals("ix_temp_tbl_2", rs.getString(6));
        assertEquals(DatabaseMetaData.tableIndexOther, rs.getShort(7));
        assertEquals(2, rs.getInt(8));
        assertEquals("rVALUE", rs.getString(9));
        assertEquals("A", rs.getString(10));
        assertNull(rs.getString(11));
        assertNull(rs.getString(12));
        assertNull(rs.getString(13));

        assertTrue(rs.next());
        assertNull(rs.getString(1));
        assertNull(rs.getString(2));
        assertEquals("temp_tbl_1", rs.getString(3));
        assertTrue(rs.getBoolean(4));
        assertNull(rs.getString(5));
        assertEquals("ix_temp_tbl_1", rs.getString(6));
        assertEquals(DatabaseMetaData.tableIndexOther, rs.getShort(7));
        assertEquals(1, rs.getInt(8));
        assertEquals("ID", rs.getString(9));
        assertEquals("A", rs.getString(10));
        assertNull(rs.getString(11));
        assertNull(rs.getString(12));
        assertNull(rs.getString(13));

        rs.close();
        
        conn.close();
    }
    
    @Test
    public void getForeignKeyList() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE REAL"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
                + "  ID INTEGER PRIMARY KEY"
                + ", SUB_ID INTEGER NOT NULL"
                + ", EXTRA_ID INTEGER NOT NULL"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);
        
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_5("
                + " TEMP_TBL_1_ID INTEGER NOT NULL"
                + ", TEMP_TBL_2_ID INTEGER NOT NULL"
                + ", TEMP_TBL_3_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_SUB_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_EXTRA_ID INTEGER NOT NULL"
                + ", VALUE2 TEXT"
                + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ", FOREIGN KEY(TEMP_TBL_2_ID) REFERENCES temp_tbl_2(ID)"
                + ", FOREIGN KEY(TEMP_TBL_3_ID) REFERENCES temp_tbl_3(ID)"
                + ", FOREIGN KEY(TEMP_TBL_4_ID, TEMP_TBL_4_SUB_ID, TEMP_TBL_4_EXTRA_ID) REFERENCES temp_tbl_4(ID, SUB_ID, EXTRA_ID)"
            + ")";
//        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
//                + " TEMP_TBL_1_ID INTEGER NOT NULL REFERENCES temp_tbl_1(ID)"
//                + ", TEMP_TBL_2_ID INTEGER NOT NULL REFERENCES temp_tbl_2(ID)"
//                + ", TEMP_TBL_3_ID INTEGER NOT NULL REFERENCES temp_tbl_3(ID)"
//                + ", VALUE2 TEXT"
//            + ")";
        stmt.executeUpdate(sql);

        List<ForeignKeyList> fil = null;
        
        fil = meta.getForeignKeyList("temp_tbl_1");
        assertEquals(0, fil.size());
        
        fil = meta.getForeignKeyList("temp_tbl_2");
        assertEquals(0, fil.size());
        
        fil = meta.getForeignKeyList("temp_tbl_3");
        assertEquals(0, fil.size());
        
        fil = meta.getForeignKeyList("temp_tbl_4");
        assertEquals(0, fil.size());
        
        fil = meta.getForeignKeyList("temp_tbl_5");
        assertEquals(6, fil.size());

        conn.close();
    }
    
    @Test
    public void getColumns() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", rVALUE REAL"
                    + ", bVALUE BLOB"
                    + ", nVALUE NULL"
                    + ", tVALUE2 text DEFAULT ''"
                    + ", tVALUE3 VarChar DEFAULT NULL"
                + ")";
        stmt.executeUpdate(sql);
        sql = "INSERT INTO temp_tbl_1("
                + "ID"
                + ", rVALUE"
                + ", bVALUE"
                + ", nVALUE"
            + ") VALUES(1, 2.345, X'E38182E38184E38186E38188E3818A00', 10)";
        stmt.executeUpdate(sql);
        stmt.close();
        
        // TODO do unit test!
        DatabaseMetaData dbMeta = conn.getMetaData();
        for (ResultSet rs = dbMeta.getColumns(null, null, "temp_tbl_1", "%VALUE%"); rs.next(); ) {
            int cntColumns = rs.getMetaData().getColumnCount();
            for (int i = 1; i <= cntColumns; ++i) {
                System.out.print(rs.getString(i) + " | ");
            }
            System.out.println();
        }
        
        conn.close();
    }

//    @Test(expected = java.sql.SQLException.class)
//    public void getPrimaryKeysThrowSQLException() throws ClassNotFoundException, SQLException {
//        final Connection conn = newConnection();
//        final DatabaseMetaData meta = conn.getMetaData();
//
//        ResultSet rs = meta.getPrimaryKeys(null, null, "temp_tbl_1");
//        assertFalse(rs.next());
//
//        conn.close();
//    }

    @Test
    public void getPrimaryKeys() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final DatabaseMetaData meta = conn.getMetaData();
        ResultSet rs = null;
        
        rs = meta.getPrimaryKeys(null, null, "temp_tbl_1");
        assertFalse(rs.next());
        rs.close();
        
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER NOT NULL"
                    + ", SUB_ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", PRIMARY KEY (ID, SUB_ID)"
                + ")";
        stmt.executeUpdate(sql);

        rs = meta.getPrimaryKeys(null, null, "temp_tbl_1");
        assertEquals(6, rs.getMetaData().getColumnCount());
        assertTrue(rs.next());
        assertEquals("temp_tbl_1", rs.getString("TABLE_NAME"));
        assertEquals("ID", rs.getString("COLUMN_NAME"));
        assertEquals(1, rs.getInt("KEY_SEQ"));
        assertTrue(rs.next());
        assertEquals("temp_tbl_1", rs.getString("TABLE_NAME"));
        assertEquals("SUB_ID", rs.getString("COLUMN_NAME"));
        assertEquals(2, rs.getInt("KEY_SEQ"));
        rs.close();

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER NOT NULL"
                + ", SUB_ID INTEGER NOT NULL"
                + ", tVALUE TEXT DEFAULT '_blank''_'"
            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getPrimaryKeys(null, null, "temp_tbl_2");
        assertEquals(6, rs.getMetaData().getColumnCount());
        assertTrue(rs.next());
        assertEquals("temp_tbl_2", rs.getString("TABLE_NAME"));
        assertEquals("_ROWID_", rs.getString("COLUMN_NAME"));
        assertEquals(1, rs.getInt("KEY_SEQ"));
        assertFalse(rs.next());
        rs.close();

//        rs = meta.getPrimaryKeys(null, null, "temp_tbl_3");
//        assertEquals(6, rs.getMetaData().getColumnCount());
//        assertFalse(rs.next());
//        rs.close();

        stmt.close();
        conn.close();
    }

    @Test
    public void getImportedKeys() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE REAL"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
                + "  ID INTEGER PRIMARY KEY"
                + ", SUB_ID INTEGER NOT NULL"
                + ", EXTRA_ID INTEGER NOT NULL"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        ResultSet rs = null;
        
        rs = meta.getImportedKeys(null, null, "temp_tbl_5");
        assertFalse(rs.next());
        ResultSetMetaData rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        rs.close();
        
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_5("
                + " TEMP_TBL_1_ID INTEGER NOT NULL"
                + ", TEMP_TBL_2_ID INTEGER NOT NULL"
                + ", TEMP_TBL_3_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_SUB_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_EXTRA_ID INTEGER NOT NULL"
                + ", VALUE2 TEXT"
                + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ", FOREIGN KEY(TEMP_TBL_2_ID) REFERENCES temp_tbl_2(ID)"
                + ", FOREIGN KEY(TEMP_TBL_3_ID) REFERENCES temp_tbl_3(ID)"
                + ", FOREIGN KEY(TEMP_TBL_4_ID, TEMP_TBL_4_SUB_ID, TEMP_TBL_4_EXTRA_ID) REFERENCES temp_tbl_4(ID, SUB_ID, EXTRA_ID)"
            + ")";
//        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
//                + " TEMP_TBL_1_ID INTEGER NOT NULL REFERENCES temp_tbl_1(ID)"
//                + ", TEMP_TBL_2_ID INTEGER NOT NULL REFERENCES temp_tbl_2(ID)"
//                + ", TEMP_TBL_3_ID INTEGER NOT NULL REFERENCES temp_tbl_3(ID)"
//                + ", VALUE2 TEXT"
//            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getImportedKeys(null, null, "temp_tbl_5");
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        
        assertEquals("temp_tbl_1", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_1_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_2", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_2_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_3", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_3_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("SUB_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_SUB_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("EXTRA_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_EXTRA_ID", rs.getString(8));
        
        rs.close();
        conn.close();
    }
    
    @Test
    public void getCrossReference() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();

        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE REAL"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
                + "  ID INTEGER PRIMARY KEY"
                + ", SUB_ID INTEGER NOT NULL"
                + ", EXTRA_ID INTEGER NOT NULL"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        ResultSet rs = null;
        
        rs = meta.getCrossReference(null, null, "temp_tbl_1", null, null, "temp_tbl_5");
        assertFalse(rs.next());
        ResultSetMetaData rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        rs.close();
                
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_5("
                + " TEMP_TBL_1_ID INTEGER NOT NULL"
                + ", TEMP_TBL_2_ID INTEGER NOT NULL"
                + ", TEMP_TBL_3_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_SUB_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_EXTRA_ID INTEGER NOT NULL"
                + ", VALUE2 TEXT"
                + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ", FOREIGN KEY(TEMP_TBL_2_ID) REFERENCES temp_tbl_2(ID)"
                + ", FOREIGN KEY(TEMP_TBL_3_ID) REFERENCES temp_tbl_3(ID)"
                + ", FOREIGN KEY(TEMP_TBL_4_ID, TEMP_TBL_4_SUB_ID, TEMP_TBL_4_EXTRA_ID) REFERENCES temp_tbl_4(ID, SUB_ID, EXTRA_ID)"
            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getCrossReference(null, null, "temp_tbl_1", null, null, "temp_tbl_5");
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        
        assertEquals("temp_tbl_1", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_1_ID", rs.getString(8));
        rs.close();
        
        rs = meta.getCrossReference(null, null, "temp_tbl_2", null, null, "temp_tbl_5");
        assertTrue(rs.next());
        assertEquals("temp_tbl_2", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_2_ID", rs.getString(8));
        rs.close();
        
        rs = meta.getCrossReference(null, null, "temp_tbl_3", null, null, "temp_tbl_5");
        assertTrue(rs.next());
        assertEquals("temp_tbl_3", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_3_ID", rs.getString(8));
        rs.close();
        
        rs = meta.getCrossReference(null, null, "temp_tbl_4", null, null, "temp_tbl_5");
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("SUB_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_SUB_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("EXTRA_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_EXTRA_ID", rs.getString(8));
        
        rs.close();
        conn.close();
    }
    
    @Test
    public void getExportedKeys() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
        ResultSet rs = null;
        ResultSetMetaData rsMeta = null;
        
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")";
        stmt.executeUpdate(sql);

        rs = meta.getExportedKeys(null, null, "temp_tbl_1");
        assertFalse(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        rs.close();
        
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE REAL"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                + "  ID INTEGER PRIMARY KEY"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_4("
                + "  ID INTEGER PRIMARY KEY"
                + ", SUB_ID INTEGER NOT NULL"
                + ", EXTRA_ID INTEGER NOT NULL"
                + ", VALUE BLOB"
            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getExportedKeys(null, null, "temp_tbl_4");
        assertFalse(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        rs.close();
        
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_5("
                + " TEMP_TBL_1_ID INTEGER NOT NULL"
                + ", TEMP_TBL_2_ID INTEGER NOT NULL"
                + ", TEMP_TBL_3_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_SUB_ID INTEGER NOT NULL"
                + ", TEMP_TBL_4_EXTRA_ID INTEGER NOT NULL"
                + ", VALUE2 TEXT"
                + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ", FOREIGN KEY(TEMP_TBL_2_ID) REFERENCES temp_tbl_2(ID)"
                + ", FOREIGN KEY(TEMP_TBL_3_ID) REFERENCES temp_tbl_3(ID)"
                + ", FOREIGN KEY(TEMP_TBL_4_ID, TEMP_TBL_4_SUB_ID, TEMP_TBL_4_EXTRA_ID) REFERENCES temp_tbl_4(ID, SUB_ID, EXTRA_ID)"
            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getExportedKeys(null, null, "temp_tbl_4");
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("PKTABLE_CAT", rsMeta.getColumnLabel(1));
        assertEquals("PKTABLE_SCHEM", rsMeta.getColumnLabel(2));
        assertEquals("PKTABLE_NAME", rsMeta.getColumnLabel(3));
        assertEquals("PKCOLUMN_NAME", rsMeta.getColumnLabel(4));
        assertEquals("FKTABLE_CAT", rsMeta.getColumnLabel(5));
        assertEquals("FKTABLE_SCHEM", rsMeta.getColumnLabel(6));
        assertEquals("FKTABLE_NAME", rsMeta.getColumnLabel(7));
        assertEquals("FKCOLUMN_NAME", rsMeta.getColumnLabel(8));
        assertEquals("KEY_SEQ", rsMeta.getColumnLabel(9));
        assertEquals("UPDATE_RULE", rsMeta.getColumnLabel(10));
        assertEquals("DELETE_RULE", rsMeta.getColumnLabel(11));
        assertEquals("FK_NAME", rsMeta.getColumnLabel(12));
        assertEquals("PK_NAME", rsMeta.getColumnLabel(13));
        assertEquals("DEFERRABILITY", rsMeta.getColumnLabel(14));
        
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("SUB_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_SUB_ID", rs.getString(8));
        
        assertTrue(rs.next());
        assertEquals("temp_tbl_4", rs.getString(3));
        assertEquals("EXTRA_ID", rs.getString(4));
        assertEquals("temp_tbl_5", rs.getString(7));
        assertEquals("TEMP_TBL_4_EXTRA_ID", rs.getString(8));
        
        rs.close();
        conn.close();
    }

    @Test
    public void getBestRowIdentifier() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final DatabaseMetaData meta = conn.getMetaData();
        ResultSet rs = null;
        ResultSetMetaData rsMeta = null;
        
        rs = meta.getBestRowIdentifier(null, null, "temp_tbl_1", DatabaseMetaData.bestRowSession, true);
        assertFalse(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("SCOPE", rsMeta.getColumnLabel(1));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(2));
        assertEquals("DATA_TYPE", rsMeta.getColumnLabel(3));
        assertEquals("TYPE_NAME", rsMeta.getColumnLabel(4));
        assertEquals("COLUMN_SIZE", rsMeta.getColumnLabel(5));
        assertEquals("BUFFER_LENGTH", rsMeta.getColumnLabel(6));
        assertEquals("DECIMAL_DIGITS", rsMeta.getColumnLabel(7));
        assertEquals("PSEUDO_COLUMN", rsMeta.getColumnLabel(8));
        rs.close();
        
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER NOT NULL"
                    + ", SUB_ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                    + ", PRIMARY KEY (ID, SUB_ID)"
                + ")";
        stmt.executeUpdate(sql);

        rs = meta.getBestRowIdentifier(null, null, "temp_tbl_1", DatabaseMetaData.bestRowSession, true);
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("SCOPE", rsMeta.getColumnLabel(1));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(2));
        assertEquals("DATA_TYPE", rsMeta.getColumnLabel(3));
        assertEquals("TYPE_NAME", rsMeta.getColumnLabel(4));
        assertEquals("COLUMN_SIZE", rsMeta.getColumnLabel(5));
        assertEquals("BUFFER_LENGTH", rsMeta.getColumnLabel(6));
        assertEquals("DECIMAL_DIGITS", rsMeta.getColumnLabel(7));
        assertEquals("PSEUDO_COLUMN", rsMeta.getColumnLabel(8));
        
        assertEquals(DatabaseMetaData.bestRowSession, rs.getInt("SCOPE"));
        assertEquals("ID", rs.getString("COLUMN_NAME"));
        assertEquals(Types.INTEGER, rs.getInt("DATA_TYPE"));
        assertEquals("INTEGER", rs.getString("TYPE_NAME"));
        assertEquals(10, rs.getInt("COLUMN_SIZE"));
        assertNull(rs.getString("BUFFER_LENGTH"));
        assertNull(rs.getString("DECIMAL_DIGITS"));
        assertEquals(DatabaseMetaData.bestRowNotPseudo, rs.getInt("PSEUDO_COLUMN"));
        rs.close();
        
        sql = "INSERT INTO temp_tbl_1 VALUES(1, 2, 'ValuE')";
        stmt.executeUpdate(sql);
        
        rs = meta.getBestRowIdentifier(null, null, "temp_tbl_1", DatabaseMetaData.bestRowSession, true);
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("SCOPE", rsMeta.getColumnLabel(1));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(2));
        assertEquals("DATA_TYPE", rsMeta.getColumnLabel(3));
        assertEquals("TYPE_NAME", rsMeta.getColumnLabel(4));
        assertEquals("COLUMN_SIZE", rsMeta.getColumnLabel(5));
        assertEquals("BUFFER_LENGTH", rsMeta.getColumnLabel(6));
        assertEquals("DECIMAL_DIGITS", rsMeta.getColumnLabel(7));
        assertEquals("PSEUDO_COLUMN", rsMeta.getColumnLabel(8));
        
        assertEquals(DatabaseMetaData.bestRowSession, rs.getInt("SCOPE"));
        assertEquals("ID", rs.getString("COLUMN_NAME"));
        assertEquals(Types.INTEGER, rs.getInt("DATA_TYPE"));
        assertEquals("INTEGER", rs.getString("TYPE_NAME"));
        assertEquals(10, rs.getInt("COLUMN_SIZE"));
        assertNull(rs.getString("BUFFER_LENGTH"));
        assertNull(rs.getString("DECIMAL_DIGITS"));
        assertEquals(DatabaseMetaData.bestRowNotPseudo, rs.getInt("PSEUDO_COLUMN"));
        
        assertTrue(rs.next());
        assertEquals(DatabaseMetaData.bestRowSession, rs.getInt("SCOPE"));
        assertEquals("SUB_ID", rs.getString("COLUMN_NAME"));
        assertEquals(Types.INTEGER, rs.getInt("DATA_TYPE"));
        assertEquals("INTEGER", rs.getString("TYPE_NAME"));
        assertEquals(10, rs.getInt("COLUMN_SIZE"));
        assertNull(rs.getString("BUFFER_LENGTH"));
        assertNull(rs.getString("DECIMAL_DIGITS"));
        assertEquals(DatabaseMetaData.bestRowNotPseudo, rs.getInt("PSEUDO_COLUMN"));
        rs.close();

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                + "  ID INTEGER NOT NULL"
                + ", SUB_ID INTEGER NOT NULL"
                + ", tVALUE TEXT DEFAULT '_blank''_'"
            + ")";
        stmt.executeUpdate(sql);

        rs = meta.getBestRowIdentifier(null, null, "temp_tbl_2", DatabaseMetaData.bestRowSession, true);
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("SCOPE", rsMeta.getColumnLabel(1));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(2));
        assertEquals("DATA_TYPE", rsMeta.getColumnLabel(3));
        assertEquals("TYPE_NAME", rsMeta.getColumnLabel(4));
        assertEquals("COLUMN_SIZE", rsMeta.getColumnLabel(5));
        assertEquals("BUFFER_LENGTH", rsMeta.getColumnLabel(6));
        assertEquals("DECIMAL_DIGITS", rsMeta.getColumnLabel(7));
        assertEquals("PSEUDO_COLUMN", rsMeta.getColumnLabel(8));
        
        assertEquals(DatabaseMetaData.bestRowSession, rs.getInt("SCOPE"));
        assertEquals("_ROWID_", rs.getString("COLUMN_NAME"));
        assertEquals(Types.INTEGER, rs.getInt("DATA_TYPE"));
        assertEquals("INTEGER", rs.getString("TYPE_NAME"));
        assertEquals(10, rs.getInt("COLUMN_SIZE"));
        assertNull(rs.getString("BUFFER_LENGTH"));
        assertNull(rs.getString("DECIMAL_DIGITS"));
        assertEquals(DatabaseMetaData.bestRowNotPseudo, rs.getInt("PSEUDO_COLUMN"));
        rs.close();

        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                + "  ID INTEGER NOT NULL"
                + ", SUB_ID INTEGER"
                + ", tVALUE TEXT DEFAULT '_blank''_'"
                + ", PRIMARY KEY (ID, SUB_ID)"
            + ")";
        stmt.executeUpdate(sql);

        sql = "INSERT INTO temp_tbl_3 VALUES(1, NULL, 'ValuE')";
        stmt.executeUpdate(sql);
        
        rs = meta.getBestRowIdentifier(null, null, "temp_tbl_3", DatabaseMetaData.bestRowSession, false);
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("SCOPE", rsMeta.getColumnLabel(1));
        assertEquals("COLUMN_NAME", rsMeta.getColumnLabel(2));
        assertEquals("DATA_TYPE", rsMeta.getColumnLabel(3));
        assertEquals("TYPE_NAME", rsMeta.getColumnLabel(4));
        assertEquals("COLUMN_SIZE", rsMeta.getColumnLabel(5));
        assertEquals("BUFFER_LENGTH", rsMeta.getColumnLabel(6));
        assertEquals("DECIMAL_DIGITS", rsMeta.getColumnLabel(7));
        assertEquals("PSEUDO_COLUMN", rsMeta.getColumnLabel(8));
        
        assertEquals(DatabaseMetaData.bestRowSession, rs.getInt("SCOPE"));
        assertEquals("ID", rs.getString("COLUMN_NAME"));
        assertEquals(Types.INTEGER, rs.getInt("DATA_TYPE"));
        assertEquals("INTEGER", rs.getString("TYPE_NAME"));
        assertEquals(10, rs.getInt("COLUMN_SIZE"));
        assertNull(rs.getString("BUFFER_LENGTH"));
        assertNull(rs.getString("DECIMAL_DIGITS"));
        assertEquals(DatabaseMetaData.bestRowNotPseudo, rs.getInt("PSEUDO_COLUMN"));
        assertFalse(rs.next());
        rs.close();
        
        stmt.close();
        conn.close();
    }
    
    @Test
    public void getDatabaseList() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
        List<DatabaseList> dl = null;
        
        dl = meta.getDatabaseList();
        assertEquals(1, dl.size());
        assertEquals(0, dl.get(0).seq);
        assertEquals("main", dl.get(0).name);
        assertEquals(DATABASE.replaceAll("/", "\\\\"), dl.get(0).file);
        
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                + ")";
        stmt.executeUpdate(sql);

        dl = meta.getDatabaseList();
        assertNotNull(dl);
        assertEquals(2, dl.size());
        
        assertEquals(0, dl.get(0).seq);
        assertEquals("main", dl.get(0).name);
        assertEquals(DATABASE.replaceAll("/", "\\\\"), dl.get(0).file);
        
        assertEquals(1, dl.get(1).seq);
        assertEquals("temp", dl.get(1).name);
        assertNotNull(dl.get(1).file);
        
        conn.close();
    }
    
    @Test
    public void getSchemas() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
        ResultSet rs = null;
        ResultSetMetaData rsMeta = null;
        
        rs = meta.getSchemas();
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("TABLE_SCHEM", rsMeta.getColumnLabel(1));
        assertEquals("TABLE_CATALOG", rsMeta.getColumnLabel(2));
        assertEquals("main", rs.getString(1));
        assertNull(rs.getString(2));
        rs.close();
        
        final Statement stmt = conn.createStatement();
        String sql
                = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                    + ", ID INTEGER NOT NULL"
                    + ", tVALUE TEXT DEFAULT '_blank''_'"
                + ")";
        stmt.executeUpdate(sql);

        rs = meta.getSchemas();
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("TABLE_SCHEM", rsMeta.getColumnLabel(1));
        assertEquals("TABLE_CATALOG", rsMeta.getColumnLabel(2));
        assertEquals("main", rs.getString(1));
        assertNull(rs.getString(2));
        
        assertTrue(rs.next());
        rsMeta = rs.getMetaData();
        assertEquals("TABLE_SCHEM", rsMeta.getColumnLabel(1));
        assertEquals("TABLE_CATALOG", rsMeta.getColumnLabel(2));
        assertEquals("temp", rs.getString(1));
        assertNull(rs.getString(2));
        
        rs.close();
        conn.close();
    }
    
    @Test
    public void getDatabaseProductVersion() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final JdbcDatabaseMetaData meta = (JdbcDatabaseMetaData) conn.getMetaData();
            assertEquals("3.5.9", meta.getDatabaseProductVersion());
            
        } finally {
            conn.close();
        }
    }
}